#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# password_page.py - Copyright (C) 2013 Red Hat, Inc.
# Written by Fabian Deutsch <fabiand@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA  02110-1301, USA.  A copy of the GNU General Public License is
# also available at http://www.gnu.org/copyleft/gpl.html.
from ovirt.node import plugins, ui, exceptions
from ovirt.node.utils.security import password_check

"""
Password page of the installer
"""


class Plugin(plugins.NodePlugin):
    _model = {}

    def name(self):
        return "Console Password"

    def rank(self):
        return 50

    def model(self):
        return self._model or {}

    def validators(self):
        return {}

    def ui_content(self):
        ws = [ui.Header("header[0]",
                        "Require a password for the admin user?"),
              ui.Divider("divider[0]"),
              ui.PasswordEntry("admin.password", "Password:"),
              ui.PasswordEntry("admin.password_confirmation",
                               "Confirm Password:"),
              ui.Divider("divider[1]"),
              ui.Label("password.info", "")
              ]
        self.widgets.add(ws)
        page = ui.Page("password", ws)
        page.buttons = [ui.QuitButton("button.quit", "Quit"),
                        ui.Button("button.back", "Back"),
                        ui.SaveButton("button.next", "Install")]
        return page

    def on_change(self, changes):
        if changes.contains_any(["admin.password",
                                 "admin.password_confirmation"]):
            self._model.update(changes)
            admin_pw, admin_pw_conf = self._model.get("admin.password", ""), \
                self._model.get("admin.password_confirmation", "")

            try:
                min_pw_length = 1
                msg = password_check(admin_pw, admin_pw_conf, min_pw_length)
                self.widgets["admin.password"].valid(True)
                self.widgets["admin.password_confirmation"].valid(True)
                if msg:
                    self.widgets["password.info"].text(msg)
                else:
                    self.widgets["password.info"].text("")
            except ValueError as e:
                self.widgets["password.info"].text("")
                raise exceptions.InvalidData(e.message)

    def on_merge(self, effective_changes):
        changes = self.pending_changes(False)
        if changes.contains_any(["button.back"]):
            self.application.ui.navigate.to_previous_plugin()
        elif changes.contains_any(["admin.password_confirmation",
                                   "button.next"]):
            self._model.update(effective_changes)
            self.application.ui.navigate.to_next_plugin()
