/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.VmBuilder;
import org.ovirt.engine.sdk4.types.Vm;

/**
 * This service manages a collection of all the virtual machines assigned to an affinity group.
 */
public interface AffinityGroupVmsService extends Service {
    /**
     * Add a virtual machine to the affinity group.
     * 
     * For example, to add the virtual machine 000-000 to affinity group 123-456 send a request to:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters/000-000/affinitygroups/123-456/vms
     * ----
     * 
     * With the following body:
     * 
     * [source,xml]
     * ----
     * <vm id="000-000"/>
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest vm(Vm vm);
        AddRequest vm(VmBuilder vm);
    }
    
    /**
     * Add a virtual machine to the affinity group.
     * 
     * For example, to add the virtual machine 000-000 to affinity group 123-456 send a request to:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters/000-000/affinitygroups/123-456/vms
     * ----
     * 
     * With the following body:
     * 
     * [source,xml]
     * ----
     * <vm id="000-000"/>
     * ----
     */
    public interface AddResponse extends Response {
        Vm vm();
    }
    
    /**
     * Add a virtual machine to the affinity group.
     * 
     * For example, to add the virtual machine 000-000 to affinity group 123-456 send a request to:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters/000-000/affinitygroups/123-456/vms
     * ----
     * 
     * With the following body:
     * 
     * [source,xml]
     * ----
     * <vm id="000-000"/>
     * ----
     */
    AddRequest add();
    
    /**
     * List all virtual machines assigned to this affinity group.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of virtual machines to return. If not specified, all the virtual machines are
         * returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of virtual machines to return. If not specified, all the virtual machines are
         * returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of virtual machines to return. If not specified, all the virtual machines are
         * returned.
         */
        ListRequest max(BigInteger max);
    }
    
    /**
     * List all virtual machines assigned to this affinity group.
     */
    public interface ListResponse extends Response {
        List<Vm> vms();
    }
    
    /**
     * List all virtual machines assigned to this affinity group.
     */
    ListRequest list();
    
    /**
     * Access the service that manages the virtual machine assignment to this affinity group.
     */
    AffinityGroupVmService vmService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

