/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.RoleBuilder;
import org.ovirt.engine.sdk4.types.Role;

/**
 * Provides read-only access to the global set of roles
 */
public interface RolesService extends Service {
    /**
     * Create a new role. The role can be administrative or non-administrative and can have different permits.
     * 
     * For example, to add the `MyRole` non-administrative role with permits to login and create virtual machines
     * send a request like this (note that you have to pass permit id):
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/roles
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <role>
     *   <name>MyRole</name>
     *   <description>My custom role to create virtual machines</description>
     *   <administrative>false</administrative>
     *   <permits>
     *     <permit id="1"/>
     *     <permit id="1300"/>
     *   </permits>
     * </group>
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest role(Role role);
        AddRequest role(RoleBuilder role);
    }
    
    /**
     * Create a new role. The role can be administrative or non-administrative and can have different permits.
     * 
     * For example, to add the `MyRole` non-administrative role with permits to login and create virtual machines
     * send a request like this (note that you have to pass permit id):
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/roles
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <role>
     *   <name>MyRole</name>
     *   <description>My custom role to create virtual machines</description>
     *   <administrative>false</administrative>
     *   <permits>
     *     <permit id="1"/>
     *     <permit id="1300"/>
     *   </permits>
     * </group>
     * ----
     */
    public interface AddResponse extends Response {
        Role role();
    }
    
    /**
     * Create a new role. The role can be administrative or non-administrative and can have different permits.
     * 
     * For example, to add the `MyRole` non-administrative role with permits to login and create virtual machines
     * send a request like this (note that you have to pass permit id):
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/roles
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <role>
     *   <name>MyRole</name>
     *   <description>My custom role to create virtual machines</description>
     *   <administrative>false</administrative>
     *   <permits>
     *     <permit id="1"/>
     *     <permit id="1300"/>
     *   </permits>
     * </group>
     * ----
     */
    AddRequest add();
    
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of roles to return. If not specified all the roles are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of roles to return. If not specified all the roles are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of roles to return. If not specified all the roles are returned.
         */
        ListRequest max(BigInteger max);
    }
    
    public interface ListResponse extends Response {
        List<Role> roles();
    }
    
    ListRequest list();
    
    /**
     * Sub-resource locator method, returns individual role resource on which the remainder of the URI is dispatched.
     */
    RoleService roleService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

