//
// Copyright (c) 2017 Joey <majunjiev@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package ovirtsdk

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httputil"
	"net/url"
	"strings"
)

//
// This annotation is intended to specify what oVirt area is the annotated concept related to. Currently the following
// areas are in use, and they are closely related to the oVirt teams, but not necessarily the same:
// - Infrastructure
// - Network
// - SLA
// - Storage
// - Virtualization
// A concept may be associated to more than one area, or to no area.
// The value of this annotation is intended for reporting only, and it doesn't affect at all the generated code or the
// validity of the model
//
type AreaService struct {
	BaseService
}

func NewAreaService(connection *Connection, path string) *AreaService {
	var result AreaService
	result.connection = connection
	result.path = path
	return &result
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AreaService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AreaService) String() string {
	return fmt.Sprintf("AreaService:%s", op.path)
}

//
//
type FollowService struct {
	BaseService
}

func NewFollowService(connection *Connection, path string) *FollowService {
	var result FollowService
	result.connection = connection
	result.path = path
	return &result
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FollowService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *FollowService) String() string {
	return fmt.Sprintf("FollowService:%s", op.path)
}

//
// This service manages a single host label assigned to an affinity group.
//
type AffinityGroupHostLabelService struct {
	BaseService
}

func NewAffinityGroupHostLabelService(connection *Connection, path string) *AffinityGroupHostLabelService {
	var result AffinityGroupHostLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Remove this label from the affinity group.
//
type AffinityGroupHostLabelServiceRemoveRequest struct {
	AffinityGroupHostLabelService *AffinityGroupHostLabelService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
}

func (p *AffinityGroupHostLabelServiceRemoveRequest) Header(key, value string) *AffinityGroupHostLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupHostLabelServiceRemoveRequest) Query(key, value string) *AffinityGroupHostLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupHostLabelServiceRemoveRequest) Async(async bool) *AffinityGroupHostLabelServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AffinityGroupHostLabelServiceRemoveRequest) Send() (*AffinityGroupHostLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupHostLabelService.connection.URL(), p.AffinityGroupHostLabelService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupHostLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupHostLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupHostLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupHostLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupHostLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityGroupHostLabelServiceRemoveResponse), nil
}

func (p *AffinityGroupHostLabelServiceRemoveRequest) MustSend() *AffinityGroupHostLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove this label from the affinity group.
//
type AffinityGroupHostLabelServiceRemoveResponse struct {
}

//
// Remove this label from the affinity group.
//
func (p *AffinityGroupHostLabelService) Remove() *AffinityGroupHostLabelServiceRemoveRequest {
	return &AffinityGroupHostLabelServiceRemoveRequest{AffinityGroupHostLabelService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupHostLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityGroupHostLabelService) String() string {
	return fmt.Sprintf("AffinityGroupHostLabelService:%s", op.path)
}

//
// This service manages a collection of all host labels assigned to an affinity group.
//
type AffinityGroupHostLabelsService struct {
	BaseService
}

func NewAffinityGroupHostLabelsService(connection *Connection, path string) *AffinityGroupHostLabelsService {
	var result AffinityGroupHostLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a host label to the affinity group.
// For example, to add the label `789` to the affinity group `456` of cluster `123`,
// send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/hostlabels
// ....
// With the following body:
// [source,xml]
// ----
// <affinity_label id="789"/>
// ----
//
type AffinityGroupHostLabelsServiceAddRequest struct {
	AffinityGroupHostLabelsService *AffinityGroupHostLabelsService
	header                         map[string]string
	query                          map[string]string
	label                          *AffinityLabel
}

func (p *AffinityGroupHostLabelsServiceAddRequest) Header(key, value string) *AffinityGroupHostLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupHostLabelsServiceAddRequest) Query(key, value string) *AffinityGroupHostLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupHostLabelsServiceAddRequest) Label(label *AffinityLabel) *AffinityGroupHostLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *AffinityGroupHostLabelsServiceAddRequest) Send() (*AffinityGroupHostLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupHostLabelsService.connection.URL(), p.AffinityGroupHostLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupHostLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupHostLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupHostLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupHostLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupHostLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupHostLabelsServiceAddResponse{label: result}, nil
}

func (p *AffinityGroupHostLabelsServiceAddRequest) MustSend() *AffinityGroupHostLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a host label to the affinity group.
// For example, to add the label `789` to the affinity group `456` of cluster `123`,
// send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/hostlabels
// ....
// With the following body:
// [source,xml]
// ----
// <affinity_label id="789"/>
// ----
//
type AffinityGroupHostLabelsServiceAddResponse struct {
	label *AffinityLabel
}

func (p *AffinityGroupHostLabelsServiceAddResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityGroupHostLabelsServiceAddResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Adds a host label to the affinity group.
// For example, to add the label `789` to the affinity group `456` of cluster `123`,
// send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/hostlabels
// ....
// With the following body:
// [source,xml]
// ----
// <affinity_label id="789"/>
// ----
//
func (p *AffinityGroupHostLabelsService) Add() *AffinityGroupHostLabelsServiceAddRequest {
	return &AffinityGroupHostLabelsServiceAddRequest{AffinityGroupHostLabelsService: p}
}

//
// List all host labels assigned to this affinity group.
// The order of the returned labels isn't guaranteed.
//
type AffinityGroupHostLabelsServiceListRequest struct {
	AffinityGroupHostLabelsService *AffinityGroupHostLabelsService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
	max                            *int64
}

func (p *AffinityGroupHostLabelsServiceListRequest) Header(key, value string) *AffinityGroupHostLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupHostLabelsServiceListRequest) Query(key, value string) *AffinityGroupHostLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupHostLabelsServiceListRequest) Follow(follow string) *AffinityGroupHostLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityGroupHostLabelsServiceListRequest) Max(max int64) *AffinityGroupHostLabelsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityGroupHostLabelsServiceListRequest) Send() (*AffinityGroupHostLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupHostLabelsService.connection.URL(), p.AffinityGroupHostLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupHostLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupHostLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupHostLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupHostLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupHostLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityGroupHostLabelsServiceListResponse{labels: result}, nil
}

func (p *AffinityGroupHostLabelsServiceListRequest) MustSend() *AffinityGroupHostLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all host labels assigned to this affinity group.
// The order of the returned labels isn't guaranteed.
//
type AffinityGroupHostLabelsServiceListResponse struct {
	labels *AffinityLabelSlice
}

func (p *AffinityGroupHostLabelsServiceListResponse) Labels() (*AffinityLabelSlice, bool) {
	if p.labels != nil {
		return p.labels, true
	}
	return nil, false
}

func (p *AffinityGroupHostLabelsServiceListResponse) MustLabels() *AffinityLabelSlice {
	if p.labels == nil {
		panic("labels in response does not exist")
	}
	return p.labels
}

//
// List all host labels assigned to this affinity group.
// The order of the returned labels isn't guaranteed.
//
func (p *AffinityGroupHostLabelsService) List() *AffinityGroupHostLabelsServiceListRequest {
	return &AffinityGroupHostLabelsServiceListRequest{AffinityGroupHostLabelsService: p}
}

//
// Access the service that manages the host label assignment to this affinity group.
//
func (op *AffinityGroupHostLabelsService) LabelService(id string) *AffinityGroupHostLabelService {
	return NewAffinityGroupHostLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupHostLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *AffinityGroupHostLabelsService) String() string {
	return fmt.Sprintf("AffinityGroupHostLabelsService:%s", op.path)
}

//
// This service manages a single host to affinity group assignment.
//
type AffinityGroupHostService struct {
	BaseService
}

func NewAffinityGroupHostService(connection *Connection, path string) *AffinityGroupHostService {
	var result AffinityGroupHostService
	result.connection = connection
	result.path = path
	return &result
}

//
// Remove host from the affinity group.
//
type AffinityGroupHostServiceRemoveRequest struct {
	AffinityGroupHostService *AffinityGroupHostService
	header                   map[string]string
	query                    map[string]string
	async                    *bool
}

func (p *AffinityGroupHostServiceRemoveRequest) Header(key, value string) *AffinityGroupHostServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupHostServiceRemoveRequest) Query(key, value string) *AffinityGroupHostServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupHostServiceRemoveRequest) Async(async bool) *AffinityGroupHostServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AffinityGroupHostServiceRemoveRequest) Send() (*AffinityGroupHostServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupHostService.connection.URL(), p.AffinityGroupHostService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityGroupHostServiceRemoveResponse), nil
}

func (p *AffinityGroupHostServiceRemoveRequest) MustSend() *AffinityGroupHostServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove host from the affinity group.
//
type AffinityGroupHostServiceRemoveResponse struct {
}

//
// Remove host from the affinity group.
//
func (p *AffinityGroupHostService) Remove() *AffinityGroupHostServiceRemoveRequest {
	return &AffinityGroupHostServiceRemoveRequest{AffinityGroupHostService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupHostService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityGroupHostService) String() string {
	return fmt.Sprintf("AffinityGroupHostService:%s", op.path)
}

//
// This service manages a collection of all hosts assigned to an affinity group.
//
type AffinityGroupHostsService struct {
	BaseService
}

func NewAffinityGroupHostsService(connection *Connection, path string) *AffinityGroupHostsService {
	var result AffinityGroupHostsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a host to the affinity group.
// For example, to add the host `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/hosts
// ....
// With the following body:
// [source,xml]
// ----
// <host id="789"/>
// ----
//
type AffinityGroupHostsServiceAddRequest struct {
	AffinityGroupHostsService *AffinityGroupHostsService
	header                    map[string]string
	query                     map[string]string
	host                      *Host
}

func (p *AffinityGroupHostsServiceAddRequest) Header(key, value string) *AffinityGroupHostsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupHostsServiceAddRequest) Query(key, value string) *AffinityGroupHostsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupHostsServiceAddRequest) Host(host *Host) *AffinityGroupHostsServiceAddRequest {
	p.host = host
	return p
}

func (p *AffinityGroupHostsServiceAddRequest) Send() (*AffinityGroupHostsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupHostsService.connection.URL(), p.AffinityGroupHostsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLHostWriteOne(writer, p.host, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupHostsServiceAddResponse{host: result}, nil
}

func (p *AffinityGroupHostsServiceAddRequest) MustSend() *AffinityGroupHostsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a host to the affinity group.
// For example, to add the host `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/hosts
// ....
// With the following body:
// [source,xml]
// ----
// <host id="789"/>
// ----
//
type AffinityGroupHostsServiceAddResponse struct {
	host *Host
}

func (p *AffinityGroupHostsServiceAddResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *AffinityGroupHostsServiceAddResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Adds a host to the affinity group.
// For example, to add the host `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/hosts
// ....
// With the following body:
// [source,xml]
// ----
// <host id="789"/>
// ----
//
func (p *AffinityGroupHostsService) Add() *AffinityGroupHostsServiceAddRequest {
	return &AffinityGroupHostsServiceAddRequest{AffinityGroupHostsService: p}
}

//
// List all hosts assigned to this affinity group.
// The order of the returned hosts isn't guaranteed.
//
type AffinityGroupHostsServiceListRequest struct {
	AffinityGroupHostsService *AffinityGroupHostsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *AffinityGroupHostsServiceListRequest) Header(key, value string) *AffinityGroupHostsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupHostsServiceListRequest) Query(key, value string) *AffinityGroupHostsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupHostsServiceListRequest) Follow(follow string) *AffinityGroupHostsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityGroupHostsServiceListRequest) Max(max int64) *AffinityGroupHostsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityGroupHostsServiceListRequest) Send() (*AffinityGroupHostsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupHostsService.connection.URL(), p.AffinityGroupHostsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityGroupHostsServiceListResponse{hosts: result}, nil
}

func (p *AffinityGroupHostsServiceListRequest) MustSend() *AffinityGroupHostsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all hosts assigned to this affinity group.
// The order of the returned hosts isn't guaranteed.
//
type AffinityGroupHostsServiceListResponse struct {
	hosts *HostSlice
}

func (p *AffinityGroupHostsServiceListResponse) Hosts() (*HostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *AffinityGroupHostsServiceListResponse) MustHosts() *HostSlice {
	if p.hosts == nil {
		panic("hosts in response does not exist")
	}
	return p.hosts
}

//
// List all hosts assigned to this affinity group.
// The order of the returned hosts isn't guaranteed.
//
func (p *AffinityGroupHostsService) List() *AffinityGroupHostsServiceListRequest {
	return &AffinityGroupHostsServiceListRequest{AffinityGroupHostsService: p}
}

//
// Access the service that manages the host assignment to this affinity group.
//
func (op *AffinityGroupHostsService) HostService(id string) *AffinityGroupHostService {
	return NewAffinityGroupHostService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupHostsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HostService(path), nil
	}
	return op.HostService(path[:index]).Service(path[index+1:])
}

func (op *AffinityGroupHostsService) String() string {
	return fmt.Sprintf("AffinityGroupHostsService:%s", op.path)
}

//
// This service manages a single affinity group.
//
type AffinityGroupService struct {
	BaseService
}

func NewAffinityGroupService(connection *Connection, path string) *AffinityGroupService {
	var result AffinityGroupService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieve the affinity group details.
// [source,xml]
// ----
// <affinity_group id="00000000-0000-0000-0000-000000000000">
//   <name>AF_GROUP_001</name>
//   <cluster id="00000000-0000-0000-0000-000000000000"/>
//   <positive>true</positive>
//   <enforcing>true</enforcing>
// </affinity_group>
// ----
//
type AffinityGroupServiceGetRequest struct {
	AffinityGroupService *AffinityGroupService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *AffinityGroupServiceGetRequest) Header(key, value string) *AffinityGroupServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupServiceGetRequest) Query(key, value string) *AffinityGroupServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupServiceGetRequest) Follow(follow string) *AffinityGroupServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AffinityGroupServiceGetRequest) Send() (*AffinityGroupServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupService.connection.URL(), p.AffinityGroupService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupServiceGetResponse{group: result}, nil
}

func (p *AffinityGroupServiceGetRequest) MustSend() *AffinityGroupServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve the affinity group details.
// [source,xml]
// ----
// <affinity_group id="00000000-0000-0000-0000-000000000000">
//   <name>AF_GROUP_001</name>
//   <cluster id="00000000-0000-0000-0000-000000000000"/>
//   <positive>true</positive>
//   <enforcing>true</enforcing>
// </affinity_group>
// ----
//
type AffinityGroupServiceGetResponse struct {
	group *AffinityGroup
}

func (p *AffinityGroupServiceGetResponse) Group() (*AffinityGroup, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

func (p *AffinityGroupServiceGetResponse) MustGroup() *AffinityGroup {
	if p.group == nil {
		panic("group in response does not exist")
	}
	return p.group
}

//
// Retrieve the affinity group details.
// [source,xml]
// ----
// <affinity_group id="00000000-0000-0000-0000-000000000000">
//   <name>AF_GROUP_001</name>
//   <cluster id="00000000-0000-0000-0000-000000000000"/>
//   <positive>true</positive>
//   <enforcing>true</enforcing>
// </affinity_group>
// ----
//
func (p *AffinityGroupService) Get() *AffinityGroupServiceGetRequest {
	return &AffinityGroupServiceGetRequest{AffinityGroupService: p}
}

//
// Remove the affinity group.
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/000-000/affinitygroups/123-456
// ----
//
type AffinityGroupServiceRemoveRequest struct {
	AffinityGroupService *AffinityGroupService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *AffinityGroupServiceRemoveRequest) Header(key, value string) *AffinityGroupServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupServiceRemoveRequest) Query(key, value string) *AffinityGroupServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupServiceRemoveRequest) Async(async bool) *AffinityGroupServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AffinityGroupServiceRemoveRequest) Send() (*AffinityGroupServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupService.connection.URL(), p.AffinityGroupService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityGroupServiceRemoveResponse), nil
}

func (p *AffinityGroupServiceRemoveRequest) MustSend() *AffinityGroupServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove the affinity group.
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/000-000/affinitygroups/123-456
// ----
//
type AffinityGroupServiceRemoveResponse struct {
}

//
// Remove the affinity group.
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/000-000/affinitygroups/123-456
// ----
//
func (p *AffinityGroupService) Remove() *AffinityGroupServiceRemoveRequest {
	return &AffinityGroupServiceRemoveRequest{AffinityGroupService: p}
}

//
// Update the affinity group.
//
type AffinityGroupServiceUpdateRequest struct {
	AffinityGroupService *AffinityGroupService
	header               map[string]string
	query                map[string]string
	async                *bool
	group                *AffinityGroup
}

func (p *AffinityGroupServiceUpdateRequest) Header(key, value string) *AffinityGroupServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupServiceUpdateRequest) Query(key, value string) *AffinityGroupServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupServiceUpdateRequest) Async(async bool) *AffinityGroupServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *AffinityGroupServiceUpdateRequest) Group(group *AffinityGroup) *AffinityGroupServiceUpdateRequest {
	p.group = group
	return p
}

func (p *AffinityGroupServiceUpdateRequest) Send() (*AffinityGroupServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupService.connection.URL(), p.AffinityGroupService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityGroupWriteOne(writer, p.group, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupServiceUpdateResponse{group: result}, nil
}

func (p *AffinityGroupServiceUpdateRequest) MustSend() *AffinityGroupServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the affinity group.
//
type AffinityGroupServiceUpdateResponse struct {
	group *AffinityGroup
}

func (p *AffinityGroupServiceUpdateResponse) Group() (*AffinityGroup, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

func (p *AffinityGroupServiceUpdateResponse) MustGroup() *AffinityGroup {
	if p.group == nil {
		panic("group in response does not exist")
	}
	return p.group
}

//
// Update the affinity group.
//
func (p *AffinityGroupService) Update() *AffinityGroupServiceUpdateRequest {
	return &AffinityGroupServiceUpdateRequest{AffinityGroupService: p}
}

//
// Returns a reference to the service that manages the
// list of all host labels attached to this affinity
// group.
//
func (op *AffinityGroupService) HostLabelsService() *AffinityGroupHostLabelsService {
	return NewAffinityGroupHostLabelsService(op.connection, fmt.Sprintf("%s/hostlabels", op.path))
}

//
// Returns a reference to the service that manages the
// list of all hosts attached to this affinity
// group.
//
func (op *AffinityGroupService) HostsService() *AffinityGroupHostsService {
	return NewAffinityGroupHostsService(op.connection, fmt.Sprintf("%s/hosts", op.path))
}

//
// Returns a reference to the service that manages the
// list of all virtual machine labels attached to this affinity
// group.
//
func (op *AffinityGroupService) VmLabelsService() *AffinityGroupVmLabelsService {
	return NewAffinityGroupVmLabelsService(op.connection, fmt.Sprintf("%s/vmlabels", op.path))
}

//
// Returns a reference to the service that manages the
// list of all virtual machines attached to this affinity
// group.
//
func (op *AffinityGroupService) VmsService() *AffinityGroupVmsService {
	return NewAffinityGroupVmsService(op.connection, fmt.Sprintf("%s/vms", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "hostlabels" {
		return op.HostLabelsService(), nil
	}
	if strings.HasPrefix(path, "hostlabels/") {
		return op.HostLabelsService().Service(path[11:])
	}
	if path == "hosts" {
		return op.HostsService(), nil
	}
	if strings.HasPrefix(path, "hosts/") {
		return op.HostsService().Service(path[6:])
	}
	if path == "vmlabels" {
		return op.VmLabelsService(), nil
	}
	if strings.HasPrefix(path, "vmlabels/") {
		return op.VmLabelsService().Service(path[9:])
	}
	if path == "vms" {
		return op.VmsService(), nil
	}
	if strings.HasPrefix(path, "vms/") {
		return op.VmsService().Service(path[4:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityGroupService) String() string {
	return fmt.Sprintf("AffinityGroupService:%s", op.path)
}

//
// This service manages a single virtual machine label assigned to an affinity group.
//
type AffinityGroupVmLabelService struct {
	BaseService
}

func NewAffinityGroupVmLabelService(connection *Connection, path string) *AffinityGroupVmLabelService {
	var result AffinityGroupVmLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Remove this label from the affinity group.
//
type AffinityGroupVmLabelServiceRemoveRequest struct {
	AffinityGroupVmLabelService *AffinityGroupVmLabelService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *AffinityGroupVmLabelServiceRemoveRequest) Header(key, value string) *AffinityGroupVmLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmLabelServiceRemoveRequest) Query(key, value string) *AffinityGroupVmLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmLabelServiceRemoveRequest) Async(async bool) *AffinityGroupVmLabelServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AffinityGroupVmLabelServiceRemoveRequest) Send() (*AffinityGroupVmLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmLabelService.connection.URL(), p.AffinityGroupVmLabelService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityGroupVmLabelServiceRemoveResponse), nil
}

func (p *AffinityGroupVmLabelServiceRemoveRequest) MustSend() *AffinityGroupVmLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove this label from the affinity group.
//
type AffinityGroupVmLabelServiceRemoveResponse struct {
}

//
// Remove this label from the affinity group.
//
func (p *AffinityGroupVmLabelService) Remove() *AffinityGroupVmLabelServiceRemoveRequest {
	return &AffinityGroupVmLabelServiceRemoveRequest{AffinityGroupVmLabelService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupVmLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityGroupVmLabelService) String() string {
	return fmt.Sprintf("AffinityGroupVmLabelService:%s", op.path)
}

//
// This service manages a collection of all virtual machine labels assigned to an affinity group.
//
type AffinityGroupVmLabelsService struct {
	BaseService
}

func NewAffinityGroupVmLabelsService(connection *Connection, path string) *AffinityGroupVmLabelsService {
	var result AffinityGroupVmLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a virtual machine label to the affinity group.
// For example, to add the label `789` to the affinity group `456` of cluster `123`,
// send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vmlabels
// ....
// With the following body:
// [source,xml]
// ----
// <affinity_label id="789"/>
// ----
//
type AffinityGroupVmLabelsServiceAddRequest struct {
	AffinityGroupVmLabelsService *AffinityGroupVmLabelsService
	header                       map[string]string
	query                        map[string]string
	label                        *AffinityLabel
}

func (p *AffinityGroupVmLabelsServiceAddRequest) Header(key, value string) *AffinityGroupVmLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmLabelsServiceAddRequest) Query(key, value string) *AffinityGroupVmLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmLabelsServiceAddRequest) Label(label *AffinityLabel) *AffinityGroupVmLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *AffinityGroupVmLabelsServiceAddRequest) Send() (*AffinityGroupVmLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmLabelsService.connection.URL(), p.AffinityGroupVmLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupVmLabelsServiceAddResponse{label: result}, nil
}

func (p *AffinityGroupVmLabelsServiceAddRequest) MustSend() *AffinityGroupVmLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a virtual machine label to the affinity group.
// For example, to add the label `789` to the affinity group `456` of cluster `123`,
// send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vmlabels
// ....
// With the following body:
// [source,xml]
// ----
// <affinity_label id="789"/>
// ----
//
type AffinityGroupVmLabelsServiceAddResponse struct {
	label *AffinityLabel
}

func (p *AffinityGroupVmLabelsServiceAddResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityGroupVmLabelsServiceAddResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Adds a virtual machine label to the affinity group.
// For example, to add the label `789` to the affinity group `456` of cluster `123`,
// send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vmlabels
// ....
// With the following body:
// [source,xml]
// ----
// <affinity_label id="789"/>
// ----
//
func (p *AffinityGroupVmLabelsService) Add() *AffinityGroupVmLabelsServiceAddRequest {
	return &AffinityGroupVmLabelsServiceAddRequest{AffinityGroupVmLabelsService: p}
}

//
// List all virtual machine labels assigned to this affinity group.
// The order of the returned labels isn't guaranteed.
//
type AffinityGroupVmLabelsServiceListRequest struct {
	AffinityGroupVmLabelsService *AffinityGroupVmLabelsService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
	max                          *int64
}

func (p *AffinityGroupVmLabelsServiceListRequest) Header(key, value string) *AffinityGroupVmLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmLabelsServiceListRequest) Query(key, value string) *AffinityGroupVmLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmLabelsServiceListRequest) Follow(follow string) *AffinityGroupVmLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityGroupVmLabelsServiceListRequest) Max(max int64) *AffinityGroupVmLabelsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityGroupVmLabelsServiceListRequest) Send() (*AffinityGroupVmLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmLabelsService.connection.URL(), p.AffinityGroupVmLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityGroupVmLabelsServiceListResponse{labels: result}, nil
}

func (p *AffinityGroupVmLabelsServiceListRequest) MustSend() *AffinityGroupVmLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all virtual machine labels assigned to this affinity group.
// The order of the returned labels isn't guaranteed.
//
type AffinityGroupVmLabelsServiceListResponse struct {
	labels *AffinityLabelSlice
}

func (p *AffinityGroupVmLabelsServiceListResponse) Labels() (*AffinityLabelSlice, bool) {
	if p.labels != nil {
		return p.labels, true
	}
	return nil, false
}

func (p *AffinityGroupVmLabelsServiceListResponse) MustLabels() *AffinityLabelSlice {
	if p.labels == nil {
		panic("labels in response does not exist")
	}
	return p.labels
}

//
// List all virtual machine labels assigned to this affinity group.
// The order of the returned labels isn't guaranteed.
//
func (p *AffinityGroupVmLabelsService) List() *AffinityGroupVmLabelsServiceListRequest {
	return &AffinityGroupVmLabelsServiceListRequest{AffinityGroupVmLabelsService: p}
}

//
// Access the service that manages the virtual machine label assignment to this affinity group.
//
func (op *AffinityGroupVmLabelsService) LabelService(id string) *AffinityGroupVmLabelService {
	return NewAffinityGroupVmLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupVmLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *AffinityGroupVmLabelsService) String() string {
	return fmt.Sprintf("AffinityGroupVmLabelsService:%s", op.path)
}

//
// This service manages a single virtual machine to affinity group assignment.
//
type AffinityGroupVmService struct {
	BaseService
}

func NewAffinityGroupVmService(connection *Connection, path string) *AffinityGroupVmService {
	var result AffinityGroupVmService
	result.connection = connection
	result.path = path
	return &result
}

//
// Remove this virtual machine from the affinity group.
//
type AffinityGroupVmServiceRemoveRequest struct {
	AffinityGroupVmService *AffinityGroupVmService
	header                 map[string]string
	query                  map[string]string
	async                  *bool
}

func (p *AffinityGroupVmServiceRemoveRequest) Header(key, value string) *AffinityGroupVmServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmServiceRemoveRequest) Query(key, value string) *AffinityGroupVmServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmServiceRemoveRequest) Async(async bool) *AffinityGroupVmServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AffinityGroupVmServiceRemoveRequest) Send() (*AffinityGroupVmServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmService.connection.URL(), p.AffinityGroupVmService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityGroupVmServiceRemoveResponse), nil
}

func (p *AffinityGroupVmServiceRemoveRequest) MustSend() *AffinityGroupVmServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove this virtual machine from the affinity group.
//
type AffinityGroupVmServiceRemoveResponse struct {
}

//
// Remove this virtual machine from the affinity group.
//
func (p *AffinityGroupVmService) Remove() *AffinityGroupVmServiceRemoveRequest {
	return &AffinityGroupVmServiceRemoveRequest{AffinityGroupVmService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupVmService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityGroupVmService) String() string {
	return fmt.Sprintf("AffinityGroupVmService:%s", op.path)
}

//
// This service manages a collection of all the virtual machines assigned to an affinity group.
//
type AffinityGroupVmsService struct {
	BaseService
}

func NewAffinityGroupVmsService(connection *Connection, path string) *AffinityGroupVmsService {
	var result AffinityGroupVmsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a virtual machine to the affinity group.
// For example, to add the virtual machine `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vms
// ....
// With the following body:
// [source,xml]
// ----
// <vm id="789"/>
// ----
//
type AffinityGroupVmsServiceAddRequest struct {
	AffinityGroupVmsService *AffinityGroupVmsService
	header                  map[string]string
	query                   map[string]string
	vm                      *Vm
}

func (p *AffinityGroupVmsServiceAddRequest) Header(key, value string) *AffinityGroupVmsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmsServiceAddRequest) Query(key, value string) *AffinityGroupVmsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmsServiceAddRequest) Vm(vm *Vm) *AffinityGroupVmsServiceAddRequest {
	p.vm = vm
	return p
}

func (p *AffinityGroupVmsServiceAddRequest) Send() (*AffinityGroupVmsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmsService.connection.URL(), p.AffinityGroupVmsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVmWriteOne(writer, p.vm, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupVmsServiceAddResponse{vm: result}, nil
}

func (p *AffinityGroupVmsServiceAddRequest) MustSend() *AffinityGroupVmsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a virtual machine to the affinity group.
// For example, to add the virtual machine `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vms
// ....
// With the following body:
// [source,xml]
// ----
// <vm id="789"/>
// ----
//
type AffinityGroupVmsServiceAddResponse struct {
	vm *Vm
}

func (p *AffinityGroupVmsServiceAddResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *AffinityGroupVmsServiceAddResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// Adds a virtual machine to the affinity group.
// For example, to add the virtual machine `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vms
// ....
// With the following body:
// [source,xml]
// ----
// <vm id="789"/>
// ----
//
func (p *AffinityGroupVmsService) Add() *AffinityGroupVmsServiceAddRequest {
	return &AffinityGroupVmsServiceAddRequest{AffinityGroupVmsService: p}
}

//
// List all virtual machines assigned to this affinity group.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityGroupVmsServiceListRequest struct {
	AffinityGroupVmsService *AffinityGroupVmsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
	max                     *int64
}

func (p *AffinityGroupVmsServiceListRequest) Header(key, value string) *AffinityGroupVmsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Query(key, value string) *AffinityGroupVmsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Follow(follow string) *AffinityGroupVmsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Max(max int64) *AffinityGroupVmsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Send() (*AffinityGroupVmsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmsService.connection.URL(), p.AffinityGroupVmsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityGroupVmsServiceListResponse{vms: result}, nil
}

func (p *AffinityGroupVmsServiceListRequest) MustSend() *AffinityGroupVmsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all virtual machines assigned to this affinity group.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityGroupVmsServiceListResponse struct {
	vms *VmSlice
}

func (p *AffinityGroupVmsServiceListResponse) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *AffinityGroupVmsServiceListResponse) MustVms() *VmSlice {
	if p.vms == nil {
		panic("vms in response does not exist")
	}
	return p.vms
}

//
// List all virtual machines assigned to this affinity group.
// The order of the returned virtual machines isn't guaranteed.
//
func (p *AffinityGroupVmsService) List() *AffinityGroupVmsServiceListRequest {
	return &AffinityGroupVmsServiceListRequest{AffinityGroupVmsService: p}
}

//
// Access the service that manages the virtual machine assignment to this affinity group.
//
func (op *AffinityGroupVmsService) VmService(id string) *AffinityGroupVmService {
	return NewAffinityGroupVmService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupVmsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VmService(path), nil
	}
	return op.VmService(path[:index]).Service(path[index+1:])
}

func (op *AffinityGroupVmsService) String() string {
	return fmt.Sprintf("AffinityGroupVmsService:%s", op.path)
}

//
// The affinity groups service manages virtual machine relationships and dependencies.
//
type AffinityGroupsService struct {
	BaseService
}

func NewAffinityGroupsService(connection *Connection, path string) *AffinityGroupsService {
	var result AffinityGroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Create a new affinity group.
// Post a request like in the example below to create a new affinity group:
// [source]
// ----
// POST /ovirt-engine/api/clusters/000-000/affinitygroups
// ----
// And use the following example in its body:
// [source,xml]
// ----
// <affinity_group>
//   <name>AF_GROUP_001</name>
//   <hosts_rule>
//     <enforcing>true</enforcing>
//     <positive>true</positive>
//   </hosts_rule>
//   <vms_rule>
//     <enabled>false</enabled>
//   </vms_rule>
// </affinity_group>
// ----
//
type AffinityGroupsServiceAddRequest struct {
	AffinityGroupsService *AffinityGroupsService
	header                map[string]string
	query                 map[string]string
	group                 *AffinityGroup
}

func (p *AffinityGroupsServiceAddRequest) Header(key, value string) *AffinityGroupsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupsServiceAddRequest) Query(key, value string) *AffinityGroupsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupsServiceAddRequest) Group(group *AffinityGroup) *AffinityGroupsServiceAddRequest {
	p.group = group
	return p
}

func (p *AffinityGroupsServiceAddRequest) Send() (*AffinityGroupsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupsService.connection.URL(), p.AffinityGroupsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityGroupWriteOne(writer, p.group, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupsServiceAddResponse{group: result}, nil
}

func (p *AffinityGroupsServiceAddRequest) MustSend() *AffinityGroupsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Create a new affinity group.
// Post a request like in the example below to create a new affinity group:
// [source]
// ----
// POST /ovirt-engine/api/clusters/000-000/affinitygroups
// ----
// And use the following example in its body:
// [source,xml]
// ----
// <affinity_group>
//   <name>AF_GROUP_001</name>
//   <hosts_rule>
//     <enforcing>true</enforcing>
//     <positive>true</positive>
//   </hosts_rule>
//   <vms_rule>
//     <enabled>false</enabled>
//   </vms_rule>
// </affinity_group>
// ----
//
type AffinityGroupsServiceAddResponse struct {
	group *AffinityGroup
}

func (p *AffinityGroupsServiceAddResponse) Group() (*AffinityGroup, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

func (p *AffinityGroupsServiceAddResponse) MustGroup() *AffinityGroup {
	if p.group == nil {
		panic("group in response does not exist")
	}
	return p.group
}

//
// Create a new affinity group.
// Post a request like in the example below to create a new affinity group:
// [source]
// ----
// POST /ovirt-engine/api/clusters/000-000/affinitygroups
// ----
// And use the following example in its body:
// [source,xml]
// ----
// <affinity_group>
//   <name>AF_GROUP_001</name>
//   <hosts_rule>
//     <enforcing>true</enforcing>
//     <positive>true</positive>
//   </hosts_rule>
//   <vms_rule>
//     <enabled>false</enabled>
//   </vms_rule>
// </affinity_group>
// ----
//
func (p *AffinityGroupsService) Add() *AffinityGroupsServiceAddRequest {
	return &AffinityGroupsServiceAddRequest{AffinityGroupsService: p}
}

//
// List existing affinity groups.
// The order of the affinity groups results isn't guaranteed.
//
type AffinityGroupsServiceListRequest struct {
	AffinityGroupsService *AffinityGroupsService
	header                map[string]string
	query                 map[string]string
	follow                *string
	max                   *int64
}

func (p *AffinityGroupsServiceListRequest) Header(key, value string) *AffinityGroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupsServiceListRequest) Query(key, value string) *AffinityGroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupsServiceListRequest) Follow(follow string) *AffinityGroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityGroupsServiceListRequest) Max(max int64) *AffinityGroupsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityGroupsServiceListRequest) Send() (*AffinityGroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupsService.connection.URL(), p.AffinityGroupsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityGroupsServiceListResponse{groups: result}, nil
}

func (p *AffinityGroupsServiceListRequest) MustSend() *AffinityGroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List existing affinity groups.
// The order of the affinity groups results isn't guaranteed.
//
type AffinityGroupsServiceListResponse struct {
	groups *AffinityGroupSlice
}

func (p *AffinityGroupsServiceListResponse) Groups() (*AffinityGroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *AffinityGroupsServiceListResponse) MustGroups() *AffinityGroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// List existing affinity groups.
// The order of the affinity groups results isn't guaranteed.
//
func (p *AffinityGroupsService) List() *AffinityGroupsServiceListRequest {
	return &AffinityGroupsServiceListRequest{AffinityGroupsService: p}
}

//
// Access the affinity group service that manages the affinity group specified by an ID.
//
func (op *AffinityGroupsService) GroupService(id string) *AffinityGroupService {
	return NewAffinityGroupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.GroupService(path), nil
	}
	return op.GroupService(path[:index]).Service(path[index+1:])
}

func (op *AffinityGroupsService) String() string {
	return fmt.Sprintf("AffinityGroupsService:%s", op.path)
}

//
// This service represents a host that has a specific
// label when accessed through the affinitylabels/hosts
// subcollection.
//
type AffinityLabelHostService struct {
	BaseService
}

func NewAffinityLabelHostService(connection *Connection, path string) *AffinityLabelHostService {
	var result AffinityLabelHostService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves details about a host that has this label assigned.
//
type AffinityLabelHostServiceGetRequest struct {
	AffinityLabelHostService *AffinityLabelHostService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *AffinityLabelHostServiceGetRequest) Header(key, value string) *AffinityLabelHostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelHostServiceGetRequest) Query(key, value string) *AffinityLabelHostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelHostServiceGetRequest) Follow(follow string) *AffinityLabelHostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelHostServiceGetRequest) Send() (*AffinityLabelHostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelHostService.connection.URL(), p.AffinityLabelHostService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelHostServiceGetResponse{host: result}, nil
}

func (p *AffinityLabelHostServiceGetRequest) MustSend() *AffinityLabelHostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves details about a host that has this label assigned.
//
type AffinityLabelHostServiceGetResponse struct {
	host *Host
}

func (p *AffinityLabelHostServiceGetResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *AffinityLabelHostServiceGetResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Retrieves details about a host that has this label assigned.
//
func (p *AffinityLabelHostService) Get() *AffinityLabelHostServiceGetRequest {
	return &AffinityLabelHostServiceGetRequest{AffinityLabelHostService: p}
}

//
// Remove a label from a host.
//
type AffinityLabelHostServiceRemoveRequest struct {
	AffinityLabelHostService *AffinityLabelHostService
	header                   map[string]string
	query                    map[string]string
}

func (p *AffinityLabelHostServiceRemoveRequest) Header(key, value string) *AffinityLabelHostServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelHostServiceRemoveRequest) Query(key, value string) *AffinityLabelHostServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelHostServiceRemoveRequest) Send() (*AffinityLabelHostServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelHostService.connection.URL(), p.AffinityLabelHostService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityLabelHostServiceRemoveResponse), nil
}

func (p *AffinityLabelHostServiceRemoveRequest) MustSend() *AffinityLabelHostServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove a label from a host.
//
type AffinityLabelHostServiceRemoveResponse struct {
}

//
// Remove a label from a host.
//
func (p *AffinityLabelHostService) Remove() *AffinityLabelHostServiceRemoveRequest {
	return &AffinityLabelHostServiceRemoveRequest{AffinityLabelHostService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelHostService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityLabelHostService) String() string {
	return fmt.Sprintf("AffinityLabelHostService:%s", op.path)
}

//
// This service represents list of hosts that have a specific
// label when accessed through the affinitylabels/hosts
// subcollection.
//
type AffinityLabelHostsService struct {
	BaseService
}

func NewAffinityLabelHostsService(connection *Connection, path string) *AffinityLabelHostsService {
	var result AffinityLabelHostsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a label to a host.
//
type AffinityLabelHostsServiceAddRequest struct {
	AffinityLabelHostsService *AffinityLabelHostsService
	header                    map[string]string
	query                     map[string]string
	host                      *Host
}

func (p *AffinityLabelHostsServiceAddRequest) Header(key, value string) *AffinityLabelHostsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelHostsServiceAddRequest) Query(key, value string) *AffinityLabelHostsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelHostsServiceAddRequest) Host(host *Host) *AffinityLabelHostsServiceAddRequest {
	p.host = host
	return p
}

func (p *AffinityLabelHostsServiceAddRequest) Send() (*AffinityLabelHostsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelHostsService.connection.URL(), p.AffinityLabelHostsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLHostWriteOne(writer, p.host, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelHostsServiceAddResponse{host: result}, nil
}

func (p *AffinityLabelHostsServiceAddRequest) MustSend() *AffinityLabelHostsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a label to a host.
//
type AffinityLabelHostsServiceAddResponse struct {
	host *Host
}

func (p *AffinityLabelHostsServiceAddResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *AffinityLabelHostsServiceAddResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Add a label to a host.
//
func (p *AffinityLabelHostsService) Add() *AffinityLabelHostsServiceAddRequest {
	return &AffinityLabelHostsServiceAddRequest{AffinityLabelHostsService: p}
}

//
// List all hosts with the label.
// The order of the returned hosts isn't guaranteed.
//
type AffinityLabelHostsServiceListRequest struct {
	AffinityLabelHostsService *AffinityLabelHostsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
}

func (p *AffinityLabelHostsServiceListRequest) Header(key, value string) *AffinityLabelHostsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelHostsServiceListRequest) Query(key, value string) *AffinityLabelHostsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelHostsServiceListRequest) Follow(follow string) *AffinityLabelHostsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelHostsServiceListRequest) Send() (*AffinityLabelHostsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelHostsService.connection.URL(), p.AffinityLabelHostsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityLabelHostsServiceListResponse{hosts: result}, nil
}

func (p *AffinityLabelHostsServiceListRequest) MustSend() *AffinityLabelHostsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all hosts with the label.
// The order of the returned hosts isn't guaranteed.
//
type AffinityLabelHostsServiceListResponse struct {
	hosts *HostSlice
}

func (p *AffinityLabelHostsServiceListResponse) Hosts() (*HostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *AffinityLabelHostsServiceListResponse) MustHosts() *HostSlice {
	if p.hosts == nil {
		panic("hosts in response does not exist")
	}
	return p.hosts
}

//
// List all hosts with the label.
// The order of the returned hosts isn't guaranteed.
//
func (p *AffinityLabelHostsService) List() *AffinityLabelHostsServiceListRequest {
	return &AffinityLabelHostsServiceListRequest{AffinityLabelHostsService: p}
}

//
// A link to the specific label-host assignment to
// allow label removal.
//
func (op *AffinityLabelHostsService) HostService(id string) *AffinityLabelHostService {
	return NewAffinityLabelHostService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelHostsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HostService(path), nil
	}
	return op.HostService(path[:index]).Service(path[index+1:])
}

func (op *AffinityLabelHostsService) String() string {
	return fmt.Sprintf("AffinityLabelHostsService:%s", op.path)
}

//
// The details of a single affinity label.
//
type AffinityLabelService struct {
	BaseService
}

func NewAffinityLabelService(connection *Connection, path string) *AffinityLabelService {
	var result AffinityLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the details of a label.
//
type AffinityLabelServiceGetRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *AffinityLabelServiceGetRequest) Header(key, value string) *AffinityLabelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceGetRequest) Query(key, value string) *AffinityLabelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceGetRequest) Follow(follow string) *AffinityLabelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelServiceGetRequest) Send() (*AffinityLabelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelServiceGetResponse{label: result}, nil
}

func (p *AffinityLabelServiceGetRequest) MustSend() *AffinityLabelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the details of a label.
//
type AffinityLabelServiceGetResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelServiceGetResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelServiceGetResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Retrieves the details of a label.
//
func (p *AffinityLabelService) Get() *AffinityLabelServiceGetRequest {
	return &AffinityLabelServiceGetRequest{AffinityLabelService: p}
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
type AffinityLabelServiceRemoveRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
}

func (p *AffinityLabelServiceRemoveRequest) Header(key, value string) *AffinityLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceRemoveRequest) Query(key, value string) *AffinityLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceRemoveRequest) Send() (*AffinityLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityLabelServiceRemoveResponse), nil
}

func (p *AffinityLabelServiceRemoveRequest) MustSend() *AffinityLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
type AffinityLabelServiceRemoveResponse struct {
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
func (p *AffinityLabelService) Remove() *AffinityLabelServiceRemoveRequest {
	return &AffinityLabelServiceRemoveRequest{AffinityLabelService: p}
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
type AffinityLabelServiceUpdateRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
	label                *AffinityLabel
}

func (p *AffinityLabelServiceUpdateRequest) Header(key, value string) *AffinityLabelServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Query(key, value string) *AffinityLabelServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Label(label *AffinityLabel) *AffinityLabelServiceUpdateRequest {
	p.label = label
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Send() (*AffinityLabelServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelServiceUpdateResponse{label: result}, nil
}

func (p *AffinityLabelServiceUpdateRequest) MustSend() *AffinityLabelServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
type AffinityLabelServiceUpdateResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelServiceUpdateResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelServiceUpdateResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
func (p *AffinityLabelService) Update() *AffinityLabelServiceUpdateRequest {
	return &AffinityLabelServiceUpdateRequest{AffinityLabelService: p}
}

//
// List all hosts with this label.
//
func (op *AffinityLabelService) HostsService() *AffinityLabelHostsService {
	return NewAffinityLabelHostsService(op.connection, fmt.Sprintf("%s/hosts", op.path))
}

//
// List all virtual machines with this label.
//
func (op *AffinityLabelService) VmsService() *AffinityLabelVmsService {
	return NewAffinityLabelVmsService(op.connection, fmt.Sprintf("%s/vms", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "hosts" {
		return op.HostsService(), nil
	}
	if strings.HasPrefix(path, "hosts/") {
		return op.HostsService().Service(path[6:])
	}
	if path == "vms" {
		return op.VmsService(), nil
	}
	if strings.HasPrefix(path, "vms/") {
		return op.VmsService().Service(path[4:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityLabelService) String() string {
	return fmt.Sprintf("AffinityLabelService:%s", op.path)
}

//
// This service represents a vm that has a specific
// label when accessed through the affinitylabels/vms
// subcollection.
//
type AffinityLabelVmService struct {
	BaseService
}

func NewAffinityLabelVmService(connection *Connection, path string) *AffinityLabelVmService {
	var result AffinityLabelVmService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves details about a vm that has this label assigned.
//
type AffinityLabelVmServiceGetRequest struct {
	AffinityLabelVmService *AffinityLabelVmService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *AffinityLabelVmServiceGetRequest) Header(key, value string) *AffinityLabelVmServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelVmServiceGetRequest) Query(key, value string) *AffinityLabelVmServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelVmServiceGetRequest) Follow(follow string) *AffinityLabelVmServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelVmServiceGetRequest) Send() (*AffinityLabelVmServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelVmService.connection.URL(), p.AffinityLabelVmService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelVmService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelVmService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelVmService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelVmService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelVmService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelVmServiceGetResponse{vm: result}, nil
}

func (p *AffinityLabelVmServiceGetRequest) MustSend() *AffinityLabelVmServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves details about a vm that has this label assigned.
//
type AffinityLabelVmServiceGetResponse struct {
	vm *Vm
}

func (p *AffinityLabelVmServiceGetResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *AffinityLabelVmServiceGetResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// Retrieves details about a vm that has this label assigned.
//
func (p *AffinityLabelVmService) Get() *AffinityLabelVmServiceGetRequest {
	return &AffinityLabelVmServiceGetRequest{AffinityLabelVmService: p}
}

//
// Remove a label from a vm.
//
type AffinityLabelVmServiceRemoveRequest struct {
	AffinityLabelVmService *AffinityLabelVmService
	header                 map[string]string
	query                  map[string]string
}

func (p *AffinityLabelVmServiceRemoveRequest) Header(key, value string) *AffinityLabelVmServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelVmServiceRemoveRequest) Query(key, value string) *AffinityLabelVmServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelVmServiceRemoveRequest) Send() (*AffinityLabelVmServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelVmService.connection.URL(), p.AffinityLabelVmService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelVmService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelVmService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelVmService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelVmService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelVmService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityLabelVmServiceRemoveResponse), nil
}

func (p *AffinityLabelVmServiceRemoveRequest) MustSend() *AffinityLabelVmServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove a label from a vm.
//
type AffinityLabelVmServiceRemoveResponse struct {
}

//
// Remove a label from a vm.
//
func (p *AffinityLabelVmService) Remove() *AffinityLabelVmServiceRemoveRequest {
	return &AffinityLabelVmServiceRemoveRequest{AffinityLabelVmService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelVmService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityLabelVmService) String() string {
	return fmt.Sprintf("AffinityLabelVmService:%s", op.path)
}

//
// This service represents list of vms that have a specific
// label when accessed through the affinitylabels/vms
// subcollection.
//
type AffinityLabelVmsService struct {
	BaseService
}

func NewAffinityLabelVmsService(connection *Connection, path string) *AffinityLabelVmsService {
	var result AffinityLabelVmsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a label to a vm.
//
type AffinityLabelVmsServiceAddRequest struct {
	AffinityLabelVmsService *AffinityLabelVmsService
	header                  map[string]string
	query                   map[string]string
	vm                      *Vm
}

func (p *AffinityLabelVmsServiceAddRequest) Header(key, value string) *AffinityLabelVmsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelVmsServiceAddRequest) Query(key, value string) *AffinityLabelVmsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelVmsServiceAddRequest) Vm(vm *Vm) *AffinityLabelVmsServiceAddRequest {
	p.vm = vm
	return p
}

func (p *AffinityLabelVmsServiceAddRequest) Send() (*AffinityLabelVmsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelVmsService.connection.URL(), p.AffinityLabelVmsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVmWriteOne(writer, p.vm, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelVmsServiceAddResponse{vm: result}, nil
}

func (p *AffinityLabelVmsServiceAddRequest) MustSend() *AffinityLabelVmsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a label to a vm.
//
type AffinityLabelVmsServiceAddResponse struct {
	vm *Vm
}

func (p *AffinityLabelVmsServiceAddResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *AffinityLabelVmsServiceAddResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// Add a label to a vm.
//
func (p *AffinityLabelVmsService) Add() *AffinityLabelVmsServiceAddRequest {
	return &AffinityLabelVmsServiceAddRequest{AffinityLabelVmsService: p}
}

//
// List all virtual machines with the label.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityLabelVmsServiceListRequest struct {
	AffinityLabelVmsService *AffinityLabelVmsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *AffinityLabelVmsServiceListRequest) Header(key, value string) *AffinityLabelVmsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelVmsServiceListRequest) Query(key, value string) *AffinityLabelVmsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelVmsServiceListRequest) Follow(follow string) *AffinityLabelVmsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelVmsServiceListRequest) Send() (*AffinityLabelVmsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelVmsService.connection.URL(), p.AffinityLabelVmsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityLabelVmsServiceListResponse{vms: result}, nil
}

func (p *AffinityLabelVmsServiceListRequest) MustSend() *AffinityLabelVmsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all virtual machines with the label.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityLabelVmsServiceListResponse struct {
	vms *VmSlice
}

func (p *AffinityLabelVmsServiceListResponse) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *AffinityLabelVmsServiceListResponse) MustVms() *VmSlice {
	if p.vms == nil {
		panic("vms in response does not exist")
	}
	return p.vms
}

//
// List all virtual machines with the label.
// The order of the returned virtual machines isn't guaranteed.
//
func (p *AffinityLabelVmsService) List() *AffinityLabelVmsServiceListRequest {
	return &AffinityLabelVmsServiceListRequest{AffinityLabelVmsService: p}
}

//
// A link to the specific label-vm assignment to
// allow label removal.
//
func (op *AffinityLabelVmsService) VmService(id string) *AffinityLabelVmService {
	return NewAffinityLabelVmService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelVmsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VmService(path), nil
	}
	return op.VmService(path[:index]).Service(path[index+1:])
}

func (op *AffinityLabelVmsService) String() string {
	return fmt.Sprintf("AffinityLabelVmsService:%s", op.path)
}

//
// Manages the affinity labels available in the system.
//
type AffinityLabelsService struct {
	BaseService
}

func NewAffinityLabelsService(connection *Connection, path string) *AffinityLabelsService {
	var result AffinityLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new label. The label is automatically attached
// to all entities mentioned in the vms or hosts lists.
//
type AffinityLabelsServiceAddRequest struct {
	AffinityLabelsService *AffinityLabelsService
	header                map[string]string
	query                 map[string]string
	label                 *AffinityLabel
}

func (p *AffinityLabelsServiceAddRequest) Header(key, value string) *AffinityLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelsServiceAddRequest) Query(key, value string) *AffinityLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelsServiceAddRequest) Label(label *AffinityLabel) *AffinityLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *AffinityLabelsServiceAddRequest) Send() (*AffinityLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelsService.connection.URL(), p.AffinityLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelsServiceAddResponse{label: result}, nil
}

func (p *AffinityLabelsServiceAddRequest) MustSend() *AffinityLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new label. The label is automatically attached
// to all entities mentioned in the vms or hosts lists.
//
type AffinityLabelsServiceAddResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelsServiceAddResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelsServiceAddResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Creates a new label. The label is automatically attached
// to all entities mentioned in the vms or hosts lists.
//
func (p *AffinityLabelsService) Add() *AffinityLabelsServiceAddRequest {
	return &AffinityLabelsServiceAddRequest{AffinityLabelsService: p}
}

//
// Lists all labels present in the system.
// The order of the returned labels isn't guaranteed.
//
type AffinityLabelsServiceListRequest struct {
	AffinityLabelsService *AffinityLabelsService
	header                map[string]string
	query                 map[string]string
	follow                *string
	max                   *int64
}

func (p *AffinityLabelsServiceListRequest) Header(key, value string) *AffinityLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelsServiceListRequest) Query(key, value string) *AffinityLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelsServiceListRequest) Follow(follow string) *AffinityLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelsServiceListRequest) Max(max int64) *AffinityLabelsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityLabelsServiceListRequest) Send() (*AffinityLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelsService.connection.URL(), p.AffinityLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityLabelsServiceListResponse{labels: result}, nil
}

func (p *AffinityLabelsServiceListRequest) MustSend() *AffinityLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all labels present in the system.
// The order of the returned labels isn't guaranteed.
//
type AffinityLabelsServiceListResponse struct {
	labels *AffinityLabelSlice
}

func (p *AffinityLabelsServiceListResponse) Labels() (*AffinityLabelSlice, bool) {
	if p.labels != nil {
		return p.labels, true
	}
	return nil, false
}

func (p *AffinityLabelsServiceListResponse) MustLabels() *AffinityLabelSlice {
	if p.labels == nil {
		panic("labels in response does not exist")
	}
	return p.labels
}

//
// Lists all labels present in the system.
// The order of the returned labels isn't guaranteed.
//
func (p *AffinityLabelsService) List() *AffinityLabelsServiceListRequest {
	return &AffinityLabelsServiceListRequest{AffinityLabelsService: p}
}

//
// Link to a single label details.
//
func (op *AffinityLabelsService) LabelService(id string) *AffinityLabelService {
	return NewAffinityLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *AffinityLabelsService) String() string {
	return fmt.Sprintf("AffinityLabelsService:%s", op.path)
}

//
// This service represents one label to entity assignment
// when accessed using the entities/affinitylabels subcollection.
//
type AssignedAffinityLabelService struct {
	BaseService
}

func NewAssignedAffinityLabelService(connection *Connection, path string) *AssignedAffinityLabelService {
	var result AssignedAffinityLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves details about the attached label.
//
type AssignedAffinityLabelServiceGetRequest struct {
	AssignedAffinityLabelService *AssignedAffinityLabelService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
}

func (p *AssignedAffinityLabelServiceGetRequest) Header(key, value string) *AssignedAffinityLabelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Query(key, value string) *AssignedAffinityLabelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Follow(follow string) *AssignedAffinityLabelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Send() (*AssignedAffinityLabelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelService.connection.URL(), p.AssignedAffinityLabelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedAffinityLabelServiceGetResponse{label: result}, nil
}

func (p *AssignedAffinityLabelServiceGetRequest) MustSend() *AssignedAffinityLabelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves details about the attached label.
//
type AssignedAffinityLabelServiceGetResponse struct {
	label *AffinityLabel
}

func (p *AssignedAffinityLabelServiceGetResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AssignedAffinityLabelServiceGetResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Retrieves details about the attached label.
//
func (p *AssignedAffinityLabelService) Get() *AssignedAffinityLabelServiceGetRequest {
	return &AssignedAffinityLabelServiceGetRequest{AssignedAffinityLabelService: p}
}

//
// Removes the label from an entity. Does not touch the label itself.
//
type AssignedAffinityLabelServiceRemoveRequest struct {
	AssignedAffinityLabelService *AssignedAffinityLabelService
	header                       map[string]string
	query                        map[string]string
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Header(key, value string) *AssignedAffinityLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Query(key, value string) *AssignedAffinityLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Send() (*AssignedAffinityLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelService.connection.URL(), p.AssignedAffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedAffinityLabelServiceRemoveResponse), nil
}

func (p *AssignedAffinityLabelServiceRemoveRequest) MustSend() *AssignedAffinityLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the label from an entity. Does not touch the label itself.
//
type AssignedAffinityLabelServiceRemoveResponse struct {
}

//
// Removes the label from an entity. Does not touch the label itself.
//
func (p *AssignedAffinityLabelService) Remove() *AssignedAffinityLabelServiceRemoveRequest {
	return &AssignedAffinityLabelServiceRemoveRequest{AssignedAffinityLabelService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedAffinityLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedAffinityLabelService) String() string {
	return fmt.Sprintf("AssignedAffinityLabelService:%s", op.path)
}

//
// This service is used to list and manipulate affinity labels that are
// assigned to supported entities when accessed using entities/affinitylabels.
//
type AssignedAffinityLabelsService struct {
	BaseService
}

func NewAssignedAffinityLabelsService(connection *Connection, path string) *AssignedAffinityLabelsService {
	var result AssignedAffinityLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Attaches a label to an entity.
//
type AssignedAffinityLabelsServiceAddRequest struct {
	AssignedAffinityLabelsService *AssignedAffinityLabelsService
	header                        map[string]string
	query                         map[string]string
	label                         *AffinityLabel
}

func (p *AssignedAffinityLabelsServiceAddRequest) Header(key, value string) *AssignedAffinityLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceAddRequest) Query(key, value string) *AssignedAffinityLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceAddRequest) Label(label *AffinityLabel) *AssignedAffinityLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *AssignedAffinityLabelsServiceAddRequest) Send() (*AssignedAffinityLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelsService.connection.URL(), p.AssignedAffinityLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedAffinityLabelsServiceAddResponse{label: result}, nil
}

func (p *AssignedAffinityLabelsServiceAddRequest) MustSend() *AssignedAffinityLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Attaches a label to an entity.
//
type AssignedAffinityLabelsServiceAddResponse struct {
	label *AffinityLabel
}

func (p *AssignedAffinityLabelsServiceAddResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AssignedAffinityLabelsServiceAddResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Attaches a label to an entity.
//
func (p *AssignedAffinityLabelsService) Add() *AssignedAffinityLabelsServiceAddRequest {
	return &AssignedAffinityLabelsServiceAddRequest{AssignedAffinityLabelsService: p}
}

//
// Lists all labels that are attached to an entity.
// The order of the returned entities isn't guaranteed.
//
type AssignedAffinityLabelsServiceListRequest struct {
	AssignedAffinityLabelsService *AssignedAffinityLabelsService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
}

func (p *AssignedAffinityLabelsServiceListRequest) Header(key, value string) *AssignedAffinityLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceListRequest) Query(key, value string) *AssignedAffinityLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceListRequest) Follow(follow string) *AssignedAffinityLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedAffinityLabelsServiceListRequest) Send() (*AssignedAffinityLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelsService.connection.URL(), p.AssignedAffinityLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedAffinityLabelsServiceListResponse{label: result}, nil
}

func (p *AssignedAffinityLabelsServiceListRequest) MustSend() *AssignedAffinityLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all labels that are attached to an entity.
// The order of the returned entities isn't guaranteed.
//
type AssignedAffinityLabelsServiceListResponse struct {
	label *AffinityLabelSlice
}

func (p *AssignedAffinityLabelsServiceListResponse) Label() (*AffinityLabelSlice, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AssignedAffinityLabelsServiceListResponse) MustLabel() *AffinityLabelSlice {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Lists all labels that are attached to an entity.
// The order of the returned entities isn't guaranteed.
//
func (p *AssignedAffinityLabelsService) List() *AssignedAffinityLabelsServiceListRequest {
	return &AssignedAffinityLabelsServiceListRequest{AssignedAffinityLabelsService: p}
}

//
// Link to the specific entity-label assignment to allow
// removal.
//
func (op *AssignedAffinityLabelsService) LabelService(id string) *AssignedAffinityLabelService {
	return NewAssignedAffinityLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedAffinityLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *AssignedAffinityLabelsService) String() string {
	return fmt.Sprintf("AssignedAffinityLabelsService:%s", op.path)
}

//
//
type AssignedCpuProfileService struct {
	BaseService
}

func NewAssignedCpuProfileService(connection *Connection, path string) *AssignedCpuProfileService {
	var result AssignedCpuProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type AssignedCpuProfileServiceGetRequest struct {
	AssignedCpuProfileService *AssignedCpuProfileService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
}

func (p *AssignedCpuProfileServiceGetRequest) Header(key, value string) *AssignedCpuProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfileServiceGetRequest) Query(key, value string) *AssignedCpuProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfileServiceGetRequest) Follow(follow string) *AssignedCpuProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedCpuProfileServiceGetRequest) Send() (*AssignedCpuProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfileService.connection.URL(), p.AssignedCpuProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedCpuProfileServiceGetResponse{profile: result}, nil
}

func (p *AssignedCpuProfileServiceGetRequest) MustSend() *AssignedCpuProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedCpuProfileServiceGetResponse struct {
	profile *CpuProfile
}

func (p *AssignedCpuProfileServiceGetResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedCpuProfileServiceGetResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *AssignedCpuProfileService) Get() *AssignedCpuProfileServiceGetRequest {
	return &AssignedCpuProfileServiceGetRequest{AssignedCpuProfileService: p}
}

//
//
type AssignedCpuProfileServiceRemoveRequest struct {
	AssignedCpuProfileService *AssignedCpuProfileService
	header                    map[string]string
	query                     map[string]string
	async                     *bool
}

func (p *AssignedCpuProfileServiceRemoveRequest) Header(key, value string) *AssignedCpuProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfileServiceRemoveRequest) Query(key, value string) *AssignedCpuProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfileServiceRemoveRequest) Async(async bool) *AssignedCpuProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AssignedCpuProfileServiceRemoveRequest) Send() (*AssignedCpuProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfileService.connection.URL(), p.AssignedCpuProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedCpuProfileServiceRemoveResponse), nil
}

func (p *AssignedCpuProfileServiceRemoveRequest) MustSend() *AssignedCpuProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedCpuProfileServiceRemoveResponse struct {
}

//
//
func (p *AssignedCpuProfileService) Remove() *AssignedCpuProfileServiceRemoveRequest {
	return &AssignedCpuProfileServiceRemoveRequest{AssignedCpuProfileService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedCpuProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedCpuProfileService) String() string {
	return fmt.Sprintf("AssignedCpuProfileService:%s", op.path)
}

//
//
type AssignedCpuProfilesService struct {
	BaseService
}

func NewAssignedCpuProfilesService(connection *Connection, path string) *AssignedCpuProfilesService {
	var result AssignedCpuProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new cpu profile for the cluster.
//
type AssignedCpuProfilesServiceAddRequest struct {
	AssignedCpuProfilesService *AssignedCpuProfilesService
	header                     map[string]string
	query                      map[string]string
	profile                    *CpuProfile
}

func (p *AssignedCpuProfilesServiceAddRequest) Header(key, value string) *AssignedCpuProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceAddRequest) Query(key, value string) *AssignedCpuProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceAddRequest) Profile(profile *CpuProfile) *AssignedCpuProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *AssignedCpuProfilesServiceAddRequest) Send() (*AssignedCpuProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfilesService.connection.URL(), p.AssignedCpuProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLCpuProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedCpuProfilesServiceAddResponse{profile: result}, nil
}

func (p *AssignedCpuProfilesServiceAddRequest) MustSend() *AssignedCpuProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new cpu profile for the cluster.
//
type AssignedCpuProfilesServiceAddResponse struct {
	profile *CpuProfile
}

func (p *AssignedCpuProfilesServiceAddResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedCpuProfilesServiceAddResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a new cpu profile for the cluster.
//
func (p *AssignedCpuProfilesService) Add() *AssignedCpuProfilesServiceAddRequest {
	return &AssignedCpuProfilesServiceAddRequest{AssignedCpuProfilesService: p}
}

//
// List the CPU profiles assigned to the cluster.
// The order of the returned CPU profiles isn't guaranteed.
//
type AssignedCpuProfilesServiceListRequest struct {
	AssignedCpuProfilesService *AssignedCpuProfilesService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
	max                        *int64
}

func (p *AssignedCpuProfilesServiceListRequest) Header(key, value string) *AssignedCpuProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Query(key, value string) *AssignedCpuProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Follow(follow string) *AssignedCpuProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Max(max int64) *AssignedCpuProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Send() (*AssignedCpuProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfilesService.connection.URL(), p.AssignedCpuProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedCpuProfilesServiceListResponse{profiles: result}, nil
}

func (p *AssignedCpuProfilesServiceListRequest) MustSend() *AssignedCpuProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the CPU profiles assigned to the cluster.
// The order of the returned CPU profiles isn't guaranteed.
//
type AssignedCpuProfilesServiceListResponse struct {
	profiles *CpuProfileSlice
}

func (p *AssignedCpuProfilesServiceListResponse) Profiles() (*CpuProfileSlice, bool) {
	if p.profiles != nil {
		return p.profiles, true
	}
	return nil, false
}

func (p *AssignedCpuProfilesServiceListResponse) MustProfiles() *CpuProfileSlice {
	if p.profiles == nil {
		panic("profiles in response does not exist")
	}
	return p.profiles
}

//
// List the CPU profiles assigned to the cluster.
// The order of the returned CPU profiles isn't guaranteed.
//
func (p *AssignedCpuProfilesService) List() *AssignedCpuProfilesServiceListRequest {
	return &AssignedCpuProfilesServiceListRequest{AssignedCpuProfilesService: p}
}

//
//
func (op *AssignedCpuProfilesService) ProfileService(id string) *AssignedCpuProfileService {
	return NewAssignedCpuProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedCpuProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProfileService(path), nil
	}
	return op.ProfileService(path[:index]).Service(path[index+1:])
}

func (op *AssignedCpuProfilesService) String() string {
	return fmt.Sprintf("AssignedCpuProfilesService:%s", op.path)
}

//
//
type AssignedDiskProfileService struct {
	BaseService
}

func NewAssignedDiskProfileService(connection *Connection, path string) *AssignedDiskProfileService {
	var result AssignedDiskProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type AssignedDiskProfileServiceGetRequest struct {
	AssignedDiskProfileService *AssignedDiskProfileService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
}

func (p *AssignedDiskProfileServiceGetRequest) Header(key, value string) *AssignedDiskProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedDiskProfileServiceGetRequest) Query(key, value string) *AssignedDiskProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedDiskProfileServiceGetRequest) Follow(follow string) *AssignedDiskProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedDiskProfileServiceGetRequest) Send() (*AssignedDiskProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedDiskProfileService.connection.URL(), p.AssignedDiskProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedDiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedDiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedDiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedDiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedDiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedDiskProfileServiceGetResponse{diskProfile: result}, nil
}

func (p *AssignedDiskProfileServiceGetRequest) MustSend() *AssignedDiskProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedDiskProfileServiceGetResponse struct {
	diskProfile *DiskProfile
}

func (p *AssignedDiskProfileServiceGetResponse) DiskProfile() (*DiskProfile, bool) {
	if p.diskProfile != nil {
		return p.diskProfile, true
	}
	return nil, false
}

func (p *AssignedDiskProfileServiceGetResponse) MustDiskProfile() *DiskProfile {
	if p.diskProfile == nil {
		panic("diskProfile in response does not exist")
	}
	return p.diskProfile
}

//
//
func (p *AssignedDiskProfileService) Get() *AssignedDiskProfileServiceGetRequest {
	return &AssignedDiskProfileServiceGetRequest{AssignedDiskProfileService: p}
}

//
//
type AssignedDiskProfileServiceRemoveRequest struct {
	AssignedDiskProfileService *AssignedDiskProfileService
	header                     map[string]string
	query                      map[string]string
	async                      *bool
}

func (p *AssignedDiskProfileServiceRemoveRequest) Header(key, value string) *AssignedDiskProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedDiskProfileServiceRemoveRequest) Query(key, value string) *AssignedDiskProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedDiskProfileServiceRemoveRequest) Async(async bool) *AssignedDiskProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AssignedDiskProfileServiceRemoveRequest) Send() (*AssignedDiskProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedDiskProfileService.connection.URL(), p.AssignedDiskProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedDiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedDiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedDiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedDiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedDiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedDiskProfileServiceRemoveResponse), nil
}

func (p *AssignedDiskProfileServiceRemoveRequest) MustSend() *AssignedDiskProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedDiskProfileServiceRemoveResponse struct {
}

//
//
func (p *AssignedDiskProfileService) Remove() *AssignedDiskProfileServiceRemoveRequest {
	return &AssignedDiskProfileServiceRemoveRequest{AssignedDiskProfileService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedDiskProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedDiskProfileService) String() string {
	return fmt.Sprintf("AssignedDiskProfileService:%s", op.path)
}

//
//
type AssignedDiskProfilesService struct {
	BaseService
}

func NewAssignedDiskProfilesService(connection *Connection, path string) *AssignedDiskProfilesService {
	var result AssignedDiskProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new disk profile for the storage domain.
//
type AssignedDiskProfilesServiceAddRequest struct {
	AssignedDiskProfilesService *AssignedDiskProfilesService
	header                      map[string]string
	query                       map[string]string
	profile                     *DiskProfile
}

func (p *AssignedDiskProfilesServiceAddRequest) Header(key, value string) *AssignedDiskProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceAddRequest) Query(key, value string) *AssignedDiskProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceAddRequest) Profile(profile *DiskProfile) *AssignedDiskProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *AssignedDiskProfilesServiceAddRequest) Send() (*AssignedDiskProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedDiskProfilesService.connection.URL(), p.AssignedDiskProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedDiskProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedDiskProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedDiskProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedDiskProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedDiskProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedDiskProfilesServiceAddResponse{profile: result}, nil
}

func (p *AssignedDiskProfilesServiceAddRequest) MustSend() *AssignedDiskProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new disk profile for the storage domain.
//
type AssignedDiskProfilesServiceAddResponse struct {
	profile *DiskProfile
}

func (p *AssignedDiskProfilesServiceAddResponse) Profile() (*DiskProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedDiskProfilesServiceAddResponse) MustProfile() *DiskProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a new disk profile for the storage domain.
//
func (p *AssignedDiskProfilesService) Add() *AssignedDiskProfilesServiceAddRequest {
	return &AssignedDiskProfilesServiceAddRequest{AssignedDiskProfilesService: p}
}

//
// Returns the list of disk profiles assigned to the storage domain.
// The order of the returned disk profiles isn't guaranteed.
//
type AssignedDiskProfilesServiceListRequest struct {
	AssignedDiskProfilesService *AssignedDiskProfilesService
	header                      map[string]string
	query                       map[string]string
	follow                      *string
	max                         *int64
}

func (p *AssignedDiskProfilesServiceListRequest) Header(key, value string) *AssignedDiskProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Query(key, value string) *AssignedDiskProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Follow(follow string) *AssignedDiskProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Max(max int64) *AssignedDiskProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Send() (*AssignedDiskProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedDiskProfilesService.connection.URL(), p.AssignedDiskProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedDiskProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedDiskProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedDiskProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedDiskProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedDiskProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedDiskProfilesServiceListResponse{profiles: result}, nil
}

func (p *AssignedDiskProfilesServiceListRequest) MustSend() *AssignedDiskProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of disk profiles assigned to the storage domain.
// The order of the returned disk profiles isn't guaranteed.
//
type AssignedDiskProfilesServiceListResponse struct {
	profiles *DiskProfileSlice
}

func (p *AssignedDiskProfilesServiceListResponse) Profiles() (*DiskProfileSlice, bool) {
	if p.profiles != nil {
		return p.profiles, true
	}
	return nil, false
}

func (p *AssignedDiskProfilesServiceListResponse) MustProfiles() *DiskProfileSlice {
	if p.profiles == nil {
		panic("profiles in response does not exist")
	}
	return p.profiles
}

//
// Returns the list of disk profiles assigned to the storage domain.
// The order of the returned disk profiles isn't guaranteed.
//
func (p *AssignedDiskProfilesService) List() *AssignedDiskProfilesServiceListRequest {
	return &AssignedDiskProfilesServiceListRequest{AssignedDiskProfilesService: p}
}

//
//
func (op *AssignedDiskProfilesService) ProfileService(id string) *AssignedDiskProfileService {
	return NewAssignedDiskProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedDiskProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProfileService(path), nil
	}
	return op.ProfileService(path[:index]).Service(path[index+1:])
}

func (op *AssignedDiskProfilesService) String() string {
	return fmt.Sprintf("AssignedDiskProfilesService:%s", op.path)
}

//
// Represents a permission sub-collection, scoped by user, group or some entity type.
//
type AssignedPermissionsService struct {
	BaseService
}

func NewAssignedPermissionsService(connection *Connection, path string) *AssignedPermissionsService {
	var result AssignedPermissionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
type AssignedPermissionsServiceAddRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddRequest) Header(key, value string) *AssignedPermissionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddRequest) Query(key, value string) *AssignedPermissionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddRequest) Send() (*AssignedPermissionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLPermissionWriteOne(writer, p.permission, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermissionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedPermissionsServiceAddResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddRequest) MustSend() *AssignedPermissionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
type AssignedPermissionsServiceAddResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
func (p *AssignedPermissionsService) Add() *AssignedPermissionsServiceAddRequest {
	return &AssignedPermissionsServiceAddRequest{AssignedPermissionsService: p}
}

//
// Add a new permission on the cluster to the group in the system.
//
type AssignedPermissionsServiceAddClusterPermissionRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddClusterPermissionRequest) Header(key, value string) *AssignedPermissionsServiceAddClusterPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddClusterPermissionRequest) Query(key, value string) *AssignedPermissionsServiceAddClusterPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddClusterPermissionRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddClusterPermissionRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddClusterPermissionRequest) Send() (*AssignedPermissionsServiceAddClusterPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/clusterpermission", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddClusterPermissionResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddClusterPermissionRequest) MustSend() *AssignedPermissionsServiceAddClusterPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the cluster to the group in the system.
//
type AssignedPermissionsServiceAddClusterPermissionResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddClusterPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddClusterPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the cluster to the group in the system.
//
func (p *AssignedPermissionsService) AddClusterPermission() *AssignedPermissionsServiceAddClusterPermissionRequest {
	return &AssignedPermissionsServiceAddClusterPermissionRequest{AssignedPermissionsService: p}
}

//
// Add a new permission on the data center to the group in the system.
//
type AssignedPermissionsServiceAddDataCenterPermissionRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddDataCenterPermissionRequest) Header(key, value string) *AssignedPermissionsServiceAddDataCenterPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddDataCenterPermissionRequest) Query(key, value string) *AssignedPermissionsServiceAddDataCenterPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddDataCenterPermissionRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddDataCenterPermissionRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddDataCenterPermissionRequest) Send() (*AssignedPermissionsServiceAddDataCenterPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/datacenterpermission", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddDataCenterPermissionResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddDataCenterPermissionRequest) MustSend() *AssignedPermissionsServiceAddDataCenterPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the data center to the group in the system.
//
type AssignedPermissionsServiceAddDataCenterPermissionResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddDataCenterPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddDataCenterPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the data center to the group in the system.
//
func (p *AssignedPermissionsService) AddDataCenterPermission() *AssignedPermissionsServiceAddDataCenterPermissionRequest {
	return &AssignedPermissionsServiceAddDataCenterPermissionRequest{AssignedPermissionsService: p}
}

//
// Add a new group level permission for a given virtual machine.
//
type AssignedPermissionsServiceAddGroupLevelRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddGroupLevelRequest) Header(key, value string) *AssignedPermissionsServiceAddGroupLevelRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddGroupLevelRequest) Query(key, value string) *AssignedPermissionsServiceAddGroupLevelRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddGroupLevelRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddGroupLevelRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddGroupLevelRequest) Send() (*AssignedPermissionsServiceAddGroupLevelResponse, error) {
	rawURL := fmt.Sprintf("%s%s/grouplevel", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddGroupLevelResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddGroupLevelRequest) MustSend() *AssignedPermissionsServiceAddGroupLevelResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new group level permission for a given virtual machine.
//
type AssignedPermissionsServiceAddGroupLevelResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddGroupLevelResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddGroupLevelResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new group level permission for a given virtual machine.
//
func (p *AssignedPermissionsService) AddGroupLevel() *AssignedPermissionsServiceAddGroupLevelRequest {
	return &AssignedPermissionsServiceAddGroupLevelRequest{AssignedPermissionsService: p}
}

//
// Add a new permission on the host to the group in the system.
//
type AssignedPermissionsServiceAddHostPermissionRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddHostPermissionRequest) Header(key, value string) *AssignedPermissionsServiceAddHostPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddHostPermissionRequest) Query(key, value string) *AssignedPermissionsServiceAddHostPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddHostPermissionRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddHostPermissionRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddHostPermissionRequest) Send() (*AssignedPermissionsServiceAddHostPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/hostpermission", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddHostPermissionResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddHostPermissionRequest) MustSend() *AssignedPermissionsServiceAddHostPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the host to the group in the system.
//
type AssignedPermissionsServiceAddHostPermissionResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddHostPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddHostPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the host to the group in the system.
//
func (p *AssignedPermissionsService) AddHostPermission() *AssignedPermissionsServiceAddHostPermissionRequest {
	return &AssignedPermissionsServiceAddHostPermissionRequest{AssignedPermissionsService: p}
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
type AssignedPermissionsServiceListRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
}

func (p *AssignedPermissionsServiceListRequest) Header(key, value string) *AssignedPermissionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceListRequest) Query(key, value string) *AssignedPermissionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceListRequest) Follow(follow string) *AssignedPermissionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedPermissionsServiceListRequest) Send() (*AssignedPermissionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermissionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedPermissionsServiceListResponse{permissions: result}, nil
}

func (p *AssignedPermissionsServiceListRequest) MustSend() *AssignedPermissionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
type AssignedPermissionsServiceListResponse struct {
	permissions *PermissionSlice
}

func (p *AssignedPermissionsServiceListResponse) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceListResponse) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("permissions in response does not exist")
	}
	return p.permissions
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
func (p *AssignedPermissionsService) List() *AssignedPermissionsServiceListRequest {
	return &AssignedPermissionsServiceListRequest{AssignedPermissionsService: p}
}

//
// Add a new permission on the storage domain to the group in the system.
//
type AssignedPermissionsServiceAddStorageDomainPermissionRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddStorageDomainPermissionRequest) Header(key, value string) *AssignedPermissionsServiceAddStorageDomainPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddStorageDomainPermissionRequest) Query(key, value string) *AssignedPermissionsServiceAddStorageDomainPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddStorageDomainPermissionRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddStorageDomainPermissionRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddStorageDomainPermissionRequest) Send() (*AssignedPermissionsServiceAddStorageDomainPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/storagedomainpermission", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddStorageDomainPermissionResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddStorageDomainPermissionRequest) MustSend() *AssignedPermissionsServiceAddStorageDomainPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the storage domain to the group in the system.
//
type AssignedPermissionsServiceAddStorageDomainPermissionResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddStorageDomainPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddStorageDomainPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the storage domain to the group in the system.
//
func (p *AssignedPermissionsService) AddStorageDomainPermission() *AssignedPermissionsServiceAddStorageDomainPermissionRequest {
	return &AssignedPermissionsServiceAddStorageDomainPermissionRequest{AssignedPermissionsService: p}
}

//
// Add a new permission on the template to the group in the system.
//
type AssignedPermissionsServiceAddTemplatePermissionRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddTemplatePermissionRequest) Header(key, value string) *AssignedPermissionsServiceAddTemplatePermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddTemplatePermissionRequest) Query(key, value string) *AssignedPermissionsServiceAddTemplatePermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddTemplatePermissionRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddTemplatePermissionRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddTemplatePermissionRequest) Send() (*AssignedPermissionsServiceAddTemplatePermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/templatepermission", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddTemplatePermissionResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddTemplatePermissionRequest) MustSend() *AssignedPermissionsServiceAddTemplatePermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the template to the group in the system.
//
type AssignedPermissionsServiceAddTemplatePermissionResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddTemplatePermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddTemplatePermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the template to the group in the system.
//
func (p *AssignedPermissionsService) AddTemplatePermission() *AssignedPermissionsServiceAddTemplatePermissionRequest {
	return &AssignedPermissionsServiceAddTemplatePermissionRequest{AssignedPermissionsService: p}
}

//
// Add a new user level permission for a given virtual machine.
//
type AssignedPermissionsServiceAddUserLevelRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddUserLevelRequest) Header(key, value string) *AssignedPermissionsServiceAddUserLevelRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddUserLevelRequest) Query(key, value string) *AssignedPermissionsServiceAddUserLevelRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddUserLevelRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddUserLevelRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddUserLevelRequest) Send() (*AssignedPermissionsServiceAddUserLevelResponse, error) {
	rawURL := fmt.Sprintf("%s%s/userlevel", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddUserLevelResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddUserLevelRequest) MustSend() *AssignedPermissionsServiceAddUserLevelResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new user level permission for a given virtual machine.
//
type AssignedPermissionsServiceAddUserLevelResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddUserLevelResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddUserLevelResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new user level permission for a given virtual machine.
//
func (p *AssignedPermissionsService) AddUserLevel() *AssignedPermissionsServiceAddUserLevelRequest {
	return &AssignedPermissionsServiceAddUserLevelRequest{AssignedPermissionsService: p}
}

//
// Add a new permission on the vm to the group in the system.
//
type AssignedPermissionsServiceAddVmPermissionRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddVmPermissionRequest) Header(key, value string) *AssignedPermissionsServiceAddVmPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddVmPermissionRequest) Query(key, value string) *AssignedPermissionsServiceAddVmPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddVmPermissionRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddVmPermissionRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddVmPermissionRequest) Send() (*AssignedPermissionsServiceAddVmPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vmpermission", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddVmPermissionResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddVmPermissionRequest) MustSend() *AssignedPermissionsServiceAddVmPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the vm to the group in the system.
//
type AssignedPermissionsServiceAddVmPermissionResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddVmPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddVmPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the vm to the group in the system.
//
func (p *AssignedPermissionsService) AddVmPermission() *AssignedPermissionsServiceAddVmPermissionRequest {
	return &AssignedPermissionsServiceAddVmPermissionRequest{AssignedPermissionsService: p}
}

//
// Add a new permission on the vm pool to the group in the system.
//
type AssignedPermissionsServiceAddVmPoolPermissionRequest struct {
	AssignedPermissionsService *AssignedPermissionsService
	header                     map[string]string
	query                      map[string]string
	permission                 *Permission
}

func (p *AssignedPermissionsServiceAddVmPoolPermissionRequest) Header(key, value string) *AssignedPermissionsServiceAddVmPoolPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddVmPoolPermissionRequest) Query(key, value string) *AssignedPermissionsServiceAddVmPoolPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedPermissionsServiceAddVmPoolPermissionRequest) Permission(permission *Permission) *AssignedPermissionsServiceAddVmPoolPermissionRequest {
	p.permission = permission
	return p
}

func (p *AssignedPermissionsServiceAddVmPoolPermissionRequest) Send() (*AssignedPermissionsServiceAddVmPoolPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vmpoolpermission", p.AssignedPermissionsService.connection.URL(), p.AssignedPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &AssignedPermissionsServiceAddVmPoolPermissionResponse{permission: result}, nil
}

func (p *AssignedPermissionsServiceAddVmPoolPermissionRequest) MustSend() *AssignedPermissionsServiceAddVmPoolPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the vm pool to the group in the system.
//
type AssignedPermissionsServiceAddVmPoolPermissionResponse struct {
	permission *Permission
}

func (p *AssignedPermissionsServiceAddVmPoolPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *AssignedPermissionsServiceAddVmPoolPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the vm pool to the group in the system.
//
func (p *AssignedPermissionsService) AddVmPoolPermission() *AssignedPermissionsServiceAddVmPoolPermissionRequest {
	return &AssignedPermissionsServiceAddVmPoolPermissionRequest{AssignedPermissionsService: p}
}

//
// Sub-resource locator method, returns individual permission resource on which the remainder of the URI is
// dispatched.
//
func (op *AssignedPermissionsService) PermissionService(id string) *PermissionService {
	return NewPermissionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedPermissionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.PermissionService(path), nil
	}
	return op.PermissionService(path[:index]).Service(path[index+1:])
}

func (op *AssignedPermissionsService) String() string {
	return fmt.Sprintf("AssignedPermissionsService:%s", op.path)
}

//
// Represents a roles sub-collection, for example scoped by user.
//
type AssignedRolesService struct {
	BaseService
}

func NewAssignedRolesService(connection *Connection, path string) *AssignedRolesService {
	var result AssignedRolesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the roles assigned to the permission.
// The order of the returned roles isn't guaranteed.
//
type AssignedRolesServiceListRequest struct {
	AssignedRolesService *AssignedRolesService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *AssignedRolesServiceListRequest) Header(key, value string) *AssignedRolesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedRolesServiceListRequest) Query(key, value string) *AssignedRolesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedRolesServiceListRequest) Follow(follow string) *AssignedRolesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedRolesServiceListRequest) Max(max int64) *AssignedRolesServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedRolesServiceListRequest) Send() (*AssignedRolesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedRolesService.connection.URL(), p.AssignedRolesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedRolesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedRolesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedRolesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedRolesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedRolesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLRoleReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedRolesServiceListResponse{roles: result}, nil
}

func (p *AssignedRolesServiceListRequest) MustSend() *AssignedRolesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the roles assigned to the permission.
// The order of the returned roles isn't guaranteed.
//
type AssignedRolesServiceListResponse struct {
	roles *RoleSlice
}

func (p *AssignedRolesServiceListResponse) Roles() (*RoleSlice, bool) {
	if p.roles != nil {
		return p.roles, true
	}
	return nil, false
}

func (p *AssignedRolesServiceListResponse) MustRoles() *RoleSlice {
	if p.roles == nil {
		panic("roles in response does not exist")
	}
	return p.roles
}

//
// Returns the roles assigned to the permission.
// The order of the returned roles isn't guaranteed.
//
func (p *AssignedRolesService) List() *AssignedRolesServiceListRequest {
	return &AssignedRolesServiceListRequest{AssignedRolesService: p}
}

//
// Sub-resource locator method, returns individual role resource on which the remainder of the URI is dispatched.
//
func (op *AssignedRolesService) RoleService(id string) *RoleService {
	return NewRoleService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedRolesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.RoleService(path), nil
	}
	return op.RoleService(path[:index]).Service(path[index+1:])
}

func (op *AssignedRolesService) String() string {
	return fmt.Sprintf("AssignedRolesService:%s", op.path)
}

//
// A service to manage assignment of specific tag to specific entities in system.
//
type AssignedTagService struct {
	BaseService
}

func NewAssignedTagService(connection *Connection, path string) *AssignedTagService {
	var result AssignedTagService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the information about the assigned tag.
// For example to retrieve the information about the tag with the id `456` which is assigned to virtual machine
// with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags/456
// ....
// [source,xml]
// ----
// <tag href="/ovirt-engine/api/tags/456" id="456">
//   <name>root</name>
//   <description>root</description>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </tag>
// ----
//
type AssignedTagServiceGetRequest struct {
	AssignedTagService *AssignedTagService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *AssignedTagServiceGetRequest) Header(key, value string) *AssignedTagServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedTagServiceGetRequest) Query(key, value string) *AssignedTagServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedTagServiceGetRequest) Follow(follow string) *AssignedTagServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedTagServiceGetRequest) Send() (*AssignedTagServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedTagService.connection.URL(), p.AssignedTagService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedTagService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedTagService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedTagService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedTagService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedTagService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedTagServiceGetResponse{tag: result}, nil
}

func (p *AssignedTagServiceGetRequest) MustSend() *AssignedTagServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the information about the assigned tag.
// For example to retrieve the information about the tag with the id `456` which is assigned to virtual machine
// with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags/456
// ....
// [source,xml]
// ----
// <tag href="/ovirt-engine/api/tags/456" id="456">
//   <name>root</name>
//   <description>root</description>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </tag>
// ----
//
type AssignedTagServiceGetResponse struct {
	tag *Tag
}

func (p *AssignedTagServiceGetResponse) Tag() (*Tag, bool) {
	if p.tag != nil {
		return p.tag, true
	}
	return nil, false
}

func (p *AssignedTagServiceGetResponse) MustTag() *Tag {
	if p.tag == nil {
		panic("tag in response does not exist")
	}
	return p.tag
}

//
// Gets the information about the assigned tag.
// For example to retrieve the information about the tag with the id `456` which is assigned to virtual machine
// with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags/456
// ....
// [source,xml]
// ----
// <tag href="/ovirt-engine/api/tags/456" id="456">
//   <name>root</name>
//   <description>root</description>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </tag>
// ----
//
func (p *AssignedTagService) Get() *AssignedTagServiceGetRequest {
	return &AssignedTagServiceGetRequest{AssignedTagService: p}
}

//
// Unassign tag from specific entity in the system.
// For example to unassign the tag with id `456` from virtual machine with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/vms/123/tags/456
// ....
//
type AssignedTagServiceRemoveRequest struct {
	AssignedTagService *AssignedTagService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *AssignedTagServiceRemoveRequest) Header(key, value string) *AssignedTagServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedTagServiceRemoveRequest) Query(key, value string) *AssignedTagServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedTagServiceRemoveRequest) Async(async bool) *AssignedTagServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AssignedTagServiceRemoveRequest) Send() (*AssignedTagServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedTagService.connection.URL(), p.AssignedTagService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedTagService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedTagService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedTagService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedTagService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedTagService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedTagServiceRemoveResponse), nil
}

func (p *AssignedTagServiceRemoveRequest) MustSend() *AssignedTagServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Unassign tag from specific entity in the system.
// For example to unassign the tag with id `456` from virtual machine with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/vms/123/tags/456
// ....
//
type AssignedTagServiceRemoveResponse struct {
}

//
// Unassign tag from specific entity in the system.
// For example to unassign the tag with id `456` from virtual machine with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/vms/123/tags/456
// ....
//
func (p *AssignedTagService) Remove() *AssignedTagServiceRemoveRequest {
	return &AssignedTagServiceRemoveRequest{AssignedTagService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedTagService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedTagService) String() string {
	return fmt.Sprintf("AssignedTagService:%s", op.path)
}

//
// A service to manage collection of assignment of tags to specific entities in system.
//
type AssignedTagsService struct {
	BaseService
}

func NewAssignedTagsService(connection *Connection, path string) *AssignedTagsService {
	var result AssignedTagsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Assign tag to specific entity in the system.
// For example to assign tag `mytag` to virtual machine with the id `123` send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
//
type AssignedTagsServiceAddRequest struct {
	AssignedTagsService *AssignedTagsService
	header              map[string]string
	query               map[string]string
	tag                 *Tag
}

func (p *AssignedTagsServiceAddRequest) Header(key, value string) *AssignedTagsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedTagsServiceAddRequest) Query(key, value string) *AssignedTagsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedTagsServiceAddRequest) Tag(tag *Tag) *AssignedTagsServiceAddRequest {
	p.tag = tag
	return p
}

func (p *AssignedTagsServiceAddRequest) Send() (*AssignedTagsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedTagsService.connection.URL(), p.AssignedTagsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLTagWriteOne(writer, p.tag, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedTagsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedTagsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedTagsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedTagsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedTagsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedTagsServiceAddResponse{tag: result}, nil
}

func (p *AssignedTagsServiceAddRequest) MustSend() *AssignedTagsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Assign tag to specific entity in the system.
// For example to assign tag `mytag` to virtual machine with the id `123` send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
//
type AssignedTagsServiceAddResponse struct {
	tag *Tag
}

func (p *AssignedTagsServiceAddResponse) Tag() (*Tag, bool) {
	if p.tag != nil {
		return p.tag, true
	}
	return nil, false
}

func (p *AssignedTagsServiceAddResponse) MustTag() *Tag {
	if p.tag == nil {
		panic("tag in response does not exist")
	}
	return p.tag
}

//
// Assign tag to specific entity in the system.
// For example to assign tag `mytag` to virtual machine with the id `123` send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
//
func (p *AssignedTagsService) Add() *AssignedTagsServiceAddRequest {
	return &AssignedTagsServiceAddRequest{AssignedTagsService: p}
}

//
// List all tags assigned to the specific entity.
// For example to list all the tags of the virtual machine with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>mytag</name>
//     <description>mytag</description>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </tag>
// </tags>
// ----
// The order of the returned tags isn't guaranteed.
//
type AssignedTagsServiceListRequest struct {
	AssignedTagsService *AssignedTagsService
	header              map[string]string
	query               map[string]string
	follow              *string
	max                 *int64
}

func (p *AssignedTagsServiceListRequest) Header(key, value string) *AssignedTagsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedTagsServiceListRequest) Query(key, value string) *AssignedTagsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedTagsServiceListRequest) Follow(follow string) *AssignedTagsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedTagsServiceListRequest) Max(max int64) *AssignedTagsServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedTagsServiceListRequest) Send() (*AssignedTagsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedTagsService.connection.URL(), p.AssignedTagsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedTagsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedTagsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedTagsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedTagsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedTagsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedTagsServiceListResponse{tags: result}, nil
}

func (p *AssignedTagsServiceListRequest) MustSend() *AssignedTagsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all tags assigned to the specific entity.
// For example to list all the tags of the virtual machine with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>mytag</name>
//     <description>mytag</description>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </tag>
// </tags>
// ----
// The order of the returned tags isn't guaranteed.
//
type AssignedTagsServiceListResponse struct {
	tags *TagSlice
}

func (p *AssignedTagsServiceListResponse) Tags() (*TagSlice, bool) {
	if p.tags != nil {
		return p.tags, true
	}
	return nil, false
}

func (p *AssignedTagsServiceListResponse) MustTags() *TagSlice {
	if p.tags == nil {
		panic("tags in response does not exist")
	}
	return p.tags
}

//
// List all tags assigned to the specific entity.
// For example to list all the tags of the virtual machine with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>mytag</name>
//     <description>mytag</description>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </tag>
// </tags>
// ----
// The order of the returned tags isn't guaranteed.
//
func (p *AssignedTagsService) List() *AssignedTagsServiceListRequest {
	return &AssignedTagsServiceListRequest{AssignedTagsService: p}
}

//
// Reference to the service that manages assignment of specific tag.
//
func (op *AssignedTagsService) TagService(id string) *AssignedTagService {
	return NewAssignedTagService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedTagsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.TagService(path), nil
	}
	return op.TagService(path[:index]).Service(path[index+1:])
}

func (op *AssignedTagsService) String() string {
	return fmt.Sprintf("AssignedTagsService:%s", op.path)
}

//
//
type AssignedVnicProfileService struct {
	BaseService
}

func NewAssignedVnicProfileService(connection *Connection, path string) *AssignedVnicProfileService {
	var result AssignedVnicProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type AssignedVnicProfileServiceGetRequest struct {
	AssignedVnicProfileService *AssignedVnicProfileService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
}

func (p *AssignedVnicProfileServiceGetRequest) Header(key, value string) *AssignedVnicProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedVnicProfileServiceGetRequest) Query(key, value string) *AssignedVnicProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedVnicProfileServiceGetRequest) Follow(follow string) *AssignedVnicProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedVnicProfileServiceGetRequest) Send() (*AssignedVnicProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedVnicProfileService.connection.URL(), p.AssignedVnicProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedVnicProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedVnicProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedVnicProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedVnicProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedVnicProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedVnicProfileServiceGetResponse{profile: result}, nil
}

func (p *AssignedVnicProfileServiceGetRequest) MustSend() *AssignedVnicProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedVnicProfileServiceGetResponse struct {
	profile *VnicProfile
}

func (p *AssignedVnicProfileServiceGetResponse) Profile() (*VnicProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedVnicProfileServiceGetResponse) MustProfile() *VnicProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *AssignedVnicProfileService) Get() *AssignedVnicProfileServiceGetRequest {
	return &AssignedVnicProfileServiceGetRequest{AssignedVnicProfileService: p}
}

//
//
type AssignedVnicProfileServiceRemoveRequest struct {
	AssignedVnicProfileService *AssignedVnicProfileService
	header                     map[string]string
	query                      map[string]string
	async                      *bool
}

func (p *AssignedVnicProfileServiceRemoveRequest) Header(key, value string) *AssignedVnicProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedVnicProfileServiceRemoveRequest) Query(key, value string) *AssignedVnicProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedVnicProfileServiceRemoveRequest) Async(async bool) *AssignedVnicProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AssignedVnicProfileServiceRemoveRequest) Send() (*AssignedVnicProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedVnicProfileService.connection.URL(), p.AssignedVnicProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedVnicProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedVnicProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedVnicProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedVnicProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedVnicProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedVnicProfileServiceRemoveResponse), nil
}

func (p *AssignedVnicProfileServiceRemoveRequest) MustSend() *AssignedVnicProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedVnicProfileServiceRemoveResponse struct {
}

//
//
func (p *AssignedVnicProfileService) Remove() *AssignedVnicProfileServiceRemoveRequest {
	return &AssignedVnicProfileServiceRemoveRequest{AssignedVnicProfileService: p}
}

//
//
func (op *AssignedVnicProfileService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedVnicProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedVnicProfileService) String() string {
	return fmt.Sprintf("AssignedVnicProfileService:%s", op.path)
}

//
//
type AssignedVnicProfilesService struct {
	BaseService
}

func NewAssignedVnicProfilesService(connection *Connection, path string) *AssignedVnicProfilesService {
	var result AssignedVnicProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new virtual network interface card profile for the network.
//
type AssignedVnicProfilesServiceAddRequest struct {
	AssignedVnicProfilesService *AssignedVnicProfilesService
	header                      map[string]string
	query                       map[string]string
	profile                     *VnicProfile
}

func (p *AssignedVnicProfilesServiceAddRequest) Header(key, value string) *AssignedVnicProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedVnicProfilesServiceAddRequest) Query(key, value string) *AssignedVnicProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedVnicProfilesServiceAddRequest) Profile(profile *VnicProfile) *AssignedVnicProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *AssignedVnicProfilesServiceAddRequest) Send() (*AssignedVnicProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedVnicProfilesService.connection.URL(), p.AssignedVnicProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVnicProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedVnicProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedVnicProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedVnicProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedVnicProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedVnicProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedVnicProfilesServiceAddResponse{profile: result}, nil
}

func (p *AssignedVnicProfilesServiceAddRequest) MustSend() *AssignedVnicProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new virtual network interface card profile for the network.
//
type AssignedVnicProfilesServiceAddResponse struct {
	profile *VnicProfile
}

func (p *AssignedVnicProfilesServiceAddResponse) Profile() (*VnicProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedVnicProfilesServiceAddResponse) MustProfile() *VnicProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a new virtual network interface card profile for the network.
//
func (p *AssignedVnicProfilesService) Add() *AssignedVnicProfilesServiceAddRequest {
	return &AssignedVnicProfilesServiceAddRequest{AssignedVnicProfilesService: p}
}

//
// Returns the list of VNIC profiles assifned to the network.
// The order of the returned VNIC profiles isn't guaranteed.
//
type AssignedVnicProfilesServiceListRequest struct {
	AssignedVnicProfilesService *AssignedVnicProfilesService
	header                      map[string]string
	query                       map[string]string
	follow                      *string
	max                         *int64
}

func (p *AssignedVnicProfilesServiceListRequest) Header(key, value string) *AssignedVnicProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedVnicProfilesServiceListRequest) Query(key, value string) *AssignedVnicProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedVnicProfilesServiceListRequest) Follow(follow string) *AssignedVnicProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedVnicProfilesServiceListRequest) Max(max int64) *AssignedVnicProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedVnicProfilesServiceListRequest) Send() (*AssignedVnicProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedVnicProfilesService.connection.URL(), p.AssignedVnicProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedVnicProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedVnicProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedVnicProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedVnicProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedVnicProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedVnicProfilesServiceListResponse{profiles: result}, nil
}

func (p *AssignedVnicProfilesServiceListRequest) MustSend() *AssignedVnicProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of VNIC profiles assifned to the network.
// The order of the returned VNIC profiles isn't guaranteed.
//
type AssignedVnicProfilesServiceListResponse struct {
	profiles *VnicProfileSlice
}

func (p *AssignedVnicProfilesServiceListResponse) Profiles() (*VnicProfileSlice, bool) {
	if p.profiles != nil {
		return p.profiles, true
	}
	return nil, false
}

func (p *AssignedVnicProfilesServiceListResponse) MustProfiles() *VnicProfileSlice {
	if p.profiles == nil {
		panic("profiles in response does not exist")
	}
	return p.profiles
}

//
// Returns the list of VNIC profiles assifned to the network.
// The order of the returned VNIC profiles isn't guaranteed.
//
func (p *AssignedVnicProfilesService) List() *AssignedVnicProfilesServiceListRequest {
	return &AssignedVnicProfilesServiceListRequest{AssignedVnicProfilesService: p}
}

//
//
func (op *AssignedVnicProfilesService) ProfileService(id string) *AssignedVnicProfileService {
	return NewAssignedVnicProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedVnicProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProfileService(path), nil
	}
	return op.ProfileService(path[:index]).Service(path[index+1:])
}

func (op *AssignedVnicProfilesService) String() string {
	return fmt.Sprintf("AssignedVnicProfilesService:%s", op.path)
}

//
// Manages a single disk available in a storage domain attached to a data center.
// IMPORTANT: Since version 4.2 of the engine this service is intended only to list disks available in the storage
// domain, and to register unregistered disks. All the other operations, like copying a disk, moving a disk, etc, have
// been deprecated and will be removed in the future. To perform those operations use the <<services/disks, service
// that manages all the disks of the system>>, or the <<services/disk, service that manages an specific disk>>.
//
type AttachedStorageDomainDiskService struct {
	BaseService
}

func NewAttachedStorageDomainDiskService(connection *Connection, path string) *AttachedStorageDomainDiskService {
	var result AttachedStorageDomainDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceCopyRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	disk                             *Disk
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Header(key, value string) *AttachedStorageDomainDiskServiceCopyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Query(key, value string) *AttachedStorageDomainDiskServiceCopyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Disk(disk *Disk) *AttachedStorageDomainDiskServiceCopyRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceCopyRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Send() (*AttachedStorageDomainDiskServiceCopyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copy", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceCopyResponse), nil
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) MustSend() *AttachedStorageDomainDiskServiceCopyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceCopyResponse struct {
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Copy() *AttachedStorageDomainDiskServiceCopyRequest {
	return &AttachedStorageDomainDiskServiceCopyRequest{AttachedStorageDomainDiskService: p}
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceExportRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Header(key, value string) *AttachedStorageDomainDiskServiceExportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Query(key, value string) *AttachedStorageDomainDiskServiceExportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceExportRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Send() (*AttachedStorageDomainDiskServiceExportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/export", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceExportResponse), nil
}

func (p *AttachedStorageDomainDiskServiceExportRequest) MustSend() *AttachedStorageDomainDiskServiceExportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceExportResponse struct {
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Export() *AttachedStorageDomainDiskServiceExportRequest {
	return &AttachedStorageDomainDiskServiceExportRequest{AttachedStorageDomainDiskService: p}
}

//
// Retrieves the description of the disk.
//
type AttachedStorageDomainDiskServiceGetRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	follow                           *string
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Header(key, value string) *AttachedStorageDomainDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Query(key, value string) *AttachedStorageDomainDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Follow(follow string) *AttachedStorageDomainDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Send() (*AttachedStorageDomainDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDiskServiceGetResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDiskServiceGetRequest) MustSend() *AttachedStorageDomainDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the description of the disk.
//
type AttachedStorageDomainDiskServiceGetResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Retrieves the description of the disk.
//
func (p *AttachedStorageDomainDiskService) Get() *AttachedStorageDomainDiskServiceGetRequest {
	return &AttachedStorageDomainDiskServiceGetRequest{AttachedStorageDomainDiskService: p}
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceMoveRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	async                            *bool
	filter                           *bool
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Header(key, value string) *AttachedStorageDomainDiskServiceMoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Query(key, value string) *AttachedStorageDomainDiskServiceMoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Async(async bool) *AttachedStorageDomainDiskServiceMoveRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Filter(filter bool) *AttachedStorageDomainDiskServiceMoveRequest {
	p.filter = &filter
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceMoveRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Send() (*AttachedStorageDomainDiskServiceMoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/move", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceMoveResponse), nil
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) MustSend() *AttachedStorageDomainDiskServiceMoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceMoveResponse struct {
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Move() *AttachedStorageDomainDiskServiceMoveRequest {
	return &AttachedStorageDomainDiskServiceMoveRequest{AttachedStorageDomainDiskService: p}
}

//
// Registers an unregistered disk.
//
type AttachedStorageDomainDiskServiceRegisterRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Header(key, value string) *AttachedStorageDomainDiskServiceRegisterRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Query(key, value string) *AttachedStorageDomainDiskServiceRegisterRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Send() (*AttachedStorageDomainDiskServiceRegisterResponse, error) {
	rawURL := fmt.Sprintf("%s%s/register", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceRegisterResponse), nil
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) MustSend() *AttachedStorageDomainDiskServiceRegisterResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Registers an unregistered disk.
//
type AttachedStorageDomainDiskServiceRegisterResponse struct {
}

//
// Registers an unregistered disk.
//
func (p *AttachedStorageDomainDiskService) Register() *AttachedStorageDomainDiskServiceRegisterRequest {
	return &AttachedStorageDomainDiskServiceRegisterRequest{AttachedStorageDomainDiskService: p}
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceRemoveRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Header(key, value string) *AttachedStorageDomainDiskServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Query(key, value string) *AttachedStorageDomainDiskServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Send() (*AttachedStorageDomainDiskServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AttachedStorageDomainDiskServiceRemoveResponse), nil
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) MustSend() *AttachedStorageDomainDiskServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceRemoveResponse struct {
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Remove() *AttachedStorageDomainDiskServiceRemoveRequest {
	return &AttachedStorageDomainDiskServiceRemoveRequest{AttachedStorageDomainDiskService: p}
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceSparsifyRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Header(key, value string) *AttachedStorageDomainDiskServiceSparsifyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Query(key, value string) *AttachedStorageDomainDiskServiceSparsifyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Send() (*AttachedStorageDomainDiskServiceSparsifyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/sparsify", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceSparsifyResponse), nil
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) MustSend() *AttachedStorageDomainDiskServiceSparsifyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceSparsifyResponse struct {
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Sparsify() *AttachedStorageDomainDiskServiceSparsifyRequest {
	return &AttachedStorageDomainDiskServiceSparsifyRequest{AttachedStorageDomainDiskService: p}
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceUpdateRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	disk                             *Disk
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Header(key, value string) *AttachedStorageDomainDiskServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Query(key, value string) *AttachedStorageDomainDiskServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Disk(disk *Disk) *AttachedStorageDomainDiskServiceUpdateRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Send() (*AttachedStorageDomainDiskServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDiskServiceUpdateResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) MustSend() *AttachedStorageDomainDiskServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceUpdateResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDiskServiceUpdateResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDiskServiceUpdateResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Update() *AttachedStorageDomainDiskServiceUpdateRequest {
	return &AttachedStorageDomainDiskServiceUpdateRequest{AttachedStorageDomainDiskService: p}
}

//
// Reference to the service that manages the permissions assigned to the disk.
//
func (op *AttachedStorageDomainDiskService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *AttachedStorageDomainDiskService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AttachedStorageDomainDiskService) String() string {
	return fmt.Sprintf("AttachedStorageDomainDiskService:%s", op.path)
}

//
// Manages the collection of disks available inside an storage domain that is attached to a data center.
//
type AttachedStorageDomainDisksService struct {
	BaseService
}

func NewAttachedStorageDomainDisksService(connection *Connection, path string) *AttachedStorageDomainDisksService {
	var result AttachedStorageDomainDisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds or registers a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To add a new disk use the <<services/disks/methods/add, add>>
// operation of the service that manages the disks of the system. To register an unregistered disk use the
// <<services/attached_storage_domain_disk/methods/register, register>> operation of the service that manages
// that disk.
//
type AttachedStorageDomainDisksServiceAddRequest struct {
	AttachedStorageDomainDisksService *AttachedStorageDomainDisksService
	header                            map[string]string
	query                             map[string]string
	disk                              *Disk
	unregistered                      *bool
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Header(key, value string) *AttachedStorageDomainDisksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Query(key, value string) *AttachedStorageDomainDisksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Disk(disk *Disk) *AttachedStorageDomainDisksServiceAddRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Unregistered(unregistered bool) *AttachedStorageDomainDisksServiceAddRequest {
	p.unregistered = &unregistered
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Send() (*AttachedStorageDomainDisksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDisksService.connection.URL(), p.AttachedStorageDomainDisksService.path)
	values := make(url.Values)
	if p.unregistered != nil {
		values["unregistered"] = []string{fmt.Sprintf("%v", *p.unregistered)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDisksServiceAddResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDisksServiceAddRequest) MustSend() *AttachedStorageDomainDisksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds or registers a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To add a new disk use the <<services/disks/methods/add, add>>
// operation of the service that manages the disks of the system. To register an unregistered disk use the
// <<services/attached_storage_domain_disk/methods/register, register>> operation of the service that manages
// that disk.
//
type AttachedStorageDomainDisksServiceAddResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDisksServiceAddResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDisksServiceAddResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Adds or registers a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To add a new disk use the <<services/disks/methods/add, add>>
// operation of the service that manages the disks of the system. To register an unregistered disk use the
// <<services/attached_storage_domain_disk/methods/register, register>> operation of the service that manages
// that disk.
//
func (p *AttachedStorageDomainDisksService) Add() *AttachedStorageDomainDisksServiceAddRequest {
	return &AttachedStorageDomainDisksServiceAddRequest{AttachedStorageDomainDisksService: p}
}

//
// Retrieve the list of disks that are available in the storage domain.
//
type AttachedStorageDomainDisksServiceListRequest struct {
	AttachedStorageDomainDisksService *AttachedStorageDomainDisksService
	header                            map[string]string
	query                             map[string]string
	follow                            *string
	max                               *int64
}

func (p *AttachedStorageDomainDisksServiceListRequest) Header(key, value string) *AttachedStorageDomainDisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Query(key, value string) *AttachedStorageDomainDisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Follow(follow string) *AttachedStorageDomainDisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Max(max int64) *AttachedStorageDomainDisksServiceListRequest {
	p.max = &max
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Send() (*AttachedStorageDomainDisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDisksService.connection.URL(), p.AttachedStorageDomainDisksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDisksServiceListResponse{disks: result}, nil
}

func (p *AttachedStorageDomainDisksServiceListRequest) MustSend() *AttachedStorageDomainDisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve the list of disks that are available in the storage domain.
//
type AttachedStorageDomainDisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *AttachedStorageDomainDisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Retrieve the list of disks that are available in the storage domain.
//
func (p *AttachedStorageDomainDisksService) List() *AttachedStorageDomainDisksServiceListRequest {
	return &AttachedStorageDomainDisksServiceListRequest{AttachedStorageDomainDisksService: p}
}

//
// Reference to the service that manages a specific disk.
//
func (op *AttachedStorageDomainDisksService) DiskService(id string) *AttachedStorageDomainDiskService {
	return NewAttachedStorageDomainDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainDisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *AttachedStorageDomainDisksService) String() string {
	return fmt.Sprintf("AttachedStorageDomainDisksService:%s", op.path)
}

//
//
type AttachedStorageDomainService struct {
	BaseService
}

func NewAttachedStorageDomainService(connection *Connection, path string) *AttachedStorageDomainService {
	var result AttachedStorageDomainService
	result.connection = connection
	result.path = path
	return &result
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type AttachedStorageDomainServiceActivateRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
}

func (p *AttachedStorageDomainServiceActivateRequest) Header(key, value string) *AttachedStorageDomainServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Query(key, value string) *AttachedStorageDomainServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Async(async bool) *AttachedStorageDomainServiceActivateRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Send() (*AttachedStorageDomainServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainServiceActivateResponse), nil
}

func (p *AttachedStorageDomainServiceActivateRequest) MustSend() *AttachedStorageDomainServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type AttachedStorageDomainServiceActivateResponse struct {
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *AttachedStorageDomainService) Activate() *AttachedStorageDomainServiceActivateRequest {
	return &AttachedStorageDomainServiceActivateRequest{AttachedStorageDomainService: p}
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
type AttachedStorageDomainServiceDeactivateRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
	force                        *bool
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Header(key, value string) *AttachedStorageDomainServiceDeactivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Query(key, value string) *AttachedStorageDomainServiceDeactivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Async(async bool) *AttachedStorageDomainServiceDeactivateRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Force(force bool) *AttachedStorageDomainServiceDeactivateRequest {
	p.force = &force
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Send() (*AttachedStorageDomainServiceDeactivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/deactivate", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainServiceDeactivateResponse), nil
}

func (p *AttachedStorageDomainServiceDeactivateRequest) MustSend() *AttachedStorageDomainServiceDeactivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
type AttachedStorageDomainServiceDeactivateResponse struct {
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
func (p *AttachedStorageDomainService) Deactivate() *AttachedStorageDomainServiceDeactivateRequest {
	return &AttachedStorageDomainServiceDeactivateRequest{AttachedStorageDomainService: p}
}

//
//
type AttachedStorageDomainServiceGetRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
}

func (p *AttachedStorageDomainServiceGetRequest) Header(key, value string) *AttachedStorageDomainServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Query(key, value string) *AttachedStorageDomainServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Follow(follow string) *AttachedStorageDomainServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Send() (*AttachedStorageDomainServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainServiceGetResponse{storageDomain: result}, nil
}

func (p *AttachedStorageDomainServiceGetRequest) MustSend() *AttachedStorageDomainServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AttachedStorageDomainServiceGetResponse struct {
	storageDomain *StorageDomain
}

func (p *AttachedStorageDomainServiceGetResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *AttachedStorageDomainServiceGetResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
//
func (p *AttachedStorageDomainService) Get() *AttachedStorageDomainServiceGetRequest {
	return &AttachedStorageDomainServiceGetRequest{AttachedStorageDomainService: p}
}

//
//
type AttachedStorageDomainServiceRemoveRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
}

func (p *AttachedStorageDomainServiceRemoveRequest) Header(key, value string) *AttachedStorageDomainServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Query(key, value string) *AttachedStorageDomainServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Async(async bool) *AttachedStorageDomainServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Send() (*AttachedStorageDomainServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AttachedStorageDomainServiceRemoveResponse), nil
}

func (p *AttachedStorageDomainServiceRemoveRequest) MustSend() *AttachedStorageDomainServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AttachedStorageDomainServiceRemoveResponse struct {
}

//
//
func (p *AttachedStorageDomainService) Remove() *AttachedStorageDomainServiceRemoveRequest {
	return &AttachedStorageDomainServiceRemoveRequest{AttachedStorageDomainService: p}
}

//
//
func (op *AttachedStorageDomainService) DisksService() *AttachedStorageDomainDisksService {
	return NewAttachedStorageDomainDisksService(op.connection, fmt.Sprintf("%s/disks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "disks" {
		return op.DisksService(), nil
	}
	if strings.HasPrefix(path, "disks/") {
		return op.DisksService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AttachedStorageDomainService) String() string {
	return fmt.Sprintf("AttachedStorageDomainService:%s", op.path)
}

//
// Manages the storage domains attached to a data center.
//
type AttachedStorageDomainsService struct {
	BaseService
}

func NewAttachedStorageDomainsService(connection *Connection, path string) *AttachedStorageDomainsService {
	var result AttachedStorageDomainsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Attaches an existing storage domain to the data center.
//
type AttachedStorageDomainsServiceAddRequest struct {
	AttachedStorageDomainsService *AttachedStorageDomainsService
	header                        map[string]string
	query                         map[string]string
	storageDomain                 *StorageDomain
}

func (p *AttachedStorageDomainsServiceAddRequest) Header(key, value string) *AttachedStorageDomainsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainsServiceAddRequest) Query(key, value string) *AttachedStorageDomainsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainsServiceAddRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainsServiceAddRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainsServiceAddRequest) Send() (*AttachedStorageDomainsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainsService.connection.URL(), p.AttachedStorageDomainsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageDomainWriteOne(writer, p.storageDomain, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainsServiceAddResponse{storageDomain: result}, nil
}

func (p *AttachedStorageDomainsServiceAddRequest) MustSend() *AttachedStorageDomainsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Attaches an existing storage domain to the data center.
//
type AttachedStorageDomainsServiceAddResponse struct {
	storageDomain *StorageDomain
}

func (p *AttachedStorageDomainsServiceAddResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *AttachedStorageDomainsServiceAddResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Attaches an existing storage domain to the data center.
//
func (p *AttachedStorageDomainsService) Add() *AttachedStorageDomainsServiceAddRequest {
	return &AttachedStorageDomainsServiceAddRequest{AttachedStorageDomainsService: p}
}

//
// Returns the list of storage domains attached to the data center.
// The order of the returned storage domains isn't guaranteed.
//
type AttachedStorageDomainsServiceListRequest struct {
	AttachedStorageDomainsService *AttachedStorageDomainsService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
	max                           *int64
}

func (p *AttachedStorageDomainsServiceListRequest) Header(key, value string) *AttachedStorageDomainsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainsServiceListRequest) Query(key, value string) *AttachedStorageDomainsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainsServiceListRequest) Follow(follow string) *AttachedStorageDomainsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainsServiceListRequest) Max(max int64) *AttachedStorageDomainsServiceListRequest {
	p.max = &max
	return p
}

func (p *AttachedStorageDomainsServiceListRequest) Send() (*AttachedStorageDomainsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainsService.connection.URL(), p.AttachedStorageDomainsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainsServiceListResponse{storageDomains: result}, nil
}

func (p *AttachedStorageDomainsServiceListRequest) MustSend() *AttachedStorageDomainsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of storage domains attached to the data center.
// The order of the returned storage domains isn't guaranteed.
//
type AttachedStorageDomainsServiceListResponse struct {
	storageDomains *StorageDomainSlice
}

func (p *AttachedStorageDomainsServiceListResponse) StorageDomains() (*StorageDomainSlice, bool) {
	if p.storageDomains != nil {
		return p.storageDomains, true
	}
	return nil, false
}

func (p *AttachedStorageDomainsServiceListResponse) MustStorageDomains() *StorageDomainSlice {
	if p.storageDomains == nil {
		panic("storageDomains in response does not exist")
	}
	return p.storageDomains
}

//
// Returns the list of storage domains attached to the data center.
// The order of the returned storage domains isn't guaranteed.
//
func (p *AttachedStorageDomainsService) List() *AttachedStorageDomainsServiceListRequest {
	return &AttachedStorageDomainsServiceListRequest{AttachedStorageDomainsService: p}
}

//
//
func (op *AttachedStorageDomainsService) StorageDomainService(id string) *AttachedStorageDomainService {
	return NewAttachedStorageDomainService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.StorageDomainService(path), nil
	}
	return op.StorageDomainService(path[:index]).Service(path[index+1:])
}

func (op *AttachedStorageDomainsService) String() string {
	return fmt.Sprintf("AttachedStorageDomainsService:%s", op.path)
}

//
//
type BalanceService struct {
	BaseService
}

func NewBalanceService(connection *Connection, path string) *BalanceService {
	var result BalanceService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type BalanceServiceGetRequest struct {
	BalanceService *BalanceService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
}

func (p *BalanceServiceGetRequest) Header(key, value string) *BalanceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalanceServiceGetRequest) Query(key, value string) *BalanceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalanceServiceGetRequest) Filter(filter bool) *BalanceServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *BalanceServiceGetRequest) Follow(follow string) *BalanceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *BalanceServiceGetRequest) Send() (*BalanceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalanceService.connection.URL(), p.BalanceService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalanceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalanceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalanceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalanceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalanceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BalanceServiceGetResponse{balance: result}, nil
}

func (p *BalanceServiceGetRequest) MustSend() *BalanceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type BalanceServiceGetResponse struct {
	balance *Balance
}

func (p *BalanceServiceGetResponse) Balance() (*Balance, bool) {
	if p.balance != nil {
		return p.balance, true
	}
	return nil, false
}

func (p *BalanceServiceGetResponse) MustBalance() *Balance {
	if p.balance == nil {
		panic("balance in response does not exist")
	}
	return p.balance
}

//
//
func (p *BalanceService) Get() *BalanceServiceGetRequest {
	return &BalanceServiceGetRequest{BalanceService: p}
}

//
//
type BalanceServiceRemoveRequest struct {
	BalanceService *BalanceService
	header         map[string]string
	query          map[string]string
	async          *bool
}

func (p *BalanceServiceRemoveRequest) Header(key, value string) *BalanceServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalanceServiceRemoveRequest) Query(key, value string) *BalanceServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalanceServiceRemoveRequest) Async(async bool) *BalanceServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *BalanceServiceRemoveRequest) Send() (*BalanceServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalanceService.connection.URL(), p.BalanceService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalanceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalanceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalanceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalanceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalanceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(BalanceServiceRemoveResponse), nil
}

func (p *BalanceServiceRemoveRequest) MustSend() *BalanceServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type BalanceServiceRemoveResponse struct {
}

//
//
func (p *BalanceService) Remove() *BalanceServiceRemoveRequest {
	return &BalanceServiceRemoveRequest{BalanceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *BalanceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *BalanceService) String() string {
	return fmt.Sprintf("BalanceService:%s", op.path)
}

//
//
type BalancesService struct {
	BaseService
}

func NewBalancesService(connection *Connection, path string) *BalancesService {
	var result BalancesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a balance module to a specified user defined scheduling policy.
//
type BalancesServiceAddRequest struct {
	BalancesService *BalancesService
	header          map[string]string
	query           map[string]string
	balance         *Balance
}

func (p *BalancesServiceAddRequest) Header(key, value string) *BalancesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalancesServiceAddRequest) Query(key, value string) *BalancesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalancesServiceAddRequest) Balance(balance *Balance) *BalancesServiceAddRequest {
	p.balance = balance
	return p
}

func (p *BalancesServiceAddRequest) Send() (*BalancesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalancesService.connection.URL(), p.BalancesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLBalanceWriteOne(writer, p.balance, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalancesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalancesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalancesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalancesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalancesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BalancesServiceAddResponse{balance: result}, nil
}

func (p *BalancesServiceAddRequest) MustSend() *BalancesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a balance module to a specified user defined scheduling policy.
//
type BalancesServiceAddResponse struct {
	balance *Balance
}

func (p *BalancesServiceAddResponse) Balance() (*Balance, bool) {
	if p.balance != nil {
		return p.balance, true
	}
	return nil, false
}

func (p *BalancesServiceAddResponse) MustBalance() *Balance {
	if p.balance == nil {
		panic("balance in response does not exist")
	}
	return p.balance
}

//
// Add a balance module to a specified user defined scheduling policy.
//
func (p *BalancesService) Add() *BalancesServiceAddRequest {
	return &BalancesServiceAddRequest{BalancesService: p}
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
type BalancesServiceListRequest struct {
	BalancesService *BalancesService
	header          map[string]string
	query           map[string]string
	filter          *bool
	follow          *string
	max             *int64
}

func (p *BalancesServiceListRequest) Header(key, value string) *BalancesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalancesServiceListRequest) Query(key, value string) *BalancesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalancesServiceListRequest) Filter(filter bool) *BalancesServiceListRequest {
	p.filter = &filter
	return p
}

func (p *BalancesServiceListRequest) Follow(follow string) *BalancesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *BalancesServiceListRequest) Max(max int64) *BalancesServiceListRequest {
	p.max = &max
	return p
}

func (p *BalancesServiceListRequest) Send() (*BalancesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalancesService.connection.URL(), p.BalancesService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalancesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalancesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalancesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalancesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalancesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &BalancesServiceListResponse{balances: result}, nil
}

func (p *BalancesServiceListRequest) MustSend() *BalancesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
type BalancesServiceListResponse struct {
	balances *BalanceSlice
}

func (p *BalancesServiceListResponse) Balances() (*BalanceSlice, bool) {
	if p.balances != nil {
		return p.balances, true
	}
	return nil, false
}

func (p *BalancesServiceListResponse) MustBalances() *BalanceSlice {
	if p.balances == nil {
		panic("balances in response does not exist")
	}
	return p.balances
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
func (p *BalancesService) List() *BalancesServiceListRequest {
	return &BalancesServiceListRequest{BalancesService: p}
}

//
//
func (op *BalancesService) BalanceService(id string) *BalanceService {
	return NewBalanceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *BalancesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.BalanceService(path), nil
	}
	return op.BalanceService(path[:index]).Service(path[index+1:])
}

func (op *BalancesService) String() string {
	return fmt.Sprintf("BalancesService:%s", op.path)
}

//
// A service to manage a bookmark.
//
type BookmarkService struct {
	BaseService
}

func NewBookmarkService(connection *Connection, path string) *BookmarkService {
	var result BookmarkService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get a bookmark.
// An example for getting a bookmark:
// [source]
// ----
// GET /ovirt-engine/api/bookmarks/123
// ----
// [source,xml]
// ----
// <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
//   <name>example_vm</name>
//   <value>vm: name=example*</value>
// </bookmark>
// ----
//
type BookmarkServiceGetRequest struct {
	BookmarkService *BookmarkService
	header          map[string]string
	query           map[string]string
	follow          *string
}

func (p *BookmarkServiceGetRequest) Header(key, value string) *BookmarkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BookmarkServiceGetRequest) Query(key, value string) *BookmarkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BookmarkServiceGetRequest) Follow(follow string) *BookmarkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *BookmarkServiceGetRequest) Send() (*BookmarkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BookmarkService.connection.URL(), p.BookmarkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BookmarkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BookmarkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BookmarkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BookmarkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BookmarkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBookmarkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BookmarkServiceGetResponse{bookmark: result}, nil
}

func (p *BookmarkServiceGetRequest) MustSend() *BookmarkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get a bookmark.
// An example for getting a bookmark:
// [source]
// ----
// GET /ovirt-engine/api/bookmarks/123
// ----
// [source,xml]
// ----
// <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
//   <name>example_vm</name>
//   <value>vm: name=example*</value>
// </bookmark>
// ----
//
type BookmarkServiceGetResponse struct {
	bookmark *Bookmark
}

func (p *BookmarkServiceGetResponse) Bookmark() (*Bookmark, bool) {
	if p.bookmark != nil {
		return p.bookmark, true
	}
	return nil, false
}

func (p *BookmarkServiceGetResponse) MustBookmark() *Bookmark {
	if p.bookmark == nil {
		panic("bookmark in response does not exist")
	}
	return p.bookmark
}

//
// Get a bookmark.
// An example for getting a bookmark:
// [source]
// ----
// GET /ovirt-engine/api/bookmarks/123
// ----
// [source,xml]
// ----
// <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
//   <name>example_vm</name>
//   <value>vm: name=example*</value>
// </bookmark>
// ----
//
func (p *BookmarkService) Get() *BookmarkServiceGetRequest {
	return &BookmarkServiceGetRequest{BookmarkService: p}
}

//
// Remove a bookmark.
// An example for removing a bookmark:
// [source]
// ----
// DELETE /ovirt-engine/api/bookmarks/123
// ----
//
type BookmarkServiceRemoveRequest struct {
	BookmarkService *BookmarkService
	header          map[string]string
	query           map[string]string
	async           *bool
}

func (p *BookmarkServiceRemoveRequest) Header(key, value string) *BookmarkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BookmarkServiceRemoveRequest) Query(key, value string) *BookmarkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BookmarkServiceRemoveRequest) Async(async bool) *BookmarkServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *BookmarkServiceRemoveRequest) Send() (*BookmarkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BookmarkService.connection.URL(), p.BookmarkService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BookmarkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BookmarkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BookmarkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BookmarkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BookmarkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(BookmarkServiceRemoveResponse), nil
}

func (p *BookmarkServiceRemoveRequest) MustSend() *BookmarkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove a bookmark.
// An example for removing a bookmark:
// [source]
// ----
// DELETE /ovirt-engine/api/bookmarks/123
// ----
//
type BookmarkServiceRemoveResponse struct {
}

//
// Remove a bookmark.
// An example for removing a bookmark:
// [source]
// ----
// DELETE /ovirt-engine/api/bookmarks/123
// ----
//
func (p *BookmarkService) Remove() *BookmarkServiceRemoveRequest {
	return &BookmarkServiceRemoveRequest{BookmarkService: p}
}

//
// Update a bookmark.
// An example for updating a bookmark:
// [source]
// ----
// PUT /ovirt-engine/api/bookmarks/123
// ----
// With the request body:
// [source,xml]
// ----
// <bookmark>
//   <name>new_example_vm</name>
//   <value>vm: name=new_example*</value>
// </bookmark>
// ----
//
type BookmarkServiceUpdateRequest struct {
	BookmarkService *BookmarkService
	header          map[string]string
	query           map[string]string
	async           *bool
	bookmark        *Bookmark
}

func (p *BookmarkServiceUpdateRequest) Header(key, value string) *BookmarkServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BookmarkServiceUpdateRequest) Query(key, value string) *BookmarkServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BookmarkServiceUpdateRequest) Async(async bool) *BookmarkServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *BookmarkServiceUpdateRequest) Bookmark(bookmark *Bookmark) *BookmarkServiceUpdateRequest {
	p.bookmark = bookmark
	return p
}

func (p *BookmarkServiceUpdateRequest) Send() (*BookmarkServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BookmarkService.connection.URL(), p.BookmarkService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLBookmarkWriteOne(writer, p.bookmark, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BookmarkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BookmarkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BookmarkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BookmarkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BookmarkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBookmarkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BookmarkServiceUpdateResponse{bookmark: result}, nil
}

func (p *BookmarkServiceUpdateRequest) MustSend() *BookmarkServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update a bookmark.
// An example for updating a bookmark:
// [source]
// ----
// PUT /ovirt-engine/api/bookmarks/123
// ----
// With the request body:
// [source,xml]
// ----
// <bookmark>
//   <name>new_example_vm</name>
//   <value>vm: name=new_example*</value>
// </bookmark>
// ----
//
type BookmarkServiceUpdateResponse struct {
	bookmark *Bookmark
}

func (p *BookmarkServiceUpdateResponse) Bookmark() (*Bookmark, bool) {
	if p.bookmark != nil {
		return p.bookmark, true
	}
	return nil, false
}

func (p *BookmarkServiceUpdateResponse) MustBookmark() *Bookmark {
	if p.bookmark == nil {
		panic("bookmark in response does not exist")
	}
	return p.bookmark
}

//
// Update a bookmark.
// An example for updating a bookmark:
// [source]
// ----
// PUT /ovirt-engine/api/bookmarks/123
// ----
// With the request body:
// [source,xml]
// ----
// <bookmark>
//   <name>new_example_vm</name>
//   <value>vm: name=new_example*</value>
// </bookmark>
// ----
//
func (p *BookmarkService) Update() *BookmarkServiceUpdateRequest {
	return &BookmarkServiceUpdateRequest{BookmarkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *BookmarkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *BookmarkService) String() string {
	return fmt.Sprintf("BookmarkService:%s", op.path)
}

//
// A service to manage bookmarks.
//
type BookmarksService struct {
	BaseService
}

func NewBookmarksService(connection *Connection, path string) *BookmarksService {
	var result BookmarksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adding a new bookmark.
// Example of adding a bookmark:
// [source]
// ----
// POST /ovirt-engine/api/bookmarks
// ----
// [source,xml]
// ----
// <bookmark>
//   <name>new_example_vm</name>
//   <value>vm: name=new_example*</value>
// </bookmark>
// ----
//
type BookmarksServiceAddRequest struct {
	BookmarksService *BookmarksService
	header           map[string]string
	query            map[string]string
	bookmark         *Bookmark
}

func (p *BookmarksServiceAddRequest) Header(key, value string) *BookmarksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BookmarksServiceAddRequest) Query(key, value string) *BookmarksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BookmarksServiceAddRequest) Bookmark(bookmark *Bookmark) *BookmarksServiceAddRequest {
	p.bookmark = bookmark
	return p
}

func (p *BookmarksServiceAddRequest) Send() (*BookmarksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BookmarksService.connection.URL(), p.BookmarksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLBookmarkWriteOne(writer, p.bookmark, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BookmarksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BookmarksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BookmarksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BookmarksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BookmarksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBookmarkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BookmarksServiceAddResponse{bookmark: result}, nil
}

func (p *BookmarksServiceAddRequest) MustSend() *BookmarksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adding a new bookmark.
// Example of adding a bookmark:
// [source]
// ----
// POST /ovirt-engine/api/bookmarks
// ----
// [source,xml]
// ----
// <bookmark>
//   <name>new_example_vm</name>
//   <value>vm: name=new_example*</value>
// </bookmark>
// ----
//
type BookmarksServiceAddResponse struct {
	bookmark *Bookmark
}

func (p *BookmarksServiceAddResponse) Bookmark() (*Bookmark, bool) {
	if p.bookmark != nil {
		return p.bookmark, true
	}
	return nil, false
}

func (p *BookmarksServiceAddResponse) MustBookmark() *Bookmark {
	if p.bookmark == nil {
		panic("bookmark in response does not exist")
	}
	return p.bookmark
}

//
// Adding a new bookmark.
// Example of adding a bookmark:
// [source]
// ----
// POST /ovirt-engine/api/bookmarks
// ----
// [source,xml]
// ----
// <bookmark>
//   <name>new_example_vm</name>
//   <value>vm: name=new_example*</value>
// </bookmark>
// ----
//
func (p *BookmarksService) Add() *BookmarksServiceAddRequest {
	return &BookmarksServiceAddRequest{BookmarksService: p}
}

//
// Listing all the available bookmarks.
// Example of listing bookmarks:
// [source]
// ----
// GET /ovirt-engine/api/bookmarks
// ----
// [source,xml]
// ----
// <bookmarks>
//   <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
//     <name>database</name>
//     <value>vm: name=database*</value>
//   </bookmark>
//   <bookmark href="/ovirt-engine/api/bookmarks/456" id="456">
//     <name>example</name>
//     <value>vm: name=example*</value>
//   </bookmark>
// </bookmarks>
// ----
// The order of the returned bookmarks isn't guaranteed.
//
type BookmarksServiceListRequest struct {
	BookmarksService *BookmarksService
	header           map[string]string
	query            map[string]string
	follow           *string
	max              *int64
}

func (p *BookmarksServiceListRequest) Header(key, value string) *BookmarksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BookmarksServiceListRequest) Query(key, value string) *BookmarksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BookmarksServiceListRequest) Follow(follow string) *BookmarksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *BookmarksServiceListRequest) Max(max int64) *BookmarksServiceListRequest {
	p.max = &max
	return p
}

func (p *BookmarksServiceListRequest) Send() (*BookmarksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BookmarksService.connection.URL(), p.BookmarksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BookmarksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BookmarksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BookmarksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BookmarksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BookmarksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBookmarkReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &BookmarksServiceListResponse{bookmarks: result}, nil
}

func (p *BookmarksServiceListRequest) MustSend() *BookmarksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Listing all the available bookmarks.
// Example of listing bookmarks:
// [source]
// ----
// GET /ovirt-engine/api/bookmarks
// ----
// [source,xml]
// ----
// <bookmarks>
//   <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
//     <name>database</name>
//     <value>vm: name=database*</value>
//   </bookmark>
//   <bookmark href="/ovirt-engine/api/bookmarks/456" id="456">
//     <name>example</name>
//     <value>vm: name=example*</value>
//   </bookmark>
// </bookmarks>
// ----
// The order of the returned bookmarks isn't guaranteed.
//
type BookmarksServiceListResponse struct {
	bookmarks *BookmarkSlice
}

func (p *BookmarksServiceListResponse) Bookmarks() (*BookmarkSlice, bool) {
	if p.bookmarks != nil {
		return p.bookmarks, true
	}
	return nil, false
}

func (p *BookmarksServiceListResponse) MustBookmarks() *BookmarkSlice {
	if p.bookmarks == nil {
		panic("bookmarks in response does not exist")
	}
	return p.bookmarks
}

//
// Listing all the available bookmarks.
// Example of listing bookmarks:
// [source]
// ----
// GET /ovirt-engine/api/bookmarks
// ----
// [source,xml]
// ----
// <bookmarks>
//   <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
//     <name>database</name>
//     <value>vm: name=database*</value>
//   </bookmark>
//   <bookmark href="/ovirt-engine/api/bookmarks/456" id="456">
//     <name>example</name>
//     <value>vm: name=example*</value>
//   </bookmark>
// </bookmarks>
// ----
// The order of the returned bookmarks isn't guaranteed.
//
func (p *BookmarksService) List() *BookmarksServiceListRequest {
	return &BookmarksServiceListRequest{BookmarksService: p}
}

//
// A reference to the service managing a specific bookmark.
//
func (op *BookmarksService) BookmarkService(id string) *BookmarkService {
	return NewBookmarkService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *BookmarksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.BookmarkService(path), nil
	}
	return op.BookmarkService(path[:index]).Service(path[index+1:])
}

func (op *BookmarksService) String() string {
	return fmt.Sprintf("BookmarksService:%s", op.path)
}

//
// Represents a feature enabled for the cluster.
//
type ClusterEnabledFeatureService struct {
	BaseService
}

func NewClusterEnabledFeatureService(connection *Connection, path string) *ClusterEnabledFeatureService {
	var result ClusterEnabledFeatureService
	result.connection = connection
	result.path = path
	return &result
}

//
// Provides the information about the cluster feature enabled.
// For example, to find details of the enabled feature `456` for cluster `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123/enabledfeatures/456
// ----
// That will return a <<types/cluster_feature, ClusterFeature>> object containing the name:
// [source,xml]
// ----
// <cluster_feature id="456">
//   <name>libgfapi_supported</name>
// </cluster_feature>
// ----
//
type ClusterEnabledFeatureServiceGetRequest struct {
	ClusterEnabledFeatureService *ClusterEnabledFeatureService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
}

func (p *ClusterEnabledFeatureServiceGetRequest) Header(key, value string) *ClusterEnabledFeatureServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterEnabledFeatureServiceGetRequest) Query(key, value string) *ClusterEnabledFeatureServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterEnabledFeatureServiceGetRequest) Follow(follow string) *ClusterEnabledFeatureServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ClusterEnabledFeatureServiceGetRequest) Send() (*ClusterEnabledFeatureServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterEnabledFeatureService.connection.URL(), p.ClusterEnabledFeatureService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterEnabledFeatureService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterEnabledFeatureService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterEnabledFeatureService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterEnabledFeatureService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterEnabledFeatureService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterFeatureReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterEnabledFeatureServiceGetResponse{feature: result}, nil
}

func (p *ClusterEnabledFeatureServiceGetRequest) MustSend() *ClusterEnabledFeatureServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Provides the information about the cluster feature enabled.
// For example, to find details of the enabled feature `456` for cluster `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123/enabledfeatures/456
// ----
// That will return a <<types/cluster_feature, ClusterFeature>> object containing the name:
// [source,xml]
// ----
// <cluster_feature id="456">
//   <name>libgfapi_supported</name>
// </cluster_feature>
// ----
//
type ClusterEnabledFeatureServiceGetResponse struct {
	feature *ClusterFeature
}

func (p *ClusterEnabledFeatureServiceGetResponse) Feature() (*ClusterFeature, bool) {
	if p.feature != nil {
		return p.feature, true
	}
	return nil, false
}

func (p *ClusterEnabledFeatureServiceGetResponse) MustFeature() *ClusterFeature {
	if p.feature == nil {
		panic("feature in response does not exist")
	}
	return p.feature
}

//
// Provides the information about the cluster feature enabled.
// For example, to find details of the enabled feature `456` for cluster `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123/enabledfeatures/456
// ----
// That will return a <<types/cluster_feature, ClusterFeature>> object containing the name:
// [source,xml]
// ----
// <cluster_feature id="456">
//   <name>libgfapi_supported</name>
// </cluster_feature>
// ----
//
func (p *ClusterEnabledFeatureService) Get() *ClusterEnabledFeatureServiceGetRequest {
	return &ClusterEnabledFeatureServiceGetRequest{ClusterEnabledFeatureService: p}
}

//
// Disables a cluster feature.
// For example, to disable the feature `456` of cluster `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/123/enabledfeatures/456
// ----
//
type ClusterEnabledFeatureServiceRemoveRequest struct {
	ClusterEnabledFeatureService *ClusterEnabledFeatureService
	header                       map[string]string
	query                        map[string]string
}

func (p *ClusterEnabledFeatureServiceRemoveRequest) Header(key, value string) *ClusterEnabledFeatureServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterEnabledFeatureServiceRemoveRequest) Query(key, value string) *ClusterEnabledFeatureServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterEnabledFeatureServiceRemoveRequest) Send() (*ClusterEnabledFeatureServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterEnabledFeatureService.connection.URL(), p.ClusterEnabledFeatureService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterEnabledFeatureService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterEnabledFeatureService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterEnabledFeatureService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterEnabledFeatureService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterEnabledFeatureService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(ClusterEnabledFeatureServiceRemoveResponse), nil
}

func (p *ClusterEnabledFeatureServiceRemoveRequest) MustSend() *ClusterEnabledFeatureServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Disables a cluster feature.
// For example, to disable the feature `456` of cluster `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/123/enabledfeatures/456
// ----
//
type ClusterEnabledFeatureServiceRemoveResponse struct {
}

//
// Disables a cluster feature.
// For example, to disable the feature `456` of cluster `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/123/enabledfeatures/456
// ----
//
func (p *ClusterEnabledFeatureService) Remove() *ClusterEnabledFeatureServiceRemoveRequest {
	return &ClusterEnabledFeatureServiceRemoveRequest{ClusterEnabledFeatureService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterEnabledFeatureService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ClusterEnabledFeatureService) String() string {
	return fmt.Sprintf("ClusterEnabledFeatureService:%s", op.path)
}

//
// Provides information about the additional features that are enabled for this cluster.
// The features that are enabled are the available features for the cluster level
//
type ClusterEnabledFeaturesService struct {
	BaseService
}

func NewClusterEnabledFeaturesService(connection *Connection, path string) *ClusterEnabledFeaturesService {
	var result ClusterEnabledFeaturesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Enable an additional feature for a cluster.
// For example, to enable a feature `456` on cluster `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/enabledfeatures
// ----
// The request body should look like this:
// [source,xml]
// ----
// <cluster_feature id="456"/>
// ----
//
type ClusterEnabledFeaturesServiceAddRequest struct {
	ClusterEnabledFeaturesService *ClusterEnabledFeaturesService
	header                        map[string]string
	query                         map[string]string
	feature                       *ClusterFeature
}

func (p *ClusterEnabledFeaturesServiceAddRequest) Header(key, value string) *ClusterEnabledFeaturesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterEnabledFeaturesServiceAddRequest) Query(key, value string) *ClusterEnabledFeaturesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterEnabledFeaturesServiceAddRequest) Feature(feature *ClusterFeature) *ClusterEnabledFeaturesServiceAddRequest {
	p.feature = feature
	return p
}

func (p *ClusterEnabledFeaturesServiceAddRequest) Send() (*ClusterEnabledFeaturesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterEnabledFeaturesService.connection.URL(), p.ClusterEnabledFeaturesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLClusterFeatureWriteOne(writer, p.feature, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterEnabledFeaturesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterEnabledFeaturesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterEnabledFeaturesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterEnabledFeaturesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterEnabledFeaturesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterFeatureReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterEnabledFeaturesServiceAddResponse{feature: result}, nil
}

func (p *ClusterEnabledFeaturesServiceAddRequest) MustSend() *ClusterEnabledFeaturesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Enable an additional feature for a cluster.
// For example, to enable a feature `456` on cluster `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/enabledfeatures
// ----
// The request body should look like this:
// [source,xml]
// ----
// <cluster_feature id="456"/>
// ----
//
type ClusterEnabledFeaturesServiceAddResponse struct {
	feature *ClusterFeature
}

func (p *ClusterEnabledFeaturesServiceAddResponse) Feature() (*ClusterFeature, bool) {
	if p.feature != nil {
		return p.feature, true
	}
	return nil, false
}

func (p *ClusterEnabledFeaturesServiceAddResponse) MustFeature() *ClusterFeature {
	if p.feature == nil {
		panic("feature in response does not exist")
	}
	return p.feature
}

//
// Enable an additional feature for a cluster.
// For example, to enable a feature `456` on cluster `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/enabledfeatures
// ----
// The request body should look like this:
// [source,xml]
// ----
// <cluster_feature id="456"/>
// ----
//
func (p *ClusterEnabledFeaturesService) Add() *ClusterEnabledFeaturesServiceAddRequest {
	return &ClusterEnabledFeaturesServiceAddRequest{ClusterEnabledFeaturesService: p}
}

//
// Lists the additional features enabled for the cluster.
// For example, to get the features enabled for cluster `123` send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123/enabledfeatures
// ----
// This will return a list of features:
// [source,xml]
// ----
// <enabled_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </enabled_features>
// ----
//
type ClusterEnabledFeaturesServiceListRequest struct {
	ClusterEnabledFeaturesService *ClusterEnabledFeaturesService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
}

func (p *ClusterEnabledFeaturesServiceListRequest) Header(key, value string) *ClusterEnabledFeaturesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterEnabledFeaturesServiceListRequest) Query(key, value string) *ClusterEnabledFeaturesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterEnabledFeaturesServiceListRequest) Follow(follow string) *ClusterEnabledFeaturesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClusterEnabledFeaturesServiceListRequest) Send() (*ClusterEnabledFeaturesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterEnabledFeaturesService.connection.URL(), p.ClusterEnabledFeaturesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterEnabledFeaturesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterEnabledFeaturesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterEnabledFeaturesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterEnabledFeaturesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterEnabledFeaturesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterFeatureReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClusterEnabledFeaturesServiceListResponse{features: result}, nil
}

func (p *ClusterEnabledFeaturesServiceListRequest) MustSend() *ClusterEnabledFeaturesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the additional features enabled for the cluster.
// For example, to get the features enabled for cluster `123` send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123/enabledfeatures
// ----
// This will return a list of features:
// [source,xml]
// ----
// <enabled_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </enabled_features>
// ----
//
type ClusterEnabledFeaturesServiceListResponse struct {
	features *ClusterFeatureSlice
}

func (p *ClusterEnabledFeaturesServiceListResponse) Features() (*ClusterFeatureSlice, bool) {
	if p.features != nil {
		return p.features, true
	}
	return nil, false
}

func (p *ClusterEnabledFeaturesServiceListResponse) MustFeatures() *ClusterFeatureSlice {
	if p.features == nil {
		panic("features in response does not exist")
	}
	return p.features
}

//
// Lists the additional features enabled for the cluster.
// For example, to get the features enabled for cluster `123` send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123/enabledfeatures
// ----
// This will return a list of features:
// [source,xml]
// ----
// <enabled_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </enabled_features>
// ----
//
func (p *ClusterEnabledFeaturesService) List() *ClusterEnabledFeaturesServiceListRequest {
	return &ClusterEnabledFeaturesServiceListRequest{ClusterEnabledFeaturesService: p}
}

//
// A reference to the service that provides information about a specific
// feature enabled for the cluster.
//
func (op *ClusterEnabledFeaturesService) FeatureService(id string) *ClusterEnabledFeatureService {
	return NewClusterEnabledFeatureService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterEnabledFeaturesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FeatureService(path), nil
	}
	return op.FeatureService(path[:index]).Service(path[index+1:])
}

func (op *ClusterEnabledFeaturesService) String() string {
	return fmt.Sprintf("ClusterEnabledFeaturesService:%s", op.path)
}

//
// This service lists external providers.
//
type ClusterExternalProvidersService struct {
	BaseService
}

func NewClusterExternalProvidersService(connection *Connection, path string) *ClusterExternalProvidersService {
	var result ClusterExternalProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of external providers.
// The order of the returned list of providers is not guaranteed.
//
type ClusterExternalProvidersServiceListRequest struct {
	ClusterExternalProvidersService *ClusterExternalProvidersService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
}

func (p *ClusterExternalProvidersServiceListRequest) Header(key, value string) *ClusterExternalProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterExternalProvidersServiceListRequest) Query(key, value string) *ClusterExternalProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterExternalProvidersServiceListRequest) Follow(follow string) *ClusterExternalProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClusterExternalProvidersServiceListRequest) Send() (*ClusterExternalProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterExternalProvidersService.connection.URL(), p.ClusterExternalProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterExternalProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterExternalProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterExternalProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterExternalProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterExternalProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClusterExternalProvidersServiceListResponse{providers: result}, nil
}

func (p *ClusterExternalProvidersServiceListRequest) MustSend() *ClusterExternalProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of external providers.
// The order of the returned list of providers is not guaranteed.
//
type ClusterExternalProvidersServiceListResponse struct {
	providers *ExternalProviderSlice
}

func (p *ClusterExternalProvidersServiceListResponse) Providers() (*ExternalProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *ClusterExternalProvidersServiceListResponse) MustProviders() *ExternalProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Returns the list of external providers.
// The order of the returned list of providers is not guaranteed.
//
func (p *ClusterExternalProvidersService) List() *ClusterExternalProvidersServiceListRequest {
	return &ClusterExternalProvidersServiceListRequest{ClusterExternalProvidersService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterExternalProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ClusterExternalProvidersService) String() string {
	return fmt.Sprintf("ClusterExternalProvidersService:%s", op.path)
}

//
// Represents a feature enabled for the cluster level
//
type ClusterFeatureService struct {
	BaseService
}

func NewClusterFeatureService(connection *Connection, path string) *ClusterFeatureService {
	var result ClusterFeatureService
	result.connection = connection
	result.path = path
	return &result
}

//
// Provides the information about the a cluster feature supported by a cluster level.
// For example, to find details of the cluster feature `456` for cluster level 4.1, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures/456
// ----
// That will return a <<types/cluster_feature, ClusterFeature>> object containing the name:
// [source,xml]
// ----
// <cluster_feature id="456">
//   <name>libgfapi_supported</name>
// </cluster_feature>
// ----
//
type ClusterFeatureServiceGetRequest struct {
	ClusterFeatureService *ClusterFeatureService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *ClusterFeatureServiceGetRequest) Header(key, value string) *ClusterFeatureServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterFeatureServiceGetRequest) Query(key, value string) *ClusterFeatureServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterFeatureServiceGetRequest) Follow(follow string) *ClusterFeatureServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ClusterFeatureServiceGetRequest) Send() (*ClusterFeatureServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterFeatureService.connection.URL(), p.ClusterFeatureService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterFeatureService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterFeatureService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterFeatureService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterFeatureService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterFeatureService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterFeatureReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterFeatureServiceGetResponse{feature: result}, nil
}

func (p *ClusterFeatureServiceGetRequest) MustSend() *ClusterFeatureServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Provides the information about the a cluster feature supported by a cluster level.
// For example, to find details of the cluster feature `456` for cluster level 4.1, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures/456
// ----
// That will return a <<types/cluster_feature, ClusterFeature>> object containing the name:
// [source,xml]
// ----
// <cluster_feature id="456">
//   <name>libgfapi_supported</name>
// </cluster_feature>
// ----
//
type ClusterFeatureServiceGetResponse struct {
	feature *ClusterFeature
}

func (p *ClusterFeatureServiceGetResponse) Feature() (*ClusterFeature, bool) {
	if p.feature != nil {
		return p.feature, true
	}
	return nil, false
}

func (p *ClusterFeatureServiceGetResponse) MustFeature() *ClusterFeature {
	if p.feature == nil {
		panic("feature in response does not exist")
	}
	return p.feature
}

//
// Provides the information about the a cluster feature supported by a cluster level.
// For example, to find details of the cluster feature `456` for cluster level 4.1, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures/456
// ----
// That will return a <<types/cluster_feature, ClusterFeature>> object containing the name:
// [source,xml]
// ----
// <cluster_feature id="456">
//   <name>libgfapi_supported</name>
// </cluster_feature>
// ----
//
func (p *ClusterFeatureService) Get() *ClusterFeatureServiceGetRequest {
	return &ClusterFeatureServiceGetRequest{ClusterFeatureService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterFeatureService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ClusterFeatureService) String() string {
	return fmt.Sprintf("ClusterFeatureService:%s", op.path)
}

//
// Provides information about the cluster features that are supported by a cluster level.
//
type ClusterFeaturesService struct {
	BaseService
}

func NewClusterFeaturesService(connection *Connection, path string) *ClusterFeaturesService {
	var result ClusterFeaturesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Lists the cluster features supported by the cluster level.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures
// ----
// This will return a list of cluster features supported by the cluster level:
// [source,xml]
// ----
// <cluster_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </cluster_features>
// ----
//
type ClusterFeaturesServiceListRequest struct {
	ClusterFeaturesService *ClusterFeaturesService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *ClusterFeaturesServiceListRequest) Header(key, value string) *ClusterFeaturesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterFeaturesServiceListRequest) Query(key, value string) *ClusterFeaturesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterFeaturesServiceListRequest) Follow(follow string) *ClusterFeaturesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClusterFeaturesServiceListRequest) Send() (*ClusterFeaturesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterFeaturesService.connection.URL(), p.ClusterFeaturesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterFeaturesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterFeaturesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterFeaturesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterFeaturesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterFeaturesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterFeatureReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClusterFeaturesServiceListResponse{features: result}, nil
}

func (p *ClusterFeaturesServiceListRequest) MustSend() *ClusterFeaturesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the cluster features supported by the cluster level.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures
// ----
// This will return a list of cluster features supported by the cluster level:
// [source,xml]
// ----
// <cluster_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </cluster_features>
// ----
//
type ClusterFeaturesServiceListResponse struct {
	features *ClusterFeatureSlice
}

func (p *ClusterFeaturesServiceListResponse) Features() (*ClusterFeatureSlice, bool) {
	if p.features != nil {
		return p.features, true
	}
	return nil, false
}

func (p *ClusterFeaturesServiceListResponse) MustFeatures() *ClusterFeatureSlice {
	if p.features == nil {
		panic("features in response does not exist")
	}
	return p.features
}

//
// Lists the cluster features supported by the cluster level.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures
// ----
// This will return a list of cluster features supported by the cluster level:
// [source,xml]
// ----
// <cluster_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </cluster_features>
// ----
//
func (p *ClusterFeaturesService) List() *ClusterFeaturesServiceListRequest {
	return &ClusterFeaturesServiceListRequest{ClusterFeaturesService: p}
}

//
// Reference to the service that provides information about a specific feature.
//
func (op *ClusterFeaturesService) FeatureService(id string) *ClusterFeatureService {
	return NewClusterFeatureService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterFeaturesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FeatureService(path), nil
	}
	return op.FeatureService(path[:index]).Service(path[index+1:])
}

func (op *ClusterFeaturesService) String() string {
	return fmt.Sprintf("ClusterFeaturesService:%s", op.path)
}

//
// Provides information about a specific cluster level. See the <<services/cluster_levels,ClusterLevels>> service for
// more information.
//
type ClusterLevelService struct {
	BaseService
}

func NewClusterLevelService(connection *Connection, path string) *ClusterLevelService {
	var result ClusterLevelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Provides the information about the capabilities of the specific cluster level managed by this service.
// For example, to find what CPU types are supported by level 3.6 you can send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/3.6
// ----
// That will return a <<types/cluster_level, ClusterLevel>> object containing the supported CPU types, and other
// information which describes the cluster level:
// [source,xml]
// ----
// <cluster_level id="3.6">
//   <cpu_types>
//     <cpu_type>
//       <name>Intel Nehalem Family</name>
//       <level>3</level>
//       <architecture>x86_64</architecture>
//     </cpu_type>
//     ...
//   </cpu_types>
//   <permits>
//     <permit id="1">
//       <name>create_vm</name>
//       <administrative>false</administrative>
//     </permit>
//     ...
//   </permits>
// </cluster_level>
// ----
//
type ClusterLevelServiceGetRequest struct {
	ClusterLevelService *ClusterLevelService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *ClusterLevelServiceGetRequest) Header(key, value string) *ClusterLevelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterLevelServiceGetRequest) Query(key, value string) *ClusterLevelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterLevelServiceGetRequest) Follow(follow string) *ClusterLevelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ClusterLevelServiceGetRequest) Send() (*ClusterLevelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterLevelService.connection.URL(), p.ClusterLevelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterLevelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterLevelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterLevelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterLevelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterLevelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterLevelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterLevelServiceGetResponse{level: result}, nil
}

func (p *ClusterLevelServiceGetRequest) MustSend() *ClusterLevelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Provides the information about the capabilities of the specific cluster level managed by this service.
// For example, to find what CPU types are supported by level 3.6 you can send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/3.6
// ----
// That will return a <<types/cluster_level, ClusterLevel>> object containing the supported CPU types, and other
// information which describes the cluster level:
// [source,xml]
// ----
// <cluster_level id="3.6">
//   <cpu_types>
//     <cpu_type>
//       <name>Intel Nehalem Family</name>
//       <level>3</level>
//       <architecture>x86_64</architecture>
//     </cpu_type>
//     ...
//   </cpu_types>
//   <permits>
//     <permit id="1">
//       <name>create_vm</name>
//       <administrative>false</administrative>
//     </permit>
//     ...
//   </permits>
// </cluster_level>
// ----
//
type ClusterLevelServiceGetResponse struct {
	level *ClusterLevel
}

func (p *ClusterLevelServiceGetResponse) Level() (*ClusterLevel, bool) {
	if p.level != nil {
		return p.level, true
	}
	return nil, false
}

func (p *ClusterLevelServiceGetResponse) MustLevel() *ClusterLevel {
	if p.level == nil {
		panic("level in response does not exist")
	}
	return p.level
}

//
// Provides the information about the capabilities of the specific cluster level managed by this service.
// For example, to find what CPU types are supported by level 3.6 you can send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/3.6
// ----
// That will return a <<types/cluster_level, ClusterLevel>> object containing the supported CPU types, and other
// information which describes the cluster level:
// [source,xml]
// ----
// <cluster_level id="3.6">
//   <cpu_types>
//     <cpu_type>
//       <name>Intel Nehalem Family</name>
//       <level>3</level>
//       <architecture>x86_64</architecture>
//     </cpu_type>
//     ...
//   </cpu_types>
//   <permits>
//     <permit id="1">
//       <name>create_vm</name>
//       <administrative>false</administrative>
//     </permit>
//     ...
//   </permits>
// </cluster_level>
// ----
//
func (p *ClusterLevelService) Get() *ClusterLevelServiceGetRequest {
	return &ClusterLevelServiceGetRequest{ClusterLevelService: p}
}

//
// Reference to the service that manages the collection of supported features for this cluster level.
//
func (op *ClusterLevelService) ClusterFeaturesService() *ClusterFeaturesService {
	return NewClusterFeaturesService(op.connection, fmt.Sprintf("%s/clusterfeatures", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterLevelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "clusterfeatures" {
		return op.ClusterFeaturesService(), nil
	}
	if strings.HasPrefix(path, "clusterfeatures/") {
		return op.ClusterFeaturesService().Service(path[16:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ClusterLevelService) String() string {
	return fmt.Sprintf("ClusterLevelService:%s", op.path)
}

//
// Provides information about the capabilities of different cluster levels supported by the engine. Version 4.0 of the
// engine supports levels 4.0 and 3.6. Each of these levels support different sets of CPU types, for example. This
// service provides that information.
//
type ClusterLevelsService struct {
	BaseService
}

func NewClusterLevelsService(connection *Connection, path string) *ClusterLevelsService {
	var result ClusterLevelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Lists the cluster levels supported by the system.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels
// ----
// This will return a list of available cluster levels.
// [source,xml]
// ----
// <cluster_levels>
//   <cluster_level id="4.0">
//      ...
//   </cluster_level>
//   ...
// </cluster_levels>
// ----
// The order of the returned cluster levels isn't guaranteed.
//
type ClusterLevelsServiceListRequest struct {
	ClusterLevelsService *ClusterLevelsService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *ClusterLevelsServiceListRequest) Header(key, value string) *ClusterLevelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterLevelsServiceListRequest) Query(key, value string) *ClusterLevelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterLevelsServiceListRequest) Follow(follow string) *ClusterLevelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClusterLevelsServiceListRequest) Send() (*ClusterLevelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterLevelsService.connection.URL(), p.ClusterLevelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterLevelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterLevelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterLevelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterLevelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterLevelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterLevelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClusterLevelsServiceListResponse{levels: result}, nil
}

func (p *ClusterLevelsServiceListRequest) MustSend() *ClusterLevelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the cluster levels supported by the system.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels
// ----
// This will return a list of available cluster levels.
// [source,xml]
// ----
// <cluster_levels>
//   <cluster_level id="4.0">
//      ...
//   </cluster_level>
//   ...
// </cluster_levels>
// ----
// The order of the returned cluster levels isn't guaranteed.
//
type ClusterLevelsServiceListResponse struct {
	levels *ClusterLevelSlice
}

func (p *ClusterLevelsServiceListResponse) Levels() (*ClusterLevelSlice, bool) {
	if p.levels != nil {
		return p.levels, true
	}
	return nil, false
}

func (p *ClusterLevelsServiceListResponse) MustLevels() *ClusterLevelSlice {
	if p.levels == nil {
		panic("levels in response does not exist")
	}
	return p.levels
}

//
// Lists the cluster levels supported by the system.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels
// ----
// This will return a list of available cluster levels.
// [source,xml]
// ----
// <cluster_levels>
//   <cluster_level id="4.0">
//      ...
//   </cluster_level>
//   ...
// </cluster_levels>
// ----
// The order of the returned cluster levels isn't guaranteed.
//
func (p *ClusterLevelsService) List() *ClusterLevelsServiceListRequest {
	return &ClusterLevelsServiceListRequest{ClusterLevelsService: p}
}

//
// Reference to the service that provides information about an specific cluster level.
//
func (op *ClusterLevelsService) LevelService(id string) *ClusterLevelService {
	return NewClusterLevelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterLevelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LevelService(path), nil
	}
	return op.LevelService(path[:index]).Service(path[index+1:])
}

func (op *ClusterLevelsService) String() string {
	return fmt.Sprintf("ClusterLevelsService:%s", op.path)
}

//
// A service to manage a specific cluster network.
//
type ClusterNetworkService struct {
	BaseService
}

func NewClusterNetworkService(connection *Connection, path string) *ClusterNetworkService {
	var result ClusterNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the cluster network details.
//
type ClusterNetworkServiceGetRequest struct {
	ClusterNetworkService *ClusterNetworkService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *ClusterNetworkServiceGetRequest) Header(key, value string) *ClusterNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworkServiceGetRequest) Query(key, value string) *ClusterNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworkServiceGetRequest) Follow(follow string) *ClusterNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ClusterNetworkServiceGetRequest) Send() (*ClusterNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworkService.connection.URL(), p.ClusterNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterNetworkServiceGetResponse{network: result}, nil
}

func (p *ClusterNetworkServiceGetRequest) MustSend() *ClusterNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the cluster network details.
//
type ClusterNetworkServiceGetResponse struct {
	network *Network
}

func (p *ClusterNetworkServiceGetResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *ClusterNetworkServiceGetResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Retrieves the cluster network details.
//
func (p *ClusterNetworkService) Get() *ClusterNetworkServiceGetRequest {
	return &ClusterNetworkServiceGetRequest{ClusterNetworkService: p}
}

//
// Unassigns the network from a cluster.
//
type ClusterNetworkServiceRemoveRequest struct {
	ClusterNetworkService *ClusterNetworkService
	header                map[string]string
	query                 map[string]string
}

func (p *ClusterNetworkServiceRemoveRequest) Header(key, value string) *ClusterNetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworkServiceRemoveRequest) Query(key, value string) *ClusterNetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworkServiceRemoveRequest) Send() (*ClusterNetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworkService.connection.URL(), p.ClusterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(ClusterNetworkServiceRemoveResponse), nil
}

func (p *ClusterNetworkServiceRemoveRequest) MustSend() *ClusterNetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Unassigns the network from a cluster.
//
type ClusterNetworkServiceRemoveResponse struct {
}

//
// Unassigns the network from a cluster.
//
func (p *ClusterNetworkService) Remove() *ClusterNetworkServiceRemoveRequest {
	return &ClusterNetworkServiceRemoveRequest{ClusterNetworkService: p}
}

//
// Updates the network in the cluster.
//
type ClusterNetworkServiceUpdateRequest struct {
	ClusterNetworkService *ClusterNetworkService
	header                map[string]string
	query                 map[string]string
	network               *Network
}

func (p *ClusterNetworkServiceUpdateRequest) Header(key, value string) *ClusterNetworkServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworkServiceUpdateRequest) Query(key, value string) *ClusterNetworkServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworkServiceUpdateRequest) Network(network *Network) *ClusterNetworkServiceUpdateRequest {
	p.network = network
	return p
}

func (p *ClusterNetworkServiceUpdateRequest) Send() (*ClusterNetworkServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworkService.connection.URL(), p.ClusterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterNetworkServiceUpdateResponse{network: result}, nil
}

func (p *ClusterNetworkServiceUpdateRequest) MustSend() *ClusterNetworkServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the network in the cluster.
//
type ClusterNetworkServiceUpdateResponse struct {
	network *Network
}

func (p *ClusterNetworkServiceUpdateResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *ClusterNetworkServiceUpdateResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Updates the network in the cluster.
//
func (p *ClusterNetworkService) Update() *ClusterNetworkServiceUpdateRequest {
	return &ClusterNetworkServiceUpdateRequest{ClusterNetworkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ClusterNetworkService) String() string {
	return fmt.Sprintf("ClusterNetworkService:%s", op.path)
}

//
// A service to manage cluster networks.
//
type ClusterNetworksService struct {
	BaseService
}

func NewClusterNetworksService(connection *Connection, path string) *ClusterNetworksService {
	var result ClusterNetworksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Assigns the network to a cluster.
// Post a request like in the example below to assign the network to a cluster:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network id="123" />
// ----
//
type ClusterNetworksServiceAddRequest struct {
	ClusterNetworksService *ClusterNetworksService
	header                 map[string]string
	query                  map[string]string
	network                *Network
}

func (p *ClusterNetworksServiceAddRequest) Header(key, value string) *ClusterNetworksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworksServiceAddRequest) Query(key, value string) *ClusterNetworksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworksServiceAddRequest) Network(network *Network) *ClusterNetworksServiceAddRequest {
	p.network = network
	return p
}

func (p *ClusterNetworksServiceAddRequest) Send() (*ClusterNetworksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworksService.connection.URL(), p.ClusterNetworksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterNetworksServiceAddResponse{network: result}, nil
}

func (p *ClusterNetworksServiceAddRequest) MustSend() *ClusterNetworksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Assigns the network to a cluster.
// Post a request like in the example below to assign the network to a cluster:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network id="123" />
// ----
//
type ClusterNetworksServiceAddResponse struct {
	network *Network
}

func (p *ClusterNetworksServiceAddResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *ClusterNetworksServiceAddResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Assigns the network to a cluster.
// Post a request like in the example below to assign the network to a cluster:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network id="123" />
// ----
//
func (p *ClusterNetworksService) Add() *ClusterNetworksServiceAddRequest {
	return &ClusterNetworksServiceAddRequest{ClusterNetworksService: p}
}

//
// Lists the networks that are assigned to the cluster.
// The order of the returned clusters isn't guaranteed.
//
type ClusterNetworksServiceListRequest struct {
	ClusterNetworksService *ClusterNetworksService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
	max                    *int64
}

func (p *ClusterNetworksServiceListRequest) Header(key, value string) *ClusterNetworksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworksServiceListRequest) Query(key, value string) *ClusterNetworksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworksServiceListRequest) Follow(follow string) *ClusterNetworksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClusterNetworksServiceListRequest) Max(max int64) *ClusterNetworksServiceListRequest {
	p.max = &max
	return p
}

func (p *ClusterNetworksServiceListRequest) Send() (*ClusterNetworksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworksService.connection.URL(), p.ClusterNetworksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClusterNetworksServiceListResponse{networks: result}, nil
}

func (p *ClusterNetworksServiceListRequest) MustSend() *ClusterNetworksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the networks that are assigned to the cluster.
// The order of the returned clusters isn't guaranteed.
//
type ClusterNetworksServiceListResponse struct {
	networks *NetworkSlice
}

func (p *ClusterNetworksServiceListResponse) Networks() (*NetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *ClusterNetworksServiceListResponse) MustNetworks() *NetworkSlice {
	if p.networks == nil {
		panic("networks in response does not exist")
	}
	return p.networks
}

//
// Lists the networks that are assigned to the cluster.
// The order of the returned clusters isn't guaranteed.
//
func (p *ClusterNetworksService) List() *ClusterNetworksServiceListRequest {
	return &ClusterNetworksServiceListRequest{ClusterNetworksService: p}
}

//
// Access the cluster network service that manages the cluster network specified by an ID.
//
func (op *ClusterNetworksService) NetworkService(id string) *ClusterNetworkService {
	return NewClusterNetworkService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterNetworksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkService(path), nil
	}
	return op.NetworkService(path[:index]).Service(path[index+1:])
}

func (op *ClusterNetworksService) String() string {
	return fmt.Sprintf("ClusterNetworksService:%s", op.path)
}

//
// A service to manage a specific cluster.
//
type ClusterService struct {
	BaseService
}

func NewClusterService(connection *Connection, path string) *ClusterService {
	var result ClusterService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets information about the cluster.
// An example of getting a cluster:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123
// ----
// [source,xml]
// ----
// <cluster href="/ovirt-engine/api/clusters/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/clusters/123/resetemulatedmachine" rel="resetemulatedmachine"/>
//   </actions>
//   <name>Default</name>
//   <description>The default server cluster</description>
//   <link href="/ovirt-engine/api/clusters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/clusters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/clusters/123/glustervolumes" rel="glustervolumes"/>
//   <link href="/ovirt-engine/api/clusters/123/glusterhooks" rel="glusterhooks"/>
//   <link href="/ovirt-engine/api/clusters/123/affinitygroups" rel="affinitygroups"/>
//   <link href="/ovirt-engine/api/clusters/123/cpuprofiles" rel="cpuprofiles"/>
//   <ballooning_enabled>false</ballooning_enabled>
//   <cpu>
//     <architecture>x86_64</architecture>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <error_handling>
//     <on_error>migrate</on_error>
//   </error_handling>
//   <fencing_policy>
//     <enabled>true</enabled>
//     <skip_if_connectivity_broken>
//       <enabled>false</enabled>
//       <threshold>50</threshold>
//     </skip_if_connectivity_broken>
//     <skip_if_sd_active>
//       <enabled>false</enabled>
//     </skip_if_sd_active>
//   </fencing_policy>
//   <gluster_service>false</gluster_service>
//   <ha_reservation>false</ha_reservation>
//   <ksm>
//     <enabled>true</enabled>
//     <merge_across_nodes>true</merge_across_nodes>
//   </ksm>
//   <memory_policy>
//     <over_commit>
//       <percent>100</percent>
//     </over_commit>
//     <transparent_hugepages>
//       <enabled>true</enabled>
//     </transparent_hugepages>
//   </memory_policy>
//   <migration>
//     <auto_converge>inherit</auto_converge>
//     <bandwidth>
//       <assignment_method>auto</assignment_method>
//     </bandwidth>
//     <compressed>inherit</compressed>
//   </migration>
//   <required_rng_sources>
//     <required_rng_source>random</required_rng_source>
//   </required_rng_sources>
//   <scheduling_policy href="/ovirt-engine/api/schedulingpolicies/456" id="456"/>
//   <threads_as_cores>false</threads_as_cores>
//   <trusted_service>false</trusted_service>
//   <tunnel_migration>false</tunnel_migration>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
//   <virt_service>true</virt_service>
//   <data_center href="/ovirt-engine/api/datacenters/111" id="111"/>
// </cluster>
// ----
//
type ClusterServiceGetRequest struct {
	ClusterService *ClusterService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
}

func (p *ClusterServiceGetRequest) Header(key, value string) *ClusterServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterServiceGetRequest) Query(key, value string) *ClusterServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterServiceGetRequest) Filter(filter bool) *ClusterServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *ClusterServiceGetRequest) Follow(follow string) *ClusterServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ClusterServiceGetRequest) Send() (*ClusterServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterService.connection.URL(), p.ClusterService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterServiceGetResponse{cluster: result}, nil
}

func (p *ClusterServiceGetRequest) MustSend() *ClusterServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets information about the cluster.
// An example of getting a cluster:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123
// ----
// [source,xml]
// ----
// <cluster href="/ovirt-engine/api/clusters/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/clusters/123/resetemulatedmachine" rel="resetemulatedmachine"/>
//   </actions>
//   <name>Default</name>
//   <description>The default server cluster</description>
//   <link href="/ovirt-engine/api/clusters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/clusters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/clusters/123/glustervolumes" rel="glustervolumes"/>
//   <link href="/ovirt-engine/api/clusters/123/glusterhooks" rel="glusterhooks"/>
//   <link href="/ovirt-engine/api/clusters/123/affinitygroups" rel="affinitygroups"/>
//   <link href="/ovirt-engine/api/clusters/123/cpuprofiles" rel="cpuprofiles"/>
//   <ballooning_enabled>false</ballooning_enabled>
//   <cpu>
//     <architecture>x86_64</architecture>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <error_handling>
//     <on_error>migrate</on_error>
//   </error_handling>
//   <fencing_policy>
//     <enabled>true</enabled>
//     <skip_if_connectivity_broken>
//       <enabled>false</enabled>
//       <threshold>50</threshold>
//     </skip_if_connectivity_broken>
//     <skip_if_sd_active>
//       <enabled>false</enabled>
//     </skip_if_sd_active>
//   </fencing_policy>
//   <gluster_service>false</gluster_service>
//   <ha_reservation>false</ha_reservation>
//   <ksm>
//     <enabled>true</enabled>
//     <merge_across_nodes>true</merge_across_nodes>
//   </ksm>
//   <memory_policy>
//     <over_commit>
//       <percent>100</percent>
//     </over_commit>
//     <transparent_hugepages>
//       <enabled>true</enabled>
//     </transparent_hugepages>
//   </memory_policy>
//   <migration>
//     <auto_converge>inherit</auto_converge>
//     <bandwidth>
//       <assignment_method>auto</assignment_method>
//     </bandwidth>
//     <compressed>inherit</compressed>
//   </migration>
//   <required_rng_sources>
//     <required_rng_source>random</required_rng_source>
//   </required_rng_sources>
//   <scheduling_policy href="/ovirt-engine/api/schedulingpolicies/456" id="456"/>
//   <threads_as_cores>false</threads_as_cores>
//   <trusted_service>false</trusted_service>
//   <tunnel_migration>false</tunnel_migration>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
//   <virt_service>true</virt_service>
//   <data_center href="/ovirt-engine/api/datacenters/111" id="111"/>
// </cluster>
// ----
//
type ClusterServiceGetResponse struct {
	cluster *Cluster
}

func (p *ClusterServiceGetResponse) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

func (p *ClusterServiceGetResponse) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("cluster in response does not exist")
	}
	return p.cluster
}

//
// Gets information about the cluster.
// An example of getting a cluster:
// [source]
// ----
// GET /ovirt-engine/api/clusters/123
// ----
// [source,xml]
// ----
// <cluster href="/ovirt-engine/api/clusters/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/clusters/123/resetemulatedmachine" rel="resetemulatedmachine"/>
//   </actions>
//   <name>Default</name>
//   <description>The default server cluster</description>
//   <link href="/ovirt-engine/api/clusters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/clusters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/clusters/123/glustervolumes" rel="glustervolumes"/>
//   <link href="/ovirt-engine/api/clusters/123/glusterhooks" rel="glusterhooks"/>
//   <link href="/ovirt-engine/api/clusters/123/affinitygroups" rel="affinitygroups"/>
//   <link href="/ovirt-engine/api/clusters/123/cpuprofiles" rel="cpuprofiles"/>
//   <ballooning_enabled>false</ballooning_enabled>
//   <cpu>
//     <architecture>x86_64</architecture>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <error_handling>
//     <on_error>migrate</on_error>
//   </error_handling>
//   <fencing_policy>
//     <enabled>true</enabled>
//     <skip_if_connectivity_broken>
//       <enabled>false</enabled>
//       <threshold>50</threshold>
//     </skip_if_connectivity_broken>
//     <skip_if_sd_active>
//       <enabled>false</enabled>
//     </skip_if_sd_active>
//   </fencing_policy>
//   <gluster_service>false</gluster_service>
//   <ha_reservation>false</ha_reservation>
//   <ksm>
//     <enabled>true</enabled>
//     <merge_across_nodes>true</merge_across_nodes>
//   </ksm>
//   <memory_policy>
//     <over_commit>
//       <percent>100</percent>
//     </over_commit>
//     <transparent_hugepages>
//       <enabled>true</enabled>
//     </transparent_hugepages>
//   </memory_policy>
//   <migration>
//     <auto_converge>inherit</auto_converge>
//     <bandwidth>
//       <assignment_method>auto</assignment_method>
//     </bandwidth>
//     <compressed>inherit</compressed>
//   </migration>
//   <required_rng_sources>
//     <required_rng_source>random</required_rng_source>
//   </required_rng_sources>
//   <scheduling_policy href="/ovirt-engine/api/schedulingpolicies/456" id="456"/>
//   <threads_as_cores>false</threads_as_cores>
//   <trusted_service>false</trusted_service>
//   <tunnel_migration>false</tunnel_migration>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
//   <virt_service>true</virt_service>
//   <data_center href="/ovirt-engine/api/datacenters/111" id="111"/>
// </cluster>
// ----
//
func (p *ClusterService) Get() *ClusterServiceGetRequest {
	return &ClusterServiceGetRequest{ClusterService: p}
}

//
// Refresh the Gluster heal info for all volumes in cluster.
// For example, Cluster `123`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/refreshglusterhealstatus
// ----
//
type ClusterServiceRefreshGlusterHealStatusRequest struct {
	ClusterService *ClusterService
	header         map[string]string
	query          map[string]string
}

func (p *ClusterServiceRefreshGlusterHealStatusRequest) Header(key, value string) *ClusterServiceRefreshGlusterHealStatusRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterServiceRefreshGlusterHealStatusRequest) Query(key, value string) *ClusterServiceRefreshGlusterHealStatusRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterServiceRefreshGlusterHealStatusRequest) Send() (*ClusterServiceRefreshGlusterHealStatusResponse, error) {
	rawURL := fmt.Sprintf("%s%s/refreshglusterhealstatus", p.ClusterService.connection.URL(), p.ClusterService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ClusterServiceRefreshGlusterHealStatusResponse), nil
}

func (p *ClusterServiceRefreshGlusterHealStatusRequest) MustSend() *ClusterServiceRefreshGlusterHealStatusResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Refresh the Gluster heal info for all volumes in cluster.
// For example, Cluster `123`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/refreshglusterhealstatus
// ----
//
type ClusterServiceRefreshGlusterHealStatusResponse struct {
}

//
// Refresh the Gluster heal info for all volumes in cluster.
// For example, Cluster `123`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/refreshglusterhealstatus
// ----
//
func (p *ClusterService) RefreshGlusterHealStatus() *ClusterServiceRefreshGlusterHealStatusRequest {
	return &ClusterServiceRefreshGlusterHealStatusRequest{ClusterService: p}
}

//
// Removes the cluster from the system.
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/00000000-0000-0000-0000-000000000000
// ----
//
type ClusterServiceRemoveRequest struct {
	ClusterService *ClusterService
	header         map[string]string
	query          map[string]string
	async          *bool
}

func (p *ClusterServiceRemoveRequest) Header(key, value string) *ClusterServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterServiceRemoveRequest) Query(key, value string) *ClusterServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterServiceRemoveRequest) Async(async bool) *ClusterServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *ClusterServiceRemoveRequest) Send() (*ClusterServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterService.connection.URL(), p.ClusterService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(ClusterServiceRemoveResponse), nil
}

func (p *ClusterServiceRemoveRequest) MustSend() *ClusterServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the cluster from the system.
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/00000000-0000-0000-0000-000000000000
// ----
//
type ClusterServiceRemoveResponse struct {
}

//
// Removes the cluster from the system.
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/00000000-0000-0000-0000-000000000000
// ----
//
func (p *ClusterService) Remove() *ClusterServiceRemoveRequest {
	return &ClusterServiceRemoveRequest{ClusterService: p}
}

//
//
type ClusterServiceResetEmulatedMachineRequest struct {
	ClusterService *ClusterService
	header         map[string]string
	query          map[string]string
	async          *bool
}

func (p *ClusterServiceResetEmulatedMachineRequest) Header(key, value string) *ClusterServiceResetEmulatedMachineRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterServiceResetEmulatedMachineRequest) Query(key, value string) *ClusterServiceResetEmulatedMachineRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterServiceResetEmulatedMachineRequest) Async(async bool) *ClusterServiceResetEmulatedMachineRequest {
	p.async = &async
	return p
}

func (p *ClusterServiceResetEmulatedMachineRequest) Send() (*ClusterServiceResetEmulatedMachineResponse, error) {
	rawURL := fmt.Sprintf("%s%s/resetemulatedmachine", p.ClusterService.connection.URL(), p.ClusterService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ClusterServiceResetEmulatedMachineResponse), nil
}

func (p *ClusterServiceResetEmulatedMachineRequest) MustSend() *ClusterServiceResetEmulatedMachineResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ClusterServiceResetEmulatedMachineResponse struct {
}

//
//
func (p *ClusterService) ResetEmulatedMachine() *ClusterServiceResetEmulatedMachineRequest {
	return &ClusterServiceResetEmulatedMachineRequest{ClusterService: p}
}

//
// Synchronizes all networks on the cluster.
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/syncallnetworks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type ClusterServiceSyncAllNetworksRequest struct {
	ClusterService *ClusterService
	header         map[string]string
	query          map[string]string
	async          *bool
}

func (p *ClusterServiceSyncAllNetworksRequest) Header(key, value string) *ClusterServiceSyncAllNetworksRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterServiceSyncAllNetworksRequest) Query(key, value string) *ClusterServiceSyncAllNetworksRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterServiceSyncAllNetworksRequest) Async(async bool) *ClusterServiceSyncAllNetworksRequest {
	p.async = &async
	return p
}

func (p *ClusterServiceSyncAllNetworksRequest) Send() (*ClusterServiceSyncAllNetworksResponse, error) {
	rawURL := fmt.Sprintf("%s%s/syncallnetworks", p.ClusterService.connection.URL(), p.ClusterService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ClusterServiceSyncAllNetworksResponse), nil
}

func (p *ClusterServiceSyncAllNetworksRequest) MustSend() *ClusterServiceSyncAllNetworksResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Synchronizes all networks on the cluster.
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/syncallnetworks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type ClusterServiceSyncAllNetworksResponse struct {
}

//
// Synchronizes all networks on the cluster.
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/syncallnetworks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *ClusterService) SyncAllNetworks() *ClusterServiceSyncAllNetworksRequest {
	return &ClusterServiceSyncAllNetworksRequest{ClusterService: p}
}

//
// Updates information about the cluster.
// Only the specified fields are updated; others remain unchanged.
// For example, to update the cluster's CPU:
// [source]
// ----
// PUT /ovirt-engine/api/clusters/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <cpu>
//     <type>Intel Haswell-noTSX Family</type>
//   </cpu>
// </cluster>
// ----
//
type ClusterServiceUpdateRequest struct {
	ClusterService *ClusterService
	header         map[string]string
	query          map[string]string
	async          *bool
	cluster        *Cluster
}

func (p *ClusterServiceUpdateRequest) Header(key, value string) *ClusterServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterServiceUpdateRequest) Query(key, value string) *ClusterServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterServiceUpdateRequest) Async(async bool) *ClusterServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *ClusterServiceUpdateRequest) Cluster(cluster *Cluster) *ClusterServiceUpdateRequest {
	p.cluster = cluster
	return p
}

func (p *ClusterServiceUpdateRequest) Send() (*ClusterServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterService.connection.URL(), p.ClusterService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLClusterWriteOne(writer, p.cluster, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterServiceUpdateResponse{cluster: result}, nil
}

func (p *ClusterServiceUpdateRequest) MustSend() *ClusterServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates information about the cluster.
// Only the specified fields are updated; others remain unchanged.
// For example, to update the cluster's CPU:
// [source]
// ----
// PUT /ovirt-engine/api/clusters/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <cpu>
//     <type>Intel Haswell-noTSX Family</type>
//   </cpu>
// </cluster>
// ----
//
type ClusterServiceUpdateResponse struct {
	cluster *Cluster
}

func (p *ClusterServiceUpdateResponse) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

func (p *ClusterServiceUpdateResponse) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("cluster in response does not exist")
	}
	return p.cluster
}

//
// Updates information about the cluster.
// Only the specified fields are updated; others remain unchanged.
// For example, to update the cluster's CPU:
// [source]
// ----
// PUT /ovirt-engine/api/clusters/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <cpu>
//     <type>Intel Haswell-noTSX Family</type>
//   </cpu>
// </cluster>
// ----
//
func (p *ClusterService) Update() *ClusterServiceUpdateRequest {
	return &ClusterServiceUpdateRequest{ClusterService: p}
}

//
// Start or finish upgrade process for the cluster based on the action value. This action marks the cluster for
// upgrade or clears the upgrade running flag on the cluster based on the action value which takes values of
// start or stop.
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/upgrade
// ----
// With a request body like this to mark the cluster for upgrade:
// [source,xml]
// ----
// <action>
//     <upgrade_action>
//         start
//     </upgrade_action>
// </action>
// ----
//
type ClusterServiceUpgradeRequest struct {
	ClusterService *ClusterService
	header         map[string]string
	query          map[string]string
	async          *bool
	upgradeAction  *ClusterUpgradeAction
}

func (p *ClusterServiceUpgradeRequest) Header(key, value string) *ClusterServiceUpgradeRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterServiceUpgradeRequest) Query(key, value string) *ClusterServiceUpgradeRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterServiceUpgradeRequest) Async(async bool) *ClusterServiceUpgradeRequest {
	p.async = &async
	return p
}

func (p *ClusterServiceUpgradeRequest) UpgradeAction(upgradeAction ClusterUpgradeAction) *ClusterServiceUpgradeRequest {
	p.upgradeAction = &upgradeAction
	return p
}

func (p *ClusterServiceUpgradeRequest) Send() (*ClusterServiceUpgradeResponse, error) {
	rawURL := fmt.Sprintf("%s%s/upgrade", p.ClusterService.connection.URL(), p.ClusterService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.upgradeAction != nil {
		actionBuilder.UpgradeAction(*p.upgradeAction)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ClusterServiceUpgradeResponse), nil
}

func (p *ClusterServiceUpgradeRequest) MustSend() *ClusterServiceUpgradeResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Start or finish upgrade process for the cluster based on the action value. This action marks the cluster for
// upgrade or clears the upgrade running flag on the cluster based on the action value which takes values of
// start or stop.
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/upgrade
// ----
// With a request body like this to mark the cluster for upgrade:
// [source,xml]
// ----
// <action>
//     <upgrade_action>
//         start
//     </upgrade_action>
// </action>
// ----
//
type ClusterServiceUpgradeResponse struct {
}

//
// Start or finish upgrade process for the cluster based on the action value. This action marks the cluster for
// upgrade or clears the upgrade running flag on the cluster based on the action value which takes values of
// start or stop.
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/upgrade
// ----
// With a request body like this to mark the cluster for upgrade:
// [source,xml]
// ----
// <action>
//     <upgrade_action>
//         start
//     </upgrade_action>
// </action>
// ----
//
func (p *ClusterService) Upgrade() *ClusterServiceUpgradeRequest {
	return &ClusterServiceUpgradeRequest{ClusterService: p}
}

//
// A reference to the service that manages affinity groups.
//
func (op *ClusterService) AffinityGroupsService() *AffinityGroupsService {
	return NewAffinityGroupsService(op.connection, fmt.Sprintf("%s/affinitygroups", op.path))
}

//
// A reference to the service that manages assigned CPU profiles for the cluster.
//
func (op *ClusterService) CpuProfilesService() *AssignedCpuProfilesService {
	return NewAssignedCpuProfilesService(op.connection, fmt.Sprintf("%s/cpuprofiles", op.path))
}

//
// A reference to the service that manages the collection of enabled features for the cluster.
//
func (op *ClusterService) EnabledFeaturesService() *ClusterEnabledFeaturesService {
	return NewClusterEnabledFeaturesService(op.connection, fmt.Sprintf("%s/enabledfeatures", op.path))
}

//
// A reference to the service that manages the collection of external network providers.
//
func (op *ClusterService) ExternalNetworkProvidersService() *ClusterExternalProvidersService {
	return NewClusterExternalProvidersService(op.connection, fmt.Sprintf("%s/externalnetworkproviders", op.path))
}

//
// A reference to the service that manages the Gluster hooks for the cluster.
//
func (op *ClusterService) GlusterHooksService() *GlusterHooksService {
	return NewGlusterHooksService(op.connection, fmt.Sprintf("%s/glusterhooks", op.path))
}

//
// A reference to the service that manages Gluster volumes for the cluster.
//
func (op *ClusterService) GlusterVolumesService() *GlusterVolumesService {
	return NewGlusterVolumesService(op.connection, fmt.Sprintf("%s/glustervolumes", op.path))
}

//
// A sub-collection with all the supported network filters for the cluster.
//
func (op *ClusterService) NetworkFiltersService() *NetworkFiltersService {
	return NewNetworkFiltersService(op.connection, fmt.Sprintf("%s/networkfilters", op.path))
}

//
// A reference to the service that manages assigned networks for the cluster.
//
func (op *ClusterService) NetworksService() *ClusterNetworksService {
	return NewClusterNetworksService(op.connection, fmt.Sprintf("%s/networks", op.path))
}

//
// A reference to permissions.
//
func (op *ClusterService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "affinitygroups" {
		return op.AffinityGroupsService(), nil
	}
	if strings.HasPrefix(path, "affinitygroups/") {
		return op.AffinityGroupsService().Service(path[15:])
	}
	if path == "cpuprofiles" {
		return op.CpuProfilesService(), nil
	}
	if strings.HasPrefix(path, "cpuprofiles/") {
		return op.CpuProfilesService().Service(path[12:])
	}
	if path == "enabledfeatures" {
		return op.EnabledFeaturesService(), nil
	}
	if strings.HasPrefix(path, "enabledfeatures/") {
		return op.EnabledFeaturesService().Service(path[16:])
	}
	if path == "externalnetworkproviders" {
		return op.ExternalNetworkProvidersService(), nil
	}
	if strings.HasPrefix(path, "externalnetworkproviders/") {
		return op.ExternalNetworkProvidersService().Service(path[25:])
	}
	if path == "glusterhooks" {
		return op.GlusterHooksService(), nil
	}
	if strings.HasPrefix(path, "glusterhooks/") {
		return op.GlusterHooksService().Service(path[13:])
	}
	if path == "glustervolumes" {
		return op.GlusterVolumesService(), nil
	}
	if strings.HasPrefix(path, "glustervolumes/") {
		return op.GlusterVolumesService().Service(path[15:])
	}
	if path == "networkfilters" {
		return op.NetworkFiltersService(), nil
	}
	if strings.HasPrefix(path, "networkfilters/") {
		return op.NetworkFiltersService().Service(path[15:])
	}
	if path == "networks" {
		return op.NetworksService(), nil
	}
	if strings.HasPrefix(path, "networks/") {
		return op.NetworksService().Service(path[9:])
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ClusterService) String() string {
	return fmt.Sprintf("ClusterService:%s", op.path)
}

//
// A service to manage clusters.
//
type ClustersService struct {
	BaseService
}

func NewClustersService(connection *Connection, path string) *ClustersService {
	var result ClustersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new cluster.
// This requires the `name`, `cpu.type`, and `data_center` attributes. Identify the data center with either the `id`
// or `name` attribute.
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
// </cluster>
// ----
// To create a cluster with an external network provider to be deployed on
// every host that is added to the cluster, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body containing a reference to the desired provider:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
//   <external_network_providers>
//     <external_provider name="ovirt-provider-ovn"/>
//   </external_network_providers>
// </cluster>
// ----
//
type ClustersServiceAddRequest struct {
	ClustersService *ClustersService
	header          map[string]string
	query           map[string]string
	cluster         *Cluster
}

func (p *ClustersServiceAddRequest) Header(key, value string) *ClustersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClustersServiceAddRequest) Query(key, value string) *ClustersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClustersServiceAddRequest) Cluster(cluster *Cluster) *ClustersServiceAddRequest {
	p.cluster = cluster
	return p
}

func (p *ClustersServiceAddRequest) Send() (*ClustersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClustersService.connection.URL(), p.ClustersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLClusterWriteOne(writer, p.cluster, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClustersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClustersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClustersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClustersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClustersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClustersServiceAddResponse{cluster: result}, nil
}

func (p *ClustersServiceAddRequest) MustSend() *ClustersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new cluster.
// This requires the `name`, `cpu.type`, and `data_center` attributes. Identify the data center with either the `id`
// or `name` attribute.
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
// </cluster>
// ----
// To create a cluster with an external network provider to be deployed on
// every host that is added to the cluster, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body containing a reference to the desired provider:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
//   <external_network_providers>
//     <external_provider name="ovirt-provider-ovn"/>
//   </external_network_providers>
// </cluster>
// ----
//
type ClustersServiceAddResponse struct {
	cluster *Cluster
}

func (p *ClustersServiceAddResponse) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

func (p *ClustersServiceAddResponse) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("cluster in response does not exist")
	}
	return p.cluster
}

//
// Creates a new cluster.
// This requires the `name`, `cpu.type`, and `data_center` attributes. Identify the data center with either the `id`
// or `name` attribute.
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
// </cluster>
// ----
// To create a cluster with an external network provider to be deployed on
// every host that is added to the cluster, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body containing a reference to the desired provider:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
//   <external_network_providers>
//     <external_provider name="ovirt-provider-ovn"/>
//   </external_network_providers>
// </cluster>
// ----
//
func (p *ClustersService) Add() *ClustersServiceAddRequest {
	return &ClustersServiceAddRequest{ClustersService: p}
}

//
// Returns the list of clusters of the system.
// The order of the returned clusters is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type ClustersServiceListRequest struct {
	ClustersService *ClustersService
	header          map[string]string
	query           map[string]string
	caseSensitive   *bool
	filter          *bool
	follow          *string
	max             *int64
	search          *string
}

func (p *ClustersServiceListRequest) Header(key, value string) *ClustersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClustersServiceListRequest) Query(key, value string) *ClustersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClustersServiceListRequest) CaseSensitive(caseSensitive bool) *ClustersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *ClustersServiceListRequest) Filter(filter bool) *ClustersServiceListRequest {
	p.filter = &filter
	return p
}

func (p *ClustersServiceListRequest) Follow(follow string) *ClustersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClustersServiceListRequest) Max(max int64) *ClustersServiceListRequest {
	p.max = &max
	return p
}

func (p *ClustersServiceListRequest) Search(search string) *ClustersServiceListRequest {
	p.search = &search
	return p
}

func (p *ClustersServiceListRequest) Send() (*ClustersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClustersService.connection.URL(), p.ClustersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClustersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClustersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClustersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClustersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClustersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClustersServiceListResponse{clusters: result}, nil
}

func (p *ClustersServiceListRequest) MustSend() *ClustersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of clusters of the system.
// The order of the returned clusters is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type ClustersServiceListResponse struct {
	clusters *ClusterSlice
}

func (p *ClustersServiceListResponse) Clusters() (*ClusterSlice, bool) {
	if p.clusters != nil {
		return p.clusters, true
	}
	return nil, false
}

func (p *ClustersServiceListResponse) MustClusters() *ClusterSlice {
	if p.clusters == nil {
		panic("clusters in response does not exist")
	}
	return p.clusters
}

//
// Returns the list of clusters of the system.
// The order of the returned clusters is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
func (p *ClustersService) List() *ClustersServiceListRequest {
	return &ClustersServiceListRequest{ClustersService: p}
}

//
// A reference to the service that manages a specific cluster.
//
func (op *ClustersService) ClusterService(id string) *ClusterService {
	return NewClusterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClustersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ClusterService(path), nil
	}
	return op.ClusterService(path[:index]).Service(path[index+1:])
}

func (op *ClustersService) String() string {
	return fmt.Sprintf("ClustersService:%s", op.path)
}

//
//
type CopyableService struct {
	BaseService
}

func NewCopyableService(connection *Connection, path string) *CopyableService {
	var result CopyableService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type CopyableServiceCopyRequest struct {
	CopyableService *CopyableService
	header          map[string]string
	query           map[string]string
	async           *bool
}

func (p *CopyableServiceCopyRequest) Header(key, value string) *CopyableServiceCopyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CopyableServiceCopyRequest) Query(key, value string) *CopyableServiceCopyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CopyableServiceCopyRequest) Async(async bool) *CopyableServiceCopyRequest {
	p.async = &async
	return p
}

func (p *CopyableServiceCopyRequest) Send() (*CopyableServiceCopyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copy", p.CopyableService.connection.URL(), p.CopyableService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CopyableService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CopyableService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CopyableService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CopyableService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CopyableService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(CopyableServiceCopyResponse), nil
}

func (p *CopyableServiceCopyRequest) MustSend() *CopyableServiceCopyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type CopyableServiceCopyResponse struct {
}

//
//
func (p *CopyableService) Copy() *CopyableServiceCopyRequest {
	return &CopyableServiceCopyRequest{CopyableService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *CopyableService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *CopyableService) String() string {
	return fmt.Sprintf("CopyableService:%s", op.path)
}

//
//
type CpuProfileService struct {
	BaseService
}

func NewCpuProfileService(connection *Connection, path string) *CpuProfileService {
	var result CpuProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type CpuProfileServiceGetRequest struct {
	CpuProfileService *CpuProfileService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *CpuProfileServiceGetRequest) Header(key, value string) *CpuProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfileServiceGetRequest) Query(key, value string) *CpuProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfileServiceGetRequest) Follow(follow string) *CpuProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *CpuProfileServiceGetRequest) Send() (*CpuProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfileService.connection.URL(), p.CpuProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &CpuProfileServiceGetResponse{profile: result}, nil
}

func (p *CpuProfileServiceGetRequest) MustSend() *CpuProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type CpuProfileServiceGetResponse struct {
	profile *CpuProfile
}

func (p *CpuProfileServiceGetResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *CpuProfileServiceGetResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *CpuProfileService) Get() *CpuProfileServiceGetRequest {
	return &CpuProfileServiceGetRequest{CpuProfileService: p}
}

//
//
type CpuProfileServiceRemoveRequest struct {
	CpuProfileService *CpuProfileService
	header            map[string]string
	query             map[string]string
	async             *bool
}

func (p *CpuProfileServiceRemoveRequest) Header(key, value string) *CpuProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfileServiceRemoveRequest) Query(key, value string) *CpuProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfileServiceRemoveRequest) Async(async bool) *CpuProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *CpuProfileServiceRemoveRequest) Send() (*CpuProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfileService.connection.URL(), p.CpuProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(CpuProfileServiceRemoveResponse), nil
}

func (p *CpuProfileServiceRemoveRequest) MustSend() *CpuProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type CpuProfileServiceRemoveResponse struct {
}

//
//
func (p *CpuProfileService) Remove() *CpuProfileServiceRemoveRequest {
	return &CpuProfileServiceRemoveRequest{CpuProfileService: p}
}

//
// Update the specified cpu profile in the system.
//
type CpuProfileServiceUpdateRequest struct {
	CpuProfileService *CpuProfileService
	header            map[string]string
	query             map[string]string
	async             *bool
	profile           *CpuProfile
}

func (p *CpuProfileServiceUpdateRequest) Header(key, value string) *CpuProfileServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfileServiceUpdateRequest) Query(key, value string) *CpuProfileServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfileServiceUpdateRequest) Async(async bool) *CpuProfileServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *CpuProfileServiceUpdateRequest) Profile(profile *CpuProfile) *CpuProfileServiceUpdateRequest {
	p.profile = profile
	return p
}

func (p *CpuProfileServiceUpdateRequest) Send() (*CpuProfileServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfileService.connection.URL(), p.CpuProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLCpuProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &CpuProfileServiceUpdateResponse{profile: result}, nil
}

func (p *CpuProfileServiceUpdateRequest) MustSend() *CpuProfileServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified cpu profile in the system.
//
type CpuProfileServiceUpdateResponse struct {
	profile *CpuProfile
}

func (p *CpuProfileServiceUpdateResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *CpuProfileServiceUpdateResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Update the specified cpu profile in the system.
//
func (p *CpuProfileService) Update() *CpuProfileServiceUpdateRequest {
	return &CpuProfileServiceUpdateRequest{CpuProfileService: p}
}

//
//
func (op *CpuProfileService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *CpuProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *CpuProfileService) String() string {
	return fmt.Sprintf("CpuProfileService:%s", op.path)
}

//
//
type CpuProfilesService struct {
	BaseService
}

func NewCpuProfilesService(connection *Connection, path string) *CpuProfilesService {
	var result CpuProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new cpu profile to the system.
//
type CpuProfilesServiceAddRequest struct {
	CpuProfilesService *CpuProfilesService
	header             map[string]string
	query              map[string]string
	profile            *CpuProfile
}

func (p *CpuProfilesServiceAddRequest) Header(key, value string) *CpuProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfilesServiceAddRequest) Query(key, value string) *CpuProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfilesServiceAddRequest) Profile(profile *CpuProfile) *CpuProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *CpuProfilesServiceAddRequest) Send() (*CpuProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfilesService.connection.URL(), p.CpuProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLCpuProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &CpuProfilesServiceAddResponse{profile: result}, nil
}

func (p *CpuProfilesServiceAddRequest) MustSend() *CpuProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new cpu profile to the system.
//
type CpuProfilesServiceAddResponse struct {
	profile *CpuProfile
}

func (p *CpuProfilesServiceAddResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *CpuProfilesServiceAddResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a new cpu profile to the system.
//
func (p *CpuProfilesService) Add() *CpuProfilesServiceAddRequest {
	return &CpuProfilesServiceAddRequest{CpuProfilesService: p}
}

//
// Returns the list of CPU profiles of the system.
// The order of the returned list of CPU profiles is random.
//
type CpuProfilesServiceListRequest struct {
	CpuProfilesService *CpuProfilesService
	header             map[string]string
	query              map[string]string
	follow             *string
	max                *int64
}

func (p *CpuProfilesServiceListRequest) Header(key, value string) *CpuProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfilesServiceListRequest) Query(key, value string) *CpuProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfilesServiceListRequest) Follow(follow string) *CpuProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *CpuProfilesServiceListRequest) Max(max int64) *CpuProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *CpuProfilesServiceListRequest) Send() (*CpuProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfilesService.connection.URL(), p.CpuProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &CpuProfilesServiceListResponse{profile: result}, nil
}

func (p *CpuProfilesServiceListRequest) MustSend() *CpuProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of CPU profiles of the system.
// The order of the returned list of CPU profiles is random.
//
type CpuProfilesServiceListResponse struct {
	profile *CpuProfileSlice
}

func (p *CpuProfilesServiceListResponse) Profile() (*CpuProfileSlice, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *CpuProfilesServiceListResponse) MustProfile() *CpuProfileSlice {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Returns the list of CPU profiles of the system.
// The order of the returned list of CPU profiles is random.
//
func (p *CpuProfilesService) List() *CpuProfilesServiceListRequest {
	return &CpuProfilesServiceListRequest{CpuProfilesService: p}
}

//
//
func (op *CpuProfilesService) ProfileService(id string) *CpuProfileService {
	return NewCpuProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *CpuProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProfileService(path), nil
	}
	return op.ProfileService(path[:index]).Service(path[index+1:])
}

func (op *CpuProfilesService) String() string {
	return fmt.Sprintf("CpuProfilesService:%s", op.path)
}

//
// A service to manage a specific data center network.
//
type DataCenterNetworkService struct {
	BaseService
}

func NewDataCenterNetworkService(connection *Connection, path string) *DataCenterNetworkService {
	var result DataCenterNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the data center network details.
//
type DataCenterNetworkServiceGetRequest struct {
	DataCenterNetworkService *DataCenterNetworkService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *DataCenterNetworkServiceGetRequest) Header(key, value string) *DataCenterNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworkServiceGetRequest) Query(key, value string) *DataCenterNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworkServiceGetRequest) Follow(follow string) *DataCenterNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DataCenterNetworkServiceGetRequest) Send() (*DataCenterNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworkService.connection.URL(), p.DataCenterNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworkServiceGetResponse{network: result}, nil
}

func (p *DataCenterNetworkServiceGetRequest) MustSend() *DataCenterNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the data center network details.
//
type DataCenterNetworkServiceGetResponse struct {
	network *Network
}

func (p *DataCenterNetworkServiceGetResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *DataCenterNetworkServiceGetResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Retrieves the data center network details.
//
func (p *DataCenterNetworkService) Get() *DataCenterNetworkServiceGetRequest {
	return &DataCenterNetworkServiceGetRequest{DataCenterNetworkService: p}
}

//
// Removes the network.
//
type DataCenterNetworkServiceRemoveRequest struct {
	DataCenterNetworkService *DataCenterNetworkService
	header                   map[string]string
	query                    map[string]string
}

func (p *DataCenterNetworkServiceRemoveRequest) Header(key, value string) *DataCenterNetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworkServiceRemoveRequest) Query(key, value string) *DataCenterNetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworkServiceRemoveRequest) Send() (*DataCenterNetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworkService.connection.URL(), p.DataCenterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DataCenterNetworkServiceRemoveResponse), nil
}

func (p *DataCenterNetworkServiceRemoveRequest) MustSend() *DataCenterNetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the network.
//
type DataCenterNetworkServiceRemoveResponse struct {
}

//
// Removes the network.
//
func (p *DataCenterNetworkService) Remove() *DataCenterNetworkServiceRemoveRequest {
	return &DataCenterNetworkServiceRemoveRequest{DataCenterNetworkService: p}
}

//
// Updates the network in the data center.
//
type DataCenterNetworkServiceUpdateRequest struct {
	DataCenterNetworkService *DataCenterNetworkService
	header                   map[string]string
	query                    map[string]string
	network                  *Network
}

func (p *DataCenterNetworkServiceUpdateRequest) Header(key, value string) *DataCenterNetworkServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworkServiceUpdateRequest) Query(key, value string) *DataCenterNetworkServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworkServiceUpdateRequest) Network(network *Network) *DataCenterNetworkServiceUpdateRequest {
	p.network = network
	return p
}

func (p *DataCenterNetworkServiceUpdateRequest) Send() (*DataCenterNetworkServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworkService.connection.URL(), p.DataCenterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworkServiceUpdateResponse{network: result}, nil
}

func (p *DataCenterNetworkServiceUpdateRequest) MustSend() *DataCenterNetworkServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the network in the data center.
//
type DataCenterNetworkServiceUpdateResponse struct {
	network *Network
}

func (p *DataCenterNetworkServiceUpdateResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *DataCenterNetworkServiceUpdateResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Updates the network in the data center.
//
func (p *DataCenterNetworkService) Update() *DataCenterNetworkServiceUpdateRequest {
	return &DataCenterNetworkServiceUpdateRequest{DataCenterNetworkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DataCenterNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DataCenterNetworkService) String() string {
	return fmt.Sprintf("DataCenterNetworkService:%s", op.path)
}

//
// A service to manage data center networks.
//
type DataCenterNetworksService struct {
	BaseService
}

func NewDataCenterNetworksService(connection *Connection, path string) *DataCenterNetworksService {
	var result DataCenterNetworksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Create a new network in a data center.
// Post a request like in the example below to create a new network in a data center with an ID of `123`.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network>
//   <name>mynetwork</name>
// </network>
// ----
//
type DataCenterNetworksServiceAddRequest struct {
	DataCenterNetworksService *DataCenterNetworksService
	header                    map[string]string
	query                     map[string]string
	network                   *Network
}

func (p *DataCenterNetworksServiceAddRequest) Header(key, value string) *DataCenterNetworksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworksServiceAddRequest) Query(key, value string) *DataCenterNetworksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworksServiceAddRequest) Network(network *Network) *DataCenterNetworksServiceAddRequest {
	p.network = network
	return p
}

func (p *DataCenterNetworksServiceAddRequest) Send() (*DataCenterNetworksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworksService.connection.URL(), p.DataCenterNetworksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworksServiceAddResponse{network: result}, nil
}

func (p *DataCenterNetworksServiceAddRequest) MustSend() *DataCenterNetworksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Create a new network in a data center.
// Post a request like in the example below to create a new network in a data center with an ID of `123`.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network>
//   <name>mynetwork</name>
// </network>
// ----
//
type DataCenterNetworksServiceAddResponse struct {
	network *Network
}

func (p *DataCenterNetworksServiceAddResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *DataCenterNetworksServiceAddResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Create a new network in a data center.
// Post a request like in the example below to create a new network in a data center with an ID of `123`.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network>
//   <name>mynetwork</name>
// </network>
// ----
//
func (p *DataCenterNetworksService) Add() *DataCenterNetworksServiceAddRequest {
	return &DataCenterNetworksServiceAddRequest{DataCenterNetworksService: p}
}

//
// Lists networks in the data center.
// The order of the returned list of networks isn't guaranteed.
//
type DataCenterNetworksServiceListRequest struct {
	DataCenterNetworksService *DataCenterNetworksService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *DataCenterNetworksServiceListRequest) Header(key, value string) *DataCenterNetworksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworksServiceListRequest) Query(key, value string) *DataCenterNetworksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworksServiceListRequest) Follow(follow string) *DataCenterNetworksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DataCenterNetworksServiceListRequest) Max(max int64) *DataCenterNetworksServiceListRequest {
	p.max = &max
	return p
}

func (p *DataCenterNetworksServiceListRequest) Send() (*DataCenterNetworksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworksService.connection.URL(), p.DataCenterNetworksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworksServiceListResponse{networks: result}, nil
}

func (p *DataCenterNetworksServiceListRequest) MustSend() *DataCenterNetworksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists networks in the data center.
// The order of the returned list of networks isn't guaranteed.
//
type DataCenterNetworksServiceListResponse struct {
	networks *NetworkSlice
}

func (p *DataCenterNetworksServiceListResponse) Networks() (*NetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *DataCenterNetworksServiceListResponse) MustNetworks() *NetworkSlice {
	if p.networks == nil {
		panic("networks in response does not exist")
	}
	return p.networks
}

//
// Lists networks in the data center.
// The order of the returned list of networks isn't guaranteed.
//
func (p *DataCenterNetworksService) List() *DataCenterNetworksServiceListRequest {
	return &DataCenterNetworksServiceListRequest{DataCenterNetworksService: p}
}

//
// Access the data center network service that manages the data center network specified by an ID.
//
func (op *DataCenterNetworksService) NetworkService(id string) *DataCenterNetworkService {
	return NewDataCenterNetworkService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DataCenterNetworksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkService(path), nil
	}
	return op.NetworkService(path[:index]).Service(path[index+1:])
}

func (op *DataCenterNetworksService) String() string {
	return fmt.Sprintf("DataCenterNetworksService:%s", op.path)
}

//
// A service to manage a data center.
//
type DataCenterService struct {
	BaseService
}

func NewDataCenterService(connection *Connection, path string) *DataCenterService {
	var result DataCenterService
	result.connection = connection
	result.path = path
	return &result
}

//
// Currently, the storage pool manager (SPM) fails to
// switch to another host if the SPM has uncleared tasks.
// Clearing all finished tasks enables the SPM switching.
// For example, to clean all the finished tasks on a data center with ID `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/cleanfinishedtasks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type DataCenterServiceCleanFinishedTasksRequest struct {
	DataCenterService *DataCenterService
	header            map[string]string
	query             map[string]string
	async             *bool
}

func (p *DataCenterServiceCleanFinishedTasksRequest) Header(key, value string) *DataCenterServiceCleanFinishedTasksRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterServiceCleanFinishedTasksRequest) Query(key, value string) *DataCenterServiceCleanFinishedTasksRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterServiceCleanFinishedTasksRequest) Async(async bool) *DataCenterServiceCleanFinishedTasksRequest {
	p.async = &async
	return p
}

func (p *DataCenterServiceCleanFinishedTasksRequest) Send() (*DataCenterServiceCleanFinishedTasksResponse, error) {
	rawURL := fmt.Sprintf("%s%s/cleanfinishedtasks", p.DataCenterService.connection.URL(), p.DataCenterService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DataCenterServiceCleanFinishedTasksResponse), nil
}

func (p *DataCenterServiceCleanFinishedTasksRequest) MustSend() *DataCenterServiceCleanFinishedTasksResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Currently, the storage pool manager (SPM) fails to
// switch to another host if the SPM has uncleared tasks.
// Clearing all finished tasks enables the SPM switching.
// For example, to clean all the finished tasks on a data center with ID `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/cleanfinishedtasks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type DataCenterServiceCleanFinishedTasksResponse struct {
}

//
// Currently, the storage pool manager (SPM) fails to
// switch to another host if the SPM has uncleared tasks.
// Clearing all finished tasks enables the SPM switching.
// For example, to clean all the finished tasks on a data center with ID `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/cleanfinishedtasks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *DataCenterService) CleanFinishedTasks() *DataCenterServiceCleanFinishedTasksRequest {
	return &DataCenterServiceCleanFinishedTasksRequest{DataCenterService: p}
}

//
// Get a data center.
// An example of getting a data center:
// [source]
// ----
// GET /ovirt-engine/api/datacenters/123
// ----
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <storage_format>v3</storage_format>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//    </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
//   <mac_pool href="/ovirt-engine/api/macpools/456" id="456"/>
// </data_center>
// ----
//
type DataCenterServiceGetRequest struct {
	DataCenterService *DataCenterService
	header            map[string]string
	query             map[string]string
	filter            *bool
	follow            *string
}

func (p *DataCenterServiceGetRequest) Header(key, value string) *DataCenterServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterServiceGetRequest) Query(key, value string) *DataCenterServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterServiceGetRequest) Filter(filter bool) *DataCenterServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *DataCenterServiceGetRequest) Follow(follow string) *DataCenterServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DataCenterServiceGetRequest) Send() (*DataCenterServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterService.connection.URL(), p.DataCenterService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDataCenterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterServiceGetResponse{dataCenter: result}, nil
}

func (p *DataCenterServiceGetRequest) MustSend() *DataCenterServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get a data center.
// An example of getting a data center:
// [source]
// ----
// GET /ovirt-engine/api/datacenters/123
// ----
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <storage_format>v3</storage_format>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//    </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
//   <mac_pool href="/ovirt-engine/api/macpools/456" id="456"/>
// </data_center>
// ----
//
type DataCenterServiceGetResponse struct {
	dataCenter *DataCenter
}

func (p *DataCenterServiceGetResponse) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *DataCenterServiceGetResponse) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("dataCenter in response does not exist")
	}
	return p.dataCenter
}

//
// Get a data center.
// An example of getting a data center:
// [source]
// ----
// GET /ovirt-engine/api/datacenters/123
// ----
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <storage_format>v3</storage_format>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//    </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
//   <mac_pool href="/ovirt-engine/api/macpools/456" id="456"/>
// </data_center>
// ----
//
func (p *DataCenterService) Get() *DataCenterServiceGetRequest {
	return &DataCenterServiceGetRequest{DataCenterService: p}
}

//
// Removes the data center.
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123
// ----
// Without any special parameters, the storage domains attached to the data center are detached and then removed
// from the storage. If something fails when performing this operation, for example if there is no host available to
// remove the storage domains from the storage, the complete operation will fail.
// If the `force` parameter is `true` then the operation will always succeed, even if something fails while removing
// one storage domain, for example. The failure is just ignored and the data center is removed from the database
// anyway.
//
type DataCenterServiceRemoveRequest struct {
	DataCenterService *DataCenterService
	header            map[string]string
	query             map[string]string
	async             *bool
	force             *bool
}

func (p *DataCenterServiceRemoveRequest) Header(key, value string) *DataCenterServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterServiceRemoveRequest) Query(key, value string) *DataCenterServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterServiceRemoveRequest) Async(async bool) *DataCenterServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *DataCenterServiceRemoveRequest) Force(force bool) *DataCenterServiceRemoveRequest {
	p.force = &force
	return p
}

func (p *DataCenterServiceRemoveRequest) Send() (*DataCenterServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterService.connection.URL(), p.DataCenterService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.force != nil {
		values["force"] = []string{fmt.Sprintf("%v", *p.force)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DataCenterServiceRemoveResponse), nil
}

func (p *DataCenterServiceRemoveRequest) MustSend() *DataCenterServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the data center.
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123
// ----
// Without any special parameters, the storage domains attached to the data center are detached and then removed
// from the storage. If something fails when performing this operation, for example if there is no host available to
// remove the storage domains from the storage, the complete operation will fail.
// If the `force` parameter is `true` then the operation will always succeed, even if something fails while removing
// one storage domain, for example. The failure is just ignored and the data center is removed from the database
// anyway.
//
type DataCenterServiceRemoveResponse struct {
}

//
// Removes the data center.
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123
// ----
// Without any special parameters, the storage domains attached to the data center are detached and then removed
// from the storage. If something fails when performing this operation, for example if there is no host available to
// remove the storage domains from the storage, the complete operation will fail.
// If the `force` parameter is `true` then the operation will always succeed, even if something fails while removing
// one storage domain, for example. The failure is just ignored and the data center is removed from the database
// anyway.
//
func (p *DataCenterService) Remove() *DataCenterServiceRemoveRequest {
	return &DataCenterServiceRemoveRequest{DataCenterService: p}
}

//
// Used for manually setting a storage domain in the data center as a master.
// For example, for setting a storage domain with ID '456' as a master on a data center with ID '123',
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/setmaster
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
// </action>
// ----
// The new master storage domain can be also specified by its name.
//
type DataCenterServiceSetMasterRequest struct {
	DataCenterService *DataCenterService
	header            map[string]string
	query             map[string]string
	async             *bool
	storageDomain     *StorageDomain
}

func (p *DataCenterServiceSetMasterRequest) Header(key, value string) *DataCenterServiceSetMasterRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterServiceSetMasterRequest) Query(key, value string) *DataCenterServiceSetMasterRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterServiceSetMasterRequest) Async(async bool) *DataCenterServiceSetMasterRequest {
	p.async = &async
	return p
}

func (p *DataCenterServiceSetMasterRequest) StorageDomain(storageDomain *StorageDomain) *DataCenterServiceSetMasterRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *DataCenterServiceSetMasterRequest) Send() (*DataCenterServiceSetMasterResponse, error) {
	rawURL := fmt.Sprintf("%s%s/setmaster", p.DataCenterService.connection.URL(), p.DataCenterService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DataCenterServiceSetMasterResponse), nil
}

func (p *DataCenterServiceSetMasterRequest) MustSend() *DataCenterServiceSetMasterResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Used for manually setting a storage domain in the data center as a master.
// For example, for setting a storage domain with ID '456' as a master on a data center with ID '123',
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/setmaster
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
// </action>
// ----
// The new master storage domain can be also specified by its name.
//
type DataCenterServiceSetMasterResponse struct {
}

//
// Used for manually setting a storage domain in the data center as a master.
// For example, for setting a storage domain with ID '456' as a master on a data center with ID '123',
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/setmaster
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
// </action>
// ----
// The new master storage domain can be also specified by its name.
//
func (p *DataCenterService) SetMaster() *DataCenterServiceSetMasterRequest {
	return &DataCenterServiceSetMasterRequest{DataCenterService: p}
}

//
// Updates the data center.
// The `name`, `description`, `storage_type`, `version`, `storage_format` and `mac_pool` elements are updatable
// post-creation. For example, to change the name and description of data center `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>myupdatedname</name>
//   <description>An updated description for the data center</description>
// </data_center>
// ----
//
type DataCenterServiceUpdateRequest struct {
	DataCenterService *DataCenterService
	header            map[string]string
	query             map[string]string
	async             *bool
	dataCenter        *DataCenter
}

func (p *DataCenterServiceUpdateRequest) Header(key, value string) *DataCenterServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterServiceUpdateRequest) Query(key, value string) *DataCenterServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterServiceUpdateRequest) Async(async bool) *DataCenterServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *DataCenterServiceUpdateRequest) DataCenter(dataCenter *DataCenter) *DataCenterServiceUpdateRequest {
	p.dataCenter = dataCenter
	return p
}

func (p *DataCenterServiceUpdateRequest) Send() (*DataCenterServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterService.connection.URL(), p.DataCenterService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDataCenterWriteOne(writer, p.dataCenter, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDataCenterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterServiceUpdateResponse{dataCenter: result}, nil
}

func (p *DataCenterServiceUpdateRequest) MustSend() *DataCenterServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the data center.
// The `name`, `description`, `storage_type`, `version`, `storage_format` and `mac_pool` elements are updatable
// post-creation. For example, to change the name and description of data center `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>myupdatedname</name>
//   <description>An updated description for the data center</description>
// </data_center>
// ----
//
type DataCenterServiceUpdateResponse struct {
	dataCenter *DataCenter
}

func (p *DataCenterServiceUpdateResponse) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *DataCenterServiceUpdateResponse) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("dataCenter in response does not exist")
	}
	return p.dataCenter
}

//
// Updates the data center.
// The `name`, `description`, `storage_type`, `version`, `storage_format` and `mac_pool` elements are updatable
// post-creation. For example, to change the name and description of data center `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>myupdatedname</name>
//   <description>An updated description for the data center</description>
// </data_center>
// ----
//
func (p *DataCenterService) Update() *DataCenterServiceUpdateRequest {
	return &DataCenterServiceUpdateRequest{DataCenterService: p}
}

//
//
func (op *DataCenterService) ClustersService() *ClustersService {
	return NewClustersService(op.connection, fmt.Sprintf("%s/clusters", op.path))
}

//
// Reference to the iSCSI bonds service.
//
func (op *DataCenterService) IscsiBondsService() *IscsiBondsService {
	return NewIscsiBondsService(op.connection, fmt.Sprintf("%s/iscsibonds", op.path))
}

//
// Returns a reference to the service, that manages the networks, that are associated with the data center.
//
func (op *DataCenterService) NetworksService() *DataCenterNetworksService {
	return NewDataCenterNetworksService(op.connection, fmt.Sprintf("%s/networks", op.path))
}

//
// Reference to the permissions service.
//
func (op *DataCenterService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Reference to the QOSs service.
//
func (op *DataCenterService) QossService() *QossService {
	return NewQossService(op.connection, fmt.Sprintf("%s/qoss", op.path))
}

//
// Reference to the quotas service.
//
func (op *DataCenterService) QuotasService() *QuotasService {
	return NewQuotasService(op.connection, fmt.Sprintf("%s/quotas", op.path))
}

//
// Attach and detach storage domains to and from a data center.
// For attaching a single storage domain we should use the following POST request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>data1</name>
// </storage_domain>
// ----
// For detaching a single storage domain we should use the following DELETE request:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123/storagedomains/123
// ----
//
func (op *DataCenterService) StorageDomainsService() *AttachedStorageDomainsService {
	return NewAttachedStorageDomainsService(op.connection, fmt.Sprintf("%s/storagedomains", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DataCenterService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "clusters" {
		return op.ClustersService(), nil
	}
	if strings.HasPrefix(path, "clusters/") {
		return op.ClustersService().Service(path[9:])
	}
	if path == "iscsibonds" {
		return op.IscsiBondsService(), nil
	}
	if strings.HasPrefix(path, "iscsibonds/") {
		return op.IscsiBondsService().Service(path[11:])
	}
	if path == "networks" {
		return op.NetworksService(), nil
	}
	if strings.HasPrefix(path, "networks/") {
		return op.NetworksService().Service(path[9:])
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "qoss" {
		return op.QossService(), nil
	}
	if strings.HasPrefix(path, "qoss/") {
		return op.QossService().Service(path[5:])
	}
	if path == "quotas" {
		return op.QuotasService(), nil
	}
	if strings.HasPrefix(path, "quotas/") {
		return op.QuotasService().Service(path[7:])
	}
	if path == "storagedomains" {
		return op.StorageDomainsService(), nil
	}
	if strings.HasPrefix(path, "storagedomains/") {
		return op.StorageDomainsService().Service(path[15:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DataCenterService) String() string {
	return fmt.Sprintf("DataCenterService:%s", op.path)
}

//
// A service to manage data centers.
//
type DataCentersService struct {
	BaseService
}

func NewDataCentersService(connection *Connection, path string) *DataCentersService {
	var result DataCentersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new data center.
// Creation of a new data center requires the `name` and `local` elements. For example, to create a data center
// named `mydc` that uses shared storage (NFS, iSCSI or fibre channel) send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>mydc</name>
//   <local>false</local>
// </data_center>
// ----
//
type DataCentersServiceAddRequest struct {
	DataCentersService *DataCentersService
	header             map[string]string
	query              map[string]string
	dataCenter         *DataCenter
}

func (p *DataCentersServiceAddRequest) Header(key, value string) *DataCentersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCentersServiceAddRequest) Query(key, value string) *DataCentersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCentersServiceAddRequest) DataCenter(dataCenter *DataCenter) *DataCentersServiceAddRequest {
	p.dataCenter = dataCenter
	return p
}

func (p *DataCentersServiceAddRequest) Send() (*DataCentersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCentersService.connection.URL(), p.DataCentersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDataCenterWriteOne(writer, p.dataCenter, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCentersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCentersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCentersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCentersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCentersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDataCenterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCentersServiceAddResponse{dataCenter: result}, nil
}

func (p *DataCentersServiceAddRequest) MustSend() *DataCentersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new data center.
// Creation of a new data center requires the `name` and `local` elements. For example, to create a data center
// named `mydc` that uses shared storage (NFS, iSCSI or fibre channel) send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>mydc</name>
//   <local>false</local>
// </data_center>
// ----
//
type DataCentersServiceAddResponse struct {
	dataCenter *DataCenter
}

func (p *DataCentersServiceAddResponse) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *DataCentersServiceAddResponse) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("dataCenter in response does not exist")
	}
	return p.dataCenter
}

//
// Creates a new data center.
// Creation of a new data center requires the `name` and `local` elements. For example, to create a data center
// named `mydc` that uses shared storage (NFS, iSCSI or fibre channel) send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>mydc</name>
//   <local>false</local>
// </data_center>
// ----
//
func (p *DataCentersService) Add() *DataCentersServiceAddRequest {
	return &DataCentersServiceAddRequest{DataCentersService: p}
}

//
// Lists the data centers.
// The following request retrieves a representation of the data centers:
// [source]
// ----
// GET /ovirt-engine/api/datacenters
// ----
// The above request performed with `curl`:
// [source,bash]
// ----
// curl \
// --request GET \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --header "Version: 4" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// https://myengine.example.com/ovirt-engine/api/datacenters
// ----
// This is what an example response could look like:
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//     </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
// </data_center>
// ----
// Note the `id` code of your `Default` data center. This code identifies this data center in relation to other
// resources of your virtual environment.
// The data center also contains a link to the storage domains collection. The data center uses this collection to
// attach storage domains from the storage domains main collection.
// The order of the returned list of data centers is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DataCentersServiceListRequest struct {
	DataCentersService *DataCentersService
	header             map[string]string
	query              map[string]string
	caseSensitive      *bool
	filter             *bool
	follow             *string
	max                *int64
	search             *string
}

func (p *DataCentersServiceListRequest) Header(key, value string) *DataCentersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCentersServiceListRequest) Query(key, value string) *DataCentersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCentersServiceListRequest) CaseSensitive(caseSensitive bool) *DataCentersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DataCentersServiceListRequest) Filter(filter bool) *DataCentersServiceListRequest {
	p.filter = &filter
	return p
}

func (p *DataCentersServiceListRequest) Follow(follow string) *DataCentersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DataCentersServiceListRequest) Max(max int64) *DataCentersServiceListRequest {
	p.max = &max
	return p
}

func (p *DataCentersServiceListRequest) Search(search string) *DataCentersServiceListRequest {
	p.search = &search
	return p
}

func (p *DataCentersServiceListRequest) Send() (*DataCentersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCentersService.connection.URL(), p.DataCentersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCentersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCentersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCentersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCentersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCentersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDataCenterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DataCentersServiceListResponse{dataCenters: result}, nil
}

func (p *DataCentersServiceListRequest) MustSend() *DataCentersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the data centers.
// The following request retrieves a representation of the data centers:
// [source]
// ----
// GET /ovirt-engine/api/datacenters
// ----
// The above request performed with `curl`:
// [source,bash]
// ----
// curl \
// --request GET \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --header "Version: 4" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// https://myengine.example.com/ovirt-engine/api/datacenters
// ----
// This is what an example response could look like:
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//     </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
// </data_center>
// ----
// Note the `id` code of your `Default` data center. This code identifies this data center in relation to other
// resources of your virtual environment.
// The data center also contains a link to the storage domains collection. The data center uses this collection to
// attach storage domains from the storage domains main collection.
// The order of the returned list of data centers is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DataCentersServiceListResponse struct {
	dataCenters *DataCenterSlice
}

func (p *DataCentersServiceListResponse) DataCenters() (*DataCenterSlice, bool) {
	if p.dataCenters != nil {
		return p.dataCenters, true
	}
	return nil, false
}

func (p *DataCentersServiceListResponse) MustDataCenters() *DataCenterSlice {
	if p.dataCenters == nil {
		panic("dataCenters in response does not exist")
	}
	return p.dataCenters
}

//
// Lists the data centers.
// The following request retrieves a representation of the data centers:
// [source]
// ----
// GET /ovirt-engine/api/datacenters
// ----
// The above request performed with `curl`:
// [source,bash]
// ----
// curl \
// --request GET \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --header "Version: 4" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// https://myengine.example.com/ovirt-engine/api/datacenters
// ----
// This is what an example response could look like:
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//     </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
// </data_center>
// ----
// Note the `id` code of your `Default` data center. This code identifies this data center in relation to other
// resources of your virtual environment.
// The data center also contains a link to the storage domains collection. The data center uses this collection to
// attach storage domains from the storage domains main collection.
// The order of the returned list of data centers is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
func (p *DataCentersService) List() *DataCentersServiceListRequest {
	return &DataCentersServiceListRequest{DataCentersService: p}
}

//
// Reference to the service that manages a specific data center.
//
func (op *DataCentersService) DataCenterService(id string) *DataCenterService {
	return NewDataCenterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DataCentersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DataCenterService(path), nil
	}
	return op.DataCenterService(path[:index]).Service(path[index+1:])
}

func (op *DataCentersService) String() string {
	return fmt.Sprintf("DataCentersService:%s", op.path)
}

//
// This service manages the attachment of a disk to a virtual machine.
//
type DiskAttachmentService struct {
	BaseService
}

func NewDiskAttachmentService(connection *Connection, path string) *DiskAttachmentService {
	var result DiskAttachmentService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the details of the attachment, including the bootable flag and link to the disk.
// An example of getting a disk attachment:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/diskattachments/456
// ----
// [source,xml]
// ----
// <disk_attachment href="/ovirt-engine/api/vms/123/diskattachments/456" id="456">
//   <active>true</active>
//   <bootable>true</bootable>
//   <interface>virtio</interface>
//   <disk href="/ovirt-engine/api/disks/456" id="456"/>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </disk_attachment>
// ----
//
type DiskAttachmentServiceGetRequest struct {
	DiskAttachmentService *DiskAttachmentService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *DiskAttachmentServiceGetRequest) Header(key, value string) *DiskAttachmentServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentServiceGetRequest) Query(key, value string) *DiskAttachmentServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentServiceGetRequest) Follow(follow string) *DiskAttachmentServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DiskAttachmentServiceGetRequest) Send() (*DiskAttachmentServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentService.connection.URL(), p.DiskAttachmentService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentServiceGetResponse{attachment: result}, nil
}

func (p *DiskAttachmentServiceGetRequest) MustSend() *DiskAttachmentServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the details of the attachment, including the bootable flag and link to the disk.
// An example of getting a disk attachment:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/diskattachments/456
// ----
// [source,xml]
// ----
// <disk_attachment href="/ovirt-engine/api/vms/123/diskattachments/456" id="456">
//   <active>true</active>
//   <bootable>true</bootable>
//   <interface>virtio</interface>
//   <disk href="/ovirt-engine/api/disks/456" id="456"/>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </disk_attachment>
// ----
//
type DiskAttachmentServiceGetResponse struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentServiceGetResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentServiceGetResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
// Returns the details of the attachment, including the bootable flag and link to the disk.
// An example of getting a disk attachment:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/diskattachments/456
// ----
// [source,xml]
// ----
// <disk_attachment href="/ovirt-engine/api/vms/123/diskattachments/456" id="456">
//   <active>true</active>
//   <bootable>true</bootable>
//   <interface>virtio</interface>
//   <disk href="/ovirt-engine/api/disks/456" id="456"/>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </disk_attachment>
// ----
//
func (p *DiskAttachmentService) Get() *DiskAttachmentServiceGetRequest {
	return &DiskAttachmentServiceGetRequest{DiskAttachmentService: p}
}

//
// Removes the disk attachment.
// This will only detach the disk from the virtual machine, but won't remove it from
// the system, unless the `detach_only` parameter is `false`.
// An example of removing a disk attachment:
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/diskattachments/456?detach_only=true
// ----
//
type DiskAttachmentServiceRemoveRequest struct {
	DiskAttachmentService *DiskAttachmentService
	header                map[string]string
	query                 map[string]string
	detachOnly            *bool
}

func (p *DiskAttachmentServiceRemoveRequest) Header(key, value string) *DiskAttachmentServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentServiceRemoveRequest) Query(key, value string) *DiskAttachmentServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentServiceRemoveRequest) DetachOnly(detachOnly bool) *DiskAttachmentServiceRemoveRequest {
	p.detachOnly = &detachOnly
	return p
}

func (p *DiskAttachmentServiceRemoveRequest) Send() (*DiskAttachmentServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentService.connection.URL(), p.DiskAttachmentService.path)
	values := make(url.Values)
	if p.detachOnly != nil {
		values["detach_only"] = []string{fmt.Sprintf("%v", *p.detachOnly)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DiskAttachmentServiceRemoveResponse), nil
}

func (p *DiskAttachmentServiceRemoveRequest) MustSend() *DiskAttachmentServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the disk attachment.
// This will only detach the disk from the virtual machine, but won't remove it from
// the system, unless the `detach_only` parameter is `false`.
// An example of removing a disk attachment:
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/diskattachments/456?detach_only=true
// ----
//
type DiskAttachmentServiceRemoveResponse struct {
}

//
// Removes the disk attachment.
// This will only detach the disk from the virtual machine, but won't remove it from
// the system, unless the `detach_only` parameter is `false`.
// An example of removing a disk attachment:
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/diskattachments/456?detach_only=true
// ----
//
func (p *DiskAttachmentService) Remove() *DiskAttachmentServiceRemoveRequest {
	return &DiskAttachmentServiceRemoveRequest{DiskAttachmentService: p}
}

//
// Update the disk attachment and the disk properties within it.
// [source]
// ----
// PUT /vms/{vm:id}/disksattachments/{attachment:id}
// <disk_attachment>
//   <bootable>true</bootable>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
//
type DiskAttachmentServiceUpdateRequest struct {
	DiskAttachmentService *DiskAttachmentService
	header                map[string]string
	query                 map[string]string
	diskAttachment        *DiskAttachment
}

func (p *DiskAttachmentServiceUpdateRequest) Header(key, value string) *DiskAttachmentServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentServiceUpdateRequest) Query(key, value string) *DiskAttachmentServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentServiceUpdateRequest) DiskAttachment(diskAttachment *DiskAttachment) *DiskAttachmentServiceUpdateRequest {
	p.diskAttachment = diskAttachment
	return p
}

func (p *DiskAttachmentServiceUpdateRequest) Send() (*DiskAttachmentServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentService.connection.URL(), p.DiskAttachmentService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskAttachmentWriteOne(writer, p.diskAttachment, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentServiceUpdateResponse{diskAttachment: result}, nil
}

func (p *DiskAttachmentServiceUpdateRequest) MustSend() *DiskAttachmentServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the disk attachment and the disk properties within it.
// [source]
// ----
// PUT /vms/{vm:id}/disksattachments/{attachment:id}
// <disk_attachment>
//   <bootable>true</bootable>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
//
type DiskAttachmentServiceUpdateResponse struct {
	diskAttachment *DiskAttachment
}

func (p *DiskAttachmentServiceUpdateResponse) DiskAttachment() (*DiskAttachment, bool) {
	if p.diskAttachment != nil {
		return p.diskAttachment, true
	}
	return nil, false
}

func (p *DiskAttachmentServiceUpdateResponse) MustDiskAttachment() *DiskAttachment {
	if p.diskAttachment == nil {
		panic("diskAttachment in response does not exist")
	}
	return p.diskAttachment
}

//
// Update the disk attachment and the disk properties within it.
// [source]
// ----
// PUT /vms/{vm:id}/disksattachments/{attachment:id}
// <disk_attachment>
//   <bootable>true</bootable>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
//
func (p *DiskAttachmentService) Update() *DiskAttachmentServiceUpdateRequest {
	return &DiskAttachmentServiceUpdateRequest{DiskAttachmentService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskAttachmentService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DiskAttachmentService) String() string {
	return fmt.Sprintf("DiskAttachmentService:%s", op.path)
}

//
// This service manages the set of disks attached to a virtual machine. Each attached disk is represented by a
// <<types/disk_attachment,DiskAttachment>>, containing the bootable flag, the disk interface and the reference to
// the disk.
//
type DiskAttachmentsService struct {
	BaseService
}

func NewDiskAttachmentsService(connection *Connection, path string) *DiskAttachmentsService {
	var result DiskAttachmentsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
type DiskAttachmentsServiceAddRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddRequest) Header(key, value string) *DiskAttachmentsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Query(key, value string) *DiskAttachmentsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddRequest {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Send() (*DiskAttachmentsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskAttachmentWriteOne(writer, p.attachment, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentsServiceAddResponse{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddRequest) MustSend() *DiskAttachmentsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
type DiskAttachmentsServiceAddResponse struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
func (p *DiskAttachmentsService) Add() *DiskAttachmentsServiceAddRequest {
	return &DiskAttachmentsServiceAddRequest{DiskAttachmentsService: p}
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
type DiskAttachmentsServiceListRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *DiskAttachmentsServiceListRequest) Header(key, value string) *DiskAttachmentsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceListRequest) Query(key, value string) *DiskAttachmentsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceListRequest) Follow(follow string) *DiskAttachmentsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DiskAttachmentsServiceListRequest) Send() (*DiskAttachmentsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentsServiceListResponse{attachments: result}, nil
}

func (p *DiskAttachmentsServiceListRequest) MustSend() *DiskAttachmentsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
type DiskAttachmentsServiceListResponse struct {
	attachments *DiskAttachmentSlice
}

func (p *DiskAttachmentsServiceListResponse) Attachments() (*DiskAttachmentSlice, bool) {
	if p.attachments != nil {
		return p.attachments, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceListResponse) MustAttachments() *DiskAttachmentSlice {
	if p.attachments == nil {
		panic("attachments in response does not exist")
	}
	return p.attachments
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
func (p *DiskAttachmentsService) List() *DiskAttachmentsServiceListRequest {
	return &DiskAttachmentsServiceListRequest{DiskAttachmentsService: p}
}

//
//
type DiskAttachmentsServiceAddProvidingDiskIdRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Header(key, value string) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Query(key, value string) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Send() (*DiskAttachmentsServiceAddProvidingDiskIdResponse, error) {
	rawURL := fmt.Sprintf("%s%s/providingdiskid", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Attachment(p.attachment)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustAttachment()
	return &DiskAttachmentsServiceAddProvidingDiskIdResponse{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) MustSend() *DiskAttachmentsServiceAddProvidingDiskIdResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskAttachmentsServiceAddProvidingDiskIdResponse struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
//
func (p *DiskAttachmentsService) AddProvidingDiskId() *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	return &DiskAttachmentsServiceAddProvidingDiskIdRequest{DiskAttachmentsService: p}
}

//
//
type DiskAttachmentsServiceAddSignature1Request struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddSignature1Request) Header(key, value string) *DiskAttachmentsServiceAddSignature1Request {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Query(key, value string) *DiskAttachmentsServiceAddSignature1Request {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddSignature1Request {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Send() (*DiskAttachmentsServiceAddSignature1Response, error) {
	rawURL := fmt.Sprintf("%s%s/signature1", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Attachment(p.attachment)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustAttachment()
	return &DiskAttachmentsServiceAddSignature1Response{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddSignature1Request) MustSend() *DiskAttachmentsServiceAddSignature1Response {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskAttachmentsServiceAddSignature1Response struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddSignature1Response) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddSignature1Response) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
//
func (p *DiskAttachmentsService) AddSignature1() *DiskAttachmentsServiceAddSignature1Request {
	return &DiskAttachmentsServiceAddSignature1Request{DiskAttachmentsService: p}
}

//
// Reference to the service that manages a specific attachment.
//
func (op *DiskAttachmentsService) AttachmentService(id string) *DiskAttachmentService {
	return NewDiskAttachmentService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskAttachmentsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.AttachmentService(path), nil
	}
	return op.AttachmentService(path[:index]).Service(path[index+1:])
}

func (op *DiskAttachmentsService) String() string {
	return fmt.Sprintf("DiskAttachmentsService:%s", op.path)
}

//
//
type DiskProfileService struct {
	BaseService
}

func NewDiskProfileService(connection *Connection, path string) *DiskProfileService {
	var result DiskProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type DiskProfileServiceGetRequest struct {
	DiskProfileService *DiskProfileService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *DiskProfileServiceGetRequest) Header(key, value string) *DiskProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfileServiceGetRequest) Query(key, value string) *DiskProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfileServiceGetRequest) Follow(follow string) *DiskProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DiskProfileServiceGetRequest) Send() (*DiskProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfileService.connection.URL(), p.DiskProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskProfileServiceGetResponse{profile: result}, nil
}

func (p *DiskProfileServiceGetRequest) MustSend() *DiskProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskProfileServiceGetResponse struct {
	profile *DiskProfile
}

func (p *DiskProfileServiceGetResponse) Profile() (*DiskProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *DiskProfileServiceGetResponse) MustProfile() *DiskProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *DiskProfileService) Get() *DiskProfileServiceGetRequest {
	return &DiskProfileServiceGetRequest{DiskProfileService: p}
}

//
//
type DiskProfileServiceRemoveRequest struct {
	DiskProfileService *DiskProfileService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *DiskProfileServiceRemoveRequest) Header(key, value string) *DiskProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfileServiceRemoveRequest) Query(key, value string) *DiskProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfileServiceRemoveRequest) Async(async bool) *DiskProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *DiskProfileServiceRemoveRequest) Send() (*DiskProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfileService.connection.URL(), p.DiskProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DiskProfileServiceRemoveResponse), nil
}

func (p *DiskProfileServiceRemoveRequest) MustSend() *DiskProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskProfileServiceRemoveResponse struct {
}

//
//
func (p *DiskProfileService) Remove() *DiskProfileServiceRemoveRequest {
	return &DiskProfileServiceRemoveRequest{DiskProfileService: p}
}

//
// Update the specified disk profile in the system.
//
type DiskProfileServiceUpdateRequest struct {
	DiskProfileService *DiskProfileService
	header             map[string]string
	query              map[string]string
	async              *bool
	profile            *DiskProfile
}

func (p *DiskProfileServiceUpdateRequest) Header(key, value string) *DiskProfileServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfileServiceUpdateRequest) Query(key, value string) *DiskProfileServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfileServiceUpdateRequest) Async(async bool) *DiskProfileServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *DiskProfileServiceUpdateRequest) Profile(profile *DiskProfile) *DiskProfileServiceUpdateRequest {
	p.profile = profile
	return p
}

func (p *DiskProfileServiceUpdateRequest) Send() (*DiskProfileServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfileService.connection.URL(), p.DiskProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskProfileServiceUpdateResponse{profile: result}, nil
}

func (p *DiskProfileServiceUpdateRequest) MustSend() *DiskProfileServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified disk profile in the system.
//
type DiskProfileServiceUpdateResponse struct {
	profile *DiskProfile
}

func (p *DiskProfileServiceUpdateResponse) Profile() (*DiskProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *DiskProfileServiceUpdateResponse) MustProfile() *DiskProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Update the specified disk profile in the system.
//
func (p *DiskProfileService) Update() *DiskProfileServiceUpdateRequest {
	return &DiskProfileServiceUpdateRequest{DiskProfileService: p}
}

//
//
func (op *DiskProfileService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DiskProfileService) String() string {
	return fmt.Sprintf("DiskProfileService:%s", op.path)
}

//
//
type DiskProfilesService struct {
	BaseService
}

func NewDiskProfilesService(connection *Connection, path string) *DiskProfilesService {
	var result DiskProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new disk profile to the system.
//
type DiskProfilesServiceAddRequest struct {
	DiskProfilesService *DiskProfilesService
	header              map[string]string
	query               map[string]string
	profile             *DiskProfile
}

func (p *DiskProfilesServiceAddRequest) Header(key, value string) *DiskProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfilesServiceAddRequest) Query(key, value string) *DiskProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfilesServiceAddRequest) Profile(profile *DiskProfile) *DiskProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *DiskProfilesServiceAddRequest) Send() (*DiskProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfilesService.connection.URL(), p.DiskProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskProfilesServiceAddResponse{profile: result}, nil
}

func (p *DiskProfilesServiceAddRequest) MustSend() *DiskProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new disk profile to the system.
//
type DiskProfilesServiceAddResponse struct {
	profile *DiskProfile
}

func (p *DiskProfilesServiceAddResponse) Profile() (*DiskProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *DiskProfilesServiceAddResponse) MustProfile() *DiskProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a new disk profile to the system.
//
func (p *DiskProfilesService) Add() *DiskProfilesServiceAddRequest {
	return &DiskProfilesServiceAddRequest{DiskProfilesService: p}
}

//
// Returns the list of disk profiles of the system.
// The order of the returned list of disk profiles isn't guaranteed.
//
type DiskProfilesServiceListRequest struct {
	DiskProfilesService *DiskProfilesService
	header              map[string]string
	query               map[string]string
	follow              *string
	max                 *int64
}

func (p *DiskProfilesServiceListRequest) Header(key, value string) *DiskProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfilesServiceListRequest) Query(key, value string) *DiskProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfilesServiceListRequest) Follow(follow string) *DiskProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DiskProfilesServiceListRequest) Max(max int64) *DiskProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *DiskProfilesServiceListRequest) Send() (*DiskProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfilesService.connection.URL(), p.DiskProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DiskProfilesServiceListResponse{profile: result}, nil
}

func (p *DiskProfilesServiceListRequest) MustSend() *DiskProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of disk profiles of the system.
// The order of the returned list of disk profiles isn't guaranteed.
//
type DiskProfilesServiceListResponse struct {
	profile *DiskProfileSlice
}

func (p *DiskProfilesServiceListResponse) Profile() (*DiskProfileSlice, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *DiskProfilesServiceListResponse) MustProfile() *DiskProfileSlice {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Returns the list of disk profiles of the system.
// The order of the returned list of disk profiles isn't guaranteed.
//
func (p *DiskProfilesService) List() *DiskProfilesServiceListRequest {
	return &DiskProfilesServiceListRequest{DiskProfilesService: p}
}

//
//
func (op *DiskProfilesService) DiskProfileService(id string) *DiskProfileService {
	return NewDiskProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskProfileService(path), nil
	}
	return op.DiskProfileService(path[:index]).Service(path[index+1:])
}

func (op *DiskProfilesService) String() string {
	return fmt.Sprintf("DiskProfilesService:%s", op.path)
}

//
// Manages a single disk.
//
type DiskService struct {
	BaseService
}

func NewDiskService(connection *Connection, path string) *DiskService {
	var result DiskService
	result.connection = connection
	result.path = path
	return &result
}

//
// This operation copies a disk to the specified storage domain.
// For example, a disk can be copied using the following request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/copy
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk>
//     <name>mydisk</name>
//   </disk>
// </action>
// ----
// If the disk profile or the quota currently used by the disk are not defined for the new storage domain, they
// can be explicitly specified. If they are not specified, the first available disk profile and the default quota are used.
// For example, to specify disk profile `987` and quota `753`, send a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk_profile id="987"/>
//   <quota id="753"/>
// </action>
// ----
//
type DiskServiceCopyRequest struct {
	DiskService   *DiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	disk          *Disk
	diskProfile   *DiskProfile
	filter        *bool
	quota         *Quota
	storageDomain *StorageDomain
}

func (p *DiskServiceCopyRequest) Header(key, value string) *DiskServiceCopyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceCopyRequest) Query(key, value string) *DiskServiceCopyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceCopyRequest) Async(async bool) *DiskServiceCopyRequest {
	p.async = &async
	return p
}

func (p *DiskServiceCopyRequest) Disk(disk *Disk) *DiskServiceCopyRequest {
	p.disk = disk
	return p
}

func (p *DiskServiceCopyRequest) DiskProfile(diskProfile *DiskProfile) *DiskServiceCopyRequest {
	p.diskProfile = diskProfile
	return p
}

func (p *DiskServiceCopyRequest) Filter(filter bool) *DiskServiceCopyRequest {
	p.filter = &filter
	return p
}

func (p *DiskServiceCopyRequest) Quota(quota *Quota) *DiskServiceCopyRequest {
	p.quota = quota
	return p
}

func (p *DiskServiceCopyRequest) StorageDomain(storageDomain *StorageDomain) *DiskServiceCopyRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *DiskServiceCopyRequest) Send() (*DiskServiceCopyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copy", p.DiskService.connection.URL(), p.DiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Disk(p.disk)
	actionBuilder.DiskProfile(p.diskProfile)
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.Quota(p.quota)
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DiskServiceCopyResponse), nil
}

func (p *DiskServiceCopyRequest) MustSend() *DiskServiceCopyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation copies a disk to the specified storage domain.
// For example, a disk can be copied using the following request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/copy
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk>
//     <name>mydisk</name>
//   </disk>
// </action>
// ----
// If the disk profile or the quota currently used by the disk are not defined for the new storage domain, they
// can be explicitly specified. If they are not specified, the first available disk profile and the default quota are used.
// For example, to specify disk profile `987` and quota `753`, send a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk_profile id="987"/>
//   <quota id="753"/>
// </action>
// ----
//
type DiskServiceCopyResponse struct {
}

//
// This operation copies a disk to the specified storage domain.
// For example, a disk can be copied using the following request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/copy
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk>
//     <name>mydisk</name>
//   </disk>
// </action>
// ----
// If the disk profile or the quota currently used by the disk are not defined for the new storage domain, they
// can be explicitly specified. If they are not specified, the first available disk profile and the default quota are used.
// For example, to specify disk profile `987` and quota `753`, send a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk_profile id="987"/>
//   <quota id="753"/>
// </action>
// ----
//
func (p *DiskService) Copy() *DiskServiceCopyRequest {
	return &DiskServiceCopyRequest{DiskService: p}
}

//
// Exports a disk to an export storage domain.
//
type DiskServiceExportRequest struct {
	DiskService   *DiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	filter        *bool
	storageDomain *StorageDomain
}

func (p *DiskServiceExportRequest) Header(key, value string) *DiskServiceExportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceExportRequest) Query(key, value string) *DiskServiceExportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceExportRequest) Async(async bool) *DiskServiceExportRequest {
	p.async = &async
	return p
}

func (p *DiskServiceExportRequest) Filter(filter bool) *DiskServiceExportRequest {
	p.filter = &filter
	return p
}

func (p *DiskServiceExportRequest) StorageDomain(storageDomain *StorageDomain) *DiskServiceExportRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *DiskServiceExportRequest) Send() (*DiskServiceExportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/export", p.DiskService.connection.URL(), p.DiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DiskServiceExportResponse), nil
}

func (p *DiskServiceExportRequest) MustSend() *DiskServiceExportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Exports a disk to an export storage domain.
//
type DiskServiceExportResponse struct {
}

//
// Exports a disk to an export storage domain.
//
func (p *DiskService) Export() *DiskServiceExportRequest {
	return &DiskServiceExportRequest{DiskService: p}
}

//
// Retrieves the description of the disk.
//
type DiskServiceGetRequest struct {
	DiskService *DiskService
	header      map[string]string
	query       map[string]string
	allContent  *bool
	follow      *string
}

func (p *DiskServiceGetRequest) Header(key, value string) *DiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceGetRequest) Query(key, value string) *DiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceGetRequest) AllContent(allContent bool) *DiskServiceGetRequest {
	p.allContent = &allContent
	return p
}

func (p *DiskServiceGetRequest) Follow(follow string) *DiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DiskServiceGetRequest) Send() (*DiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskService.connection.URL(), p.DiskService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskServiceGetResponse{disk: result}, nil
}

func (p *DiskServiceGetRequest) MustSend() *DiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the description of the disk.
//
type DiskServiceGetResponse struct {
	disk *Disk
}

func (p *DiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Retrieves the description of the disk.
//
func (p *DiskService) Get() *DiskServiceGetRequest {
	return &DiskServiceGetRequest{DiskService: p}
}

//
// Moves a disk to another storage domain.
// For example, to move the disk with identifier `123` to a storage domain with identifier `456` send the following
// request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/move
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
// </action>
// ----
// If the disk profile or the quota used currently by
// the disk aren't defined for the new storage domain,
// then they can be explicitly specified. If they aren't
// then the first available disk profile and the default
// quota are used.
// For example, to explicitly use disk profile `987` and
// quota `753` send a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk_profile id="987"/>
//   <quota id="753"/>
// </action>
// ----
//
type DiskServiceMoveRequest struct {
	DiskService   *DiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	diskProfile   *DiskProfile
	filter        *bool
	quota         *Quota
	storageDomain *StorageDomain
}

func (p *DiskServiceMoveRequest) Header(key, value string) *DiskServiceMoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceMoveRequest) Query(key, value string) *DiskServiceMoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceMoveRequest) Async(async bool) *DiskServiceMoveRequest {
	p.async = &async
	return p
}

func (p *DiskServiceMoveRequest) DiskProfile(diskProfile *DiskProfile) *DiskServiceMoveRequest {
	p.diskProfile = diskProfile
	return p
}

func (p *DiskServiceMoveRequest) Filter(filter bool) *DiskServiceMoveRequest {
	p.filter = &filter
	return p
}

func (p *DiskServiceMoveRequest) Quota(quota *Quota) *DiskServiceMoveRequest {
	p.quota = quota
	return p
}

func (p *DiskServiceMoveRequest) StorageDomain(storageDomain *StorageDomain) *DiskServiceMoveRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *DiskServiceMoveRequest) Send() (*DiskServiceMoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/move", p.DiskService.connection.URL(), p.DiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.DiskProfile(p.diskProfile)
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.Quota(p.quota)
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DiskServiceMoveResponse), nil
}

func (p *DiskServiceMoveRequest) MustSend() *DiskServiceMoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Moves a disk to another storage domain.
// For example, to move the disk with identifier `123` to a storage domain with identifier `456` send the following
// request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/move
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
// </action>
// ----
// If the disk profile or the quota used currently by
// the disk aren't defined for the new storage domain,
// then they can be explicitly specified. If they aren't
// then the first available disk profile and the default
// quota are used.
// For example, to explicitly use disk profile `987` and
// quota `753` send a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk_profile id="987"/>
//   <quota id="753"/>
// </action>
// ----
//
type DiskServiceMoveResponse struct {
}

//
// Moves a disk to another storage domain.
// For example, to move the disk with identifier `123` to a storage domain with identifier `456` send the following
// request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/move
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
// </action>
// ----
// If the disk profile or the quota used currently by
// the disk aren't defined for the new storage domain,
// then they can be explicitly specified. If they aren't
// then the first available disk profile and the default
// quota are used.
// For example, to explicitly use disk profile `987` and
// quota `753` send a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain id="456"/>
//   <disk_profile id="987"/>
//   <quota id="753"/>
// </action>
// ----
//
func (p *DiskService) Move() *DiskServiceMoveRequest {
	return &DiskServiceMoveRequest{DiskService: p}
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
type DiskServiceReduceRequest struct {
	DiskService *DiskService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *DiskServiceReduceRequest) Header(key, value string) *DiskServiceReduceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceReduceRequest) Query(key, value string) *DiskServiceReduceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceReduceRequest) Async(async bool) *DiskServiceReduceRequest {
	p.async = &async
	return p
}

func (p *DiskServiceReduceRequest) Send() (*DiskServiceReduceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/reduce", p.DiskService.connection.URL(), p.DiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DiskServiceReduceResponse), nil
}

func (p *DiskServiceReduceRequest) MustSend() *DiskServiceReduceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
type DiskServiceReduceResponse struct {
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
func (p *DiskService) Reduce() *DiskServiceReduceRequest {
	return &DiskServiceReduceRequest{DiskService: p}
}

//
// Refreshes a direct LUN disk with up-to-date information from the storage.
// Refreshing a direct LUN disk is useful when:
// - The LUN was added using the API without the host parameter, and therefore does not contain
//   any information from the storage (see <<services/disks/methods/add, DisksService::add>>).
// - New information about the LUN is available on the storage and you want to update the LUN with it.
// To refresh direct LUN disk `123` using host `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/refreshlun
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <host id='456'/>
// </action>
// ----
//
type DiskServiceRefreshLunRequest struct {
	DiskService *DiskService
	header      map[string]string
	query       map[string]string
	host        *Host
}

func (p *DiskServiceRefreshLunRequest) Header(key, value string) *DiskServiceRefreshLunRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceRefreshLunRequest) Query(key, value string) *DiskServiceRefreshLunRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceRefreshLunRequest) Host(host *Host) *DiskServiceRefreshLunRequest {
	p.host = host
	return p
}

func (p *DiskServiceRefreshLunRequest) Send() (*DiskServiceRefreshLunResponse, error) {
	rawURL := fmt.Sprintf("%s%s/refreshlun", p.DiskService.connection.URL(), p.DiskService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Host(p.host)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DiskServiceRefreshLunResponse), nil
}

func (p *DiskServiceRefreshLunRequest) MustSend() *DiskServiceRefreshLunResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Refreshes a direct LUN disk with up-to-date information from the storage.
// Refreshing a direct LUN disk is useful when:
// - The LUN was added using the API without the host parameter, and therefore does not contain
//   any information from the storage (see <<services/disks/methods/add, DisksService::add>>).
// - New information about the LUN is available on the storage and you want to update the LUN with it.
// To refresh direct LUN disk `123` using host `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/refreshlun
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <host id='456'/>
// </action>
// ----
//
type DiskServiceRefreshLunResponse struct {
}

//
// Refreshes a direct LUN disk with up-to-date information from the storage.
// Refreshing a direct LUN disk is useful when:
// - The LUN was added using the API without the host parameter, and therefore does not contain
//   any information from the storage (see <<services/disks/methods/add, DisksService::add>>).
// - New information about the LUN is available on the storage and you want to update the LUN with it.
// To refresh direct LUN disk `123` using host `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/disks/123/refreshlun
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <host id='456'/>
// </action>
// ----
//
func (p *DiskService) RefreshLun() *DiskServiceRefreshLunRequest {
	return &DiskServiceRefreshLunRequest{DiskService: p}
}

//
// Removes a disk.
//
type DiskServiceRemoveRequest struct {
	DiskService *DiskService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *DiskServiceRemoveRequest) Header(key, value string) *DiskServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceRemoveRequest) Query(key, value string) *DiskServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceRemoveRequest) Async(async bool) *DiskServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *DiskServiceRemoveRequest) Send() (*DiskServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskService.connection.URL(), p.DiskService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DiskServiceRemoveResponse), nil
}

func (p *DiskServiceRemoveRequest) MustSend() *DiskServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a disk.
//
type DiskServiceRemoveResponse struct {
}

//
// Removes a disk.
//
func (p *DiskService) Remove() *DiskServiceRemoveRequest {
	return &DiskServiceRemoveRequest{DiskService: p}
}

//
// Sparsify the disk.
// Sparsification frees space in the disk image that is not used by its
// filesystem. As a result, the image will occupy less space on the storage.
// Currently sparsification works only on disks without snapshots. Disks
// having derived disks are also not allowed.
//
type DiskServiceSparsifyRequest struct {
	DiskService *DiskService
	header      map[string]string
	query       map[string]string
}

func (p *DiskServiceSparsifyRequest) Header(key, value string) *DiskServiceSparsifyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceSparsifyRequest) Query(key, value string) *DiskServiceSparsifyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceSparsifyRequest) Send() (*DiskServiceSparsifyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/sparsify", p.DiskService.connection.URL(), p.DiskService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(DiskServiceSparsifyResponse), nil
}

func (p *DiskServiceSparsifyRequest) MustSend() *DiskServiceSparsifyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Sparsify the disk.
// Sparsification frees space in the disk image that is not used by its
// filesystem. As a result, the image will occupy less space on the storage.
// Currently sparsification works only on disks without snapshots. Disks
// having derived disks are also not allowed.
//
type DiskServiceSparsifyResponse struct {
}

//
// Sparsify the disk.
// Sparsification frees space in the disk image that is not used by its
// filesystem. As a result, the image will occupy less space on the storage.
// Currently sparsification works only on disks without snapshots. Disks
// having derived disks are also not allowed.
//
func (p *DiskService) Sparsify() *DiskServiceSparsifyRequest {
	return &DiskServiceSparsifyRequest{DiskService: p}
}

//
// Updates the parameters of the specified disk.
// This operation allows updating the following floating disk properties:
// * For Image disks: `provisioned_size`, `alias`, `description`, `wipe_after_delete`, `shareable`, `backup` and `disk_profile`.
// * For LUN disks: `alias`, `description` and `shareable`.
// * For Cinder and Managed Block disks: `provisioned_size`, `alias` and `description`.
// * For VM attached disks, the `qcow_version` can also be updated.
// For example, a disk's update can be done by using the following request:
// [source]
// ----
// PUT /ovirt-engine/api/disks/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <disk>
//   <qcow_version>qcow2_v3</qcow_version>
//   <alias>new-alias</alias>
//   <description>new-desc</description>
// </disk>
// ----
// Since the backend operation is asynchronous, the disk element that is returned
// to the user might not be synced with the changed properties.
//
type DiskServiceUpdateRequest struct {
	DiskService *DiskService
	header      map[string]string
	query       map[string]string
	disk        *Disk
}

func (p *DiskServiceUpdateRequest) Header(key, value string) *DiskServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskServiceUpdateRequest) Query(key, value string) *DiskServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskServiceUpdateRequest) Disk(disk *Disk) *DiskServiceUpdateRequest {
	p.disk = disk
	return p
}

func (p *DiskServiceUpdateRequest) Send() (*DiskServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskService.connection.URL(), p.DiskService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskServiceUpdateResponse{disk: result}, nil
}

func (p *DiskServiceUpdateRequest) MustSend() *DiskServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the parameters of the specified disk.
// This operation allows updating the following floating disk properties:
// * For Image disks: `provisioned_size`, `alias`, `description`, `wipe_after_delete`, `shareable`, `backup` and `disk_profile`.
// * For LUN disks: `alias`, `description` and `shareable`.
// * For Cinder and Managed Block disks: `provisioned_size`, `alias` and `description`.
// * For VM attached disks, the `qcow_version` can also be updated.
// For example, a disk's update can be done by using the following request:
// [source]
// ----
// PUT /ovirt-engine/api/disks/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <disk>
//   <qcow_version>qcow2_v3</qcow_version>
//   <alias>new-alias</alias>
//   <description>new-desc</description>
// </disk>
// ----
// Since the backend operation is asynchronous, the disk element that is returned
// to the user might not be synced with the changed properties.
//
type DiskServiceUpdateResponse struct {
	disk *Disk
}

func (p *DiskServiceUpdateResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DiskServiceUpdateResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Updates the parameters of the specified disk.
// This operation allows updating the following floating disk properties:
// * For Image disks: `provisioned_size`, `alias`, `description`, `wipe_after_delete`, `shareable`, `backup` and `disk_profile`.
// * For LUN disks: `alias`, `description` and `shareable`.
// * For Cinder and Managed Block disks: `provisioned_size`, `alias` and `description`.
// * For VM attached disks, the `qcow_version` can also be updated.
// For example, a disk's update can be done by using the following request:
// [source]
// ----
// PUT /ovirt-engine/api/disks/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <disk>
//   <qcow_version>qcow2_v3</qcow_version>
//   <alias>new-alias</alias>
//   <description>new-desc</description>
// </disk>
// ----
// Since the backend operation is asynchronous, the disk element that is returned
// to the user might not be synced with the changed properties.
//
func (p *DiskService) Update() *DiskServiceUpdateRequest {
	return &DiskServiceUpdateRequest{DiskService: p}
}

//
// Reference to the service that manages the DiskSnapshots.
// For example, to list all disk snapshots under the disks resource '123':
// ....
// GET /ovirt-engine/api/disks/123/disksnapshots
// ....
// For example, to retrieve a specific disk snapshot '789' under the disk resource '123':
// ....
// GET /ovirt-engine/api/disks/123/disksnapshots/789
// ....
//
func (op *DiskService) DiskSnapshotsService() *DiskSnapshotsService {
	return NewDiskSnapshotsService(op.connection, fmt.Sprintf("%s/disksnapshots", op.path))
}

//
// Reference to the service that manages the permissions assigned to the disk.
//
func (op *DiskService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *DiskService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "disksnapshots" {
		return op.DiskSnapshotsService(), nil
	}
	if strings.HasPrefix(path, "disksnapshots/") {
		return op.DiskSnapshotsService().Service(path[14:])
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DiskService) String() string {
	return fmt.Sprintf("DiskService:%s", op.path)
}

//
//
type DiskSnapshotService struct {
	BaseService
}

func NewDiskSnapshotService(connection *Connection, path string) *DiskSnapshotService {
	var result DiskSnapshotService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type DiskSnapshotServiceGetRequest struct {
	DiskSnapshotService *DiskSnapshotService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *DiskSnapshotServiceGetRequest) Header(key, value string) *DiskSnapshotServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskSnapshotServiceGetRequest) Query(key, value string) *DiskSnapshotServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskSnapshotServiceGetRequest) Follow(follow string) *DiskSnapshotServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DiskSnapshotServiceGetRequest) Send() (*DiskSnapshotServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskSnapshotService.connection.URL(), p.DiskSnapshotService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskSnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskSnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskSnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskSnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskSnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskSnapshotReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskSnapshotServiceGetResponse{snapshot: result}, nil
}

func (p *DiskSnapshotServiceGetRequest) MustSend() *DiskSnapshotServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskSnapshotServiceGetResponse struct {
	snapshot *DiskSnapshot
}

func (p *DiskSnapshotServiceGetResponse) Snapshot() (*DiskSnapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

func (p *DiskSnapshotServiceGetResponse) MustSnapshot() *DiskSnapshot {
	if p.snapshot == nil {
		panic("snapshot in response does not exist")
	}
	return p.snapshot
}

//
//
func (p *DiskSnapshotService) Get() *DiskSnapshotServiceGetRequest {
	return &DiskSnapshotServiceGetRequest{DiskSnapshotService: p}
}

//
//
type DiskSnapshotServiceRemoveRequest struct {
	DiskSnapshotService *DiskSnapshotService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *DiskSnapshotServiceRemoveRequest) Header(key, value string) *DiskSnapshotServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Query(key, value string) *DiskSnapshotServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Async(async bool) *DiskSnapshotServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Send() (*DiskSnapshotServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskSnapshotService.connection.URL(), p.DiskSnapshotService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskSnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskSnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskSnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskSnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskSnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DiskSnapshotServiceRemoveResponse), nil
}

func (p *DiskSnapshotServiceRemoveRequest) MustSend() *DiskSnapshotServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskSnapshotServiceRemoveResponse struct {
}

//
//
func (p *DiskSnapshotService) Remove() *DiskSnapshotServiceRemoveRequest {
	return &DiskSnapshotServiceRemoveRequest{DiskSnapshotService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskSnapshotService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DiskSnapshotService) String() string {
	return fmt.Sprintf("DiskSnapshotService:%s", op.path)
}

//
// Manages the collection of disk snapshots available in an storage domain.
//
type DiskSnapshotsService struct {
	BaseService
}

func NewDiskSnapshotsService(connection *Connection, path string) *DiskSnapshotsService {
	var result DiskSnapshotsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of disk snapshots of the storage domain.
// The order of the returned list of disk snapshots isn't guaranteed.
//
type DiskSnapshotsServiceListRequest struct {
	DiskSnapshotsService *DiskSnapshotsService
	header               map[string]string
	query                map[string]string
	follow               *string
	includeActive        *bool
	max                  *int64
}

func (p *DiskSnapshotsServiceListRequest) Header(key, value string) *DiskSnapshotsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskSnapshotsServiceListRequest) Query(key, value string) *DiskSnapshotsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskSnapshotsServiceListRequest) Follow(follow string) *DiskSnapshotsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DiskSnapshotsServiceListRequest) IncludeActive(includeActive bool) *DiskSnapshotsServiceListRequest {
	p.includeActive = &includeActive
	return p
}

func (p *DiskSnapshotsServiceListRequest) Max(max int64) *DiskSnapshotsServiceListRequest {
	p.max = &max
	return p
}

func (p *DiskSnapshotsServiceListRequest) Send() (*DiskSnapshotsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskSnapshotsService.connection.URL(), p.DiskSnapshotsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.includeActive != nil {
		values["include_active"] = []string{fmt.Sprintf("%v", *p.includeActive)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskSnapshotsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskSnapshotsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskSnapshotsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskSnapshotsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskSnapshotsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskSnapshotReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DiskSnapshotsServiceListResponse{snapshots: result}, nil
}

func (p *DiskSnapshotsServiceListRequest) MustSend() *DiskSnapshotsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of disk snapshots of the storage domain.
// The order of the returned list of disk snapshots isn't guaranteed.
//
type DiskSnapshotsServiceListResponse struct {
	snapshots *DiskSnapshotSlice
}

func (p *DiskSnapshotsServiceListResponse) Snapshots() (*DiskSnapshotSlice, bool) {
	if p.snapshots != nil {
		return p.snapshots, true
	}
	return nil, false
}

func (p *DiskSnapshotsServiceListResponse) MustSnapshots() *DiskSnapshotSlice {
	if p.snapshots == nil {
		panic("snapshots in response does not exist")
	}
	return p.snapshots
}

//
// Returns the list of disk snapshots of the storage domain.
// The order of the returned list of disk snapshots isn't guaranteed.
//
func (p *DiskSnapshotsService) List() *DiskSnapshotsServiceListRequest {
	return &DiskSnapshotsServiceListRequest{DiskSnapshotsService: p}
}

//
//
func (op *DiskSnapshotsService) SnapshotService(id string) *DiskSnapshotService {
	return NewDiskSnapshotService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskSnapshotsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.SnapshotService(path), nil
	}
	return op.SnapshotService(path[:index]).Service(path[index+1:])
}

func (op *DiskSnapshotsService) String() string {
	return fmt.Sprintf("DiskSnapshotsService:%s", op.path)
}

//
// Manages the collection of disks available in the system.
//
type DisksService struct {
	BaseService
}

func NewDisksService(connection *Connection, path string) *DisksService {
	var result DisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
type DisksServiceAddRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddRequest) Header(key, value string) *DisksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddRequest) Query(key, value string) *DisksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddRequest) Disk(disk *Disk) *DisksServiceAddRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddRequest) Send() (*DisksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DisksService.connection.URL(), p.DisksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DisksServiceAddResponse{disk: result}, nil
}

func (p *DisksServiceAddRequest) MustSend() *DisksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
type DisksServiceAddResponse struct {
	disk *Disk
}

func (p *DisksServiceAddResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
func (p *DisksService) Add() *DisksServiceAddRequest {
	return &DisksServiceAddRequest{DisksService: p}
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DisksServiceListRequest struct {
	DisksService  *DisksService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *DisksServiceListRequest) Header(key, value string) *DisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceListRequest) Query(key, value string) *DisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceListRequest) CaseSensitive(caseSensitive bool) *DisksServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DisksServiceListRequest) Follow(follow string) *DisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DisksServiceListRequest) Max(max int64) *DisksServiceListRequest {
	p.max = &max
	return p
}

func (p *DisksServiceListRequest) Search(search string) *DisksServiceListRequest {
	p.search = &search
	return p
}

func (p *DisksServiceListRequest) Send() (*DisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DisksService.connection.URL(), p.DisksService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DisksServiceListResponse{disks: result}, nil
}

func (p *DisksServiceListRequest) MustSend() *DisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *DisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *DisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
func (p *DisksService) List() *DisksServiceListRequest {
	return &DisksServiceListRequest{DisksService: p}
}

//
// Add a new lun disk to the storage domain.
//
type DisksServiceAddLunRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddLunRequest) Header(key, value string) *DisksServiceAddLunRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddLunRequest) Query(key, value string) *DisksServiceAddLunRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddLunRequest) Disk(disk *Disk) *DisksServiceAddLunRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddLunRequest) Send() (*DisksServiceAddLunResponse, error) {
	rawURL := fmt.Sprintf("%s%s/lun", p.DisksService.connection.URL(), p.DisksService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDisk()
	return &DisksServiceAddLunResponse{disk: result}, nil
}

func (p *DisksServiceAddLunRequest) MustSend() *DisksServiceAddLunResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new lun disk to the storage domain.
//
type DisksServiceAddLunResponse struct {
	disk *Disk
}

func (p *DisksServiceAddLunResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddLunResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Add a new lun disk to the storage domain.
//
func (p *DisksService) AddLun() *DisksServiceAddLunRequest {
	return &DisksServiceAddLunRequest{DisksService: p}
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
type DisksServiceAddOnStorageDomainRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddOnStorageDomainRequest) Header(key, value string) *DisksServiceAddOnStorageDomainRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Query(key, value string) *DisksServiceAddOnStorageDomainRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Disk(disk *Disk) *DisksServiceAddOnStorageDomainRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Send() (*DisksServiceAddOnStorageDomainResponse, error) {
	rawURL := fmt.Sprintf("%s%s/onstoragedomain", p.DisksService.connection.URL(), p.DisksService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDisk()
	return &DisksServiceAddOnStorageDomainResponse{disk: result}, nil
}

func (p *DisksServiceAddOnStorageDomainRequest) MustSend() *DisksServiceAddOnStorageDomainResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
type DisksServiceAddOnStorageDomainResponse struct {
	disk *Disk
}

func (p *DisksServiceAddOnStorageDomainResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddOnStorageDomainResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
func (p *DisksService) AddOnStorageDomain() *DisksServiceAddOnStorageDomainRequest {
	return &DisksServiceAddOnStorageDomainRequest{DisksService: p}
}

//
// Reference to a service managing a specific disk.
//
func (op *DisksService) DiskService(id string) *DiskService {
	return NewDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *DisksService) String() string {
	return fmt.Sprintf("DisksService:%s", op.path)
}

//
// A service to manage an event in the system.
//
type EventService struct {
	BaseService
}

func NewEventService(connection *Connection, path string) *EventService {
	var result EventService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get an event.
// An example of getting an event:
// [source]
// ----
// GET /ovirt-engine/api/events/123
// ----
// [source,xml]
// ----
// <event href="/ovirt-engine/api/events/123" id="123">
//   <description>Host example.com was added by admin@internal-authz.</description>
//   <code>42</code>
//   <correlation_id>135</correlation_id>
//   <custom_id>-1</custom_id>
//   <flood_rate>30</flood_rate>
//   <origin>oVirt</origin>
//   <severity>normal</severity>
//   <time>2016-12-11T11:13:44.654+02:00</time>
//   <cluster href="/ovirt-engine/api/clusters/456" id="456"/>
//   <host href="/ovirt-engine/api/hosts/789" id="789"/>
//   <user href="/ovirt-engine/api/users/987" id="987"/>
// </event>
// ----
// Note that the number of fields changes according to the information that resides on the event.
// For example, for storage domain related events you will get the storage domain reference,
// as well as the reference for the data center this storage domain resides in.
//
type EventServiceGetRequest struct {
	EventService *EventService
	header       map[string]string
	query        map[string]string
	follow       *string
}

func (p *EventServiceGetRequest) Header(key, value string) *EventServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventServiceGetRequest) Query(key, value string) *EventServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventServiceGetRequest) Follow(follow string) *EventServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *EventServiceGetRequest) Send() (*EventServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventService.connection.URL(), p.EventService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &EventServiceGetResponse{event: result}, nil
}

func (p *EventServiceGetRequest) MustSend() *EventServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get an event.
// An example of getting an event:
// [source]
// ----
// GET /ovirt-engine/api/events/123
// ----
// [source,xml]
// ----
// <event href="/ovirt-engine/api/events/123" id="123">
//   <description>Host example.com was added by admin@internal-authz.</description>
//   <code>42</code>
//   <correlation_id>135</correlation_id>
//   <custom_id>-1</custom_id>
//   <flood_rate>30</flood_rate>
//   <origin>oVirt</origin>
//   <severity>normal</severity>
//   <time>2016-12-11T11:13:44.654+02:00</time>
//   <cluster href="/ovirt-engine/api/clusters/456" id="456"/>
//   <host href="/ovirt-engine/api/hosts/789" id="789"/>
//   <user href="/ovirt-engine/api/users/987" id="987"/>
// </event>
// ----
// Note that the number of fields changes according to the information that resides on the event.
// For example, for storage domain related events you will get the storage domain reference,
// as well as the reference for the data center this storage domain resides in.
//
type EventServiceGetResponse struct {
	event *Event
}

func (p *EventServiceGetResponse) Event() (*Event, bool) {
	if p.event != nil {
		return p.event, true
	}
	return nil, false
}

func (p *EventServiceGetResponse) MustEvent() *Event {
	if p.event == nil {
		panic("event in response does not exist")
	}
	return p.event
}

//
// Get an event.
// An example of getting an event:
// [source]
// ----
// GET /ovirt-engine/api/events/123
// ----
// [source,xml]
// ----
// <event href="/ovirt-engine/api/events/123" id="123">
//   <description>Host example.com was added by admin@internal-authz.</description>
//   <code>42</code>
//   <correlation_id>135</correlation_id>
//   <custom_id>-1</custom_id>
//   <flood_rate>30</flood_rate>
//   <origin>oVirt</origin>
//   <severity>normal</severity>
//   <time>2016-12-11T11:13:44.654+02:00</time>
//   <cluster href="/ovirt-engine/api/clusters/456" id="456"/>
//   <host href="/ovirt-engine/api/hosts/789" id="789"/>
//   <user href="/ovirt-engine/api/users/987" id="987"/>
// </event>
// ----
// Note that the number of fields changes according to the information that resides on the event.
// For example, for storage domain related events you will get the storage domain reference,
// as well as the reference for the data center this storage domain resides in.
//
func (p *EventService) Get() *EventServiceGetRequest {
	return &EventServiceGetRequest{EventService: p}
}

//
// Removes an event from internal audit log.
// An event can be removed by sending following request
// [source]
// ----
// DELETE /ovirt-engine/api/events/123
// ----
//
type EventServiceRemoveRequest struct {
	EventService *EventService
	header       map[string]string
	query        map[string]string
	async        *bool
}

func (p *EventServiceRemoveRequest) Header(key, value string) *EventServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventServiceRemoveRequest) Query(key, value string) *EventServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventServiceRemoveRequest) Async(async bool) *EventServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *EventServiceRemoveRequest) Send() (*EventServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventService.connection.URL(), p.EventService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(EventServiceRemoveResponse), nil
}

func (p *EventServiceRemoveRequest) MustSend() *EventServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes an event from internal audit log.
// An event can be removed by sending following request
// [source]
// ----
// DELETE /ovirt-engine/api/events/123
// ----
//
type EventServiceRemoveResponse struct {
}

//
// Removes an event from internal audit log.
// An event can be removed by sending following request
// [source]
// ----
// DELETE /ovirt-engine/api/events/123
// ----
//
func (p *EventService) Remove() *EventServiceRemoveRequest {
	return &EventServiceRemoveRequest{EventService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EventService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *EventService) String() string {
	return fmt.Sprintf("EventService:%s", op.path)
}

//
// A service to manage a specific event-subscription in the system.
//
type EventSubscriptionService struct {
	BaseService
}

func NewEventSubscriptionService(connection *Connection, path string) *EventSubscriptionService {
	var result EventSubscriptionService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the information about the event-subscription.
// For example to retrieve the information about the subscription of user '123' to
// the event 'vm_console_detected':
// ....
// GET /ovirt-engine/api/users/123/vm_console_detected
// ....
// [source,xml]
// ----
// <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_console_detected">
//   <event>vm_console_detected</event>
//   <notification_method>smtp</notification_method>
//   <user href="/ovirt-engine/api/users/123" id="123"/>
//   <address>a@b.com</address>
// </event-subscription>
// ----
//
type EventSubscriptionServiceGetRequest struct {
	EventSubscriptionService *EventSubscriptionService
	header                   map[string]string
	query                    map[string]string
}

func (p *EventSubscriptionServiceGetRequest) Header(key, value string) *EventSubscriptionServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionServiceGetRequest) Query(key, value string) *EventSubscriptionServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionServiceGetRequest) Send() (*EventSubscriptionServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionService.connection.URL(), p.EventSubscriptionService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventSubscriptionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &EventSubscriptionServiceGetResponse{eventSubscription: result}, nil
}

func (p *EventSubscriptionServiceGetRequest) MustSend() *EventSubscriptionServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the information about the event-subscription.
// For example to retrieve the information about the subscription of user '123' to
// the event 'vm_console_detected':
// ....
// GET /ovirt-engine/api/users/123/vm_console_detected
// ....
// [source,xml]
// ----
// <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_console_detected">
//   <event>vm_console_detected</event>
//   <notification_method>smtp</notification_method>
//   <user href="/ovirt-engine/api/users/123" id="123"/>
//   <address>a@b.com</address>
// </event-subscription>
// ----
//
type EventSubscriptionServiceGetResponse struct {
	eventSubscription *EventSubscription
}

func (p *EventSubscriptionServiceGetResponse) EventSubscription() (*EventSubscription, bool) {
	if p.eventSubscription != nil {
		return p.eventSubscription, true
	}
	return nil, false
}

func (p *EventSubscriptionServiceGetResponse) MustEventSubscription() *EventSubscription {
	if p.eventSubscription == nil {
		panic("eventSubscription in response does not exist")
	}
	return p.eventSubscription
}

//
// Gets the information about the event-subscription.
// For example to retrieve the information about the subscription of user '123' to
// the event 'vm_console_detected':
// ....
// GET /ovirt-engine/api/users/123/vm_console_detected
// ....
// [source,xml]
// ----
// <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_console_detected">
//   <event>vm_console_detected</event>
//   <notification_method>smtp</notification_method>
//   <user href="/ovirt-engine/api/users/123" id="123"/>
//   <address>a@b.com</address>
// </event-subscription>
// ----
//
func (p *EventSubscriptionService) Get() *EventSubscriptionServiceGetRequest {
	return &EventSubscriptionServiceGetRequest{EventSubscriptionService: p}
}

//
// Removes the event-subscription from the system.
// For example to remove user 123's subscription to `vm_console_detected` event:
// ....
// DELETE /ovirt-engine/api/users/123/vm_console_detected
// ....
//
type EventSubscriptionServiceRemoveRequest struct {
	EventSubscriptionService *EventSubscriptionService
	header                   map[string]string
	query                    map[string]string
	async                    *bool
}

func (p *EventSubscriptionServiceRemoveRequest) Header(key, value string) *EventSubscriptionServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionServiceRemoveRequest) Query(key, value string) *EventSubscriptionServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionServiceRemoveRequest) Async(async bool) *EventSubscriptionServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *EventSubscriptionServiceRemoveRequest) Send() (*EventSubscriptionServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionService.connection.URL(), p.EventSubscriptionService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(EventSubscriptionServiceRemoveResponse), nil
}

func (p *EventSubscriptionServiceRemoveRequest) MustSend() *EventSubscriptionServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the event-subscription from the system.
// For example to remove user 123's subscription to `vm_console_detected` event:
// ....
// DELETE /ovirt-engine/api/users/123/vm_console_detected
// ....
//
type EventSubscriptionServiceRemoveResponse struct {
}

//
// Removes the event-subscription from the system.
// For example to remove user 123's subscription to `vm_console_detected` event:
// ....
// DELETE /ovirt-engine/api/users/123/vm_console_detected
// ....
//
func (p *EventSubscriptionService) Remove() *EventSubscriptionServiceRemoveRequest {
	return &EventSubscriptionServiceRemoveRequest{EventSubscriptionService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EventSubscriptionService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *EventSubscriptionService) String() string {
	return fmt.Sprintf("EventSubscriptionService:%s", op.path)
}

//
// Represents a service to manage collection of event-subscription of a user.
//
type EventSubscriptionsService struct {
	BaseService
}

func NewEventSubscriptionsService(connection *Connection, path string) *EventSubscriptionsService {
	var result EventSubscriptionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new event-subscription to the system.
// An event-subscription is always added in the context of a user. For example, to add new
// event-subscription for `host_high_cpu_use` for user `123`, and have the notification
// sent to the e-mail address: `a@b.com`, send a request like this:
// ....
// POST /ovirt-engine/api/users/123/eventsubscriptions
// ....
// With a request body like this:
// [source,xml]
// ----
// <event_subscription>
//     <event>host_high_cpu_use</event>
//     <address>a@b.com</address>
// </event_subscription>
// ----
// The event name will become the ID of the new event-subscription entity:
// GET .../api/users/123/eventsubscriptions/host_high_cpu_use
// Note that no user id is provided in the request body. This is because the user-id (in this case 123)
// is already known to the API from the context. Note also that event-subscription entity contains
// notification-method field, but it is not provided either in the request body. This is because currently
// it's always set to SMTP as SNMP notifications are still unsupported by the API layer.
//
type EventSubscriptionsServiceAddRequest struct {
	EventSubscriptionsService *EventSubscriptionsService
	header                    map[string]string
	query                     map[string]string
	eventSubscription         *EventSubscription
}

func (p *EventSubscriptionsServiceAddRequest) Header(key, value string) *EventSubscriptionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionsServiceAddRequest) Query(key, value string) *EventSubscriptionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionsServiceAddRequest) EventSubscription(eventSubscription *EventSubscription) *EventSubscriptionsServiceAddRequest {
	p.eventSubscription = eventSubscription
	return p
}

func (p *EventSubscriptionsServiceAddRequest) Send() (*EventSubscriptionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionsService.connection.URL(), p.EventSubscriptionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLEventSubscriptionWriteOne(writer, p.eventSubscription, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventSubscriptionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &EventSubscriptionsServiceAddResponse{eventSubscription: result}, nil
}

func (p *EventSubscriptionsServiceAddRequest) MustSend() *EventSubscriptionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new event-subscription to the system.
// An event-subscription is always added in the context of a user. For example, to add new
// event-subscription for `host_high_cpu_use` for user `123`, and have the notification
// sent to the e-mail address: `a@b.com`, send a request like this:
// ....
// POST /ovirt-engine/api/users/123/eventsubscriptions
// ....
// With a request body like this:
// [source,xml]
// ----
// <event_subscription>
//     <event>host_high_cpu_use</event>
//     <address>a@b.com</address>
// </event_subscription>
// ----
// The event name will become the ID of the new event-subscription entity:
// GET .../api/users/123/eventsubscriptions/host_high_cpu_use
// Note that no user id is provided in the request body. This is because the user-id (in this case 123)
// is already known to the API from the context. Note also that event-subscription entity contains
// notification-method field, but it is not provided either in the request body. This is because currently
// it's always set to SMTP as SNMP notifications are still unsupported by the API layer.
//
type EventSubscriptionsServiceAddResponse struct {
	eventSubscription *EventSubscription
}

func (p *EventSubscriptionsServiceAddResponse) EventSubscription() (*EventSubscription, bool) {
	if p.eventSubscription != nil {
		return p.eventSubscription, true
	}
	return nil, false
}

func (p *EventSubscriptionsServiceAddResponse) MustEventSubscription() *EventSubscription {
	if p.eventSubscription == nil {
		panic("eventSubscription in response does not exist")
	}
	return p.eventSubscription
}

//
// Add a new event-subscription to the system.
// An event-subscription is always added in the context of a user. For example, to add new
// event-subscription for `host_high_cpu_use` for user `123`, and have the notification
// sent to the e-mail address: `a@b.com`, send a request like this:
// ....
// POST /ovirt-engine/api/users/123/eventsubscriptions
// ....
// With a request body like this:
// [source,xml]
// ----
// <event_subscription>
//     <event>host_high_cpu_use</event>
//     <address>a@b.com</address>
// </event_subscription>
// ----
// The event name will become the ID of the new event-subscription entity:
// GET .../api/users/123/eventsubscriptions/host_high_cpu_use
// Note that no user id is provided in the request body. This is because the user-id (in this case 123)
// is already known to the API from the context. Note also that event-subscription entity contains
// notification-method field, but it is not provided either in the request body. This is because currently
// it's always set to SMTP as SNMP notifications are still unsupported by the API layer.
//
func (p *EventSubscriptionsService) Add() *EventSubscriptionsServiceAddRequest {
	return &EventSubscriptionsServiceAddRequest{EventSubscriptionsService: p}
}

//
// List the event-subscriptions for the provided user.
// For example to list event-subscriptions for user `123`:
// ....
// GET /ovirt-engine/api/users/123/event-subscriptions
// ....
// [source,xml]
// ----
// <event-subscriptions>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/host_install_failed">
//     <event>host_install_failed</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_paused">
//     <event>vm_paused</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
// </event-subscriptions>
// ----
//
type EventSubscriptionsServiceListRequest struct {
	EventSubscriptionsService *EventSubscriptionsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *EventSubscriptionsServiceListRequest) Header(key, value string) *EventSubscriptionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionsServiceListRequest) Query(key, value string) *EventSubscriptionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionsServiceListRequest) Follow(follow string) *EventSubscriptionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *EventSubscriptionsServiceListRequest) Max(max int64) *EventSubscriptionsServiceListRequest {
	p.max = &max
	return p
}

func (p *EventSubscriptionsServiceListRequest) Send() (*EventSubscriptionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionsService.connection.URL(), p.EventSubscriptionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventSubscriptionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &EventSubscriptionsServiceListResponse{eventSubscriptions: result}, nil
}

func (p *EventSubscriptionsServiceListRequest) MustSend() *EventSubscriptionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the event-subscriptions for the provided user.
// For example to list event-subscriptions for user `123`:
// ....
// GET /ovirt-engine/api/users/123/event-subscriptions
// ....
// [source,xml]
// ----
// <event-subscriptions>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/host_install_failed">
//     <event>host_install_failed</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_paused">
//     <event>vm_paused</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
// </event-subscriptions>
// ----
//
type EventSubscriptionsServiceListResponse struct {
	eventSubscriptions *EventSubscriptionSlice
}

func (p *EventSubscriptionsServiceListResponse) EventSubscriptions() (*EventSubscriptionSlice, bool) {
	if p.eventSubscriptions != nil {
		return p.eventSubscriptions, true
	}
	return nil, false
}

func (p *EventSubscriptionsServiceListResponse) MustEventSubscriptions() *EventSubscriptionSlice {
	if p.eventSubscriptions == nil {
		panic("eventSubscriptions in response does not exist")
	}
	return p.eventSubscriptions
}

//
// List the event-subscriptions for the provided user.
// For example to list event-subscriptions for user `123`:
// ....
// GET /ovirt-engine/api/users/123/event-subscriptions
// ....
// [source,xml]
// ----
// <event-subscriptions>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/host_install_failed">
//     <event>host_install_failed</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_paused">
//     <event>vm_paused</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
// </event-subscriptions>
// ----
//
func (p *EventSubscriptionsService) List() *EventSubscriptionsServiceListRequest {
	return &EventSubscriptionsServiceListRequest{EventSubscriptionsService: p}
}

//
// Reference to the service that manages a specific event-subscription.
//
func (op *EventSubscriptionsService) EventSubscriptionService(id string) *EventSubscriptionService {
	return NewEventSubscriptionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EventSubscriptionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.EventSubscriptionService(path), nil
	}
	return op.EventSubscriptionService(path[:index]).Service(path[index+1:])
}

func (op *EventSubscriptionsService) String() string {
	return fmt.Sprintf("EventSubscriptionsService:%s", op.path)
}

//
// A service to manage events in the system.
//
type EventsService struct {
	BaseService
}

func NewEventsService(connection *Connection, path string) *EventsService {
	var result EventsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds an external event to the internal audit log.
// This is intended for integration with external systems that detect or produce events relevant for the
// administrator of the system. For example, an external monitoring tool may be able to detect that a file system
// is full inside the guest operating system of a virtual machine. This event can be added to the internal audit
// log sending a request like this:
// [source]
// ----
// POST /ovirt-engine/api/events
// <event>
//   <description>File system /home is full</description>
//   <severity>alert</severity>
//   <origin>mymonitor</origin>
//   <custom_id>1467879754</custom_id>
// </event>
// ----
// Events can also be linked to specific objects. For example, the above event could be linked to the specific
// virtual machine where it happened, using the `vm` link:
// [source]
// ----
// POST /ovirt-engine/api/events
// <event>
//   <description>File system /home is full</description>
//   <severity>alert</severity>
//   <origin>mymonitor</origin>
//   <custom_id>1467879754</custom_id>
//   <vm id="aae98225-5b73-490d-a252-899209af17e9"/>
// </event>
// ----
// NOTE: When using links, like the `vm` in the previous example, only the `id` attribute is accepted. The `name`
// attribute, if provided, is simply ignored.
//
type EventsServiceAddRequest struct {
	EventsService *EventsService
	header        map[string]string
	query         map[string]string
	event         *Event
}

func (p *EventsServiceAddRequest) Header(key, value string) *EventsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventsServiceAddRequest) Query(key, value string) *EventsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventsServiceAddRequest) Event(event *Event) *EventsServiceAddRequest {
	p.event = event
	return p
}

func (p *EventsServiceAddRequest) Send() (*EventsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventsService.connection.URL(), p.EventsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLEventWriteOne(writer, p.event, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &EventsServiceAddResponse{event: result}, nil
}

func (p *EventsServiceAddRequest) MustSend() *EventsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds an external event to the internal audit log.
// This is intended for integration with external systems that detect or produce events relevant for the
// administrator of the system. For example, an external monitoring tool may be able to detect that a file system
// is full inside the guest operating system of a virtual machine. This event can be added to the internal audit
// log sending a request like this:
// [source]
// ----
// POST /ovirt-engine/api/events
// <event>
//   <description>File system /home is full</description>
//   <severity>alert</severity>
//   <origin>mymonitor</origin>
//   <custom_id>1467879754</custom_id>
// </event>
// ----
// Events can also be linked to specific objects. For example, the above event could be linked to the specific
// virtual machine where it happened, using the `vm` link:
// [source]
// ----
// POST /ovirt-engine/api/events
// <event>
//   <description>File system /home is full</description>
//   <severity>alert</severity>
//   <origin>mymonitor</origin>
//   <custom_id>1467879754</custom_id>
//   <vm id="aae98225-5b73-490d-a252-899209af17e9"/>
// </event>
// ----
// NOTE: When using links, like the `vm` in the previous example, only the `id` attribute is accepted. The `name`
// attribute, if provided, is simply ignored.
//
type EventsServiceAddResponse struct {
	event *Event
}

func (p *EventsServiceAddResponse) Event() (*Event, bool) {
	if p.event != nil {
		return p.event, true
	}
	return nil, false
}

func (p *EventsServiceAddResponse) MustEvent() *Event {
	if p.event == nil {
		panic("event in response does not exist")
	}
	return p.event
}

//
// Adds an external event to the internal audit log.
// This is intended for integration with external systems that detect or produce events relevant for the
// administrator of the system. For example, an external monitoring tool may be able to detect that a file system
// is full inside the guest operating system of a virtual machine. This event can be added to the internal audit
// log sending a request like this:
// [source]
// ----
// POST /ovirt-engine/api/events
// <event>
//   <description>File system /home is full</description>
//   <severity>alert</severity>
//   <origin>mymonitor</origin>
//   <custom_id>1467879754</custom_id>
// </event>
// ----
// Events can also be linked to specific objects. For example, the above event could be linked to the specific
// virtual machine where it happened, using the `vm` link:
// [source]
// ----
// POST /ovirt-engine/api/events
// <event>
//   <description>File system /home is full</description>
//   <severity>alert</severity>
//   <origin>mymonitor</origin>
//   <custom_id>1467879754</custom_id>
//   <vm id="aae98225-5b73-490d-a252-899209af17e9"/>
// </event>
// ----
// NOTE: When using links, like the `vm` in the previous example, only the `id` attribute is accepted. The `name`
// attribute, if provided, is simply ignored.
//
func (p *EventsService) Add() *EventsServiceAddRequest {
	return &EventsServiceAddRequest{EventsService: p}
}

//
// Get list of events.
// [source]
// ----
// GET /ovirt-engine/api/events
// ----
// To the above request we get following response:
// [source,xml]
// ----
// <events>
//   <event href="/ovirt-engine/api/events/2" id="2">
//     <description>User admin@internal-authz logged out.</description>
//     <code>31</code>
//     <correlation_id>1e892ea9</correlation_id>
//     <custom_id>-1</custom_id>
//     <flood_rate>30</flood_rate>
//     <origin>oVirt</origin>
//     <severity>normal</severity>
//     <time>2016-09-14T12:14:34.541+02:00</time>
//     <user href="/ovirt-engine/api/users/57d91d48-00da-0137-0138-000000000244" id="57d91d48-00da-0137-0138-000000000244"/>
//   </event>
//   <event href="/ovirt-engine/api/events/1" id="1">
//     <description>User admin logged in.</description>
//     <code>30</code>
//     <correlation_id>1fbd81f4</correlation_id>
//     <custom_id>-1</custom_id>
//     <flood_rate>30</flood_rate>
//     <origin>oVirt</origin>
//     <severity>normal</severity>
//     <time>2016-09-14T11:54:35.229+02:00</time>
//     <user href="/ovirt-engine/api/users/57d91d48-00da-0137-0138-000000000244" id="57d91d48-00da-0137-0138-000000000244"/>
//   </event>
// </events>
// ----
// The following events occur:
// * id="1" - The API logs in the admin user account.
// * id="2" - The API logs out of the admin user account.
// The order of the returned list of events is always garanteed. If the `sortby` clause is included in the
// `search` parameter, then the events will be ordered according to that clause. If the `sortby` clause isn't
// included, then the events will be sorted by the numeric value of the `id` attribute, starting with the
// highest value. This, combined with the `max` parameter, simplifies obtaining the most recent event:
// ....
// GET /ovirt-engine/api/events?max=1
// ....
//
type EventsServiceListRequest struct {
	EventsService *EventsService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	from          *int64
	max           *int64
	search        *string
}

func (p *EventsServiceListRequest) Header(key, value string) *EventsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventsServiceListRequest) Query(key, value string) *EventsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventsServiceListRequest) CaseSensitive(caseSensitive bool) *EventsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *EventsServiceListRequest) Follow(follow string) *EventsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *EventsServiceListRequest) From(from int64) *EventsServiceListRequest {
	p.from = &from
	return p
}

func (p *EventsServiceListRequest) Max(max int64) *EventsServiceListRequest {
	p.max = &max
	return p
}

func (p *EventsServiceListRequest) Search(search string) *EventsServiceListRequest {
	p.search = &search
	return p
}

func (p *EventsServiceListRequest) Send() (*EventsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventsService.connection.URL(), p.EventsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.from != nil {
		values["from"] = []string{fmt.Sprintf("%v", *p.from)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &EventsServiceListResponse{events: result}, nil
}

func (p *EventsServiceListRequest) MustSend() *EventsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of events.
// [source]
// ----
// GET /ovirt-engine/api/events
// ----
// To the above request we get following response:
// [source,xml]
// ----
// <events>
//   <event href="/ovirt-engine/api/events/2" id="2">
//     <description>User admin@internal-authz logged out.</description>
//     <code>31</code>
//     <correlation_id>1e892ea9</correlation_id>
//     <custom_id>-1</custom_id>
//     <flood_rate>30</flood_rate>
//     <origin>oVirt</origin>
//     <severity>normal</severity>
//     <time>2016-09-14T12:14:34.541+02:00</time>
//     <user href="/ovirt-engine/api/users/57d91d48-00da-0137-0138-000000000244" id="57d91d48-00da-0137-0138-000000000244"/>
//   </event>
//   <event href="/ovirt-engine/api/events/1" id="1">
//     <description>User admin logged in.</description>
//     <code>30</code>
//     <correlation_id>1fbd81f4</correlation_id>
//     <custom_id>-1</custom_id>
//     <flood_rate>30</flood_rate>
//     <origin>oVirt</origin>
//     <severity>normal</severity>
//     <time>2016-09-14T11:54:35.229+02:00</time>
//     <user href="/ovirt-engine/api/users/57d91d48-00da-0137-0138-000000000244" id="57d91d48-00da-0137-0138-000000000244"/>
//   </event>
// </events>
// ----
// The following events occur:
// * id="1" - The API logs in the admin user account.
// * id="2" - The API logs out of the admin user account.
// The order of the returned list of events is always garanteed. If the `sortby` clause is included in the
// `search` parameter, then the events will be ordered according to that clause. If the `sortby` clause isn't
// included, then the events will be sorted by the numeric value of the `id` attribute, starting with the
// highest value. This, combined with the `max` parameter, simplifies obtaining the most recent event:
// ....
// GET /ovirt-engine/api/events?max=1
// ....
//
type EventsServiceListResponse struct {
	events *EventSlice
}

func (p *EventsServiceListResponse) Events() (*EventSlice, bool) {
	if p.events != nil {
		return p.events, true
	}
	return nil, false
}

func (p *EventsServiceListResponse) MustEvents() *EventSlice {
	if p.events == nil {
		panic("events in response does not exist")
	}
	return p.events
}

//
// Get list of events.
// [source]
// ----
// GET /ovirt-engine/api/events
// ----
// To the above request we get following response:
// [source,xml]
// ----
// <events>
//   <event href="/ovirt-engine/api/events/2" id="2">
//     <description>User admin@internal-authz logged out.</description>
//     <code>31</code>
//     <correlation_id>1e892ea9</correlation_id>
//     <custom_id>-1</custom_id>
//     <flood_rate>30</flood_rate>
//     <origin>oVirt</origin>
//     <severity>normal</severity>
//     <time>2016-09-14T12:14:34.541+02:00</time>
//     <user href="/ovirt-engine/api/users/57d91d48-00da-0137-0138-000000000244" id="57d91d48-00da-0137-0138-000000000244"/>
//   </event>
//   <event href="/ovirt-engine/api/events/1" id="1">
//     <description>User admin logged in.</description>
//     <code>30</code>
//     <correlation_id>1fbd81f4</correlation_id>
//     <custom_id>-1</custom_id>
//     <flood_rate>30</flood_rate>
//     <origin>oVirt</origin>
//     <severity>normal</severity>
//     <time>2016-09-14T11:54:35.229+02:00</time>
//     <user href="/ovirt-engine/api/users/57d91d48-00da-0137-0138-000000000244" id="57d91d48-00da-0137-0138-000000000244"/>
//   </event>
// </events>
// ----
// The following events occur:
// * id="1" - The API logs in the admin user account.
// * id="2" - The API logs out of the admin user account.
// The order of the returned list of events is always garanteed. If the `sortby` clause is included in the
// `search` parameter, then the events will be ordered according to that clause. If the `sortby` clause isn't
// included, then the events will be sorted by the numeric value of the `id` attribute, starting with the
// highest value. This, combined with the `max` parameter, simplifies obtaining the most recent event:
// ....
// GET /ovirt-engine/api/events?max=1
// ....
//
func (p *EventsService) List() *EventsServiceListRequest {
	return &EventsServiceListRequest{EventsService: p}
}

//
//
type EventsServiceUndeleteRequest struct {
	EventsService *EventsService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *EventsServiceUndeleteRequest) Header(key, value string) *EventsServiceUndeleteRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventsServiceUndeleteRequest) Query(key, value string) *EventsServiceUndeleteRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventsServiceUndeleteRequest) Async(async bool) *EventsServiceUndeleteRequest {
	p.async = &async
	return p
}

func (p *EventsServiceUndeleteRequest) Send() (*EventsServiceUndeleteResponse, error) {
	rawURL := fmt.Sprintf("%s%s/undelete", p.EventsService.connection.URL(), p.EventsService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(EventsServiceUndeleteResponse), nil
}

func (p *EventsServiceUndeleteRequest) MustSend() *EventsServiceUndeleteResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type EventsServiceUndeleteResponse struct {
}

//
//
func (p *EventsService) Undelete() *EventsServiceUndeleteRequest {
	return &EventsServiceUndeleteRequest{EventsService: p}
}

//
// Reference to the service that manages a specific event.
//
func (op *EventsService) EventService(id string) *EventService {
	return NewEventService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EventsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.EventService(path), nil
	}
	return op.EventService(path[:index]).Service(path[index+1:])
}

func (op *EventsService) String() string {
	return fmt.Sprintf("EventsService:%s", op.path)
}

//
// Describes how an external network provider is provisioned by the system on the host.
//
type ExternalNetworkProviderConfigurationService struct {
	BaseService
}

func NewExternalNetworkProviderConfigurationService(connection *Connection, path string) *ExternalNetworkProviderConfigurationService {
	var result ExternalNetworkProviderConfigurationService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the information about an external network provider on the host.
//
type ExternalNetworkProviderConfigurationServiceGetRequest struct {
	ExternalNetworkProviderConfigurationService *ExternalNetworkProviderConfigurationService
	header                                      map[string]string
	query                                       map[string]string
	follow                                      *string
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Header(key, value string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Query(key, value string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Follow(follow string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Send() (*ExternalNetworkProviderConfigurationServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalNetworkProviderConfigurationService.connection.URL(), p.ExternalNetworkProviderConfigurationService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalNetworkProviderConfigurationService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalNetworkProviderConfigurationService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalNetworkProviderConfigurationService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalNetworkProviderConfigurationService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalNetworkProviderConfigurationService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalNetworkProviderConfigurationReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalNetworkProviderConfigurationServiceGetResponse{configuration: result}, nil
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) MustSend() *ExternalNetworkProviderConfigurationServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the information about an external network provider on the host.
//
type ExternalNetworkProviderConfigurationServiceGetResponse struct {
	configuration *ExternalNetworkProviderConfiguration
}

func (p *ExternalNetworkProviderConfigurationServiceGetResponse) Configuration() (*ExternalNetworkProviderConfiguration, bool) {
	if p.configuration != nil {
		return p.configuration, true
	}
	return nil, false
}

func (p *ExternalNetworkProviderConfigurationServiceGetResponse) MustConfiguration() *ExternalNetworkProviderConfiguration {
	if p.configuration == nil {
		panic("configuration in response does not exist")
	}
	return p.configuration
}

//
// Returns the information about an external network provider on the host.
//
func (p *ExternalNetworkProviderConfigurationService) Get() *ExternalNetworkProviderConfigurationServiceGetRequest {
	return &ExternalNetworkProviderConfigurationServiceGetRequest{ExternalNetworkProviderConfigurationService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalNetworkProviderConfigurationService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalNetworkProviderConfigurationService) String() string {
	return fmt.Sprintf("ExternalNetworkProviderConfigurationService:%s", op.path)
}

//
// A service to list all external network providers provisioned by the system on the host.
//
type ExternalNetworkProviderConfigurationsService struct {
	BaseService
}

func NewExternalNetworkProviderConfigurationsService(connection *Connection, path string) *ExternalNetworkProviderConfigurationsService {
	var result ExternalNetworkProviderConfigurationsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of all external network providers on the host.
// The order of the returned list of networks is not guaranteed.
//
type ExternalNetworkProviderConfigurationsServiceListRequest struct {
	ExternalNetworkProviderConfigurationsService *ExternalNetworkProviderConfigurationsService
	header                                       map[string]string
	query                                        map[string]string
	follow                                       *string
}

func (p *ExternalNetworkProviderConfigurationsServiceListRequest) Header(key, value string) *ExternalNetworkProviderConfigurationsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationsServiceListRequest) Query(key, value string) *ExternalNetworkProviderConfigurationsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationsServiceListRequest) Follow(follow string) *ExternalNetworkProviderConfigurationsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalNetworkProviderConfigurationsServiceListRequest) Send() (*ExternalNetworkProviderConfigurationsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalNetworkProviderConfigurationsService.connection.URL(), p.ExternalNetworkProviderConfigurationsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalNetworkProviderConfigurationsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalNetworkProviderConfigurationsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalNetworkProviderConfigurationsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalNetworkProviderConfigurationsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalNetworkProviderConfigurationsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalNetworkProviderConfigurationReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalNetworkProviderConfigurationsServiceListResponse{configurations: result}, nil
}

func (p *ExternalNetworkProviderConfigurationsServiceListRequest) MustSend() *ExternalNetworkProviderConfigurationsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of all external network providers on the host.
// The order of the returned list of networks is not guaranteed.
//
type ExternalNetworkProviderConfigurationsServiceListResponse struct {
	configurations *ExternalNetworkProviderConfigurationSlice
}

func (p *ExternalNetworkProviderConfigurationsServiceListResponse) Configurations() (*ExternalNetworkProviderConfigurationSlice, bool) {
	if p.configurations != nil {
		return p.configurations, true
	}
	return nil, false
}

func (p *ExternalNetworkProviderConfigurationsServiceListResponse) MustConfigurations() *ExternalNetworkProviderConfigurationSlice {
	if p.configurations == nil {
		panic("configurations in response does not exist")
	}
	return p.configurations
}

//
// Returns the list of all external network providers on the host.
// The order of the returned list of networks is not guaranteed.
//
func (p *ExternalNetworkProviderConfigurationsService) List() *ExternalNetworkProviderConfigurationsServiceListRequest {
	return &ExternalNetworkProviderConfigurationsServiceListRequest{ExternalNetworkProviderConfigurationsService: p}
}

//
//
func (op *ExternalNetworkProviderConfigurationsService) ConfigurationService(id string) *ExternalNetworkProviderConfigurationService {
	return NewExternalNetworkProviderConfigurationService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalNetworkProviderConfigurationsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ConfigurationService(path), nil
	}
	return op.ConfigurationService(path[:index]).Service(path[index+1:])
}

func (op *ExternalNetworkProviderConfigurationsService) String() string {
	return fmt.Sprintf("ExternalNetworkProviderConfigurationsService:%s", op.path)
}

//
// A service to view specific certificate for external provider.
//
type ExternalProviderCertificateService struct {
	BaseService
}

func NewExternalProviderCertificateService(connection *Connection, path string) *ExternalProviderCertificateService {
	var result ExternalProviderCertificateService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get specific certificate.
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/certificate/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <certificate id="0">
//   <organization>provider.example.com</organization>
//   <subject>CN=provider.example.com</subject>
//   <content>...</content>
// </certificate>
// ----
//
type ExternalProviderCertificateServiceGetRequest struct {
	ExternalProviderCertificateService *ExternalProviderCertificateService
	header                             map[string]string
	query                              map[string]string
	follow                             *string
}

func (p *ExternalProviderCertificateServiceGetRequest) Header(key, value string) *ExternalProviderCertificateServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalProviderCertificateServiceGetRequest) Query(key, value string) *ExternalProviderCertificateServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalProviderCertificateServiceGetRequest) Follow(follow string) *ExternalProviderCertificateServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalProviderCertificateServiceGetRequest) Send() (*ExternalProviderCertificateServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalProviderCertificateService.connection.URL(), p.ExternalProviderCertificateService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalProviderCertificateService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalProviderCertificateService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalProviderCertificateService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalProviderCertificateService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalProviderCertificateService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCertificateReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalProviderCertificateServiceGetResponse{certificate: result}, nil
}

func (p *ExternalProviderCertificateServiceGetRequest) MustSend() *ExternalProviderCertificateServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get specific certificate.
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/certificate/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <certificate id="0">
//   <organization>provider.example.com</organization>
//   <subject>CN=provider.example.com</subject>
//   <content>...</content>
// </certificate>
// ----
//
type ExternalProviderCertificateServiceGetResponse struct {
	certificate *Certificate
}

func (p *ExternalProviderCertificateServiceGetResponse) Certificate() (*Certificate, bool) {
	if p.certificate != nil {
		return p.certificate, true
	}
	return nil, false
}

func (p *ExternalProviderCertificateServiceGetResponse) MustCertificate() *Certificate {
	if p.certificate == nil {
		panic("certificate in response does not exist")
	}
	return p.certificate
}

//
// Get specific certificate.
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/certificate/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <certificate id="0">
//   <organization>provider.example.com</organization>
//   <subject>CN=provider.example.com</subject>
//   <content>...</content>
// </certificate>
// ----
//
func (p *ExternalProviderCertificateService) Get() *ExternalProviderCertificateServiceGetRequest {
	return &ExternalProviderCertificateServiceGetRequest{ExternalProviderCertificateService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalProviderCertificateService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalProviderCertificateService) String() string {
	return fmt.Sprintf("ExternalProviderCertificateService:%s", op.path)
}

//
// A service to view certificates for external provider.
//
type ExternalProviderCertificatesService struct {
	BaseService
}

func NewExternalProviderCertificatesService(connection *Connection, path string) *ExternalProviderCertificatesService {
	var result ExternalProviderCertificatesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the chain of certificates presented by the external provider.
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/certificates
// ----
// And here is sample response:
// [source,xml]
// ----
// <certificates>
//   <certificate id="789">...</certificate>
//   ...
// </certificates>
// ----
// The order of the returned certificates is always guaranteed to be the sign order: the first is the
// certificate of the server itself, the second the certificate of the CA that signs the first, so on.
//
type ExternalProviderCertificatesServiceListRequest struct {
	ExternalProviderCertificatesService *ExternalProviderCertificatesService
	header                              map[string]string
	query                               map[string]string
	follow                              *string
	max                                 *int64
}

func (p *ExternalProviderCertificatesServiceListRequest) Header(key, value string) *ExternalProviderCertificatesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalProviderCertificatesServiceListRequest) Query(key, value string) *ExternalProviderCertificatesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalProviderCertificatesServiceListRequest) Follow(follow string) *ExternalProviderCertificatesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalProviderCertificatesServiceListRequest) Max(max int64) *ExternalProviderCertificatesServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalProviderCertificatesServiceListRequest) Send() (*ExternalProviderCertificatesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalProviderCertificatesService.connection.URL(), p.ExternalProviderCertificatesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalProviderCertificatesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalProviderCertificatesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalProviderCertificatesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalProviderCertificatesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalProviderCertificatesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCertificateReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalProviderCertificatesServiceListResponse{certificates: result}, nil
}

func (p *ExternalProviderCertificatesServiceListRequest) MustSend() *ExternalProviderCertificatesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the chain of certificates presented by the external provider.
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/certificates
// ----
// And here is sample response:
// [source,xml]
// ----
// <certificates>
//   <certificate id="789">...</certificate>
//   ...
// </certificates>
// ----
// The order of the returned certificates is always guaranteed to be the sign order: the first is the
// certificate of the server itself, the second the certificate of the CA that signs the first, so on.
//
type ExternalProviderCertificatesServiceListResponse struct {
	certificates *CertificateSlice
}

func (p *ExternalProviderCertificatesServiceListResponse) Certificates() (*CertificateSlice, bool) {
	if p.certificates != nil {
		return p.certificates, true
	}
	return nil, false
}

func (p *ExternalProviderCertificatesServiceListResponse) MustCertificates() *CertificateSlice {
	if p.certificates == nil {
		panic("certificates in response does not exist")
	}
	return p.certificates
}

//
// Returns the chain of certificates presented by the external provider.
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/certificates
// ----
// And here is sample response:
// [source,xml]
// ----
// <certificates>
//   <certificate id="789">...</certificate>
//   ...
// </certificates>
// ----
// The order of the returned certificates is always guaranteed to be the sign order: the first is the
// certificate of the server itself, the second the certificate of the CA that signs the first, so on.
//
func (p *ExternalProviderCertificatesService) List() *ExternalProviderCertificatesServiceListRequest {
	return &ExternalProviderCertificatesServiceListRequest{ExternalProviderCertificatesService: p}
}

//
// Reference to service that manages a specific certificate
// for this external provider.
//
func (op *ExternalProviderCertificatesService) CertificateService(id string) *ExternalProviderCertificateService {
	return NewExternalProviderCertificateService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalProviderCertificatesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.CertificateService(path), nil
	}
	return op.CertificateService(path[:index]).Service(path[index+1:])
}

func (op *ExternalProviderCertificatesService) String() string {
	return fmt.Sprintf("ExternalProviderCertificatesService:%s", op.path)
}

//
// Provides capability to manage external providers.
//
type ExternalProviderService struct {
	BaseService
}

func NewExternalProviderService(connection *Connection, path string) *ExternalProviderService {
	var result ExternalProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalProviderServiceImportCertificatesRequest struct {
	ExternalProviderService *ExternalProviderService
	header                  map[string]string
	query                   map[string]string
	certificates            *CertificateSlice
}

func (p *ExternalProviderServiceImportCertificatesRequest) Header(key, value string) *ExternalProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) Query(key, value string) *ExternalProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *ExternalProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *ExternalProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) Send() (*ExternalProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.ExternalProviderService.connection.URL(), p.ExternalProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalProviderServiceImportCertificatesResponse), nil
}

func (p *ExternalProviderServiceImportCertificatesRequest) MustSend() *ExternalProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *ExternalProviderService) ImportCertificates() *ExternalProviderServiceImportCertificatesRequest {
	return &ExternalProviderServiceImportCertificatesRequest{ExternalProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalProviderServiceTestConnectivityRequest struct {
	ExternalProviderService *ExternalProviderService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
}

func (p *ExternalProviderServiceTestConnectivityRequest) Header(key, value string) *ExternalProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalProviderServiceTestConnectivityRequest) Query(key, value string) *ExternalProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalProviderServiceTestConnectivityRequest) Async(async bool) *ExternalProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *ExternalProviderServiceTestConnectivityRequest) Send() (*ExternalProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.ExternalProviderService.connection.URL(), p.ExternalProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalProviderServiceTestConnectivityResponse), nil
}

func (p *ExternalProviderServiceTestConnectivityRequest) MustSend() *ExternalProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *ExternalProviderService) TestConnectivity() *ExternalProviderServiceTestConnectivityRequest {
	return &ExternalProviderServiceTestConnectivityRequest{ExternalProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *ExternalProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalProviderService) String() string {
	return fmt.Sprintf("ExternalProviderService:%s", op.path)
}

//
// Provides capability to import external virtual machines.
//
type ExternalVmImportsService struct {
	BaseService
}

func NewExternalVmImportsService(connection *Connection, path string) *ExternalVmImportsService {
	var result ExternalVmImportsService
	result.connection = connection
	result.path = path
	return &result
}

//
// This operation is used to import a virtual machine from external hypervisor, such as KVM, XEN or VMware.
// For example import of a virtual machine from VMware can be facilitated using the following request:
// [source]
// ----
// POST /externalvmimports
// ----
// With request body of type <<types/external_vm_import,ExternalVmImport>>, for example:
// [source,xml]
// ----
// <external_vm_import>
//   <vm>
//     <name>my_vm</name>
//   </vm>
//   <cluster id="360014051136c20574f743bdbd28177fd" />
//   <storage_domain id="8bb5ade5-e988-4000-8b93-dbfc6717fe50" />
//   <name>vm_name_as_is_in_vmware</name>
//   <sparse>true</sparse>
//   <username>vmware_user</username>
//   <password>123456</password>
//   <provider>VMWARE</provider>
//   <url>vpx://wmware_user@vcenter-host/DataCenter/Cluster/esxi-host?no_verify=1</url>
//   <drivers_iso id="virtio-win-1.6.7.iso" />
// </external_vm_import>
// ----
//
type ExternalVmImportsServiceAddRequest struct {
	ExternalVmImportsService *ExternalVmImportsService
	header                   map[string]string
	query                    map[string]string
	import_                  *ExternalVmImport
}

func (p *ExternalVmImportsServiceAddRequest) Header(key, value string) *ExternalVmImportsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalVmImportsServiceAddRequest) Query(key, value string) *ExternalVmImportsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalVmImportsServiceAddRequest) Import(import_ *ExternalVmImport) *ExternalVmImportsServiceAddRequest {
	p.import_ = import_
	return p
}

func (p *ExternalVmImportsServiceAddRequest) Send() (*ExternalVmImportsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalVmImportsService.connection.URL(), p.ExternalVmImportsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLExternalVmImportWriteOne(writer, p.import_, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalVmImportsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalVmImportsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalVmImportsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalVmImportsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalVmImportsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalVmImportReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalVmImportsServiceAddResponse{import_: result}, nil
}

func (p *ExternalVmImportsServiceAddRequest) MustSend() *ExternalVmImportsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation is used to import a virtual machine from external hypervisor, such as KVM, XEN or VMware.
// For example import of a virtual machine from VMware can be facilitated using the following request:
// [source]
// ----
// POST /externalvmimports
// ----
// With request body of type <<types/external_vm_import,ExternalVmImport>>, for example:
// [source,xml]
// ----
// <external_vm_import>
//   <vm>
//     <name>my_vm</name>
//   </vm>
//   <cluster id="360014051136c20574f743bdbd28177fd" />
//   <storage_domain id="8bb5ade5-e988-4000-8b93-dbfc6717fe50" />
//   <name>vm_name_as_is_in_vmware</name>
//   <sparse>true</sparse>
//   <username>vmware_user</username>
//   <password>123456</password>
//   <provider>VMWARE</provider>
//   <url>vpx://wmware_user@vcenter-host/DataCenter/Cluster/esxi-host?no_verify=1</url>
//   <drivers_iso id="virtio-win-1.6.7.iso" />
// </external_vm_import>
// ----
//
type ExternalVmImportsServiceAddResponse struct {
	import_ *ExternalVmImport
}

func (p *ExternalVmImportsServiceAddResponse) Import() (*ExternalVmImport, bool) {
	if p.import_ != nil {
		return p.import_, true
	}
	return nil, false
}

func (p *ExternalVmImportsServiceAddResponse) MustImport() *ExternalVmImport {
	if p.import_ == nil {
		panic("import_ in response does not exist")
	}
	return p.import_
}

//
// This operation is used to import a virtual machine from external hypervisor, such as KVM, XEN or VMware.
// For example import of a virtual machine from VMware can be facilitated using the following request:
// [source]
// ----
// POST /externalvmimports
// ----
// With request body of type <<types/external_vm_import,ExternalVmImport>>, for example:
// [source,xml]
// ----
// <external_vm_import>
//   <vm>
//     <name>my_vm</name>
//   </vm>
//   <cluster id="360014051136c20574f743bdbd28177fd" />
//   <storage_domain id="8bb5ade5-e988-4000-8b93-dbfc6717fe50" />
//   <name>vm_name_as_is_in_vmware</name>
//   <sparse>true</sparse>
//   <username>vmware_user</username>
//   <password>123456</password>
//   <provider>VMWARE</provider>
//   <url>vpx://wmware_user@vcenter-host/DataCenter/Cluster/esxi-host?no_verify=1</url>
//   <drivers_iso id="virtio-win-1.6.7.iso" />
// </external_vm_import>
// ----
//
func (p *ExternalVmImportsService) Add() *ExternalVmImportsServiceAddRequest {
	return &ExternalVmImportsServiceAddRequest{ExternalVmImportsService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalVmImportsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalVmImportsService) String() string {
	return fmt.Sprintf("ExternalVmImportsService:%s", op.path)
}

//
// A service to manage fence agent for a specific host.
//
type FenceAgentService struct {
	BaseService
}

func NewFenceAgentService(connection *Connection, path string) *FenceAgentService {
	var result FenceAgentService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets details of this fence agent.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <agent id="0">
//   <type>apc</type>
//   <order>1</order>
//   <ip>192.168.1.101</ip>
//   <user>user</user>
//   <password>xxx</password>
//   <port>9</port>
//   <options>name1=value1, name2=value2</options>
// </agent>
// ----
//
type FenceAgentServiceGetRequest struct {
	FenceAgentService *FenceAgentService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *FenceAgentServiceGetRequest) Header(key, value string) *FenceAgentServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentServiceGetRequest) Query(key, value string) *FenceAgentServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentServiceGetRequest) Follow(follow string) *FenceAgentServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *FenceAgentServiceGetRequest) Send() (*FenceAgentServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentService.connection.URL(), p.FenceAgentService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAgentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FenceAgentServiceGetResponse{agent: result}, nil
}

func (p *FenceAgentServiceGetRequest) MustSend() *FenceAgentServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets details of this fence agent.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <agent id="0">
//   <type>apc</type>
//   <order>1</order>
//   <ip>192.168.1.101</ip>
//   <user>user</user>
//   <password>xxx</password>
//   <port>9</port>
//   <options>name1=value1, name2=value2</options>
// </agent>
// ----
//
type FenceAgentServiceGetResponse struct {
	agent *Agent
}

func (p *FenceAgentServiceGetResponse) Agent() (*Agent, bool) {
	if p.agent != nil {
		return p.agent, true
	}
	return nil, false
}

func (p *FenceAgentServiceGetResponse) MustAgent() *Agent {
	if p.agent == nil {
		panic("agent in response does not exist")
	}
	return p.agent
}

//
// Gets details of this fence agent.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <agent id="0">
//   <type>apc</type>
//   <order>1</order>
//   <ip>192.168.1.101</ip>
//   <user>user</user>
//   <password>xxx</password>
//   <port>9</port>
//   <options>name1=value1, name2=value2</options>
// </agent>
// ----
//
func (p *FenceAgentService) Get() *FenceAgentServiceGetRequest {
	return &FenceAgentServiceGetRequest{FenceAgentService: p}
}

//
// Removes a fence agent for a specific host.
// [source]
// ----
// DELETE /ovirt-engine/api/hosts/123/fenceagents/0
// ----
//
type FenceAgentServiceRemoveRequest struct {
	FenceAgentService *FenceAgentService
	header            map[string]string
	query             map[string]string
	async             *bool
}

func (p *FenceAgentServiceRemoveRequest) Header(key, value string) *FenceAgentServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentServiceRemoveRequest) Query(key, value string) *FenceAgentServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentServiceRemoveRequest) Async(async bool) *FenceAgentServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *FenceAgentServiceRemoveRequest) Send() (*FenceAgentServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentService.connection.URL(), p.FenceAgentService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(FenceAgentServiceRemoveResponse), nil
}

func (p *FenceAgentServiceRemoveRequest) MustSend() *FenceAgentServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a fence agent for a specific host.
// [source]
// ----
// DELETE /ovirt-engine/api/hosts/123/fenceagents/0
// ----
//
type FenceAgentServiceRemoveResponse struct {
}

//
// Removes a fence agent for a specific host.
// [source]
// ----
// DELETE /ovirt-engine/api/hosts/123/fenceagents/0
// ----
//
func (p *FenceAgentService) Remove() *FenceAgentServiceRemoveRequest {
	return &FenceAgentServiceRemoveRequest{FenceAgentService: p}
}

//
// Update a fencing-agent.
//
type FenceAgentServiceUpdateRequest struct {
	FenceAgentService *FenceAgentService
	header            map[string]string
	query             map[string]string
	agent             *Agent
	async             *bool
}

func (p *FenceAgentServiceUpdateRequest) Header(key, value string) *FenceAgentServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentServiceUpdateRequest) Query(key, value string) *FenceAgentServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentServiceUpdateRequest) Agent(agent *Agent) *FenceAgentServiceUpdateRequest {
	p.agent = agent
	return p
}

func (p *FenceAgentServiceUpdateRequest) Async(async bool) *FenceAgentServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *FenceAgentServiceUpdateRequest) Send() (*FenceAgentServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentService.connection.URL(), p.FenceAgentService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAgentWriteOne(writer, p.agent, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAgentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FenceAgentServiceUpdateResponse{agent: result}, nil
}

func (p *FenceAgentServiceUpdateRequest) MustSend() *FenceAgentServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update a fencing-agent.
//
type FenceAgentServiceUpdateResponse struct {
	agent *Agent
}

func (p *FenceAgentServiceUpdateResponse) Agent() (*Agent, bool) {
	if p.agent != nil {
		return p.agent, true
	}
	return nil, false
}

func (p *FenceAgentServiceUpdateResponse) MustAgent() *Agent {
	if p.agent == nil {
		panic("agent in response does not exist")
	}
	return p.agent
}

//
// Update a fencing-agent.
//
func (p *FenceAgentService) Update() *FenceAgentServiceUpdateRequest {
	return &FenceAgentServiceUpdateRequest{FenceAgentService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FenceAgentService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *FenceAgentService) String() string {
	return fmt.Sprintf("FenceAgentService:%s", op.path)
}

//
// A service to manage fence agents for a specific host.
//
type FenceAgentsService struct {
	BaseService
}

func NewFenceAgentsService(connection *Connection, path string) *FenceAgentsService {
	var result FenceAgentsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new fencing-agent to the host.
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/fenceagents
// You should consult the /usr/sbin/fence_<agent_name> manual page for
// the legal parameters to [name1=value1, name2=value2,...] in the options field.
// If any parameter in options appears by name that means that it is mandatory.
// For example in <options>slot=7[,name1=value1, name2=value2,...]</options>
// slot is mandatory.
// ----
// apc, bladecenter, wti fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>apc</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>slot=7[,name1=value1, name2=value2,...]</options>
//   </agent>
// apc_snmp, hpblade, ilo, ilo2, ilo_ssh, redfish, rsa fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>apc_snmp</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>[name1=value1, name2=value2,...]</options>
//   </agent>
// cisco_ucs, drac5, eps fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>cisco_ucs</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <options>slot=7[,name1=value1, name2=value2,...]</options>
//   </agent>
// drac7, ilo3, ilo4, ipmilan, rsb fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>drac7</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <options>[name1=value1, name2=value2,...]</options>
//   </agent>
//
type FenceAgentsServiceAddRequest struct {
	FenceAgentsService *FenceAgentsService
	header             map[string]string
	query              map[string]string
	agent              *Agent
}

func (p *FenceAgentsServiceAddRequest) Header(key, value string) *FenceAgentsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentsServiceAddRequest) Query(key, value string) *FenceAgentsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentsServiceAddRequest) Agent(agent *Agent) *FenceAgentsServiceAddRequest {
	p.agent = agent
	return p
}

func (p *FenceAgentsServiceAddRequest) Send() (*FenceAgentsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentsService.connection.URL(), p.FenceAgentsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAgentWriteOne(writer, p.agent, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAgentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FenceAgentsServiceAddResponse{agent: result}, nil
}

func (p *FenceAgentsServiceAddRequest) MustSend() *FenceAgentsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new fencing-agent to the host.
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/fenceagents
// You should consult the /usr/sbin/fence_<agent_name> manual page for
// the legal parameters to [name1=value1, name2=value2,...] in the options field.
// If any parameter in options appears by name that means that it is mandatory.
// For example in <options>slot=7[,name1=value1, name2=value2,...]</options>
// slot is mandatory.
// ----
// apc, bladecenter, wti fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>apc</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>slot=7[,name1=value1, name2=value2,...]</options>
//   </agent>
// apc_snmp, hpblade, ilo, ilo2, ilo_ssh, redfish, rsa fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>apc_snmp</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>[name1=value1, name2=value2,...]</options>
//   </agent>
// cisco_ucs, drac5, eps fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>cisco_ucs</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <options>slot=7[,name1=value1, name2=value2,...]</options>
//   </agent>
// drac7, ilo3, ilo4, ipmilan, rsb fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>drac7</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <options>[name1=value1, name2=value2,...]</options>
//   </agent>
//
type FenceAgentsServiceAddResponse struct {
	agent *Agent
}

func (p *FenceAgentsServiceAddResponse) Agent() (*Agent, bool) {
	if p.agent != nil {
		return p.agent, true
	}
	return nil, false
}

func (p *FenceAgentsServiceAddResponse) MustAgent() *Agent {
	if p.agent == nil {
		panic("agent in response does not exist")
	}
	return p.agent
}

//
// Add a new fencing-agent to the host.
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/fenceagents
// You should consult the /usr/sbin/fence_<agent_name> manual page for
// the legal parameters to [name1=value1, name2=value2,...] in the options field.
// If any parameter in options appears by name that means that it is mandatory.
// For example in <options>slot=7[,name1=value1, name2=value2,...]</options>
// slot is mandatory.
// ----
// apc, bladecenter, wti fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>apc</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>slot=7[,name1=value1, name2=value2,...]</options>
//   </agent>
// apc_snmp, hpblade, ilo, ilo2, ilo_ssh, redfish, rsa fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>apc_snmp</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>[name1=value1, name2=value2,...]</options>
//   </agent>
// cisco_ucs, drac5, eps fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>cisco_ucs</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <options>slot=7[,name1=value1, name2=value2,...]</options>
//   </agent>
// drac7, ilo3, ilo4, ipmilan, rsb fencing agent/s sample request:
// [source,xml]
//   <agent>
//     <type>drac7</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <options>[name1=value1, name2=value2,...]</options>
//   </agent>
//
func (p *FenceAgentsService) Add() *FenceAgentsServiceAddRequest {
	return &FenceAgentsServiceAddRequest{FenceAgentsService: p}
}

//
// Returns the list of fencing agents configured for the host.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents
// ----
// And here is sample response:
// [source,xml]
// ----
// <agents>
//   <agent id="0">
//     <type>apc</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>name1=value1, name2=value2</options>
//   </agent>
// </agents>
// ----
// The order of the returned list of fencing agents isn't guaranteed.
//
type FenceAgentsServiceListRequest struct {
	FenceAgentsService *FenceAgentsService
	header             map[string]string
	query              map[string]string
	follow             *string
	max                *int64
}

func (p *FenceAgentsServiceListRequest) Header(key, value string) *FenceAgentsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentsServiceListRequest) Query(key, value string) *FenceAgentsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentsServiceListRequest) Follow(follow string) *FenceAgentsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *FenceAgentsServiceListRequest) Max(max int64) *FenceAgentsServiceListRequest {
	p.max = &max
	return p
}

func (p *FenceAgentsServiceListRequest) Send() (*FenceAgentsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentsService.connection.URL(), p.FenceAgentsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAgentReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &FenceAgentsServiceListResponse{agents: result}, nil
}

func (p *FenceAgentsServiceListRequest) MustSend() *FenceAgentsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of fencing agents configured for the host.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents
// ----
// And here is sample response:
// [source,xml]
// ----
// <agents>
//   <agent id="0">
//     <type>apc</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>name1=value1, name2=value2</options>
//   </agent>
// </agents>
// ----
// The order of the returned list of fencing agents isn't guaranteed.
//
type FenceAgentsServiceListResponse struct {
	agents *AgentSlice
}

func (p *FenceAgentsServiceListResponse) Agents() (*AgentSlice, bool) {
	if p.agents != nil {
		return p.agents, true
	}
	return nil, false
}

func (p *FenceAgentsServiceListResponse) MustAgents() *AgentSlice {
	if p.agents == nil {
		panic("agents in response does not exist")
	}
	return p.agents
}

//
// Returns the list of fencing agents configured for the host.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents
// ----
// And here is sample response:
// [source,xml]
// ----
// <agents>
//   <agent id="0">
//     <type>apc</type>
//     <order>1</order>
//     <ip>192.168.1.101</ip>
//     <user>user</user>
//     <password>xxx</password>
//     <port>9</port>
//     <options>name1=value1, name2=value2</options>
//   </agent>
// </agents>
// ----
// The order of the returned list of fencing agents isn't guaranteed.
//
func (p *FenceAgentsService) List() *FenceAgentsServiceListRequest {
	return &FenceAgentsServiceListRequest{FenceAgentsService: p}
}

//
// Reference to service that manages a specific fence agent
// for this host.
//
func (op *FenceAgentsService) AgentService(id string) *FenceAgentService {
	return NewFenceAgentService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FenceAgentsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.AgentService(path), nil
	}
	return op.AgentService(path[:index]).Service(path[index+1:])
}

func (op *FenceAgentsService) String() string {
	return fmt.Sprintf("FenceAgentsService:%s", op.path)
}

//
//
type FileService struct {
	BaseService
}

func NewFileService(connection *Connection, path string) *FileService {
	var result FileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type FileServiceGetRequest struct {
	FileService *FileService
	header      map[string]string
	query       map[string]string
	follow      *string
}

func (p *FileServiceGetRequest) Header(key, value string) *FileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FileServiceGetRequest) Query(key, value string) *FileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FileServiceGetRequest) Follow(follow string) *FileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *FileServiceGetRequest) Send() (*FileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FileService.connection.URL(), p.FileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FileServiceGetResponse{file: result}, nil
}

func (p *FileServiceGetRequest) MustSend() *FileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type FileServiceGetResponse struct {
	file *File
}

func (p *FileServiceGetResponse) File() (*File, bool) {
	if p.file != nil {
		return p.file, true
	}
	return nil, false
}

func (p *FileServiceGetResponse) MustFile() *File {
	if p.file == nil {
		panic("file in response does not exist")
	}
	return p.file
}

//
//
func (p *FileService) Get() *FileServiceGetRequest {
	return &FileServiceGetRequest{FileService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *FileService) String() string {
	return fmt.Sprintf("FileService:%s", op.path)
}

//
// Provides a way for clients to list available files.
// This service is specifically targeted to ISO storage domains, which contain ISO images and virtual floppy disks
// (VFDs) that an administrator uploads.
// The addition of a CD-ROM device to a virtual machine requires an ISO image from the files of an ISO storage domain.
//
type FilesService struct {
	BaseService
}

func NewFilesService(connection *Connection, path string) *FilesService {
	var result FilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
type FilesServiceListRequest struct {
	FilesService  *FilesService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	refresh       *bool
	search        *string
}

func (p *FilesServiceListRequest) Header(key, value string) *FilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FilesServiceListRequest) Query(key, value string) *FilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FilesServiceListRequest) CaseSensitive(caseSensitive bool) *FilesServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *FilesServiceListRequest) Follow(follow string) *FilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *FilesServiceListRequest) Max(max int64) *FilesServiceListRequest {
	p.max = &max
	return p
}

func (p *FilesServiceListRequest) Refresh(refresh bool) *FilesServiceListRequest {
	p.refresh = &refresh
	return p
}

func (p *FilesServiceListRequest) Search(search string) *FilesServiceListRequest {
	p.search = &search
	return p
}

func (p *FilesServiceListRequest) Send() (*FilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FilesService.connection.URL(), p.FilesService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.refresh != nil {
		values["refresh"] = []string{fmt.Sprintf("%v", *p.refresh)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &FilesServiceListResponse{file: result}, nil
}

func (p *FilesServiceListRequest) MustSend() *FilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
type FilesServiceListResponse struct {
	file *FileSlice
}

func (p *FilesServiceListResponse) File() (*FileSlice, bool) {
	if p.file != nil {
		return p.file, true
	}
	return nil, false
}

func (p *FilesServiceListResponse) MustFile() *FileSlice {
	if p.file == nil {
		panic("file in response does not exist")
	}
	return p.file
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
func (p *FilesService) List() *FilesServiceListRequest {
	return &FilesServiceListRequest{FilesService: p}
}

//
//
func (op *FilesService) FileService(id string) *FileService {
	return NewFileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FileService(path), nil
	}
	return op.FileService(path[:index]).Service(path[index+1:])
}

func (op *FilesService) String() string {
	return fmt.Sprintf("FilesService:%s", op.path)
}

//
//
type FilterService struct {
	BaseService
}

func NewFilterService(connection *Connection, path string) *FilterService {
	var result FilterService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type FilterServiceGetRequest struct {
	FilterService *FilterService
	header        map[string]string
	query         map[string]string
	filter        *bool
	follow        *string
}

func (p *FilterServiceGetRequest) Header(key, value string) *FilterServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FilterServiceGetRequest) Query(key, value string) *FilterServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FilterServiceGetRequest) Filter(filter bool) *FilterServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *FilterServiceGetRequest) Follow(follow string) *FilterServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *FilterServiceGetRequest) Send() (*FilterServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FilterService.connection.URL(), p.FilterService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FilterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FilterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FilterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FilterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FilterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFilterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FilterServiceGetResponse{result: result}, nil
}

func (p *FilterServiceGetRequest) MustSend() *FilterServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type FilterServiceGetResponse struct {
	result *Filter
}

func (p *FilterServiceGetResponse) Result() (*Filter, bool) {
	if p.result != nil {
		return p.result, true
	}
	return nil, false
}

func (p *FilterServiceGetResponse) MustResult() *Filter {
	if p.result == nil {
		panic("result in response does not exist")
	}
	return p.result
}

//
//
func (p *FilterService) Get() *FilterServiceGetRequest {
	return &FilterServiceGetRequest{FilterService: p}
}

//
//
type FilterServiceRemoveRequest struct {
	FilterService *FilterService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *FilterServiceRemoveRequest) Header(key, value string) *FilterServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FilterServiceRemoveRequest) Query(key, value string) *FilterServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FilterServiceRemoveRequest) Async(async bool) *FilterServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *FilterServiceRemoveRequest) Send() (*FilterServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FilterService.connection.URL(), p.FilterService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FilterService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FilterService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FilterService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FilterService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FilterService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(FilterServiceRemoveResponse), nil
}

func (p *FilterServiceRemoveRequest) MustSend() *FilterServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type FilterServiceRemoveResponse struct {
}

//
//
func (p *FilterService) Remove() *FilterServiceRemoveRequest {
	return &FilterServiceRemoveRequest{FilterService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FilterService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *FilterService) String() string {
	return fmt.Sprintf("FilterService:%s", op.path)
}

//
// Manages the filters used by an scheduling policy.
//
type FiltersService struct {
	BaseService
}

func NewFiltersService(connection *Connection, path string) *FiltersService {
	var result FiltersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a filter to a specified user defined scheduling policy.
//
type FiltersServiceAddRequest struct {
	FiltersService *FiltersService
	header         map[string]string
	query          map[string]string
	filter         *Filter
}

func (p *FiltersServiceAddRequest) Header(key, value string) *FiltersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FiltersServiceAddRequest) Query(key, value string) *FiltersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FiltersServiceAddRequest) Filter(filter *Filter) *FiltersServiceAddRequest {
	p.filter = filter
	return p
}

func (p *FiltersServiceAddRequest) Send() (*FiltersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FiltersService.connection.URL(), p.FiltersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLFilterWriteOne(writer, p.filter, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFilterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FiltersServiceAddResponse{filter: result}, nil
}

func (p *FiltersServiceAddRequest) MustSend() *FiltersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a filter to a specified user defined scheduling policy.
//
type FiltersServiceAddResponse struct {
	filter *Filter
}

func (p *FiltersServiceAddResponse) Filter() (*Filter, bool) {
	if p.filter != nil {
		return p.filter, true
	}
	return nil, false
}

func (p *FiltersServiceAddResponse) MustFilter() *Filter {
	if p.filter == nil {
		panic("filter in response does not exist")
	}
	return p.filter
}

//
// Add a filter to a specified user defined scheduling policy.
//
func (p *FiltersService) Add() *FiltersServiceAddRequest {
	return &FiltersServiceAddRequest{FiltersService: p}
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
type FiltersServiceListRequest struct {
	FiltersService *FiltersService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
	max            *int64
}

func (p *FiltersServiceListRequest) Header(key, value string) *FiltersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FiltersServiceListRequest) Query(key, value string) *FiltersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FiltersServiceListRequest) Filter(filter bool) *FiltersServiceListRequest {
	p.filter = &filter
	return p
}

func (p *FiltersServiceListRequest) Follow(follow string) *FiltersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *FiltersServiceListRequest) Max(max int64) *FiltersServiceListRequest {
	p.max = &max
	return p
}

func (p *FiltersServiceListRequest) Send() (*FiltersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FiltersService.connection.URL(), p.FiltersService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFilterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &FiltersServiceListResponse{filters: result}, nil
}

func (p *FiltersServiceListRequest) MustSend() *FiltersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
type FiltersServiceListResponse struct {
	filters *FilterSlice
}

func (p *FiltersServiceListResponse) Filters() (*FilterSlice, bool) {
	if p.filters != nil {
		return p.filters, true
	}
	return nil, false
}

func (p *FiltersServiceListResponse) MustFilters() *FilterSlice {
	if p.filters == nil {
		panic("filters in response does not exist")
	}
	return p.filters
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
func (p *FiltersService) List() *FiltersServiceListRequest {
	return &FiltersServiceListRequest{FiltersService: p}
}

//
//
func (op *FiltersService) FilterService(id string) *FilterService {
	return NewFilterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FiltersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FilterService(path), nil
	}
	return op.FilterService(path[:index]).Service(path[index+1:])
}

func (op *FiltersService) String() string {
	return fmt.Sprintf("FiltersService:%s", op.path)
}

//
// A service to access a particular device of a host.
//
type HostDeviceService struct {
	BaseService
}

func NewHostDeviceService(connection *Connection, path string) *HostDeviceService {
	var result HostDeviceService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
type HostDeviceServiceGetRequest struct {
	HostDeviceService *HostDeviceService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *HostDeviceServiceGetRequest) Header(key, value string) *HostDeviceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostDeviceServiceGetRequest) Query(key, value string) *HostDeviceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostDeviceServiceGetRequest) Follow(follow string) *HostDeviceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostDeviceServiceGetRequest) Send() (*HostDeviceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostDeviceService.connection.URL(), p.HostDeviceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostDeviceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostDeviceServiceGetResponse{device: result}, nil
}

func (p *HostDeviceServiceGetRequest) MustSend() *HostDeviceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
type HostDeviceServiceGetResponse struct {
	device *HostDevice
}

func (p *HostDeviceServiceGetResponse) Device() (*HostDevice, bool) {
	if p.device != nil {
		return p.device, true
	}
	return nil, false
}

func (p *HostDeviceServiceGetResponse) MustDevice() *HostDevice {
	if p.device == nil {
		panic("device in response does not exist")
	}
	return p.device
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
func (p *HostDeviceService) Get() *HostDeviceServiceGetRequest {
	return &HostDeviceServiceGetRequest{HostDeviceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostDeviceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostDeviceService) String() string {
	return fmt.Sprintf("HostDeviceService:%s", op.path)
}

//
// A service to access host devices.
//
type HostDevicesService struct {
	BaseService
}

func NewHostDevicesService(connection *Connection, path string) *HostDevicesService {
	var result HostDevicesService
	result.connection = connection
	result.path = path
	return &result
}

//
// List the devices of a host.
// The order of the returned list of devices isn't guaranteed.
//
type HostDevicesServiceListRequest struct {
	HostDevicesService *HostDevicesService
	header             map[string]string
	query              map[string]string
	follow             *string
	max                *int64
}

func (p *HostDevicesServiceListRequest) Header(key, value string) *HostDevicesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostDevicesServiceListRequest) Query(key, value string) *HostDevicesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostDevicesServiceListRequest) Follow(follow string) *HostDevicesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostDevicesServiceListRequest) Max(max int64) *HostDevicesServiceListRequest {
	p.max = &max
	return p
}

func (p *HostDevicesServiceListRequest) Send() (*HostDevicesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostDevicesService.connection.URL(), p.HostDevicesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostDevicesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostDevicesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostDevicesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostDevicesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostDevicesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostDeviceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostDevicesServiceListResponse{devices: result}, nil
}

func (p *HostDevicesServiceListRequest) MustSend() *HostDevicesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the devices of a host.
// The order of the returned list of devices isn't guaranteed.
//
type HostDevicesServiceListResponse struct {
	devices *HostDeviceSlice
}

func (p *HostDevicesServiceListResponse) Devices() (*HostDeviceSlice, bool) {
	if p.devices != nil {
		return p.devices, true
	}
	return nil, false
}

func (p *HostDevicesServiceListResponse) MustDevices() *HostDeviceSlice {
	if p.devices == nil {
		panic("devices in response does not exist")
	}
	return p.devices
}

//
// List the devices of a host.
// The order of the returned list of devices isn't guaranteed.
//
func (p *HostDevicesService) List() *HostDevicesServiceListRequest {
	return &HostDevicesServiceListRequest{HostDevicesService: p}
}

//
// Reference to the service that can be used to access a specific host device.
//
func (op *HostDevicesService) DeviceService(id string) *HostDeviceService {
	return NewHostDeviceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostDevicesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DeviceService(path), nil
	}
	return op.DeviceService(path[:index]).Service(path[index+1:])
}

func (op *HostDevicesService) String() string {
	return fmt.Sprintf("HostDevicesService:%s", op.path)
}

//
//
type HostHookService struct {
	BaseService
}

func NewHostHookService(connection *Connection, path string) *HostHookService {
	var result HostHookService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type HostHookServiceGetRequest struct {
	HostHookService *HostHookService
	header          map[string]string
	query           map[string]string
	follow          *string
}

func (p *HostHookServiceGetRequest) Header(key, value string) *HostHookServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostHookServiceGetRequest) Query(key, value string) *HostHookServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostHookServiceGetRequest) Follow(follow string) *HostHookServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostHookServiceGetRequest) Send() (*HostHookServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostHookService.connection.URL(), p.HostHookService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHookReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostHookServiceGetResponse{hook: result}, nil
}

func (p *HostHookServiceGetRequest) MustSend() *HostHookServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type HostHookServiceGetResponse struct {
	hook *Hook
}

func (p *HostHookServiceGetResponse) Hook() (*Hook, bool) {
	if p.hook != nil {
		return p.hook, true
	}
	return nil, false
}

func (p *HostHookServiceGetResponse) MustHook() *Hook {
	if p.hook == nil {
		panic("hook in response does not exist")
	}
	return p.hook
}

//
//
func (p *HostHookService) Get() *HostHookServiceGetRequest {
	return &HostHookServiceGetRequest{HostHookService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostHookService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostHookService) String() string {
	return fmt.Sprintf("HostHookService:%s", op.path)
}

//
//
type HostHooksService struct {
	BaseService
}

func NewHostHooksService(connection *Connection, path string) *HostHooksService {
	var result HostHooksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of hooks configured for the host.
// The order of the returned list of hooks is random.
//
type HostHooksServiceListRequest struct {
	HostHooksService *HostHooksService
	header           map[string]string
	query            map[string]string
	follow           *string
	max              *int64
}

func (p *HostHooksServiceListRequest) Header(key, value string) *HostHooksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostHooksServiceListRequest) Query(key, value string) *HostHooksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostHooksServiceListRequest) Follow(follow string) *HostHooksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostHooksServiceListRequest) Max(max int64) *HostHooksServiceListRequest {
	p.max = &max
	return p
}

func (p *HostHooksServiceListRequest) Send() (*HostHooksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostHooksService.connection.URL(), p.HostHooksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostHooksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostHooksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostHooksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostHooksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostHooksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHookReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostHooksServiceListResponse{hooks: result}, nil
}

func (p *HostHooksServiceListRequest) MustSend() *HostHooksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of hooks configured for the host.
// The order of the returned list of hooks is random.
//
type HostHooksServiceListResponse struct {
	hooks *HookSlice
}

func (p *HostHooksServiceListResponse) Hooks() (*HookSlice, bool) {
	if p.hooks != nil {
		return p.hooks, true
	}
	return nil, false
}

func (p *HostHooksServiceListResponse) MustHooks() *HookSlice {
	if p.hooks == nil {
		panic("hooks in response does not exist")
	}
	return p.hooks
}

//
// Returns the list of hooks configured for the host.
// The order of the returned list of hooks is random.
//
func (p *HostHooksService) List() *HostHooksServiceListRequest {
	return &HostHooksServiceListRequest{HostHooksService: p}
}

//
//
func (op *HostHooksService) HookService(id string) *HostHookService {
	return NewHostHookService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostHooksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HookService(path), nil
	}
	return op.HookService(path[:index]).Service(path[index+1:])
}

func (op *HostHooksService) String() string {
	return fmt.Sprintf("HostHooksService:%s", op.path)
}

//
// A service to manage a network interface of a host.
//
type HostNicService struct {
	BaseService
}

func NewHostNicService(connection *Connection, path string) *HostNicService {
	var result HostNicService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type HostNicServiceGetRequest struct {
	HostNicService *HostNicService
	header         map[string]string
	query          map[string]string
	allContent     *bool
	follow         *string
}

func (p *HostNicServiceGetRequest) Header(key, value string) *HostNicServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNicServiceGetRequest) Query(key, value string) *HostNicServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNicServiceGetRequest) AllContent(allContent bool) *HostNicServiceGetRequest {
	p.allContent = &allContent
	return p
}

func (p *HostNicServiceGetRequest) Follow(follow string) *HostNicServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostNicServiceGetRequest) Send() (*HostNicServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostNicService.connection.URL(), p.HostNicService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostNicServiceGetResponse{nic: result}, nil
}

func (p *HostNicServiceGetRequest) MustSend() *HostNicServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type HostNicServiceGetResponse struct {
	nic *HostNic
}

func (p *HostNicServiceGetResponse) Nic() (*HostNic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *HostNicServiceGetResponse) MustNic() *HostNic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
//
func (p *HostNicService) Get() *HostNicServiceGetRequest {
	return &HostNicServiceGetRequest{HostNicService: p}
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
type HostNicServiceUpdateVirtualFunctionsConfigurationRequest struct {
	HostNicService                *HostNicService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
	virtualFunctionsConfiguration *HostNicVirtualFunctionsConfiguration
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Header(key, value string) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Query(key, value string) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Async(async bool) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	p.async = &async
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) VirtualFunctionsConfiguration(virtualFunctionsConfiguration *HostNicVirtualFunctionsConfiguration) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	p.virtualFunctionsConfiguration = virtualFunctionsConfiguration
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Send() (*HostNicServiceUpdateVirtualFunctionsConfigurationResponse, error) {
	rawURL := fmt.Sprintf("%s%s/updatevirtualfunctionsconfiguration", p.HostNicService.connection.URL(), p.HostNicService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.VirtualFunctionsConfiguration(p.virtualFunctionsConfiguration)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostNicServiceUpdateVirtualFunctionsConfigurationResponse), nil
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) MustSend() *HostNicServiceUpdateVirtualFunctionsConfigurationResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
type HostNicServiceUpdateVirtualFunctionsConfigurationResponse struct {
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
func (p *HostNicService) UpdateVirtualFunctionsConfiguration() *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	return &HostNicServiceUpdateVirtualFunctionsConfigurationRequest{HostNicService: p}
}

//
// A reference to information elements received by LLDP on the NIC.
//
func (op *HostNicService) LinkLayerDiscoveryProtocolElementsService() *LinkLayerDiscoveryProtocolService {
	return NewLinkLayerDiscoveryProtocolService(op.connection, fmt.Sprintf("%s/linklayerdiscoveryprotocolelements", op.path))
}

//
// Reference to the service that manages the network attachments assigned to this network interface.
//
func (op *HostNicService) NetworkAttachmentsService() *NetworkAttachmentsService {
	return NewNetworkAttachmentsService(op.connection, fmt.Sprintf("%s/networkattachments", op.path))
}

//
// Reference to the service that manages the network labels assigned to this network interface.
//
func (op *HostNicService) NetworkLabelsService() *NetworkLabelsService {
	return NewNetworkLabelsService(op.connection, fmt.Sprintf("%s/networklabels", op.path))
}

//
//
func (op *HostNicService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Retrieves sub-collection resource of network labels that are allowed on an the virtual functions
// in case that the current resource represents an SR-IOV physical function NIC.
//
func (op *HostNicService) VirtualFunctionAllowedLabelsService() *NetworkLabelsService {
	return NewNetworkLabelsService(op.connection, fmt.Sprintf("%s/virtualfunctionallowedlabels", op.path))
}

//
// Retrieves sub-collection resource of networks that are allowed on an the virtual functions
// in case that the current resource represents an SR-IOV physical function NIC.
//
func (op *HostNicService) VirtualFunctionAllowedNetworksService() *VirtualFunctionAllowedNetworksService {
	return NewVirtualFunctionAllowedNetworksService(op.connection, fmt.Sprintf("%s/virtualfunctionallowednetworks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostNicService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "linklayerdiscoveryprotocolelements" {
		return op.LinkLayerDiscoveryProtocolElementsService(), nil
	}
	if strings.HasPrefix(path, "linklayerdiscoveryprotocolelements/") {
		return op.LinkLayerDiscoveryProtocolElementsService().Service(path[35:])
	}
	if path == "networkattachments" {
		return op.NetworkAttachmentsService(), nil
	}
	if strings.HasPrefix(path, "networkattachments/") {
		return op.NetworkAttachmentsService().Service(path[19:])
	}
	if path == "networklabels" {
		return op.NetworkLabelsService(), nil
	}
	if strings.HasPrefix(path, "networklabels/") {
		return op.NetworkLabelsService().Service(path[14:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	if path == "virtualfunctionallowedlabels" {
		return op.VirtualFunctionAllowedLabelsService(), nil
	}
	if strings.HasPrefix(path, "virtualfunctionallowedlabels/") {
		return op.VirtualFunctionAllowedLabelsService().Service(path[29:])
	}
	if path == "virtualfunctionallowednetworks" {
		return op.VirtualFunctionAllowedNetworksService(), nil
	}
	if strings.HasPrefix(path, "virtualfunctionallowednetworks/") {
		return op.VirtualFunctionAllowedNetworksService().Service(path[31:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostNicService) String() string {
	return fmt.Sprintf("HostNicService:%s", op.path)
}

//
// A service to manage the network interfaces of a host.
//
type HostNicsService struct {
	BaseService
}

func NewHostNicsService(connection *Connection, path string) *HostNicsService {
	var result HostNicsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of network interfaces of the host.
// The order of the returned list of network interfaces isn't guaranteed.
//
type HostNicsServiceListRequest struct {
	HostNicsService *HostNicsService
	header          map[string]string
	query           map[string]string
	allContent      *bool
	follow          *string
	max             *int64
}

func (p *HostNicsServiceListRequest) Header(key, value string) *HostNicsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNicsServiceListRequest) Query(key, value string) *HostNicsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNicsServiceListRequest) AllContent(allContent bool) *HostNicsServiceListRequest {
	p.allContent = &allContent
	return p
}

func (p *HostNicsServiceListRequest) Follow(follow string) *HostNicsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostNicsServiceListRequest) Max(max int64) *HostNicsServiceListRequest {
	p.max = &max
	return p
}

func (p *HostNicsServiceListRequest) Send() (*HostNicsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostNicsService.connection.URL(), p.HostNicsService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostNicReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostNicsServiceListResponse{nics: result}, nil
}

func (p *HostNicsServiceListRequest) MustSend() *HostNicsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of network interfaces of the host.
// The order of the returned list of network interfaces isn't guaranteed.
//
type HostNicsServiceListResponse struct {
	nics *HostNicSlice
}

func (p *HostNicsServiceListResponse) Nics() (*HostNicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *HostNicsServiceListResponse) MustNics() *HostNicSlice {
	if p.nics == nil {
		panic("nics in response does not exist")
	}
	return p.nics
}

//
// Returns the list of network interfaces of the host.
// The order of the returned list of network interfaces isn't guaranteed.
//
func (p *HostNicsService) List() *HostNicsServiceListRequest {
	return &HostNicsServiceListRequest{HostNicsService: p}
}

//
// Reference to the service that manages a single network interface.
//
func (op *HostNicsService) NicService(id string) *HostNicService {
	return NewHostNicService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostNicsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NicService(path), nil
	}
	return op.NicService(path[:index]).Service(path[index+1:])
}

func (op *HostNicsService) String() string {
	return fmt.Sprintf("HostNicsService:%s", op.path)
}

//
//
type HostNumaNodeService struct {
	BaseService
}

func NewHostNumaNodeService(connection *Connection, path string) *HostNumaNodeService {
	var result HostNumaNodeService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type HostNumaNodeServiceGetRequest struct {
	HostNumaNodeService *HostNumaNodeService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *HostNumaNodeServiceGetRequest) Header(key, value string) *HostNumaNodeServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNumaNodeServiceGetRequest) Query(key, value string) *HostNumaNodeServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNumaNodeServiceGetRequest) Follow(follow string) *HostNumaNodeServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostNumaNodeServiceGetRequest) Send() (*HostNumaNodeServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostNumaNodeService.connection.URL(), p.HostNumaNodeService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNumaNodeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNumaNodeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNumaNodeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNumaNodeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNumaNodeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNumaNodeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostNumaNodeServiceGetResponse{node: result}, nil
}

func (p *HostNumaNodeServiceGetRequest) MustSend() *HostNumaNodeServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type HostNumaNodeServiceGetResponse struct {
	node *NumaNode
}

func (p *HostNumaNodeServiceGetResponse) Node() (*NumaNode, bool) {
	if p.node != nil {
		return p.node, true
	}
	return nil, false
}

func (p *HostNumaNodeServiceGetResponse) MustNode() *NumaNode {
	if p.node == nil {
		panic("node in response does not exist")
	}
	return p.node
}

//
//
func (p *HostNumaNodeService) Get() *HostNumaNodeServiceGetRequest {
	return &HostNumaNodeServiceGetRequest{HostNumaNodeService: p}
}

//
//
func (op *HostNumaNodeService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostNumaNodeService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostNumaNodeService) String() string {
	return fmt.Sprintf("HostNumaNodeService:%s", op.path)
}

//
//
type HostNumaNodesService struct {
	BaseService
}

func NewHostNumaNodesService(connection *Connection, path string) *HostNumaNodesService {
	var result HostNumaNodesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of NUMA nodes of the host.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
type HostNumaNodesServiceListRequest struct {
	HostNumaNodesService *HostNumaNodesService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *HostNumaNodesServiceListRequest) Header(key, value string) *HostNumaNodesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNumaNodesServiceListRequest) Query(key, value string) *HostNumaNodesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNumaNodesServiceListRequest) Follow(follow string) *HostNumaNodesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostNumaNodesServiceListRequest) Max(max int64) *HostNumaNodesServiceListRequest {
	p.max = &max
	return p
}

func (p *HostNumaNodesServiceListRequest) Send() (*HostNumaNodesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostNumaNodesService.connection.URL(), p.HostNumaNodesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNumaNodesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNumaNodesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNumaNodesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNumaNodesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNumaNodesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNumaNodeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostNumaNodesServiceListResponse{nodes: result}, nil
}

func (p *HostNumaNodesServiceListRequest) MustSend() *HostNumaNodesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of NUMA nodes of the host.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
type HostNumaNodesServiceListResponse struct {
	nodes *NumaNodeSlice
}

func (p *HostNumaNodesServiceListResponse) Nodes() (*NumaNodeSlice, bool) {
	if p.nodes != nil {
		return p.nodes, true
	}
	return nil, false
}

func (p *HostNumaNodesServiceListResponse) MustNodes() *NumaNodeSlice {
	if p.nodes == nil {
		panic("nodes in response does not exist")
	}
	return p.nodes
}

//
// Returns the list of NUMA nodes of the host.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
func (p *HostNumaNodesService) List() *HostNumaNodesServiceListRequest {
	return &HostNumaNodesServiceListRequest{HostNumaNodesService: p}
}

//
//
func (op *HostNumaNodesService) NodeService(id string) *HostNumaNodeService {
	return NewHostNumaNodeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostNumaNodesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NodeService(path), nil
	}
	return op.NodeService(path[:index]).Service(path[index+1:])
}

func (op *HostNumaNodesService) String() string {
	return fmt.Sprintf("HostNumaNodesService:%s", op.path)
}

//
// A service to manage a host.
//
type HostService struct {
	BaseService
}

func NewHostService(connection *Connection, path string) *HostService {
	var result HostService
	result.connection = connection
	result.path = path
	return &result
}

//
// Activates the host for use, for example to run virtual machines.
//
type HostServiceActivateRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceActivateRequest) Header(key, value string) *HostServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceActivateRequest) Query(key, value string) *HostServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceActivateRequest) Async(async bool) *HostServiceActivateRequest {
	p.async = &async
	return p
}

func (p *HostServiceActivateRequest) Send() (*HostServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceActivateResponse), nil
}

func (p *HostServiceActivateRequest) MustSend() *HostServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Activates the host for use, for example to run virtual machines.
//
type HostServiceActivateResponse struct {
}

//
// Activates the host for use, for example to run virtual machines.
//
func (p *HostService) Activate() *HostServiceActivateRequest {
	return &HostServiceActivateRequest{HostService: p}
}

//
// Approve a pre-installed Hypervisor host for usage in the virtualization environment.
// This action also accepts an optional cluster element to define the target cluster for this host.
//
type HostServiceApproveRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	activate    *bool
	async       *bool
	cluster     *Cluster
	host        *Host
	reboot      *bool
}

func (p *HostServiceApproveRequest) Header(key, value string) *HostServiceApproveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceApproveRequest) Query(key, value string) *HostServiceApproveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceApproveRequest) Activate(activate bool) *HostServiceApproveRequest {
	p.activate = &activate
	return p
}

func (p *HostServiceApproveRequest) Async(async bool) *HostServiceApproveRequest {
	p.async = &async
	return p
}

func (p *HostServiceApproveRequest) Cluster(cluster *Cluster) *HostServiceApproveRequest {
	p.cluster = cluster
	return p
}

func (p *HostServiceApproveRequest) Host(host *Host) *HostServiceApproveRequest {
	p.host = host
	return p
}

func (p *HostServiceApproveRequest) Reboot(reboot bool) *HostServiceApproveRequest {
	p.reboot = &reboot
	return p
}

func (p *HostServiceApproveRequest) Send() (*HostServiceApproveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/approve", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.activate != nil {
		actionBuilder.Activate(*p.activate)
	}
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Cluster(p.cluster)
	actionBuilder.Host(p.host)
	if p.reboot != nil {
		actionBuilder.Reboot(*p.reboot)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceApproveResponse), nil
}

func (p *HostServiceApproveRequest) MustSend() *HostServiceApproveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Approve a pre-installed Hypervisor host for usage in the virtualization environment.
// This action also accepts an optional cluster element to define the target cluster for this host.
//
type HostServiceApproveResponse struct {
}

//
// Approve a pre-installed Hypervisor host for usage in the virtualization environment.
// This action also accepts an optional cluster element to define the target cluster for this host.
//
func (p *HostService) Approve() *HostServiceApproveRequest {
	return &HostServiceApproveRequest{HostService: p}
}

//
// Marks the network configuration as good and persists it inside the host.
// An API user commits the network configuration to persist a host network interface attachment or detachment, or
// persist the creation and deletion of a bonded interface.
// IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
// not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
// automatically reverts to the previous networking configuration.
// For example, to commit the network configuration of host with id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/commitnetconfig
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// IMPORTANT: Since {engine-name} 4.3, it is possible to also specify `commit_on_success` in
// the <<services/host/methods/setup_networks, setupnetworks>> request, in which case the new
// configuration is automatically saved in the {hypervisor-name} upon completing the setup and
// re-establishing connectivity between the {hypervisor-name} and {engine-name}, and without
// waiting for a separate <<services/host/methods/commit_net_config, commitnetconfig>> request.
//
type HostServiceCommitNetConfigRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceCommitNetConfigRequest) Header(key, value string) *HostServiceCommitNetConfigRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceCommitNetConfigRequest) Query(key, value string) *HostServiceCommitNetConfigRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceCommitNetConfigRequest) Async(async bool) *HostServiceCommitNetConfigRequest {
	p.async = &async
	return p
}

func (p *HostServiceCommitNetConfigRequest) Send() (*HostServiceCommitNetConfigResponse, error) {
	rawURL := fmt.Sprintf("%s%s/commitnetconfig", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceCommitNetConfigResponse), nil
}

func (p *HostServiceCommitNetConfigRequest) MustSend() *HostServiceCommitNetConfigResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Marks the network configuration as good and persists it inside the host.
// An API user commits the network configuration to persist a host network interface attachment or detachment, or
// persist the creation and deletion of a bonded interface.
// IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
// not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
// automatically reverts to the previous networking configuration.
// For example, to commit the network configuration of host with id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/commitnetconfig
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// IMPORTANT: Since {engine-name} 4.3, it is possible to also specify `commit_on_success` in
// the <<services/host/methods/setup_networks, setupnetworks>> request, in which case the new
// configuration is automatically saved in the {hypervisor-name} upon completing the setup and
// re-establishing connectivity between the {hypervisor-name} and {engine-name}, and without
// waiting for a separate <<services/host/methods/commit_net_config, commitnetconfig>> request.
//
type HostServiceCommitNetConfigResponse struct {
}

//
// Marks the network configuration as good and persists it inside the host.
// An API user commits the network configuration to persist a host network interface attachment or detachment, or
// persist the creation and deletion of a bonded interface.
// IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
// not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
// automatically reverts to the previous networking configuration.
// For example, to commit the network configuration of host with id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/commitnetconfig
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// IMPORTANT: Since {engine-name} 4.3, it is possible to also specify `commit_on_success` in
// the <<services/host/methods/setup_networks, setupnetworks>> request, in which case the new
// configuration is automatically saved in the {hypervisor-name} upon completing the setup and
// re-establishing connectivity between the {hypervisor-name} and {engine-name}, and without
// waiting for a separate <<services/host/methods/commit_net_config, commitnetconfig>> request.
//
func (p *HostService) CommitNetConfig() *HostServiceCommitNetConfigRequest {
	return &HostServiceCommitNetConfigRequest{HostService: p}
}

//
// Copy the network configuration of the specified host to current host.
// IMPORTANT: Any network attachments that are not present on the source host will be erased from the target host
// by the copy operation.
// To copy networks from another host, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/copyhostnetworks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//    <source_host id="456"/>
// </action>
// ----
//
type HostServiceCopyHostNetworksRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
	sourceHost  *Host
}

func (p *HostServiceCopyHostNetworksRequest) Header(key, value string) *HostServiceCopyHostNetworksRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceCopyHostNetworksRequest) Query(key, value string) *HostServiceCopyHostNetworksRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceCopyHostNetworksRequest) Async(async bool) *HostServiceCopyHostNetworksRequest {
	p.async = &async
	return p
}

func (p *HostServiceCopyHostNetworksRequest) SourceHost(sourceHost *Host) *HostServiceCopyHostNetworksRequest {
	p.sourceHost = sourceHost
	return p
}

func (p *HostServiceCopyHostNetworksRequest) Send() (*HostServiceCopyHostNetworksResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copyhostnetworks", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.SourceHost(p.sourceHost)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceCopyHostNetworksResponse), nil
}

func (p *HostServiceCopyHostNetworksRequest) MustSend() *HostServiceCopyHostNetworksResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Copy the network configuration of the specified host to current host.
// IMPORTANT: Any network attachments that are not present on the source host will be erased from the target host
// by the copy operation.
// To copy networks from another host, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/copyhostnetworks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//    <source_host id="456"/>
// </action>
// ----
//
type HostServiceCopyHostNetworksResponse struct {
}

//
// Copy the network configuration of the specified host to current host.
// IMPORTANT: Any network attachments that are not present on the source host will be erased from the target host
// by the copy operation.
// To copy networks from another host, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/copyhostnetworks
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//    <source_host id="456"/>
// </action>
// ----
//
func (p *HostService) CopyHostNetworks() *HostServiceCopyHostNetworksRequest {
	return &HostServiceCopyHostNetworksRequest{HostService: p}
}

//
// Deactivates the host to perform maintenance tasks.
//
type HostServiceDeactivateRequest struct {
	HostService        *HostService
	header             map[string]string
	query              map[string]string
	async              *bool
	reason             *string
	stopGlusterService *bool
}

func (p *HostServiceDeactivateRequest) Header(key, value string) *HostServiceDeactivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceDeactivateRequest) Query(key, value string) *HostServiceDeactivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceDeactivateRequest) Async(async bool) *HostServiceDeactivateRequest {
	p.async = &async
	return p
}

func (p *HostServiceDeactivateRequest) Reason(reason string) *HostServiceDeactivateRequest {
	p.reason = &reason
	return p
}

func (p *HostServiceDeactivateRequest) StopGlusterService(stopGlusterService bool) *HostServiceDeactivateRequest {
	p.stopGlusterService = &stopGlusterService
	return p
}

func (p *HostServiceDeactivateRequest) Send() (*HostServiceDeactivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/deactivate", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.reason != nil {
		actionBuilder.Reason(*p.reason)
	}
	if p.stopGlusterService != nil {
		actionBuilder.StopGlusterService(*p.stopGlusterService)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceDeactivateResponse), nil
}

func (p *HostServiceDeactivateRequest) MustSend() *HostServiceDeactivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Deactivates the host to perform maintenance tasks.
//
type HostServiceDeactivateResponse struct {
}

//
// Deactivates the host to perform maintenance tasks.
//
func (p *HostService) Deactivate() *HostServiceDeactivateRequest {
	return &HostServiceDeactivateRequest{HostService: p}
}

//
// Discovers iSCSI targets on the host, using the initiator details.
// Returns a list of IscsiDetails objects containing the discovered data.
// For example, to discover iSCSI targets available in `myiscsi.example.com`,
// from host `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/discoveriscsi
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <iscsi>
//     <address>myiscsi.example.com</address>
//   </iscsi>
// </action>
// ----
// The result will be like this:
// [source,xml]
// ----
// <discovered_targets>
//   <iscsi_details>
//     <address>10.35.1.72</address>
//     <port>3260</port>
//     <portal>10.35.1.72:3260,1</portal>
//     <target>iqn.2015-08.com.tgt:444</target>
//   </iscsi_details>
// </discovered_targets>
// ----
//
type HostServiceDiscoverIscsiRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
	iscsi       *IscsiDetails
}

func (p *HostServiceDiscoverIscsiRequest) Header(key, value string) *HostServiceDiscoverIscsiRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceDiscoverIscsiRequest) Query(key, value string) *HostServiceDiscoverIscsiRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceDiscoverIscsiRequest) Async(async bool) *HostServiceDiscoverIscsiRequest {
	p.async = &async
	return p
}

func (p *HostServiceDiscoverIscsiRequest) Iscsi(iscsi *IscsiDetails) *HostServiceDiscoverIscsiRequest {
	p.iscsi = iscsi
	return p
}

func (p *HostServiceDiscoverIscsiRequest) Send() (*HostServiceDiscoverIscsiResponse, error) {
	rawURL := fmt.Sprintf("%s%s/discoveriscsi", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Iscsi(p.iscsi)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDiscoveredTargets()
	return &HostServiceDiscoverIscsiResponse{discoveredTargets: result}, nil
}

func (p *HostServiceDiscoverIscsiRequest) MustSend() *HostServiceDiscoverIscsiResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Discovers iSCSI targets on the host, using the initiator details.
// Returns a list of IscsiDetails objects containing the discovered data.
// For example, to discover iSCSI targets available in `myiscsi.example.com`,
// from host `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/discoveriscsi
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <iscsi>
//     <address>myiscsi.example.com</address>
//   </iscsi>
// </action>
// ----
// The result will be like this:
// [source,xml]
// ----
// <discovered_targets>
//   <iscsi_details>
//     <address>10.35.1.72</address>
//     <port>3260</port>
//     <portal>10.35.1.72:3260,1</portal>
//     <target>iqn.2015-08.com.tgt:444</target>
//   </iscsi_details>
// </discovered_targets>
// ----
//
type HostServiceDiscoverIscsiResponse struct {
	discoveredTargets *IscsiDetailsSlice
}

func (p *HostServiceDiscoverIscsiResponse) DiscoveredTargets() (*IscsiDetailsSlice, bool) {
	if p.discoveredTargets != nil {
		return p.discoveredTargets, true
	}
	return nil, false
}

func (p *HostServiceDiscoverIscsiResponse) MustDiscoveredTargets() *IscsiDetailsSlice {
	if p.discoveredTargets == nil {
		panic("discoveredTargets in response does not exist")
	}
	return p.discoveredTargets
}

//
// Discovers iSCSI targets on the host, using the initiator details.
// Returns a list of IscsiDetails objects containing the discovered data.
// For example, to discover iSCSI targets available in `myiscsi.example.com`,
// from host `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/discoveriscsi
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <iscsi>
//     <address>myiscsi.example.com</address>
//   </iscsi>
// </action>
// ----
// The result will be like this:
// [source,xml]
// ----
// <discovered_targets>
//   <iscsi_details>
//     <address>10.35.1.72</address>
//     <port>3260</port>
//     <portal>10.35.1.72:3260,1</portal>
//     <target>iqn.2015-08.com.tgt:444</target>
//   </iscsi_details>
// </discovered_targets>
// ----
//
func (p *HostService) DiscoverIscsi() *HostServiceDiscoverIscsiRequest {
	return &HostServiceDiscoverIscsiRequest{HostService: p}
}

//
// Enrolls the certificate of the host. Useful in case you get a warning that it is about to expire or has already
// expired.
//
type HostServiceEnrollCertificateRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceEnrollCertificateRequest) Header(key, value string) *HostServiceEnrollCertificateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceEnrollCertificateRequest) Query(key, value string) *HostServiceEnrollCertificateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceEnrollCertificateRequest) Async(async bool) *HostServiceEnrollCertificateRequest {
	p.async = &async
	return p
}

func (p *HostServiceEnrollCertificateRequest) Send() (*HostServiceEnrollCertificateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/enrollcertificate", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceEnrollCertificateResponse), nil
}

func (p *HostServiceEnrollCertificateRequest) MustSend() *HostServiceEnrollCertificateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Enrolls the certificate of the host. Useful in case you get a warning that it is about to expire or has already
// expired.
//
type HostServiceEnrollCertificateResponse struct {
}

//
// Enrolls the certificate of the host. Useful in case you get a warning that it is about to expire or has already
// expired.
//
func (p *HostService) EnrollCertificate() *HostServiceEnrollCertificateRequest {
	return &HostServiceEnrollCertificateRequest{HostService: p}
}

//
// Controls the host's power management device.
// For example, to start the host. This can be done via:
// [source]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <action>
//   <fence_type>start</fence_type>
// </action>
// ' \
// "${url}/hosts/123/fence"
// ----
//
type HostServiceFenceRequest struct {
	HostService             *HostService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
	fenceType               *string
	maintenanceAfterRestart *bool
}

func (p *HostServiceFenceRequest) Header(key, value string) *HostServiceFenceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceFenceRequest) Query(key, value string) *HostServiceFenceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceFenceRequest) Async(async bool) *HostServiceFenceRequest {
	p.async = &async
	return p
}

func (p *HostServiceFenceRequest) FenceType(fenceType string) *HostServiceFenceRequest {
	p.fenceType = &fenceType
	return p
}

func (p *HostServiceFenceRequest) MaintenanceAfterRestart(maintenanceAfterRestart bool) *HostServiceFenceRequest {
	p.maintenanceAfterRestart = &maintenanceAfterRestart
	return p
}

func (p *HostServiceFenceRequest) Send() (*HostServiceFenceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/fence", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.fenceType != nil {
		actionBuilder.FenceType(*p.fenceType)
	}
	if p.maintenanceAfterRestart != nil {
		actionBuilder.MaintenanceAfterRestart(*p.maintenanceAfterRestart)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPowerManagement()
	return &HostServiceFenceResponse{powerManagement: result}, nil
}

func (p *HostServiceFenceRequest) MustSend() *HostServiceFenceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Controls the host's power management device.
// For example, to start the host. This can be done via:
// [source]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <action>
//   <fence_type>start</fence_type>
// </action>
// ' \
// "${url}/hosts/123/fence"
// ----
//
type HostServiceFenceResponse struct {
	powerManagement *PowerManagement
}

func (p *HostServiceFenceResponse) PowerManagement() (*PowerManagement, bool) {
	if p.powerManagement != nil {
		return p.powerManagement, true
	}
	return nil, false
}

func (p *HostServiceFenceResponse) MustPowerManagement() *PowerManagement {
	if p.powerManagement == nil {
		panic("powerManagement in response does not exist")
	}
	return p.powerManagement
}

//
// Controls the host's power management device.
// For example, to start the host. This can be done via:
// [source]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <action>
//   <fence_type>start</fence_type>
// </action>
// ' \
// "${url}/hosts/123/fence"
// ----
//
func (p *HostService) Fence() *HostServiceFenceRequest {
	return &HostServiceFenceRequest{HostService: p}
}

//
// To manually set a host as the storage pool manager (SPM).
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/forceselectspm
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type HostServiceForceSelectSpmRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceForceSelectSpmRequest) Header(key, value string) *HostServiceForceSelectSpmRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceForceSelectSpmRequest) Query(key, value string) *HostServiceForceSelectSpmRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceForceSelectSpmRequest) Async(async bool) *HostServiceForceSelectSpmRequest {
	p.async = &async
	return p
}

func (p *HostServiceForceSelectSpmRequest) Send() (*HostServiceForceSelectSpmResponse, error) {
	rawURL := fmt.Sprintf("%s%s/forceselectspm", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceForceSelectSpmResponse), nil
}

func (p *HostServiceForceSelectSpmRequest) MustSend() *HostServiceForceSelectSpmResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// To manually set a host as the storage pool manager (SPM).
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/forceselectspm
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type HostServiceForceSelectSpmResponse struct {
}

//
// To manually set a host as the storage pool manager (SPM).
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/forceselectspm
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *HostService) ForceSelectSpm() *HostServiceForceSelectSpmRequest {
	return &HostServiceForceSelectSpmRequest{HostService: p}
}

//
// Gets the host details.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123
// ----
//
type HostServiceGetRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	allContent  *bool
	filter      *bool
	follow      *string
}

func (p *HostServiceGetRequest) Header(key, value string) *HostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceGetRequest) Query(key, value string) *HostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceGetRequest) AllContent(allContent bool) *HostServiceGetRequest {
	p.allContent = &allContent
	return p
}

func (p *HostServiceGetRequest) Filter(filter bool) *HostServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *HostServiceGetRequest) Follow(follow string) *HostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostServiceGetRequest) Send() (*HostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostService.connection.URL(), p.HostService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostServiceGetResponse{host: result}, nil
}

func (p *HostServiceGetRequest) MustSend() *HostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the host details.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123
// ----
//
type HostServiceGetResponse struct {
	host *Host
}

func (p *HostServiceGetResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *HostServiceGetResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
/