//
// Copyright (c) 2017 Joey <majunjiev@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package ovirtsdk

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httputil"
	"net/url"
	"strings"
)

//
//
type FollowService struct {
	BaseService
}

func NewFollowService(connection *Connection, path string) *FollowService {
	var result FollowService
	result.connection = connection
	result.path = path
	return &result
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FollowService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *FollowService) String() string {
	return fmt.Sprintf("FollowService:%s", op.path)
}

//
// A service to manage jobs.
//
type JobsService struct {
	BaseService
}

func NewJobsService(connection *Connection, path string) *JobsService {
	var result JobsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add an external job.
// For example, to add a job with the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs
// ----
// With the following request body:
// [source,xml]
// ----
// <job>
//   <description>Doing some work</description>
//   <auto_cleared>true</auto_cleared>
// </job>
// ----
// The response should look like:
// [source,xml]
// ----
// <job href="/ovirt-engine/api/jobs/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//     <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//   </actions>
//   <description>Doing some work</description>
//   <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//   <auto_cleared>true</auto_cleared>
//   <external>true</external>
//   <last_updated>2016-12-13T02:15:42.130+02:00</last_updated>
//   <start_time>2016-12-13T02:15:42.130+02:00</start_time>
//   <status>started</status>
//   <owner href="/ovirt-engine/api/users/456" id="456"/>
// </job>
// ----
//
type JobsServiceAddRequest struct {
	JobsService *JobsService
	header      map[string]string
	query       map[string]string
	job         *Job
}

func (p *JobsServiceAddRequest) Header(key, value string) *JobsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *JobsServiceAddRequest) Query(key, value string) *JobsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *JobsServiceAddRequest) Job(job *Job) *JobsServiceAddRequest {
	p.job = job
	return p
}

func (p *JobsServiceAddRequest) Send() (*JobsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.JobsService.connection.URL(), p.JobsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLJobWriteOne(writer, p.job, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.JobsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.JobsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.JobsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.JobsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.JobsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLJobReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &JobsServiceAddResponse{job: result}, nil
}

func (p *JobsServiceAddRequest) MustSend() *JobsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add an external job.
// For example, to add a job with the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs
// ----
// With the following request body:
// [source,xml]
// ----
// <job>
//   <description>Doing some work</description>
//   <auto_cleared>true</auto_cleared>
// </job>
// ----
// The response should look like:
// [source,xml]
// ----
// <job href="/ovirt-engine/api/jobs/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//     <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//   </actions>
//   <description>Doing some work</description>
//   <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//   <auto_cleared>true</auto_cleared>
//   <external>true</external>
//   <last_updated>2016-12-13T02:15:42.130+02:00</last_updated>
//   <start_time>2016-12-13T02:15:42.130+02:00</start_time>
//   <status>started</status>
//   <owner href="/ovirt-engine/api/users/456" id="456"/>
// </job>
// ----
//
type JobsServiceAddResponse struct {
	job *Job
}

func (p *JobsServiceAddResponse) Job() (*Job, bool) {
	if p.job != nil {
		return p.job, true
	}
	return nil, false
}

func (p *JobsServiceAddResponse) MustJob() *Job {
	if p.job == nil {
		panic("job in response does not exist")
	}
	return p.job
}

//
// Add an external job.
// For example, to add a job with the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs
// ----
// With the following request body:
// [source,xml]
// ----
// <job>
//   <description>Doing some work</description>
//   <auto_cleared>true</auto_cleared>
// </job>
// ----
// The response should look like:
// [source,xml]
// ----
// <job href="/ovirt-engine/api/jobs/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//     <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//   </actions>
//   <description>Doing some work</description>
//   <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//   <auto_cleared>true</auto_cleared>
//   <external>true</external>
//   <last_updated>2016-12-13T02:15:42.130+02:00</last_updated>
//   <start_time>2016-12-13T02:15:42.130+02:00</start_time>
//   <status>started</status>
//   <owner href="/ovirt-engine/api/users/456" id="456"/>
// </job>
// ----
//
func (p *JobsService) Add() *JobsServiceAddRequest {
	return &JobsServiceAddRequest{JobsService: p}
}

//
// Retrieves the representation of the jobs.
// [source]
// ----
// GET /ovirt-engine/api/jobs
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <jobs>
//   <job href="/ovirt-engine/api/jobs/123" id="123">
//     <actions>
//       <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//       <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//     </actions>
//     <description>Adding Disk</description>
//     <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//     <auto_cleared>true</auto_cleared>
//     <end_time>2016-12-12T23:07:29.758+02:00</end_time>
//     <external>false</external>
//     <last_updated>2016-12-12T23:07:29.758+02:00</last_updated>
//     <start_time>2016-12-12T23:07:26.593+02:00</start_time>
//     <status>failed</status>
//     <owner href="/ovirt-engine/api/users/456" id="456"/>
//   </job>
//   ...
// </jobs>
// ----
// The order of the returned list of jobs isn't guaranteed.
//
type JobsServiceListRequest struct {
	JobsService   *JobsService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *JobsServiceListRequest) Header(key, value string) *JobsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *JobsServiceListRequest) Query(key, value string) *JobsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *JobsServiceListRequest) CaseSensitive(caseSensitive bool) *JobsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *JobsServiceListRequest) Follow(follow string) *JobsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *JobsServiceListRequest) Max(max int64) *JobsServiceListRequest {
	p.max = &max
	return p
}

func (p *JobsServiceListRequest) Search(search string) *JobsServiceListRequest {
	p.search = &search
	return p
}

func (p *JobsServiceListRequest) Send() (*JobsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.JobsService.connection.URL(), p.JobsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.JobsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.JobsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.JobsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.JobsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.JobsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLJobReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &JobsServiceListResponse{jobs: result}, nil
}

func (p *JobsServiceListRequest) MustSend() *JobsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representation of the jobs.
// [source]
// ----
// GET /ovirt-engine/api/jobs
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <jobs>
//   <job href="/ovirt-engine/api/jobs/123" id="123">
//     <actions>
//       <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//       <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//     </actions>
//     <description>Adding Disk</description>
//     <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//     <auto_cleared>true</auto_cleared>
//     <end_time>2016-12-12T23:07:29.758+02:00</end_time>
//     <external>false</external>
//     <last_updated>2016-12-12T23:07:29.758+02:00</last_updated>
//     <start_time>2016-12-12T23:07:26.593+02:00</start_time>
//     <status>failed</status>
//     <owner href="/ovirt-engine/api/users/456" id="456"/>
//   </job>
//   ...
// </jobs>
// ----
// The order of the returned list of jobs isn't guaranteed.
//
type JobsServiceListResponse struct {
	jobs *JobSlice
}

func (p *JobsServiceListResponse) Jobs() (*JobSlice, bool) {
	if p.jobs != nil {
		return p.jobs, true
	}
	return nil, false
}

func (p *JobsServiceListResponse) MustJobs() *JobSlice {
	if p.jobs == nil {
		panic("jobs in response does not exist")
	}
	return p.jobs
}

//
// Retrieves the representation of the jobs.
// [source]
// ----
// GET /ovirt-engine/api/jobs
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <jobs>
//   <job href="/ovirt-engine/api/jobs/123" id="123">
//     <actions>
//       <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//       <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//     </actions>
//     <description>Adding Disk</description>
//     <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//     <auto_cleared>true</auto_cleared>
//     <end_time>2016-12-12T23:07:29.758+02:00</end_time>
//     <external>false</external>
//     <last_updated>2016-12-12T23:07:29.758+02:00</last_updated>
//     <start_time>2016-12-12T23:07:26.593+02:00</start_time>
//     <status>failed</status>
//     <owner href="/ovirt-engine/api/users/456" id="456"/>
//   </job>
//   ...
// </jobs>
// ----
// The order of the returned list of jobs isn't guaranteed.
//
func (p *JobsService) List() *JobsServiceListRequest {
	return &JobsServiceListRequest{JobsService: p}
}

//
// Reference to the job service.
//
func (op *JobsService) JobService(id string) *JobService {
	return NewJobService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *JobsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.JobService(path), nil
	}
	return op.JobService(path[:index]).Service(path[index+1:])
}

func (op *JobsService) String() string {
	return fmt.Sprintf("JobsService:%s", op.path)
}

//
// A service to manage data centers.
//
type DataCentersService struct {
	BaseService
}

func NewDataCentersService(connection *Connection, path string) *DataCentersService {
	var result DataCentersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new data center.
// Creation of a new data center requires the `name` and `local` elements. For example, to create a data center
// named `mydc` that uses shared storage (NFS, iSCSI or fibre channel) send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>mydc</name>
//   <local>false</local>
// </data_center>
// ----
//
type DataCentersServiceAddRequest struct {
	DataCentersService *DataCentersService
	header             map[string]string
	query              map[string]string
	dataCenter         *DataCenter
}

func (p *DataCentersServiceAddRequest) Header(key, value string) *DataCentersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCentersServiceAddRequest) Query(key, value string) *DataCentersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCentersServiceAddRequest) DataCenter(dataCenter *DataCenter) *DataCentersServiceAddRequest {
	p.dataCenter = dataCenter
	return p
}

func (p *DataCentersServiceAddRequest) Send() (*DataCentersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCentersService.connection.URL(), p.DataCentersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDataCenterWriteOne(writer, p.dataCenter, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCentersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCentersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCentersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCentersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCentersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDataCenterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCentersServiceAddResponse{dataCenter: result}, nil
}

func (p *DataCentersServiceAddRequest) MustSend() *DataCentersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new data center.
// Creation of a new data center requires the `name` and `local` elements. For example, to create a data center
// named `mydc` that uses shared storage (NFS, iSCSI or fibre channel) send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>mydc</name>
//   <local>false</local>
// </data_center>
// ----
//
type DataCentersServiceAddResponse struct {
	dataCenter *DataCenter
}

func (p *DataCentersServiceAddResponse) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *DataCentersServiceAddResponse) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("dataCenter in response does not exist")
	}
	return p.dataCenter
}

//
// Creates a new data center.
// Creation of a new data center requires the `name` and `local` elements. For example, to create a data center
// named `mydc` that uses shared storage (NFS, iSCSI or fibre channel) send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters
// ----
// With a request body like this:
// [source,xml]
// ----
// <data_center>
//   <name>mydc</name>
//   <local>false</local>
// </data_center>
// ----
//
func (p *DataCentersService) Add() *DataCentersServiceAddRequest {
	return &DataCentersServiceAddRequest{DataCentersService: p}
}

//
// Lists the data centers.
// The following request retrieves a representation of the data centers:
// [source]
// ----
// GET /ovirt-engine/api/datacenters
// ----
// The above request performed with `curl`:
// [source,bash]
// ----
// curl \
// --request GET \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --header "Version: 4" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// https://myengine.example.com/ovirt-engine/api/datacenters
// ----
// This is what an example response could look like:
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//     </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
// </data_center>
// ----
// Note the `id` code of your `Default` data center. This code identifies this data center in relation to other
// resources of your virtual environment.
// The data center also contains a link to the storage domains collection. The data center uses this collection to
// attach storage domains from the storage domains main collection.
// The order of the returned list of data centers is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DataCentersServiceListRequest struct {
	DataCentersService *DataCentersService
	header             map[string]string
	query              map[string]string
	caseSensitive      *bool
	filter             *bool
	follow             *string
	max                *int64
	search             *string
}

func (p *DataCentersServiceListRequest) Header(key, value string) *DataCentersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCentersServiceListRequest) Query(key, value string) *DataCentersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCentersServiceListRequest) CaseSensitive(caseSensitive bool) *DataCentersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DataCentersServiceListRequest) Filter(filter bool) *DataCentersServiceListRequest {
	p.filter = &filter
	return p
}

func (p *DataCentersServiceListRequest) Follow(follow string) *DataCentersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DataCentersServiceListRequest) Max(max int64) *DataCentersServiceListRequest {
	p.max = &max
	return p
}

func (p *DataCentersServiceListRequest) Search(search string) *DataCentersServiceListRequest {
	p.search = &search
	return p
}

func (p *DataCentersServiceListRequest) Send() (*DataCentersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCentersService.connection.URL(), p.DataCentersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCentersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCentersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCentersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCentersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCentersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDataCenterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DataCentersServiceListResponse{dataCenters: result}, nil
}

func (p *DataCentersServiceListRequest) MustSend() *DataCentersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the data centers.
// The following request retrieves a representation of the data centers:
// [source]
// ----
// GET /ovirt-engine/api/datacenters
// ----
// The above request performed with `curl`:
// [source,bash]
// ----
// curl \
// --request GET \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --header "Version: 4" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// https://myengine.example.com/ovirt-engine/api/datacenters
// ----
// This is what an example response could look like:
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//     </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
// </data_center>
// ----
// Note the `id` code of your `Default` data center. This code identifies this data center in relation to other
// resources of your virtual environment.
// The data center also contains a link to the storage domains collection. The data center uses this collection to
// attach storage domains from the storage domains main collection.
// The order of the returned list of data centers is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DataCentersServiceListResponse struct {
	dataCenters *DataCenterSlice
}

func (p *DataCentersServiceListResponse) DataCenters() (*DataCenterSlice, bool) {
	if p.dataCenters != nil {
		return p.dataCenters, true
	}
	return nil, false
}

func (p *DataCentersServiceListResponse) MustDataCenters() *DataCenterSlice {
	if p.dataCenters == nil {
		panic("dataCenters in response does not exist")
	}
	return p.dataCenters
}

//
// Lists the data centers.
// The following request retrieves a representation of the data centers:
// [source]
// ----
// GET /ovirt-engine/api/datacenters
// ----
// The above request performed with `curl`:
// [source,bash]
// ----
// curl \
// --request GET \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --header "Version: 4" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// https://myengine.example.com/ovirt-engine/api/datacenters
// ----
// This is what an example response could look like:
// [source,xml]
// ----
// <data_center href="/ovirt-engine/api/datacenters/123" id="123">
//   <name>Default</name>
//   <description>The default Data Center</description>
//   <link href="/ovirt-engine/api/datacenters/123/networks" rel="networks"/>
//   <link href="/ovirt-engine/api/datacenters/123/storagedomains" rel="storagedomains"/>
//   <link href="/ovirt-engine/api/datacenters/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/datacenters/123/clusters" rel="clusters"/>
//   <link href="/ovirt-engine/api/datacenters/123/qoss" rel="qoss"/>
//   <link href="/ovirt-engine/api/datacenters/123/iscsibonds" rel="iscsibonds"/>
//   <link href="/ovirt-engine/api/datacenters/123/quotas" rel="quotas"/>
//   <local>false</local>
//   <quota_mode>disabled</quota_mode>
//   <status>up</status>
//   <supported_versions>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//     </version>
//   </supported_versions>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//   </version>
// </data_center>
// ----
// Note the `id` code of your `Default` data center. This code identifies this data center in relation to other
// resources of your virtual environment.
// The data center also contains a link to the storage domains collection. The data center uses this collection to
// attach storage domains from the storage domains main collection.
// The order of the returned list of data centers is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
func (p *DataCentersService) List() *DataCentersServiceListRequest {
	return &DataCentersServiceListRequest{DataCentersService: p}
}

//
// Reference to the service that manages a specific data center.
//
func (op *DataCentersService) DataCenterService(id string) *DataCenterService {
	return NewDataCenterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DataCentersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DataCenterService(path), nil
	}
	return op.DataCenterService(path[:index]).Service(path[index+1:])
}

func (op *DataCentersService) String() string {
	return fmt.Sprintf("DataCentersService:%s", op.path)
}

//
//
type PermissionService struct {
	BaseService
}

func NewPermissionService(connection *Connection, path string) *PermissionService {
	var result PermissionService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type PermissionServiceGetRequest struct {
	PermissionService *PermissionService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *PermissionServiceGetRequest) Header(key, value string) *PermissionServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *PermissionServiceGetRequest) Query(key, value string) *PermissionServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *PermissionServiceGetRequest) Follow(follow string) *PermissionServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *PermissionServiceGetRequest) Send() (*PermissionServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.PermissionService.connection.URL(), p.PermissionService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.PermissionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.PermissionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.PermissionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.PermissionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.PermissionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermissionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &PermissionServiceGetResponse{permission: result}, nil
}

func (p *PermissionServiceGetRequest) MustSend() *PermissionServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type PermissionServiceGetResponse struct {
	permission *Permission
}

func (p *PermissionServiceGetResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *PermissionServiceGetResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
//
func (p *PermissionService) Get() *PermissionServiceGetRequest {
	return &PermissionServiceGetRequest{PermissionService: p}
}

//
//
type PermissionServiceRemoveRequest struct {
	PermissionService *PermissionService
	header            map[string]string
	query             map[string]string
	async             *bool
}

func (p *PermissionServiceRemoveRequest) Header(key, value string) *PermissionServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *PermissionServiceRemoveRequest) Query(key, value string) *PermissionServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *PermissionServiceRemoveRequest) Async(async bool) *PermissionServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *PermissionServiceRemoveRequest) Send() (*PermissionServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.PermissionService.connection.URL(), p.PermissionService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.PermissionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.PermissionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.PermissionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.PermissionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.PermissionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(PermissionServiceRemoveResponse), nil
}

func (p *PermissionServiceRemoveRequest) MustSend() *PermissionServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type PermissionServiceRemoveResponse struct {
}

//
//
func (p *PermissionService) Remove() *PermissionServiceRemoveRequest {
	return &PermissionServiceRemoveRequest{PermissionService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *PermissionService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *PermissionService) String() string {
	return fmt.Sprintf("PermissionService:%s", op.path)
}

//
//
type StorageService struct {
	BaseService
}

func NewStorageService(connection *Connection, path string) *StorageService {
	var result StorageService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageServiceGetRequest struct {
	StorageService *StorageService
	header         map[string]string
	query          map[string]string
	follow         *string
	reportStatus   *bool
}

func (p *StorageServiceGetRequest) Header(key, value string) *StorageServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServiceGetRequest) Query(key, value string) *StorageServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServiceGetRequest) Follow(follow string) *StorageServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageServiceGetRequest) ReportStatus(reportStatus bool) *StorageServiceGetRequest {
	p.reportStatus = &reportStatus
	return p
}

func (p *StorageServiceGetRequest) Send() (*StorageServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageService.connection.URL(), p.StorageService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.reportStatus != nil {
		values["report_status"] = []string{fmt.Sprintf("%v", *p.reportStatus)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostStorageReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageServiceGetResponse{storage: result}, nil
}

func (p *StorageServiceGetRequest) MustSend() *StorageServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageServiceGetResponse struct {
	storage *HostStorage
}

func (p *StorageServiceGetResponse) Storage() (*HostStorage, bool) {
	if p.storage != nil {
		return p.storage, true
	}
	return nil, false
}

func (p *StorageServiceGetResponse) MustStorage() *HostStorage {
	if p.storage == nil {
		panic("storage in response does not exist")
	}
	return p.storage
}

//
//
func (p *StorageService) Get() *StorageServiceGetRequest {
	return &StorageServiceGetRequest{StorageService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageService) String() string {
	return fmt.Sprintf("StorageService:%s", op.path)
}

//
// A service to manage collection of assignment of tags to specific entities in system.
//
type AssignedTagsService struct {
	BaseService
}

func NewAssignedTagsService(connection *Connection, path string) *AssignedTagsService {
	var result AssignedTagsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Assign tag to specific entity in the system.
// For example to assign tag `mytag` to virtual machine with the id `123` send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
//
type AssignedTagsServiceAddRequest struct {
	AssignedTagsService *AssignedTagsService
	header              map[string]string
	query               map[string]string
	tag                 *Tag
}

func (p *AssignedTagsServiceAddRequest) Header(key, value string) *AssignedTagsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedTagsServiceAddRequest) Query(key, value string) *AssignedTagsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedTagsServiceAddRequest) Tag(tag *Tag) *AssignedTagsServiceAddRequest {
	p.tag = tag
	return p
}

func (p *AssignedTagsServiceAddRequest) Send() (*AssignedTagsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedTagsService.connection.URL(), p.AssignedTagsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLTagWriteOne(writer, p.tag, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedTagsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedTagsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedTagsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedTagsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedTagsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedTagsServiceAddResponse{tag: result}, nil
}

func (p *AssignedTagsServiceAddRequest) MustSend() *AssignedTagsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Assign tag to specific entity in the system.
// For example to assign tag `mytag` to virtual machine with the id `123` send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
//
type AssignedTagsServiceAddResponse struct {
	tag *Tag
}

func (p *AssignedTagsServiceAddResponse) Tag() (*Tag, bool) {
	if p.tag != nil {
		return p.tag, true
	}
	return nil, false
}

func (p *AssignedTagsServiceAddResponse) MustTag() *Tag {
	if p.tag == nil {
		panic("tag in response does not exist")
	}
	return p.tag
}

//
// Assign tag to specific entity in the system.
// For example to assign tag `mytag` to virtual machine with the id `123` send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
//
func (p *AssignedTagsService) Add() *AssignedTagsServiceAddRequest {
	return &AssignedTagsServiceAddRequest{AssignedTagsService: p}
}

//
// List all tags assigned to the specific entity.
// For example to list all the tags of the virtual machine with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>mytag</name>
//     <description>mytag</description>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </tag>
// </tags>
// ----
// The order of the returned tags isn't guaranteed.
//
type AssignedTagsServiceListRequest struct {
	AssignedTagsService *AssignedTagsService
	header              map[string]string
	query               map[string]string
	follow              *string
	max                 *int64
}

func (p *AssignedTagsServiceListRequest) Header(key, value string) *AssignedTagsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedTagsServiceListRequest) Query(key, value string) *AssignedTagsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedTagsServiceListRequest) Follow(follow string) *AssignedTagsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedTagsServiceListRequest) Max(max int64) *AssignedTagsServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedTagsServiceListRequest) Send() (*AssignedTagsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedTagsService.connection.URL(), p.AssignedTagsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedTagsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedTagsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedTagsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedTagsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedTagsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedTagsServiceListResponse{tags: result}, nil
}

func (p *AssignedTagsServiceListRequest) MustSend() *AssignedTagsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all tags assigned to the specific entity.
// For example to list all the tags of the virtual machine with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>mytag</name>
//     <description>mytag</description>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </tag>
// </tags>
// ----
// The order of the returned tags isn't guaranteed.
//
type AssignedTagsServiceListResponse struct {
	tags *TagSlice
}

func (p *AssignedTagsServiceListResponse) Tags() (*TagSlice, bool) {
	if p.tags != nil {
		return p.tags, true
	}
	return nil, false
}

func (p *AssignedTagsServiceListResponse) MustTags() *TagSlice {
	if p.tags == nil {
		panic("tags in response does not exist")
	}
	return p.tags
}

//
// List all tags assigned to the specific entity.
// For example to list all the tags of the virtual machine with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/vms/123/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>mytag</name>
//     <description>mytag</description>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </tag>
// </tags>
// ----
// The order of the returned tags isn't guaranteed.
//
func (p *AssignedTagsService) List() *AssignedTagsServiceListRequest {
	return &AssignedTagsServiceListRequest{AssignedTagsService: p}
}

//
// Reference to the service that manages assignment of specific tag.
//
func (op *AssignedTagsService) TagService(id string) *AssignedTagService {
	return NewAssignedTagService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedTagsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.TagService(path), nil
	}
	return op.TagService(path[:index]).Service(path[index+1:])
}

func (op *AssignedTagsService) String() string {
	return fmt.Sprintf("AssignedTagsService:%s", op.path)
}

//
// A service to fetch information elements received by Link Layer Discovery Protocol (LLDP).
//
type LinkLayerDiscoveryProtocolService struct {
	BaseService
}

func NewLinkLayerDiscoveryProtocolService(connection *Connection, path string) *LinkLayerDiscoveryProtocolService {
	var result LinkLayerDiscoveryProtocolService
	result.connection = connection
	result.path = path
	return &result
}

//
// Fetches information elements received by LLDP.
//
type LinkLayerDiscoveryProtocolServiceListRequest struct {
	LinkLayerDiscoveryProtocolService *LinkLayerDiscoveryProtocolService
	header                            map[string]string
	query                             map[string]string
	follow                            *string
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Header(key, value string) *LinkLayerDiscoveryProtocolServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Query(key, value string) *LinkLayerDiscoveryProtocolServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Follow(follow string) *LinkLayerDiscoveryProtocolServiceListRequest {
	p.follow = &follow
	return p
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Send() (*LinkLayerDiscoveryProtocolServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.LinkLayerDiscoveryProtocolService.connection.URL(), p.LinkLayerDiscoveryProtocolService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.LinkLayerDiscoveryProtocolService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.LinkLayerDiscoveryProtocolService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.LinkLayerDiscoveryProtocolService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.LinkLayerDiscoveryProtocolService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.LinkLayerDiscoveryProtocolService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLLinkLayerDiscoveryProtocolElementReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &LinkLayerDiscoveryProtocolServiceListResponse{elements: result}, nil
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) MustSend() *LinkLayerDiscoveryProtocolServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Fetches information elements received by LLDP.
//
type LinkLayerDiscoveryProtocolServiceListResponse struct {
	elements *LinkLayerDiscoveryProtocolElementSlice
}

func (p *LinkLayerDiscoveryProtocolServiceListResponse) Elements() (*LinkLayerDiscoveryProtocolElementSlice, bool) {
	if p.elements != nil {
		return p.elements, true
	}
	return nil, false
}

func (p *LinkLayerDiscoveryProtocolServiceListResponse) MustElements() *LinkLayerDiscoveryProtocolElementSlice {
	if p.elements == nil {
		panic("elements in response does not exist")
	}
	return p.elements
}

//
// Fetches information elements received by LLDP.
//
func (p *LinkLayerDiscoveryProtocolService) List() *LinkLayerDiscoveryProtocolServiceListRequest {
	return &LinkLayerDiscoveryProtocolServiceListRequest{LinkLayerDiscoveryProtocolService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *LinkLayerDiscoveryProtocolService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *LinkLayerDiscoveryProtocolService) String() string {
	return fmt.Sprintf("LinkLayerDiscoveryProtocolService:%s", op.path)
}

//
// This service doesn't add any new methods, it is just a placeholder for the annotation that specifies the path of the
// resource that manages the permissions assigned to the system object.
//
type SystemPermissionsService struct {
	BaseService
}

func NewSystemPermissionsService(connection *Connection, path string) *SystemPermissionsService {
	var result SystemPermissionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
type SystemPermissionsServiceAddRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddRequest) Header(key, value string) *SystemPermissionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddRequest) Query(key, value string) *SystemPermissionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddRequest) Permission(permission *Permission) *SystemPermissionsServiceAddRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddRequest) Send() (*SystemPermissionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLPermissionWriteOne(writer, p.permission, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermissionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SystemPermissionsServiceAddResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddRequest) MustSend() *SystemPermissionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
type SystemPermissionsServiceAddResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
func (p *SystemPermissionsService) Add() *SystemPermissionsServiceAddRequest {
	return &SystemPermissionsServiceAddRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the cluster to the group in the system.
//
type SystemPermissionsServiceAddClusterPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddClusterPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddClusterPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddClusterPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Send() (*SystemPermissionsServiceAddClusterPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/clusterpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddClusterPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) MustSend() *SystemPermissionsServiceAddClusterPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the cluster to the group in the system.
//
type SystemPermissionsServiceAddClusterPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddClusterPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddClusterPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the cluster to the group in the system.
//
func (p *SystemPermissionsService) AddClusterPermission() *SystemPermissionsServiceAddClusterPermissionRequest {
	return &SystemPermissionsServiceAddClusterPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the data center to the group in the system.
//
type SystemPermissionsServiceAddDataCenterPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddDataCenterPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddDataCenterPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddDataCenterPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Send() (*SystemPermissionsServiceAddDataCenterPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/datacenterpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddDataCenterPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) MustSend() *SystemPermissionsServiceAddDataCenterPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the data center to the group in the system.
//
type SystemPermissionsServiceAddDataCenterPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddDataCenterPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddDataCenterPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the data center to the group in the system.
//
func (p *SystemPermissionsService) AddDataCenterPermission() *SystemPermissionsServiceAddDataCenterPermissionRequest {
	return &SystemPermissionsServiceAddDataCenterPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new group level permission for a given virtual machine.
//
type SystemPermissionsServiceAddGroupLevelRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Header(key, value string) *SystemPermissionsServiceAddGroupLevelRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Query(key, value string) *SystemPermissionsServiceAddGroupLevelRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Permission(permission *Permission) *SystemPermissionsServiceAddGroupLevelRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Send() (*SystemPermissionsServiceAddGroupLevelResponse, error) {
	rawURL := fmt.Sprintf("%s%s/grouplevel", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddGroupLevelResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) MustSend() *SystemPermissionsServiceAddGroupLevelResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new group level permission for a given virtual machine.
//
type SystemPermissionsServiceAddGroupLevelResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddGroupLevelResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddGroupLevelResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new group level permission for a given virtual machine.
//
func (p *SystemPermissionsService) AddGroupLevel() *SystemPermissionsServiceAddGroupLevelRequest {
	return &SystemPermissionsServiceAddGroupLevelRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the host to the group in the system.
//
type SystemPermissionsServiceAddHostPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddHostPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddHostPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddHostPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Send() (*SystemPermissionsServiceAddHostPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/hostpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddHostPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) MustSend() *SystemPermissionsServiceAddHostPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the host to the group in the system.
//
type SystemPermissionsServiceAddHostPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddHostPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddHostPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the host to the group in the system.
//
func (p *SystemPermissionsService) AddHostPermission() *SystemPermissionsServiceAddHostPermissionRequest {
	return &SystemPermissionsServiceAddHostPermissionRequest{SystemPermissionsService: p}
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
type SystemPermissionsServiceListRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *SystemPermissionsServiceListRequest) Header(key, value string) *SystemPermissionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceListRequest) Query(key, value string) *SystemPermissionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceListRequest) Follow(follow string) *SystemPermissionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SystemPermissionsServiceListRequest) Send() (*SystemPermissionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermissionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SystemPermissionsServiceListResponse{permissions: result}, nil
}

func (p *SystemPermissionsServiceListRequest) MustSend() *SystemPermissionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
type SystemPermissionsServiceListResponse struct {
	permissions *PermissionSlice
}

func (p *SystemPermissionsServiceListResponse) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceListResponse) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("permissions in response does not exist")
	}
	return p.permissions
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
func (p *SystemPermissionsService) List() *SystemPermissionsServiceListRequest {
	return &SystemPermissionsServiceListRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the storage domain to the group in the system.
//
type SystemPermissionsServiceAddStorageDomainPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Send() (*SystemPermissionsServiceAddStorageDomainPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/storagedomainpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddStorageDomainPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) MustSend() *SystemPermissionsServiceAddStorageDomainPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the storage domain to the group in the system.
//
type SystemPermissionsServiceAddStorageDomainPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the storage domain to the group in the system.
//
func (p *SystemPermissionsService) AddStorageDomainPermission() *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	return &SystemPermissionsServiceAddStorageDomainPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the template to the group in the system.
//
type SystemPermissionsServiceAddTemplatePermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Header(key, value string) *SystemPermissionsServiceAddTemplatePermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Query(key, value string) *SystemPermissionsServiceAddTemplatePermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddTemplatePermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Send() (*SystemPermissionsServiceAddTemplatePermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/templatepermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddTemplatePermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) MustSend() *SystemPermissionsServiceAddTemplatePermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the template to the group in the system.
//
type SystemPermissionsServiceAddTemplatePermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddTemplatePermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddTemplatePermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the template to the group in the system.
//
func (p *SystemPermissionsService) AddTemplatePermission() *SystemPermissionsServiceAddTemplatePermissionRequest {
	return &SystemPermissionsServiceAddTemplatePermissionRequest{SystemPermissionsService: p}
}

//
// Add a new user level permission for a given virtual machine.
//
type SystemPermissionsServiceAddUserLevelRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Header(key, value string) *SystemPermissionsServiceAddUserLevelRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Query(key, value string) *SystemPermissionsServiceAddUserLevelRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Permission(permission *Permission) *SystemPermissionsServiceAddUserLevelRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Send() (*SystemPermissionsServiceAddUserLevelResponse, error) {
	rawURL := fmt.Sprintf("%s%s/userlevel", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddUserLevelResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddUserLevelRequest) MustSend() *SystemPermissionsServiceAddUserLevelResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new user level permission for a given virtual machine.
//
type SystemPermissionsServiceAddUserLevelResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddUserLevelResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddUserLevelResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new user level permission for a given virtual machine.
//
func (p *SystemPermissionsService) AddUserLevel() *SystemPermissionsServiceAddUserLevelRequest {
	return &SystemPermissionsServiceAddUserLevelRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the vm to the group in the system.
//
type SystemPermissionsServiceAddVmPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddVmPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddVmPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddVmPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Send() (*SystemPermissionsServiceAddVmPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vmpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddVmPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) MustSend() *SystemPermissionsServiceAddVmPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the vm to the group in the system.
//
type SystemPermissionsServiceAddVmPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddVmPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddVmPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the vm to the group in the system.
//
func (p *SystemPermissionsService) AddVmPermission() *SystemPermissionsServiceAddVmPermissionRequest {
	return &SystemPermissionsServiceAddVmPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the vm pool to the group in the system.
//
type SystemPermissionsServiceAddVmPoolPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddVmPoolPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddVmPoolPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddVmPoolPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Send() (*SystemPermissionsServiceAddVmPoolPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vmpoolpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddVmPoolPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) MustSend() *SystemPermissionsServiceAddVmPoolPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the vm pool to the group in the system.
//
type SystemPermissionsServiceAddVmPoolPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddVmPoolPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddVmPoolPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the vm pool to the group in the system.
//
func (p *SystemPermissionsService) AddVmPoolPermission() *SystemPermissionsServiceAddVmPoolPermissionRequest {
	return &SystemPermissionsServiceAddVmPoolPermissionRequest{SystemPermissionsService: p}
}

//
// Sub-resource locator method, returns individual permission resource on which the remainder of the URI is
// dispatched.
//
func (op *SystemPermissionsService) PermissionService(id string) *PermissionService {
	return NewPermissionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SystemPermissionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.PermissionService(path), nil
	}
	return op.PermissionService(path[:index]).Service(path[index+1:])
}

func (op *SystemPermissionsService) String() string {
	return fmt.Sprintf("SystemPermissionsService:%s", op.path)
}

//
// Manages the set of scheduling policies available in the system.
//
type SchedulingPoliciesService struct {
	BaseService
}

func NewSchedulingPoliciesService(connection *Connection, path string) *SchedulingPoliciesService {
	var result SchedulingPoliciesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new scheduling policy to the system.
//
type SchedulingPoliciesServiceAddRequest struct {
	SchedulingPoliciesService *SchedulingPoliciesService
	header                    map[string]string
	query                     map[string]string
	policy                    *SchedulingPolicy
}

func (p *SchedulingPoliciesServiceAddRequest) Header(key, value string) *SchedulingPoliciesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPoliciesServiceAddRequest) Query(key, value string) *SchedulingPoliciesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPoliciesServiceAddRequest) Policy(policy *SchedulingPolicy) *SchedulingPoliciesServiceAddRequest {
	p.policy = policy
	return p
}

func (p *SchedulingPoliciesServiceAddRequest) Send() (*SchedulingPoliciesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPoliciesService.connection.URL(), p.SchedulingPoliciesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSchedulingPolicyWriteOne(writer, p.policy, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPoliciesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPoliciesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPoliciesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPoliciesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPoliciesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SchedulingPoliciesServiceAddResponse{policy: result}, nil
}

func (p *SchedulingPoliciesServiceAddRequest) MustSend() *SchedulingPoliciesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new scheduling policy to the system.
//
type SchedulingPoliciesServiceAddResponse struct {
	policy *SchedulingPolicy
}

func (p *SchedulingPoliciesServiceAddResponse) Policy() (*SchedulingPolicy, bool) {
	if p.policy != nil {
		return p.policy, true
	}
	return nil, false
}

func (p *SchedulingPoliciesServiceAddResponse) MustPolicy() *SchedulingPolicy {
	if p.policy == nil {
		panic("policy in response does not exist")
	}
	return p.policy
}

//
// Add a new scheduling policy to the system.
//
func (p *SchedulingPoliciesService) Add() *SchedulingPoliciesServiceAddRequest {
	return &SchedulingPoliciesServiceAddRequest{SchedulingPoliciesService: p}
}

//
// Returns the list of scheduling policies available in the system.
// The order of the returned list of scheduling policies isn't guaranteed.
//
type SchedulingPoliciesServiceListRequest struct {
	SchedulingPoliciesService *SchedulingPoliciesService
	header                    map[string]string
	query                     map[string]string
	filter                    *bool
	follow                    *string
	max                       *int64
}

func (p *SchedulingPoliciesServiceListRequest) Header(key, value string) *SchedulingPoliciesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Query(key, value string) *SchedulingPoliciesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Filter(filter bool) *SchedulingPoliciesServiceListRequest {
	p.filter = &filter
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Follow(follow string) *SchedulingPoliciesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Max(max int64) *SchedulingPoliciesServiceListRequest {
	p.max = &max
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Send() (*SchedulingPoliciesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPoliciesService.connection.URL(), p.SchedulingPoliciesService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPoliciesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPoliciesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPoliciesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPoliciesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPoliciesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SchedulingPoliciesServiceListResponse{policies: result}, nil
}

func (p *SchedulingPoliciesServiceListRequest) MustSend() *SchedulingPoliciesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of scheduling policies available in the system.
// The order of the returned list of scheduling policies isn't guaranteed.
//
type SchedulingPoliciesServiceListResponse struct {
	policies *SchedulingPolicySlice
}

func (p *SchedulingPoliciesServiceListResponse) Policies() (*SchedulingPolicySlice, bool) {
	if p.policies != nil {
		return p.policies, true
	}
	return nil, false
}

func (p *SchedulingPoliciesServiceListResponse) MustPolicies() *SchedulingPolicySlice {
	if p.policies == nil {
		panic("policies in response does not exist")
	}
	return p.policies
}

//
// Returns the list of scheduling policies available in the system.
// The order of the returned list of scheduling policies isn't guaranteed.
//
func (p *SchedulingPoliciesService) List() *SchedulingPoliciesServiceListRequest {
	return &SchedulingPoliciesServiceListRequest{SchedulingPoliciesService: p}
}

//
//
func (op *SchedulingPoliciesService) PolicyService(id string) *SchedulingPolicyService {
	return NewSchedulingPolicyService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SchedulingPoliciesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.PolicyService(path), nil
	}
	return op.PolicyService(path[:index]).Service(path[index+1:])
}

func (op *SchedulingPoliciesService) String() string {
	return fmt.Sprintf("SchedulingPoliciesService:%s", op.path)
}

//
// A service to manage a specific cluster network.
//
type ClusterNetworkService struct {
	BaseService
}

func NewClusterNetworkService(connection *Connection, path string) *ClusterNetworkService {
	var result ClusterNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the cluster network details.
//
type ClusterNetworkServiceGetRequest struct {
	ClusterNetworkService *ClusterNetworkService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *ClusterNetworkServiceGetRequest) Header(key, value string) *ClusterNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworkServiceGetRequest) Query(key, value string) *ClusterNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworkServiceGetRequest) Follow(follow string) *ClusterNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ClusterNetworkServiceGetRequest) Send() (*ClusterNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworkService.connection.URL(), p.ClusterNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterNetworkServiceGetResponse{network: result}, nil
}

func (p *ClusterNetworkServiceGetRequest) MustSend() *ClusterNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the cluster network details.
//
type ClusterNetworkServiceGetResponse struct {
	network *Network
}

func (p *ClusterNetworkServiceGetResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *ClusterNetworkServiceGetResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Retrieves the cluster network details.
//
func (p *ClusterNetworkService) Get() *ClusterNetworkServiceGetRequest {
	return &ClusterNetworkServiceGetRequest{ClusterNetworkService: p}
}

//
// Unassigns the network from a cluster.
//
type ClusterNetworkServiceRemoveRequest struct {
	ClusterNetworkService *ClusterNetworkService
	header                map[string]string
	query                 map[string]string
}

func (p *ClusterNetworkServiceRemoveRequest) Header(key, value string) *ClusterNetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworkServiceRemoveRequest) Query(key, value string) *ClusterNetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworkServiceRemoveRequest) Send() (*ClusterNetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworkService.connection.URL(), p.ClusterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(ClusterNetworkServiceRemoveResponse), nil
}

func (p *ClusterNetworkServiceRemoveRequest) MustSend() *ClusterNetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Unassigns the network from a cluster.
//
type ClusterNetworkServiceRemoveResponse struct {
}

//
// Unassigns the network from a cluster.
//
func (p *ClusterNetworkService) Remove() *ClusterNetworkServiceRemoveRequest {
	return &ClusterNetworkServiceRemoveRequest{ClusterNetworkService: p}
}

//
// Updates the network in the cluster.
//
type ClusterNetworkServiceUpdateRequest struct {
	ClusterNetworkService *ClusterNetworkService
	header                map[string]string
	query                 map[string]string
	network               *Network
}

func (p *ClusterNetworkServiceUpdateRequest) Header(key, value string) *ClusterNetworkServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterNetworkServiceUpdateRequest) Query(key, value string) *ClusterNetworkServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterNetworkServiceUpdateRequest) Network(network *Network) *ClusterNetworkServiceUpdateRequest {
	p.network = network
	return p
}

func (p *ClusterNetworkServiceUpdateRequest) Send() (*ClusterNetworkServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterNetworkService.connection.URL(), p.ClusterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClusterNetworkServiceUpdateResponse{network: result}, nil
}

func (p *ClusterNetworkServiceUpdateRequest) MustSend() *ClusterNetworkServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the network in the cluster.
//
type ClusterNetworkServiceUpdateResponse struct {
	network *Network
}

func (p *ClusterNetworkServiceUpdateResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *ClusterNetworkServiceUpdateResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Updates the network in the cluster.
//
func (p *ClusterNetworkService) Update() *ClusterNetworkServiceUpdateRequest {
	return &ClusterNetworkServiceUpdateRequest{ClusterNetworkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ClusterNetworkService) String() string {
	return fmt.Sprintf("ClusterNetworkService:%s", op.path)
}

//
// A service managing a CD-ROM device on templates.
//
type TemplateCdromService struct {
	BaseService
}

func NewTemplateCdromService(connection *Connection, path string) *TemplateCdromService {
	var result TemplateCdromService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the information about this CD-ROM device.
// For example, to get information about the CD-ROM device of template `123` send a request like:
// [source]
// ----
// GET /ovirt-engine/api/templates/123/cdroms/
// ----
//
type TemplateCdromServiceGetRequest struct {
	TemplateCdromService *TemplateCdromService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *TemplateCdromServiceGetRequest) Header(key, value string) *TemplateCdromServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateCdromServiceGetRequest) Query(key, value string) *TemplateCdromServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateCdromServiceGetRequest) Follow(follow string) *TemplateCdromServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *TemplateCdromServiceGetRequest) Send() (*TemplateCdromServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateCdromService.connection.URL(), p.TemplateCdromService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateCdromService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateCdromService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateCdromService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateCdromService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateCdromService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCdromReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TemplateCdromServiceGetResponse{cdrom: result}, nil
}

func (p *TemplateCdromServiceGetRequest) MustSend() *TemplateCdromServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the information about this CD-ROM device.
// For example, to get information about the CD-ROM device of template `123` send a request like:
// [source]
// ----
// GET /ovirt-engine/api/templates/123/cdroms/
// ----
//
type TemplateCdromServiceGetResponse struct {
	cdrom *Cdrom
}

func (p *TemplateCdromServiceGetResponse) Cdrom() (*Cdrom, bool) {
	if p.cdrom != nil {
		return p.cdrom, true
	}
	return nil, false
}

func (p *TemplateCdromServiceGetResponse) MustCdrom() *Cdrom {
	if p.cdrom == nil {
		panic("cdrom in response does not exist")
	}
	return p.cdrom
}

//
// Returns the information about this CD-ROM device.
// For example, to get information about the CD-ROM device of template `123` send a request like:
// [source]
// ----
// GET /ovirt-engine/api/templates/123/cdroms/
// ----
//
func (p *TemplateCdromService) Get() *TemplateCdromServiceGetRequest {
	return &TemplateCdromServiceGetRequest{TemplateCdromService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *TemplateCdromService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *TemplateCdromService) String() string {
	return fmt.Sprintf("TemplateCdromService:%s", op.path)
}

//
// Manages the set of network attachments of a host or host NIC.
//
type NetworkAttachmentsService struct {
	BaseService
}

func NewNetworkAttachmentsService(connection *Connection, path string) *NetworkAttachmentsService {
	var result NetworkAttachmentsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new network attachment to the network interface.
//
type NetworkAttachmentsServiceAddRequest struct {
	NetworkAttachmentsService *NetworkAttachmentsService
	header                    map[string]string
	query                     map[string]string
	attachment                *NetworkAttachment
}

func (p *NetworkAttachmentsServiceAddRequest) Header(key, value string) *NetworkAttachmentsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkAttachmentsServiceAddRequest) Query(key, value string) *NetworkAttachmentsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkAttachmentsServiceAddRequest) Attachment(attachment *NetworkAttachment) *NetworkAttachmentsServiceAddRequest {
	p.attachment = attachment
	return p
}

func (p *NetworkAttachmentsServiceAddRequest) Send() (*NetworkAttachmentsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkAttachmentsService.connection.URL(), p.NetworkAttachmentsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkAttachmentWriteOne(writer, p.attachment, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkAttachmentsServiceAddResponse{attachment: result}, nil
}

func (p *NetworkAttachmentsServiceAddRequest) MustSend() *NetworkAttachmentsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new network attachment to the network interface.
//
type NetworkAttachmentsServiceAddResponse struct {
	attachment *NetworkAttachment
}

func (p *NetworkAttachmentsServiceAddResponse) Attachment() (*NetworkAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *NetworkAttachmentsServiceAddResponse) MustAttachment() *NetworkAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
// Add a new network attachment to the network interface.
//
func (p *NetworkAttachmentsService) Add() *NetworkAttachmentsServiceAddRequest {
	return &NetworkAttachmentsServiceAddRequest{NetworkAttachmentsService: p}
}

//
// Returns the list of network attachments of the host or host NIC.
// The order of the returned list of network attachments isn't guaranteed.
//
type NetworkAttachmentsServiceListRequest struct {
	NetworkAttachmentsService *NetworkAttachmentsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *NetworkAttachmentsServiceListRequest) Header(key, value string) *NetworkAttachmentsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkAttachmentsServiceListRequest) Query(key, value string) *NetworkAttachmentsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkAttachmentsServiceListRequest) Follow(follow string) *NetworkAttachmentsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *NetworkAttachmentsServiceListRequest) Max(max int64) *NetworkAttachmentsServiceListRequest {
	p.max = &max
	return p
}

func (p *NetworkAttachmentsServiceListRequest) Send() (*NetworkAttachmentsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkAttachmentsService.connection.URL(), p.NetworkAttachmentsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkAttachmentReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &NetworkAttachmentsServiceListResponse{attachments: result}, nil
}

func (p *NetworkAttachmentsServiceListRequest) MustSend() *NetworkAttachmentsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of network attachments of the host or host NIC.
// The order of the returned list of network attachments isn't guaranteed.
//
type NetworkAttachmentsServiceListResponse struct {
	attachments *NetworkAttachmentSlice
}

func (p *NetworkAttachmentsServiceListResponse) Attachments() (*NetworkAttachmentSlice, bool) {
	if p.attachments != nil {
		return p.attachments, true
	}
	return nil, false
}

func (p *NetworkAttachmentsServiceListResponse) MustAttachments() *NetworkAttachmentSlice {
	if p.attachments == nil {
		panic("attachments in response does not exist")
	}
	return p.attachments
}

//
// Returns the list of network attachments of the host or host NIC.
// The order of the returned list of network attachments isn't guaranteed.
//
func (p *NetworkAttachmentsService) List() *NetworkAttachmentsServiceListRequest {
	return &NetworkAttachmentsServiceListRequest{NetworkAttachmentsService: p}
}

//
//
func (op *NetworkAttachmentsService) AttachmentService(id string) *NetworkAttachmentService {
	return NewNetworkAttachmentService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkAttachmentsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.AttachmentService(path), nil
	}
	return op.AttachmentService(path[:index]).Service(path[index+1:])
}

func (op *NetworkAttachmentsService) String() string {
	return fmt.Sprintf("NetworkAttachmentsService:%s", op.path)
}

//
//
type UnmanagedNetworkService struct {
	BaseService
}

func NewUnmanagedNetworkService(connection *Connection, path string) *UnmanagedNetworkService {
	var result UnmanagedNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type UnmanagedNetworkServiceGetRequest struct {
	UnmanagedNetworkService *UnmanagedNetworkService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *UnmanagedNetworkServiceGetRequest) Header(key, value string) *UnmanagedNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UnmanagedNetworkServiceGetRequest) Query(key, value string) *UnmanagedNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UnmanagedNetworkServiceGetRequest) Follow(follow string) *UnmanagedNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *UnmanagedNetworkServiceGetRequest) Send() (*UnmanagedNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UnmanagedNetworkService.connection.URL(), p.UnmanagedNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UnmanagedNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UnmanagedNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UnmanagedNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UnmanagedNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UnmanagedNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUnmanagedNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &UnmanagedNetworkServiceGetResponse{network: result}, nil
}

func (p *UnmanagedNetworkServiceGetRequest) MustSend() *UnmanagedNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type UnmanagedNetworkServiceGetResponse struct {
	network *UnmanagedNetwork
}

func (p *UnmanagedNetworkServiceGetResponse) Network() (*UnmanagedNetwork, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *UnmanagedNetworkServiceGetResponse) MustNetwork() *UnmanagedNetwork {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
//
func (p *UnmanagedNetworkService) Get() *UnmanagedNetworkServiceGetRequest {
	return &UnmanagedNetworkServiceGetRequest{UnmanagedNetworkService: p}
}

//
//
type UnmanagedNetworkServiceRemoveRequest struct {
	UnmanagedNetworkService *UnmanagedNetworkService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
}

func (p *UnmanagedNetworkServiceRemoveRequest) Header(key, value string) *UnmanagedNetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UnmanagedNetworkServiceRemoveRequest) Query(key, value string) *UnmanagedNetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UnmanagedNetworkServiceRemoveRequest) Async(async bool) *UnmanagedNetworkServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *UnmanagedNetworkServiceRemoveRequest) Send() (*UnmanagedNetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UnmanagedNetworkService.connection.URL(), p.UnmanagedNetworkService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UnmanagedNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UnmanagedNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UnmanagedNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UnmanagedNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UnmanagedNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(UnmanagedNetworkServiceRemoveResponse), nil
}

func (p *UnmanagedNetworkServiceRemoveRequest) MustSend() *UnmanagedNetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type UnmanagedNetworkServiceRemoveResponse struct {
}

//
//
func (p *UnmanagedNetworkService) Remove() *UnmanagedNetworkServiceRemoveRequest {
	return &UnmanagedNetworkServiceRemoveRequest{UnmanagedNetworkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *UnmanagedNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *UnmanagedNetworkService) String() string {
	return fmt.Sprintf("UnmanagedNetworkService:%s", op.path)
}

//
// Manages the set of NICs of an snapshot.
//
type SnapshotNicsService struct {
	BaseService
}

func NewSnapshotNicsService(connection *Connection, path string) *SnapshotNicsService {
	var result SnapshotNicsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of NICs of the snapshot.
// The order of the returned list of NICs isn't guaranteed.
//
type SnapshotNicsServiceListRequest struct {
	SnapshotNicsService *SnapshotNicsService
	header              map[string]string
	query               map[string]string
	follow              *string
	max                 *int64
}

func (p *SnapshotNicsServiceListRequest) Header(key, value string) *SnapshotNicsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotNicsServiceListRequest) Query(key, value string) *SnapshotNicsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotNicsServiceListRequest) Follow(follow string) *SnapshotNicsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SnapshotNicsServiceListRequest) Max(max int64) *SnapshotNicsServiceListRequest {
	p.max = &max
	return p
}

func (p *SnapshotNicsServiceListRequest) Send() (*SnapshotNicsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotNicsService.connection.URL(), p.SnapshotNicsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SnapshotNicsServiceListResponse{nics: result}, nil
}

func (p *SnapshotNicsServiceListRequest) MustSend() *SnapshotNicsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of NICs of the snapshot.
// The order of the returned list of NICs isn't guaranteed.
//
type SnapshotNicsServiceListResponse struct {
	nics *NicSlice
}

func (p *SnapshotNicsServiceListResponse) Nics() (*NicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *SnapshotNicsServiceListResponse) MustNics() *NicSlice {
	if p.nics == nil {
		panic("nics in response does not exist")
	}
	return p.nics
}

//
// Returns the list of NICs of the snapshot.
// The order of the returned list of NICs isn't guaranteed.
//
func (p *SnapshotNicsService) List() *SnapshotNicsServiceListRequest {
	return &SnapshotNicsServiceListRequest{SnapshotNicsService: p}
}

//
//
func (op *SnapshotNicsService) NicService(id string) *SnapshotNicService {
	return NewSnapshotNicService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SnapshotNicsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NicService(path), nil
	}
	return op.NicService(path[:index]).Service(path[index+1:])
}

func (op *SnapshotNicsService) String() string {
	return fmt.Sprintf("SnapshotNicsService:%s", op.path)
}

//
// A service to manage host storages.
//
type HostStorageService struct {
	BaseService
}

func NewHostStorageService(connection *Connection, path string) *HostStorageService {
	var result HostStorageService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get list of storages.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/storage
// ----
// The XML response you get will be like this one:
// [source,xml]
// ----
// <host_storages>
//   <host_storage id="123">
//     ...
//   </host_storage>
//   ...
// </host_storages>
// ----
// The order of the returned list of storages isn't guaranteed.
//
type HostStorageServiceListRequest struct {
	HostStorageService *HostStorageService
	header             map[string]string
	query              map[string]string
	follow             *string
	reportStatus       *bool
}

func (p *HostStorageServiceListRequest) Header(key, value string) *HostStorageServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostStorageServiceListRequest) Query(key, value string) *HostStorageServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostStorageServiceListRequest) Follow(follow string) *HostStorageServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostStorageServiceListRequest) ReportStatus(reportStatus bool) *HostStorageServiceListRequest {
	p.reportStatus = &reportStatus
	return p
}

func (p *HostStorageServiceListRequest) Send() (*HostStorageServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostStorageService.connection.URL(), p.HostStorageService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.reportStatus != nil {
		values["report_status"] = []string{fmt.Sprintf("%v", *p.reportStatus)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostStorageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostStorageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostStorageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostStorageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostStorageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostStorageReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostStorageServiceListResponse{storages: result}, nil
}

func (p *HostStorageServiceListRequest) MustSend() *HostStorageServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of storages.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/storage
// ----
// The XML response you get will be like this one:
// [source,xml]
// ----
// <host_storages>
//   <host_storage id="123">
//     ...
//   </host_storage>
//   ...
// </host_storages>
// ----
// The order of the returned list of storages isn't guaranteed.
//
type HostStorageServiceListResponse struct {
	storages *HostStorageSlice
}

func (p *HostStorageServiceListResponse) Storages() (*HostStorageSlice, bool) {
	if p.storages != nil {
		return p.storages, true
	}
	return nil, false
}

func (p *HostStorageServiceListResponse) MustStorages() *HostStorageSlice {
	if p.storages == nil {
		panic("storages in response does not exist")
	}
	return p.storages
}

//
// Get list of storages.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/storage
// ----
// The XML response you get will be like this one:
// [source,xml]
// ----
// <host_storages>
//   <host_storage id="123">
//     ...
//   </host_storage>
//   ...
// </host_storages>
// ----
// The order of the returned list of storages isn't guaranteed.
//
func (p *HostStorageService) List() *HostStorageServiceListRequest {
	return &HostStorageServiceListRequest{HostStorageService: p}
}

//
// Reference to a service managing the storage.
//
func (op *HostStorageService) StorageService(id string) *StorageService {
	return NewStorageService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostStorageService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.StorageService(path), nil
	}
	return op.StorageService(path[:index]).Service(path[index+1:])
}

func (op *HostStorageService) String() string {
	return fmt.Sprintf("HostStorageService:%s", op.path)
}

//
// Represents a service to manage collection of the tags in the system.
//
type TagsService struct {
	BaseService
}

func NewTagsService(connection *Connection, path string) *TagsService {
	var result TagsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new tag to the system.
// For example, to add new tag with name `mytag` to the system send a request like this:
// ....
// POST /ovirt-engine/api/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
// NOTE: The root tag is a special pseudo-tag assumed as the default parent tag if no parent tag is specified.
// The root tag cannot be deleted nor assigned a parent tag.
// To create new tag with specific parent tag send a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
//   <parent>
//     <name>myparenttag</name>
//   </parent>
// </tag>
// ----
//
type TagsServiceAddRequest struct {
	TagsService *TagsService
	header      map[string]string
	query       map[string]string
	tag         *Tag
}

func (p *TagsServiceAddRequest) Header(key, value string) *TagsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TagsServiceAddRequest) Query(key, value string) *TagsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TagsServiceAddRequest) Tag(tag *Tag) *TagsServiceAddRequest {
	p.tag = tag
	return p
}

func (p *TagsServiceAddRequest) Send() (*TagsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TagsService.connection.URL(), p.TagsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLTagWriteOne(writer, p.tag, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TagsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TagsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TagsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TagsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TagsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TagsServiceAddResponse{tag: result}, nil
}

func (p *TagsServiceAddRequest) MustSend() *TagsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new tag to the system.
// For example, to add new tag with name `mytag` to the system send a request like this:
// ....
// POST /ovirt-engine/api/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
// NOTE: The root tag is a special pseudo-tag assumed as the default parent tag if no parent tag is specified.
// The root tag cannot be deleted nor assigned a parent tag.
// To create new tag with specific parent tag send a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
//   <parent>
//     <name>myparenttag</name>
//   </parent>
// </tag>
// ----
//
type TagsServiceAddResponse struct {
	tag *Tag
}

func (p *TagsServiceAddResponse) Tag() (*Tag, bool) {
	if p.tag != nil {
		return p.tag, true
	}
	return nil, false
}

func (p *TagsServiceAddResponse) MustTag() *Tag {
	if p.tag == nil {
		panic("tag in response does not exist")
	}
	return p.tag
}

//
// Add a new tag to the system.
// For example, to add new tag with name `mytag` to the system send a request like this:
// ....
// POST /ovirt-engine/api/tags
// ....
// With a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
// </tag>
// ----
// NOTE: The root tag is a special pseudo-tag assumed as the default parent tag if no parent tag is specified.
// The root tag cannot be deleted nor assigned a parent tag.
// To create new tag with specific parent tag send a request body like this:
// [source,xml]
// ----
// <tag>
//   <name>mytag</name>
//   <parent>
//     <name>myparenttag</name>
//   </parent>
// </tag>
// ----
//
func (p *TagsService) Add() *TagsServiceAddRequest {
	return &TagsServiceAddRequest{TagsService: p}
}

//
// List the tags in the system.
// For example to list the full hierarchy of the tags in the system send a request like this:
// ....
// GET /ovirt-engine/api/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>root2</name>
//     <description>root2</description>
//     <parent href="/ovirt-engine/api/tags/111" id="111"/>
//   </tag>
//   <tag href="/ovirt-engine/api/tags/333" id="333">
//     <name>root3</name>
//     <description>root3</description>
//     <parent href="/ovirt-engine/api/tags/222" id="222"/>
//   </tag>
//   <tag href="/ovirt-engine/api/tags/111" id="111">
//     <name>root</name>
//     <description>root</description>
//   </tag>
// </tags>
// ----
// In the previous XML output you can see the following hierarchy of the tags:
// ....
// root:        (id: 111)
//   - root2    (id: 222)
//     - root3  (id: 333)
// ....
// The order of the returned list of tags isn't guaranteed.
//
type TagsServiceListRequest struct {
	TagsService *TagsService
	header      map[string]string
	query       map[string]string
	follow      *string
	max         *int64
}

func (p *TagsServiceListRequest) Header(key, value string) *TagsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TagsServiceListRequest) Query(key, value string) *TagsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TagsServiceListRequest) Follow(follow string) *TagsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *TagsServiceListRequest) Max(max int64) *TagsServiceListRequest {
	p.max = &max
	return p
}

func (p *TagsServiceListRequest) Send() (*TagsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TagsService.connection.URL(), p.TagsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TagsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TagsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TagsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TagsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TagsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &TagsServiceListResponse{tags: result}, nil
}

func (p *TagsServiceListRequest) MustSend() *TagsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the tags in the system.
// For example to list the full hierarchy of the tags in the system send a request like this:
// ....
// GET /ovirt-engine/api/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>root2</name>
//     <description>root2</description>
//     <parent href="/ovirt-engine/api/tags/111" id="111"/>
//   </tag>
//   <tag href="/ovirt-engine/api/tags/333" id="333">
//     <name>root3</name>
//     <description>root3</description>
//     <parent href="/ovirt-engine/api/tags/222" id="222"/>
//   </tag>
//   <tag href="/ovirt-engine/api/tags/111" id="111">
//     <name>root</name>
//     <description>root</description>
//   </tag>
// </tags>
// ----
// In the previous XML output you can see the following hierarchy of the tags:
// ....
// root:        (id: 111)
//   - root2    (id: 222)
//     - root3  (id: 333)
// ....
// The order of the returned list of tags isn't guaranteed.
//
type TagsServiceListResponse struct {
	tags *TagSlice
}

func (p *TagsServiceListResponse) Tags() (*TagSlice, bool) {
	if p.tags != nil {
		return p.tags, true
	}
	return nil, false
}

func (p *TagsServiceListResponse) MustTags() *TagSlice {
	if p.tags == nil {
		panic("tags in response does not exist")
	}
	return p.tags
}

//
// List the tags in the system.
// For example to list the full hierarchy of the tags in the system send a request like this:
// ....
// GET /ovirt-engine/api/tags
// ....
// [source,xml]
// ----
// <tags>
//   <tag href="/ovirt-engine/api/tags/222" id="222">
//     <name>root2</name>
//     <description>root2</description>
//     <parent href="/ovirt-engine/api/tags/111" id="111"/>
//   </tag>
//   <tag href="/ovirt-engine/api/tags/333" id="333">
//     <name>root3</name>
//     <description>root3</description>
//     <parent href="/ovirt-engine/api/tags/222" id="222"/>
//   </tag>
//   <tag href="/ovirt-engine/api/tags/111" id="111">
//     <name>root</name>
//     <description>root</description>
//   </tag>
// </tags>
// ----
// In the previous XML output you can see the following hierarchy of the tags:
// ....
// root:        (id: 111)
//   - root2    (id: 222)
//     - root3  (id: 333)
// ....
// The order of the returned list of tags isn't guaranteed.
//
func (p *TagsService) List() *TagsServiceListRequest {
	return &TagsServiceListRequest{TagsService: p}
}

//
// Reference to the service that manages a specific tag.
//
func (op *TagsService) TagService(id string) *TagService {
	return NewTagService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *TagsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.TagService(path), nil
	}
	return op.TagService(path[:index]).Service(path[index+1:])
}

func (op *TagsService) String() string {
	return fmt.Sprintf("TagsService:%s", op.path)
}

//
//
type VmBackupDisksService struct {
	BaseService
}

func NewVmBackupDisksService(connection *Connection, path string) *VmBackupDisksService {
	var result VmBackupDisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of disks in backup.
//
type VmBackupDisksServiceListRequest struct {
	VmBackupDisksService *VmBackupDisksService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *VmBackupDisksServiceListRequest) Header(key, value string) *VmBackupDisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmBackupDisksServiceListRequest) Query(key, value string) *VmBackupDisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmBackupDisksServiceListRequest) Follow(follow string) *VmBackupDisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmBackupDisksServiceListRequest) Max(max int64) *VmBackupDisksServiceListRequest {
	p.max = &max
	return p
}

func (p *VmBackupDisksServiceListRequest) Send() (*VmBackupDisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmBackupDisksService.connection.URL(), p.VmBackupDisksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmBackupDisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmBackupDisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmBackupDisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmBackupDisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmBackupDisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmBackupDisksServiceListResponse{disks: result}, nil
}

func (p *VmBackupDisksServiceListRequest) MustSend() *VmBackupDisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of disks in backup.
//
type VmBackupDisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *VmBackupDisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *VmBackupDisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Returns the list of disks in backup.
//
func (p *VmBackupDisksService) List() *VmBackupDisksServiceListRequest {
	return &VmBackupDisksServiceListRequest{VmBackupDisksService: p}
}

//
// A reference to the service that manages a specific disk.
//
func (op *VmBackupDisksService) DiskService(id string) *VmBackupDiskService {
	return NewVmBackupDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmBackupDisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *VmBackupDisksService) String() string {
	return fmt.Sprintf("VmBackupDisksService:%s", op.path)
}

//
// A service that manages hosts.
//
type HostsService struct {
	BaseService
}

func NewHostsService(connection *Connection, path string) *HostsService {
	var result HostsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new host.
// The host is created based on the attributes of the `host` parameter. The `name`, `address`, and `root_password`
// properties are required.
// For example, to add a host, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/hosts
// ----
// With the following request body:
// [source,xml]
// ----
// <host>
//   <name>myhost</name>
//   <address>myhost.example.com</address>
//   <root_password>myrootpassword</root_password>
// </host>
// ----
// NOTE: The `root_password` element is only included in the client-provided initial representation and is not
// exposed in the representations returned from subsequent requests.
// IMPORTANT: Since version 4.1.2 of the engine, when a host is newly added, the host's firewall
// definitions are overridden by default.
// To add a hosted engine host, use the optional `deploy_hosted_engine` parameter:
// [source]
// ----
// POST /ovirt-engine/api/hosts?deploy_hosted_engine=true
// ----
// If the cluster has a default external network provider that is supported for automatic deployment,
// the external network provider is deployed when adding the host.
// Only external network providers for OVN are supported for the automatic deployment.
// To deploy an external network provider other than the one defined in the clusters, overwrite the external
// network provider when adding hosts, by sending the following request:
// [source]
// ----
// POST /ovirt-engine/api/hosts
// ----
// With a request body that contains a reference to the desired provider in the
// `external_network_provider_configuration`:
// [source,xml]
// ----
// <host>
//   <name>myhost</name>
//   <address>myhost.example.com</address>
//   <root_password>123456</root_password>
//   <external_network_provider_configurations>
//     <external_network_provider_configuration>
//       <external_network_provider name="ovirt-provider-ovn"/>
//     </external_network_provider_configuration>
//   </external_network_provider_configurations>
// </host>
// ----
//
type HostsServiceAddRequest struct {
	HostsService         *HostsService
	header               map[string]string
	query                map[string]string
	activate             *bool
	deployHostedEngine   *bool
	host                 *Host
	undeployHostedEngine *bool
}

func (p *HostsServiceAddRequest) Header(key, value string) *HostsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostsServiceAddRequest) Query(key, value string) *HostsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostsServiceAddRequest) Activate(activate bool) *HostsServiceAddRequest {
	p.activate = &activate
	return p
}

func (p *HostsServiceAddRequest) DeployHostedEngine(deployHostedEngine bool) *HostsServiceAddRequest {
	p.deployHostedEngine = &deployHostedEngine
	return p
}

func (p *HostsServiceAddRequest) Host(host *Host) *HostsServiceAddRequest {
	p.host = host
	return p
}

func (p *HostsServiceAddRequest) UndeployHostedEngine(undeployHostedEngine bool) *HostsServiceAddRequest {
	p.undeployHostedEngine = &undeployHostedEngine
	return p
}

func (p *HostsServiceAddRequest) Send() (*HostsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostsService.connection.URL(), p.HostsService.path)
	values := make(url.Values)
	if p.activate != nil {
		values["activate"] = []string{fmt.Sprintf("%v", *p.activate)}
	}

	if p.deployHostedEngine != nil {
		values["deploy_hosted_engine"] = []string{fmt.Sprintf("%v", *p.deployHostedEngine)}
	}

	if p.undeployHostedEngine != nil {
		values["undeploy_hosted_engine"] = []string{fmt.Sprintf("%v", *p.undeployHostedEngine)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLHostWriteOne(writer, p.host, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostsServiceAddResponse{host: result}, nil
}

func (p *HostsServiceAddRequest) MustSend() *HostsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new host.
// The host is created based on the attributes of the `host` parameter. The `name`, `address`, and `root_password`
// properties are required.
// For example, to add a host, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/hosts
// ----
// With the following request body:
// [source,xml]
// ----
// <host>
//   <name>myhost</name>
//   <address>myhost.example.com</address>
//   <root_password>myrootpassword</root_password>
// </host>
// ----
// NOTE: The `root_password` element is only included in the client-provided initial representation and is not
// exposed in the representations returned from subsequent requests.
// IMPORTANT: Since version 4.1.2 of the engine, when a host is newly added, the host's firewall
// definitions are overridden by default.
// To add a hosted engine host, use the optional `deploy_hosted_engine` parameter:
// [source]
// ----
// POST /ovirt-engine/api/hosts?deploy_hosted_engine=true
// ----
// If the cluster has a default external network provider that is supported for automatic deployment,
// the external network provider is deployed when adding the host.
// Only external network providers for OVN are supported for the automatic deployment.
// To deploy an external network provider other than the one defined in the clusters, overwrite the external
// network provider when adding hosts, by sending the following request:
// [source]
// ----
// POST /ovirt-engine/api/hosts
// ----
// With a request body that contains a reference to the desired provider in the
// `external_network_provider_configuration`:
// [source,xml]
// ----
// <host>
//   <name>myhost</name>
//   <address>myhost.example.com</address>
//   <root_password>123456</root_password>
//   <external_network_provider_configurations>
//     <external_network_provider_configuration>
//       <external_network_provider name="ovirt-provider-ovn"/>
//     </external_network_provider_configuration>
//   </external_network_provider_configurations>
// </host>
// ----
//
type HostsServiceAddResponse struct {
	host *Host
}

func (p *HostsServiceAddResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *HostsServiceAddResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Creates a new host.
// The host is created based on the attributes of the `host` parameter. The `name`, `address`, and `root_password`
// properties are required.
// For example, to add a host, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/hosts
// ----
// With the following request body:
// [source,xml]
// ----
// <host>
//   <name>myhost</name>
//   <address>myhost.example.com</address>
//   <root_password>myrootpassword</root_password>
// </host>
// ----
// NOTE: The `root_password` element is only included in the client-provided initial representation and is not
// exposed in the representations returned from subsequent requests.
// IMPORTANT: Since version 4.1.2 of the engine, when a host is newly added, the host's firewall
// definitions are overridden by default.
// To add a hosted engine host, use the optional `deploy_hosted_engine` parameter:
// [source]
// ----
// POST /ovirt-engine/api/hosts?deploy_hosted_engine=true
// ----
// If the cluster has a default external network provider that is supported for automatic deployment,
// the external network provider is deployed when adding the host.
// Only external network providers for OVN are supported for the automatic deployment.
// To deploy an external network provider other than the one defined in the clusters, overwrite the external
// network provider when adding hosts, by sending the following request:
// [source]
// ----
// POST /ovirt-engine/api/hosts
// ----
// With a request body that contains a reference to the desired provider in the
// `external_network_provider_configuration`:
// [source,xml]
// ----
// <host>
//   <name>myhost</name>
//   <address>myhost.example.com</address>
//   <root_password>123456</root_password>
//   <external_network_provider_configurations>
//     <external_network_provider_configuration>
//       <external_network_provider name="ovirt-provider-ovn"/>
//     </external_network_provider_configuration>
//   </external_network_provider_configurations>
// </host>
// ----
//
func (p *HostsService) Add() *HostsServiceAddRequest {
	return &HostsServiceAddRequest{HostsService: p}
}

//
// Get a list of all available hosts.
// For example, to list the hosts send the following request:
// ....
// GET /ovirt-engine/api/hosts
// ....
// The response body will be similar to this:
// [source,xml]
// ----
// <hosts>
//   <host href="/ovirt-engine/api/hosts/123" id="123">
//     ...
//   </host>
//   <host href="/ovirt-engine/api/hosts/456" id="456">
//     ...
//   </host>
//   ...
// </host>
// ----
// The order of the returned list of hosts is guaranteed only if the `sortby` clause is included in
// the `search` parameter.
//
type HostsServiceListRequest struct {
	HostsService      *HostsService
	header            map[string]string
	query             map[string]string
	allContent        *bool
	caseSensitive     *bool
	filter            *bool
	follow            *string
	max               *int64
	migrationTargetOf *string
	search            *string
}

func (p *HostsServiceListRequest) Header(key, value string) *HostsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostsServiceListRequest) Query(key, value string) *HostsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostsServiceListRequest) AllContent(allContent bool) *HostsServiceListRequest {
	p.allContent = &allContent
	return p
}

func (p *HostsServiceListRequest) CaseSensitive(caseSensitive bool) *HostsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *HostsServiceListRequest) Filter(filter bool) *HostsServiceListRequest {
	p.filter = &filter
	return p
}

func (p *HostsServiceListRequest) Follow(follow string) *HostsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostsServiceListRequest) Max(max int64) *HostsServiceListRequest {
	p.max = &max
	return p
}

func (p *HostsServiceListRequest) MigrationTargetOf(migrationTargetOf string) *HostsServiceListRequest {
	p.migrationTargetOf = &migrationTargetOf
	return p
}

func (p *HostsServiceListRequest) Search(search string) *HostsServiceListRequest {
	p.search = &search
	return p
}

func (p *HostsServiceListRequest) Send() (*HostsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostsService.connection.URL(), p.HostsService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.migrationTargetOf != nil {
		values["migration_target_of"] = []string{fmt.Sprintf("%v", *p.migrationTargetOf)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostsServiceListResponse{hosts: result}, nil
}

func (p *HostsServiceListRequest) MustSend() *HostsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get a list of all available hosts.
// For example, to list the hosts send the following request:
// ....
// GET /ovirt-engine/api/hosts
// ....
// The response body will be similar to this:
// [source,xml]
// ----
// <hosts>
//   <host href="/ovirt-engine/api/hosts/123" id="123">
//     ...
//   </host>
//   <host href="/ovirt-engine/api/hosts/456" id="456">
//     ...
//   </host>
//   ...
// </host>
// ----
// The order of the returned list of hosts is guaranteed only if the `sortby` clause is included in
// the `search` parameter.
//
type HostsServiceListResponse struct {
	hosts *HostSlice
}

func (p *HostsServiceListResponse) Hosts() (*HostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *HostsServiceListResponse) MustHosts() *HostSlice {
	if p.hosts == nil {
		panic("hosts in response does not exist")
	}
	return p.hosts
}

//
// Get a list of all available hosts.
// For example, to list the hosts send the following request:
// ....
// GET /ovirt-engine/api/hosts
// ....
// The response body will be similar to this:
// [source,xml]
// ----
// <hosts>
//   <host href="/ovirt-engine/api/hosts/123" id="123">
//     ...
//   </host>
//   <host href="/ovirt-engine/api/hosts/456" id="456">
//     ...
//   </host>
//   ...
// </host>
// ----
// The order of the returned list of hosts is guaranteed only if the `sortby` clause is included in
// the `search` parameter.
//
func (p *HostsService) List() *HostsServiceListRequest {
	return &HostsServiceListRequest{HostsService: p}
}

//
// Add a new host to the system providing the host root password. This has been deprecated and provided for backwards compatibility.
//
type HostsServiceAddUsingRootPasswordRequest struct {
	HostsService         *HostsService
	header               map[string]string
	query                map[string]string
	activate             *bool
	deployHostedEngine   *bool
	host                 *Host
	undeployHostedEngine *bool
}

func (p *HostsServiceAddUsingRootPasswordRequest) Header(key, value string) *HostsServiceAddUsingRootPasswordRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostsServiceAddUsingRootPasswordRequest) Query(key, value string) *HostsServiceAddUsingRootPasswordRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostsServiceAddUsingRootPasswordRequest) Activate(activate bool) *HostsServiceAddUsingRootPasswordRequest {
	p.activate = &activate
	return p
}

func (p *HostsServiceAddUsingRootPasswordRequest) DeployHostedEngine(deployHostedEngine bool) *HostsServiceAddUsingRootPasswordRequest {
	p.deployHostedEngine = &deployHostedEngine
	return p
}

func (p *HostsServiceAddUsingRootPasswordRequest) Host(host *Host) *HostsServiceAddUsingRootPasswordRequest {
	p.host = host
	return p
}

func (p *HostsServiceAddUsingRootPasswordRequest) UndeployHostedEngine(undeployHostedEngine bool) *HostsServiceAddUsingRootPasswordRequest {
	p.undeployHostedEngine = &undeployHostedEngine
	return p
}

func (p *HostsServiceAddUsingRootPasswordRequest) Send() (*HostsServiceAddUsingRootPasswordResponse, error) {
	rawURL := fmt.Sprintf("%s%s/usingrootpassword", p.HostsService.connection.URL(), p.HostsService.path)
	actionBuilder := NewActionBuilder()
	if p.activate != nil {
		actionBuilder.Activate(*p.activate)
	}
	if p.deployHostedEngine != nil {
		actionBuilder.DeployHostedEngine(*p.deployHostedEngine)
	}
	actionBuilder.Host(p.host)
	if p.undeployHostedEngine != nil {
		actionBuilder.UndeployHostedEngine(*p.undeployHostedEngine)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustHost()
	return &HostsServiceAddUsingRootPasswordResponse{host: result}, nil
}

func (p *HostsServiceAddUsingRootPasswordRequest) MustSend() *HostsServiceAddUsingRootPasswordResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new host to the system providing the host root password. This has been deprecated and provided for backwards compatibility.
//
type HostsServiceAddUsingRootPasswordResponse struct {
	host *Host
}

func (p *HostsServiceAddUsingRootPasswordResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *HostsServiceAddUsingRootPasswordResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Add a new host to the system providing the host root password. This has been deprecated and provided for backwards compatibility.
//
func (p *HostsService) AddUsingRootPassword() *HostsServiceAddUsingRootPasswordRequest {
	return &HostsServiceAddUsingRootPasswordRequest{HostsService: p}
}

//
// Add a new host to the system providing the ssh password or fingerprint.
//
type HostsServiceAddUsingSshRequest struct {
	HostsService         *HostsService
	header               map[string]string
	query                map[string]string
	activate             *bool
	deployHostedEngine   *bool
	host                 *Host
	undeployHostedEngine *bool
}

func (p *HostsServiceAddUsingSshRequest) Header(key, value string) *HostsServiceAddUsingSshRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostsServiceAddUsingSshRequest) Query(key, value string) *HostsServiceAddUsingSshRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostsServiceAddUsingSshRequest) Activate(activate bool) *HostsServiceAddUsingSshRequest {
	p.activate = &activate
	return p
}

func (p *HostsServiceAddUsingSshRequest) DeployHostedEngine(deployHostedEngine bool) *HostsServiceAddUsingSshRequest {
	p.deployHostedEngine = &deployHostedEngine
	return p
}

func (p *HostsServiceAddUsingSshRequest) Host(host *Host) *HostsServiceAddUsingSshRequest {
	p.host = host
	return p
}

func (p *HostsServiceAddUsingSshRequest) UndeployHostedEngine(undeployHostedEngine bool) *HostsServiceAddUsingSshRequest {
	p.undeployHostedEngine = &undeployHostedEngine
	return p
}

func (p *HostsServiceAddUsingSshRequest) Send() (*HostsServiceAddUsingSshResponse, error) {
	rawURL := fmt.Sprintf("%s%s/usingssh", p.HostsService.connection.URL(), p.HostsService.path)
	actionBuilder := NewActionBuilder()
	if p.activate != nil {
		actionBuilder.Activate(*p.activate)
	}
	if p.deployHostedEngine != nil {
		actionBuilder.DeployHostedEngine(*p.deployHostedEngine)
	}
	actionBuilder.Host(p.host)
	if p.undeployHostedEngine != nil {
		actionBuilder.UndeployHostedEngine(*p.undeployHostedEngine)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustHost()
	return &HostsServiceAddUsingSshResponse{host: result}, nil
}

func (p *HostsServiceAddUsingSshRequest) MustSend() *HostsServiceAddUsingSshResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new host to the system providing the ssh password or fingerprint.
//
type HostsServiceAddUsingSshResponse struct {
	host *Host
}

func (p *HostsServiceAddUsingSshResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *HostsServiceAddUsingSshResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Add a new host to the system providing the ssh password or fingerprint.
//
func (p *HostsService) AddUsingSsh() *HostsServiceAddUsingSshRequest {
	return &HostsServiceAddUsingSshRequest{HostsService: p}
}

//
// A Reference to service managing a specific host.
//
func (op *HostsService) HostService(id string) *HostService {
	return NewHostService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HostService(path), nil
	}
	return op.HostService(path[:index]).Service(path[index+1:])
}

func (op *HostsService) String() string {
	return fmt.Sprintf("HostsService:%s", op.path)
}

//
//
type BalancesService struct {
	BaseService
}

func NewBalancesService(connection *Connection, path string) *BalancesService {
	var result BalancesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a balance module to a specified user defined scheduling policy.
//
type BalancesServiceAddRequest struct {
	BalancesService *BalancesService
	header          map[string]string
	query           map[string]string
	balance         *Balance
}

func (p *BalancesServiceAddRequest) Header(key, value string) *BalancesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalancesServiceAddRequest) Query(key, value string) *BalancesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalancesServiceAddRequest) Balance(balance *Balance) *BalancesServiceAddRequest {
	p.balance = balance
	return p
}

func (p *BalancesServiceAddRequest) Send() (*BalancesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalancesService.connection.URL(), p.BalancesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLBalanceWriteOne(writer, p.balance, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalancesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalancesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalancesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalancesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalancesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BalancesServiceAddResponse{balance: result}, nil
}

func (p *BalancesServiceAddRequest) MustSend() *BalancesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a balance module to a specified user defined scheduling policy.
//
type BalancesServiceAddResponse struct {
	balance *Balance
}

func (p *BalancesServiceAddResponse) Balance() (*Balance, bool) {
	if p.balance != nil {
		return p.balance, true
	}
	return nil, false
}

func (p *BalancesServiceAddResponse) MustBalance() *Balance {
	if p.balance == nil {
		panic("balance in response does not exist")
	}
	return p.balance
}

//
// Add a balance module to a specified user defined scheduling policy.
//
func (p *BalancesService) Add() *BalancesServiceAddRequest {
	return &BalancesServiceAddRequest{BalancesService: p}
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
type BalancesServiceListRequest struct {
	BalancesService *BalancesService
	header          map[string]string
	query           map[string]string
	filter          *bool
	follow          *string
	max             *int64
}

func (p *BalancesServiceListRequest) Header(key, value string) *BalancesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalancesServiceListRequest) Query(key, value string) *BalancesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalancesServiceListRequest) Filter(filter bool) *BalancesServiceListRequest {
	p.filter = &filter
	return p
}

func (p *BalancesServiceListRequest) Follow(follow string) *BalancesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *BalancesServiceListRequest) Max(max int64) *BalancesServiceListRequest {
	p.max = &max
	return p
}

func (p *BalancesServiceListRequest) Send() (*BalancesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalancesService.connection.URL(), p.BalancesService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalancesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalancesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalancesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalancesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalancesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &BalancesServiceListResponse{balances: result}, nil
}

func (p *BalancesServiceListRequest) MustSend() *BalancesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
type BalancesServiceListResponse struct {
	balances *BalanceSlice
}

func (p *BalancesServiceListResponse) Balances() (*BalanceSlice, bool) {
	if p.balances != nil {
		return p.balances, true
	}
	return nil, false
}

func (p *BalancesServiceListResponse) MustBalances() *BalanceSlice {
	if p.balances == nil {
		panic("balances in response does not exist")
	}
	return p.balances
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
func (p *BalancesService) List() *BalancesServiceListRequest {
	return &BalancesServiceListRequest{BalancesService: p}
}

//
//
func (op *BalancesService) BalanceService(id string) *BalanceService {
	return NewBalanceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *BalancesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.BalanceService(path), nil
	}
	return op.BalanceService(path[:index]).Service(path[index+1:])
}

func (op *BalancesService) String() string {
	return fmt.Sprintf("BalancesService:%s", op.path)
}

//
//
type DiskSnapshotService struct {
	BaseService
}

func NewDiskSnapshotService(connection *Connection, path string) *DiskSnapshotService {
	var result DiskSnapshotService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type DiskSnapshotServiceGetRequest struct {
	DiskSnapshotService *DiskSnapshotService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *DiskSnapshotServiceGetRequest) Header(key, value string) *DiskSnapshotServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskSnapshotServiceGetRequest) Query(key, value string) *DiskSnapshotServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskSnapshotServiceGetRequest) Follow(follow string) *DiskSnapshotServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DiskSnapshotServiceGetRequest) Send() (*DiskSnapshotServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskSnapshotService.connection.URL(), p.DiskSnapshotService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskSnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskSnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskSnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskSnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskSnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskSnapshotReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskSnapshotServiceGetResponse{snapshot: result}, nil
}

func (p *DiskSnapshotServiceGetRequest) MustSend() *DiskSnapshotServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskSnapshotServiceGetResponse struct {
	snapshot *DiskSnapshot
}

func (p *DiskSnapshotServiceGetResponse) Snapshot() (*DiskSnapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

func (p *DiskSnapshotServiceGetResponse) MustSnapshot() *DiskSnapshot {
	if p.snapshot == nil {
		panic("snapshot in response does not exist")
	}
	return p.snapshot
}

//
//
func (p *DiskSnapshotService) Get() *DiskSnapshotServiceGetRequest {
	return &DiskSnapshotServiceGetRequest{DiskSnapshotService: p}
}

//
//
type DiskSnapshotServiceRemoveRequest struct {
	DiskSnapshotService *DiskSnapshotService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *DiskSnapshotServiceRemoveRequest) Header(key, value string) *DiskSnapshotServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Query(key, value string) *DiskSnapshotServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Async(async bool) *DiskSnapshotServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Send() (*DiskSnapshotServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskSnapshotService.connection.URL(), p.DiskSnapshotService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskSnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskSnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskSnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskSnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskSnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DiskSnapshotServiceRemoveResponse), nil
}

func (p *DiskSnapshotServiceRemoveRequest) MustSend() *DiskSnapshotServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskSnapshotServiceRemoveResponse struct {
}

//
//
func (p *DiskSnapshotService) Remove() *DiskSnapshotServiceRemoveRequest {
	return &DiskSnapshotServiceRemoveRequest{DiskSnapshotService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskSnapshotService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DiskSnapshotService) String() string {
	return fmt.Sprintf("DiskSnapshotService:%s", op.path)
}

//
//
type StorageDomainServerConnectionService struct {
	BaseService
}

func NewStorageDomainServerConnectionService(connection *Connection, path string) *StorageDomainServerConnectionService {
	var result StorageDomainServerConnectionService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageDomainServerConnectionServiceGetRequest struct {
	StorageDomainServerConnectionService *StorageDomainServerConnectionService
	header                               map[string]string
	query                                map[string]string
	follow                               *string
}

func (p *StorageDomainServerConnectionServiceGetRequest) Header(key, value string) *StorageDomainServerConnectionServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Query(key, value string) *StorageDomainServerConnectionServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Follow(follow string) *StorageDomainServerConnectionServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Send() (*StorageDomainServerConnectionServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainServerConnectionService.connection.URL(), p.StorageDomainServerConnectionService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainServerConnectionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainServerConnectionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainServerConnectionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainServerConnectionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainServerConnectionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainServerConnectionServiceGetResponse{connection: result}, nil
}

func (p *StorageDomainServerConnectionServiceGetRequest) MustSend() *StorageDomainServerConnectionServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageDomainServerConnectionServiceGetResponse struct {
	connection *StorageConnection
}

func (p *StorageDomainServerConnectionServiceGetResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageDomainServerConnectionServiceGetResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
//
func (p *StorageDomainServerConnectionService) Get() *StorageDomainServerConnectionServiceGetRequest {
	return &StorageDomainServerConnectionServiceGetRequest{StorageDomainServerConnectionService: p}
}

//
// Detaches a storage connection from storage.
//
type StorageDomainServerConnectionServiceRemoveRequest struct {
	StorageDomainServerConnectionService *StorageDomainServerConnectionService
	header                               map[string]string
	query                                map[string]string
	async                                *bool
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Header(key, value string) *StorageDomainServerConnectionServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Query(key, value string) *StorageDomainServerConnectionServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Async(async bool) *StorageDomainServerConnectionServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Send() (*StorageDomainServerConnectionServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainServerConnectionService.connection.URL(), p.StorageDomainServerConnectionService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainServerConnectionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainServerConnectionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainServerConnectionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainServerConnectionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainServerConnectionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(StorageDomainServerConnectionServiceRemoveResponse), nil
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) MustSend() *StorageDomainServerConnectionServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Detaches a storage connection from storage.
//
type StorageDomainServerConnectionServiceRemoveResponse struct {
}

//
// Detaches a storage connection from storage.
//
func (p *StorageDomainServerConnectionService) Remove() *StorageDomainServerConnectionServiceRemoveRequest {
	return &StorageDomainServerConnectionServiceRemoveRequest{StorageDomainServerConnectionService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainServerConnectionService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageDomainServerConnectionService) String() string {
	return fmt.Sprintf("StorageDomainServerConnectionService:%s", op.path)
}

//
//
type OpenstackSubnetsService struct {
	BaseService
}

func NewOpenstackSubnetsService(connection *Connection, path string) *OpenstackSubnetsService {
	var result OpenstackSubnetsService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackSubnetsServiceAddRequest struct {
	OpenstackSubnetsService *OpenstackSubnetsService
	header                  map[string]string
	query                   map[string]string
	subnet                  *OpenStackSubnet
}

func (p *OpenstackSubnetsServiceAddRequest) Header(key, value string) *OpenstackSubnetsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetsServiceAddRequest) Query(key, value string) *OpenstackSubnetsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetsServiceAddRequest) Subnet(subnet *OpenStackSubnet) *OpenstackSubnetsServiceAddRequest {
	p.subnet = subnet
	return p
}

func (p *OpenstackSubnetsServiceAddRequest) Send() (*OpenstackSubnetsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetsService.connection.URL(), p.OpenstackSubnetsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackSubnetWriteOne(writer, p.subnet, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackSubnetReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackSubnetsServiceAddResponse{subnet: result}, nil
}

func (p *OpenstackSubnetsServiceAddRequest) MustSend() *OpenstackSubnetsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackSubnetsServiceAddResponse struct {
	subnet *OpenStackSubnet
}

func (p *OpenstackSubnetsServiceAddResponse) Subnet() (*OpenStackSubnet, bool) {
	if p.subnet != nil {
		return p.subnet, true
	}
	return nil, false
}

func (p *OpenstackSubnetsServiceAddResponse) MustSubnet() *OpenStackSubnet {
	if p.subnet == nil {
		panic("subnet in response does not exist")
	}
	return p.subnet
}

//
//
func (p *OpenstackSubnetsService) Add() *OpenstackSubnetsServiceAddRequest {
	return &OpenstackSubnetsServiceAddRequest{OpenstackSubnetsService: p}
}

//
// Returns the list of sub-networks.
// The order of the returned list of sub-networks isn't guaranteed.
//
type OpenstackSubnetsServiceListRequest struct {
	OpenstackSubnetsService *OpenstackSubnetsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
	max                     *int64
}

func (p *OpenstackSubnetsServiceListRequest) Header(key, value string) *OpenstackSubnetsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Query(key, value string) *OpenstackSubnetsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Follow(follow string) *OpenstackSubnetsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Max(max int64) *OpenstackSubnetsServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Send() (*OpenstackSubnetsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetsService.connection.URL(), p.OpenstackSubnetsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackSubnetReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackSubnetsServiceListResponse{subnets: result}, nil
}

func (p *OpenstackSubnetsServiceListRequest) MustSend() *OpenstackSubnetsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of sub-networks.
// The order of the returned list of sub-networks isn't guaranteed.
//
type OpenstackSubnetsServiceListResponse struct {
	subnets *OpenStackSubnetSlice
}

func (p *OpenstackSubnetsServiceListResponse) Subnets() (*OpenStackSubnetSlice, bool) {
	if p.subnets != nil {
		return p.subnets, true
	}
	return nil, false
}

func (p *OpenstackSubnetsServiceListResponse) MustSubnets() *OpenStackSubnetSlice {
	if p.subnets == nil {
		panic("subnets in response does not exist")
	}
	return p.subnets
}

//
// Returns the list of sub-networks.
// The order of the returned list of sub-networks isn't guaranteed.
//
func (p *OpenstackSubnetsService) List() *OpenstackSubnetsServiceListRequest {
	return &OpenstackSubnetsServiceListRequest{OpenstackSubnetsService: p}
}

//
//
func (op *OpenstackSubnetsService) SubnetService(id string) *OpenstackSubnetService {
	return NewOpenstackSubnetService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackSubnetsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.SubnetService(path), nil
	}
	return op.SubnetService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackSubnetsService) String() string {
	return fmt.Sprintf("OpenstackSubnetsService:%s", op.path)
}

//
//
type OpenstackVolumeTypeService struct {
	BaseService
}

func NewOpenstackVolumeTypeService(connection *Connection, path string) *OpenstackVolumeTypeService {
	var result OpenstackVolumeTypeService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackVolumeTypeServiceGetRequest struct {
	OpenstackVolumeTypeService *OpenstackVolumeTypeService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
}

func (p *OpenstackVolumeTypeServiceGetRequest) Header(key, value string) *OpenstackVolumeTypeServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeTypeServiceGetRequest) Query(key, value string) *OpenstackVolumeTypeServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeTypeServiceGetRequest) Follow(follow string) *OpenstackVolumeTypeServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeTypeServiceGetRequest) Send() (*OpenstackVolumeTypeServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeTypeService.connection.URL(), p.OpenstackVolumeTypeService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeTypeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeTypeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeTypeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeTypeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeTypeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeTypeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeTypeServiceGetResponse{type_: result}, nil
}

func (p *OpenstackVolumeTypeServiceGetRequest) MustSend() *OpenstackVolumeTypeServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeTypeServiceGetResponse struct {
	type_ *OpenStackVolumeType
}

func (p *OpenstackVolumeTypeServiceGetResponse) Type() (*OpenStackVolumeType, bool) {
	if p.type_ != nil {
		return p.type_, true
	}
	return nil, false
}

func (p *OpenstackVolumeTypeServiceGetResponse) MustType() *OpenStackVolumeType {
	if p.type_ == nil {
		panic("type_ in response does not exist")
	}
	return p.type_
}

//
//
func (p *OpenstackVolumeTypeService) Get() *OpenstackVolumeTypeServiceGetRequest {
	return &OpenstackVolumeTypeServiceGetRequest{OpenstackVolumeTypeService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeTypeService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackVolumeTypeService) String() string {
	return fmt.Sprintf("OpenstackVolumeTypeService:%s", op.path)
}

//
//
type OpenstackVolumeAuthenticationKeyService struct {
	BaseService
}

func NewOpenstackVolumeAuthenticationKeyService(connection *Connection, path string) *OpenstackVolumeAuthenticationKeyService {
	var result OpenstackVolumeAuthenticationKeyService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackVolumeAuthenticationKeyServiceGetRequest struct {
	OpenstackVolumeAuthenticationKeyService *OpenstackVolumeAuthenticationKeyService
	header                                  map[string]string
	query                                   map[string]string
	follow                                  *string
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Follow(follow string) *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Send() (*OpenstackVolumeAuthenticationKeyServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeyService.connection.URL(), p.OpenstackVolumeAuthenticationKeyService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeyServiceGetResponse{key: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) MustSend() *OpenstackVolumeAuthenticationKeyServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeAuthenticationKeyServiceGetResponse struct {
	key *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetResponse) Key() (*OpenstackVolumeAuthenticationKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetResponse) MustKey() *OpenstackVolumeAuthenticationKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *OpenstackVolumeAuthenticationKeyService) Get() *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	return &OpenstackVolumeAuthenticationKeyServiceGetRequest{OpenstackVolumeAuthenticationKeyService: p}
}

//
//
type OpenstackVolumeAuthenticationKeyServiceRemoveRequest struct {
	OpenstackVolumeAuthenticationKeyService *OpenstackVolumeAuthenticationKeyService
	header                                  map[string]string
	query                                   map[string]string
	async                                   *bool
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Async(async bool) *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Send() (*OpenstackVolumeAuthenticationKeyServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeyService.connection.URL(), p.OpenstackVolumeAuthenticationKeyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackVolumeAuthenticationKeyServiceRemoveResponse), nil
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) MustSend() *OpenstackVolumeAuthenticationKeyServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeAuthenticationKeyServiceRemoveResponse struct {
}

//
//
func (p *OpenstackVolumeAuthenticationKeyService) Remove() *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	return &OpenstackVolumeAuthenticationKeyServiceRemoveRequest{OpenstackVolumeAuthenticationKeyService: p}
}

//
// Update the specified authentication key.
//
type OpenstackVolumeAuthenticationKeyServiceUpdateRequest struct {
	OpenstackVolumeAuthenticationKeyService *OpenstackVolumeAuthenticationKeyService
	header                                  map[string]string
	query                                   map[string]string
	key                                     *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Key(key *OpenstackVolumeAuthenticationKey) *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	p.key = key
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Send() (*OpenstackVolumeAuthenticationKeyServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeyService.connection.URL(), p.OpenstackVolumeAuthenticationKeyService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenstackVolumeAuthenticationKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeyServiceUpdateResponse{key: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) MustSend() *OpenstackVolumeAuthenticationKeyServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified authentication key.
//
type OpenstackVolumeAuthenticationKeyServiceUpdateResponse struct {
	key *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateResponse) Key() (*OpenstackVolumeAuthenticationKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateResponse) MustKey() *OpenstackVolumeAuthenticationKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
// Update the specified authentication key.
//
func (p *OpenstackVolumeAuthenticationKeyService) Update() *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	return &OpenstackVolumeAuthenticationKeyServiceUpdateRequest{OpenstackVolumeAuthenticationKeyService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeAuthenticationKeyService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackVolumeAuthenticationKeyService) String() string {
	return fmt.Sprintf("OpenstackVolumeAuthenticationKeyService:%s", op.path)
}

//
//
type OpenstackImageProvidersService struct {
	BaseService
}

func NewOpenstackImageProvidersService(connection *Connection, path string) *OpenstackImageProvidersService {
	var result OpenstackImageProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new OpenStack image provider to the system.
//
type OpenstackImageProvidersServiceAddRequest struct {
	OpenstackImageProvidersService *OpenstackImageProvidersService
	header                         map[string]string
	query                          map[string]string
	provider                       *OpenStackImageProvider
}

func (p *OpenstackImageProvidersServiceAddRequest) Header(key, value string) *OpenstackImageProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceAddRequest) Query(key, value string) *OpenstackImageProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceAddRequest) Provider(provider *OpenStackImageProvider) *OpenstackImageProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *OpenstackImageProvidersServiceAddRequest) Send() (*OpenstackImageProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProvidersService.connection.URL(), p.OpenstackImageProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackImageProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProvidersServiceAddResponse{provider: result}, nil
}

func (p *OpenstackImageProvidersServiceAddRequest) MustSend() *OpenstackImageProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new OpenStack image provider to the system.
//
type OpenstackImageProvidersServiceAddResponse struct {
	provider *OpenStackImageProvider
}

func (p *OpenstackImageProvidersServiceAddResponse) Provider() (*OpenStackImageProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackImageProvidersServiceAddResponse) MustProvider() *OpenStackImageProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Add a new OpenStack image provider to the system.
//
func (p *OpenstackImageProvidersService) Add() *OpenstackImageProvidersServiceAddRequest {
	return &OpenstackImageProvidersServiceAddRequest{OpenstackImageProvidersService: p}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackImageProvidersServiceListRequest struct {
	OpenstackImageProvidersService *OpenstackImageProvidersService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
	max                            *int64
	search                         *string
}

func (p *OpenstackImageProvidersServiceListRequest) Header(key, value string) *OpenstackImageProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Query(key, value string) *OpenstackImageProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Follow(follow string) *OpenstackImageProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Max(max int64) *OpenstackImageProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Search(search string) *OpenstackImageProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Send() (*OpenstackImageProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProvidersService.connection.URL(), p.OpenstackImageProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProvidersServiceListResponse{providers: result}, nil
}

func (p *OpenstackImageProvidersServiceListRequest) MustSend() *OpenstackImageProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackImageProvidersServiceListResponse struct {
	providers *OpenStackImageProviderSlice
}

func (p *OpenstackImageProvidersServiceListResponse) Providers() (*OpenStackImageProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *OpenstackImageProvidersServiceListResponse) MustProviders() *OpenStackImageProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
func (p *OpenstackImageProvidersService) List() *OpenstackImageProvidersServiceListRequest {
	return &OpenstackImageProvidersServiceListRequest{OpenstackImageProvidersService: p}
}

//
//
func (op *OpenstackImageProvidersService) ProviderService(id string) *OpenstackImageProviderService {
	return NewOpenstackImageProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImageProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackImageProvidersService) String() string {
	return fmt.Sprintf("OpenstackImageProvidersService:%s", op.path)
}

//
//
type OpenstackImageProviderService struct {
	BaseService
}

func NewOpenstackImageProviderService(connection *Connection, path string) *OpenstackImageProviderService {
	var result OpenstackImageProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackImageProviderServiceGetRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
}

func (p *OpenstackImageProviderServiceGetRequest) Header(key, value string) *OpenstackImageProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceGetRequest) Query(key, value string) *OpenstackImageProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceGetRequest) Follow(follow string) *OpenstackImageProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImageProviderServiceGetRequest) Send() (*OpenstackImageProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProviderServiceGetResponse{provider: result}, nil
}

func (p *OpenstackImageProviderServiceGetRequest) MustSend() *OpenstackImageProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackImageProviderServiceGetResponse struct {
	provider *OpenStackImageProvider
}

func (p *OpenstackImageProviderServiceGetResponse) Provider() (*OpenStackImageProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackImageProviderServiceGetResponse) MustProvider() *OpenStackImageProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
//
func (p *OpenstackImageProviderService) Get() *OpenstackImageProviderServiceGetRequest {
	return &OpenstackImageProviderServiceGetRequest{OpenstackImageProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackImageProviderServiceImportCertificatesRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	certificates                  *CertificateSlice
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Header(key, value string) *OpenstackImageProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Query(key, value string) *OpenstackImageProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *OpenstackImageProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *OpenstackImageProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Send() (*OpenstackImageProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackImageProviderServiceImportCertificatesResponse), nil
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) MustSend() *OpenstackImageProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackImageProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *OpenstackImageProviderService) ImportCertificates() *OpenstackImageProviderServiceImportCertificatesRequest {
	return &OpenstackImageProviderServiceImportCertificatesRequest{OpenstackImageProviderService: p}
}

//
//
type OpenstackImageProviderServiceRemoveRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
}

func (p *OpenstackImageProviderServiceRemoveRequest) Header(key, value string) *OpenstackImageProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceRemoveRequest) Query(key, value string) *OpenstackImageProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceRemoveRequest) Async(async bool) *OpenstackImageProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageProviderServiceRemoveRequest) Send() (*OpenstackImageProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackImageProviderServiceRemoveResponse), nil
}

func (p *OpenstackImageProviderServiceRemoveRequest) MustSend() *OpenstackImageProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackImageProviderServiceRemoveResponse struct {
}

//
//
func (p *OpenstackImageProviderService) Remove() *OpenstackImageProviderServiceRemoveRequest {
	return &OpenstackImageProviderServiceRemoveRequest{OpenstackImageProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackImageProviderServiceTestConnectivityRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Header(key, value string) *OpenstackImageProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Query(key, value string) *OpenstackImageProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Async(async bool) *OpenstackImageProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Send() (*OpenstackImageProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackImageProviderServiceTestConnectivityResponse), nil
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) MustSend() *OpenstackImageProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackImageProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *OpenstackImageProviderService) TestConnectivity() *OpenstackImageProviderServiceTestConnectivityRequest {
	return &OpenstackImageProviderServiceTestConnectivityRequest{OpenstackImageProviderService: p}
}

//
// Update the specified OpenStack image provider in the system.
//
type OpenstackImageProviderServiceUpdateRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
	provider                      *OpenStackImageProvider
}

func (p *OpenstackImageProviderServiceUpdateRequest) Header(key, value string) *OpenstackImageProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Query(key, value string) *OpenstackImageProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Async(async bool) *OpenstackImageProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Provider(provider *OpenStackImageProvider) *OpenstackImageProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Send() (*OpenstackImageProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackImageProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProviderServiceUpdateResponse{provider: result}, nil
}

func (p *OpenstackImageProviderServiceUpdateRequest) MustSend() *OpenstackImageProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified OpenStack image provider in the system.
//
type OpenstackImageProviderServiceUpdateResponse struct {
	provider *OpenStackImageProvider
}

func (p *OpenstackImageProviderServiceUpdateResponse) Provider() (*OpenStackImageProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackImageProviderServiceUpdateResponse) MustProvider() *OpenStackImageProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Update the specified OpenStack image provider in the system.
//
func (p *OpenstackImageProviderService) Update() *OpenstackImageProviderServiceUpdateRequest {
	return &OpenstackImageProviderServiceUpdateRequest{OpenstackImageProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *OpenstackImageProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
//
func (op *OpenstackImageProviderService) ImagesService() *OpenstackImagesService {
	return NewOpenstackImagesService(op.connection, fmt.Sprintf("%s/images", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImageProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "images" {
		return op.ImagesService(), nil
	}
	if strings.HasPrefix(path, "images/") {
		return op.ImagesService().Service(path[7:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackImageProviderService) String() string {
	return fmt.Sprintf("OpenstackImageProviderService:%s", op.path)
}

//
//
type OpenstackImageService struct {
	BaseService
}

func NewOpenstackImageService(connection *Connection, path string) *OpenstackImageService {
	var result OpenstackImageService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackImageServiceGetRequest struct {
	OpenstackImageService *OpenstackImageService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *OpenstackImageServiceGetRequest) Header(key, value string) *OpenstackImageServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageServiceGetRequest) Query(key, value string) *OpenstackImageServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageServiceGetRequest) Follow(follow string) *OpenstackImageServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImageServiceGetRequest) Send() (*OpenstackImageServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageService.connection.URL(), p.OpenstackImageService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageServiceGetResponse{image: result}, nil
}

func (p *OpenstackImageServiceGetRequest) MustSend() *OpenstackImageServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackImageServiceGetResponse struct {
	image *OpenStackImage
}

func (p *OpenstackImageServiceGetResponse) Image() (*OpenStackImage, bool) {
	if p.image != nil {
		return p.image, true
	}
	return nil, false
}

func (p *OpenstackImageServiceGetResponse) MustImage() *OpenStackImage {
	if p.image == nil {
		panic("image in response does not exist")
	}
	return p.image
}

//
//
func (p *OpenstackImageService) Get() *OpenstackImageServiceGetRequest {
	return &OpenstackImageServiceGetRequest{OpenstackImageService: p}
}

//
// Imports a virtual machine from a Glance image storage domain.
// For example, to import the image with identifier `456` from the
// storage domain with identifier `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/openstackimageproviders/123/images/456/import
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain>
//     <name>images0</name>
//   </storage_domain>
//   <cluster>
//     <name>images0</name>
//   </cluster>
// </action>
// ----
//
type OpenstackImageServiceImportRequest struct {
	OpenstackImageService *OpenstackImageService
	header                map[string]string
	query                 map[string]string
	async                 *bool
	cluster               *Cluster
	disk                  *Disk
	importAsTemplate      *bool
	storageDomain         *StorageDomain
	template              *Template
}

func (p *OpenstackImageServiceImportRequest) Header(key, value string) *OpenstackImageServiceImportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageServiceImportRequest) Query(key, value string) *OpenstackImageServiceImportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageServiceImportRequest) Async(async bool) *OpenstackImageServiceImportRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageServiceImportRequest) Cluster(cluster *Cluster) *OpenstackImageServiceImportRequest {
	p.cluster = cluster
	return p
}

func (p *OpenstackImageServiceImportRequest) Disk(disk *Disk) *OpenstackImageServiceImportRequest {
	p.disk = disk
	return p
}

func (p *OpenstackImageServiceImportRequest) ImportAsTemplate(importAsTemplate bool) *OpenstackImageServiceImportRequest {
	p.importAsTemplate = &importAsTemplate
	return p
}

func (p *OpenstackImageServiceImportRequest) StorageDomain(storageDomain *StorageDomain) *OpenstackImageServiceImportRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *OpenstackImageServiceImportRequest) Template(template *Template) *OpenstackImageServiceImportRequest {
	p.template = template
	return p
}

func (p *OpenstackImageServiceImportRequest) Send() (*OpenstackImageServiceImportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/import", p.OpenstackImageService.connection.URL(), p.OpenstackImageService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Cluster(p.cluster)
	actionBuilder.Disk(p.disk)
	if p.importAsTemplate != nil {
		actionBuilder.ImportAsTemplate(*p.importAsTemplate)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	actionBuilder.Template(p.template)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackImageServiceImportResponse), nil
}

func (p *OpenstackImageServiceImportRequest) MustSend() *OpenstackImageServiceImportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Imports a virtual machine from a Glance image storage domain.
// For example, to import the image with identifier `456` from the
// storage domain with identifier `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/openstackimageproviders/123/images/456/import
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain>
//     <name>images0</name>
//   </storage_domain>
//   <cluster>
//     <name>images0</name>
//   </cluster>
// </action>
// ----
//
type OpenstackImageServiceImportResponse struct {
}

//
// Imports a virtual machine from a Glance image storage domain.
// For example, to import the image with identifier `456` from the
// storage domain with identifier `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/openstackimageproviders/123/images/456/import
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain>
//     <name>images0</name>
//   </storage_domain>
//   <cluster>
//     <name>images0</name>
//   </cluster>
// </action>
// ----
//
func (p *OpenstackImageService) Import() *OpenstackImageServiceImportRequest {
	return &OpenstackImageServiceImportRequest{OpenstackImageService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImageService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackImageService) String() string {
	return fmt.Sprintf("OpenstackImageService:%s", op.path)
}

//
//
type OpenstackVolumeTypesService struct {
	BaseService
}

func NewOpenstackVolumeTypesService(connection *Connection, path string) *OpenstackVolumeTypesService {
	var result OpenstackVolumeTypesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of volume types.
// The order of the returned list of volume types isn't guaranteed.
//
type OpenstackVolumeTypesServiceListRequest struct {
	OpenstackVolumeTypesService *OpenstackVolumeTypesService
	header                      map[string]string
	query                       map[string]string
	follow                      *string
	max                         *int64
}

func (p *OpenstackVolumeTypesServiceListRequest) Header(key, value string) *OpenstackVolumeTypesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Query(key, value string) *OpenstackVolumeTypesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Follow(follow string) *OpenstackVolumeTypesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Max(max int64) *OpenstackVolumeTypesServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Send() (*OpenstackVolumeTypesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeTypesService.connection.URL(), p.OpenstackVolumeTypesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeTypesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeTypesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeTypesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeTypesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeTypesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeTypeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeTypesServiceListResponse{types: result}, nil
}

func (p *OpenstackVolumeTypesServiceListRequest) MustSend() *OpenstackVolumeTypesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of volume types.
// The order of the returned list of volume types isn't guaranteed.
//
type OpenstackVolumeTypesServiceListResponse struct {
	types *OpenStackVolumeTypeSlice
}

func (p *OpenstackVolumeTypesServiceListResponse) Types() (*OpenStackVolumeTypeSlice, bool) {
	if p.types != nil {
		return p.types, true
	}
	return nil, false
}

func (p *OpenstackVolumeTypesServiceListResponse) MustTypes() *OpenStackVolumeTypeSlice {
	if p.types == nil {
		panic("types in response does not exist")
	}
	return p.types
}

//
// Returns the list of volume types.
// The order of the returned list of volume types isn't guaranteed.
//
func (p *OpenstackVolumeTypesService) List() *OpenstackVolumeTypesServiceListRequest {
	return &OpenstackVolumeTypesServiceListRequest{OpenstackVolumeTypesService: p}
}

//
//
func (op *OpenstackVolumeTypesService) TypeService(id string) *OpenstackVolumeTypeService {
	return NewOpenstackVolumeTypeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeTypesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.TypeService(path), nil
	}
	return op.TypeService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackVolumeTypesService) String() string {
	return fmt.Sprintf("OpenstackVolumeTypesService:%s", op.path)
}

//
//
type OpenstackSubnetService struct {
	BaseService
}

func NewOpenstackSubnetService(connection *Connection, path string) *OpenstackSubnetService {
	var result OpenstackSubnetService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackSubnetServiceGetRequest struct {
	OpenstackSubnetService *OpenstackSubnetService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *OpenstackSubnetServiceGetRequest) Header(key, value string) *OpenstackSubnetServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetServiceGetRequest) Query(key, value string) *OpenstackSubnetServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetServiceGetRequest) Follow(follow string) *OpenstackSubnetServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackSubnetServiceGetRequest) Send() (*OpenstackSubnetServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetService.connection.URL(), p.OpenstackSubnetService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackSubnetReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackSubnetServiceGetResponse{subnet: result}, nil
}

func (p *OpenstackSubnetServiceGetRequest) MustSend() *OpenstackSubnetServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackSubnetServiceGetResponse struct {
	subnet *OpenStackSubnet
}

func (p *OpenstackSubnetServiceGetResponse) Subnet() (*OpenStackSubnet, bool) {
	if p.subnet != nil {
		return p.subnet, true
	}
	return nil, false
}

func (p *OpenstackSubnetServiceGetResponse) MustSubnet() *OpenStackSubnet {
	if p.subnet == nil {
		panic("subnet in response does not exist")
	}
	return p.subnet
}

//
//
func (p *OpenstackSubnetService) Get() *OpenstackSubnetServiceGetRequest {
	return &OpenstackSubnetServiceGetRequest{OpenstackSubnetService: p}
}

//
//
type OpenstackSubnetServiceRemoveRequest struct {
	OpenstackSubnetService *OpenstackSubnetService
	header                 map[string]string
	query                  map[string]string
	async                  *bool
}

func (p *OpenstackSubnetServiceRemoveRequest) Header(key, value string) *OpenstackSubnetServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetServiceRemoveRequest) Query(key, value string) *OpenstackSubnetServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetServiceRemoveRequest) Async(async bool) *OpenstackSubnetServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackSubnetServiceRemoveRequest) Send() (*OpenstackSubnetServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetService.connection.URL(), p.OpenstackSubnetService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackSubnetServiceRemoveResponse), nil
}

func (p *OpenstackSubnetServiceRemoveRequest) MustSend() *OpenstackSubnetServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackSubnetServiceRemoveResponse struct {
}

//
//
func (p *OpenstackSubnetService) Remove() *OpenstackSubnetServiceRemoveRequest {
	return &OpenstackSubnetServiceRemoveRequest{OpenstackSubnetService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackSubnetService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackSubnetService) String() string {
	return fmt.Sprintf("OpenstackSubnetService:%s", op.path)
}

//
//
type OpenstackNetworksService struct {
	BaseService
}

func NewOpenstackNetworksService(connection *Connection, path string) *OpenstackNetworksService {
	var result OpenstackNetworksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
type OpenstackNetworksServiceListRequest struct {
	OpenstackNetworksService *OpenstackNetworksService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
	max                      *int64
}

func (p *OpenstackNetworksServiceListRequest) Header(key, value string) *OpenstackNetworksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworksServiceListRequest) Query(key, value string) *OpenstackNetworksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworksServiceListRequest) Follow(follow string) *OpenstackNetworksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworksServiceListRequest) Max(max int64) *OpenstackNetworksServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackNetworksServiceListRequest) Send() (*OpenstackNetworksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworksService.connection.URL(), p.OpenstackNetworksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworksServiceListResponse{networks: result}, nil
}

func (p *OpenstackNetworksServiceListRequest) MustSend() *OpenstackNetworksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
type OpenstackNetworksServiceListResponse struct {
	networks *OpenStackNetworkSlice
}

func (p *OpenstackNetworksServiceListResponse) Networks() (*OpenStackNetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *OpenstackNetworksServiceListResponse) MustNetworks() *OpenStackNetworkSlice {
	if p.networks == nil {
		panic("networks in response does not exist")
	}
	return p.networks
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
func (p *OpenstackNetworksService) List() *OpenstackNetworksServiceListRequest {
	return &OpenstackNetworksServiceListRequest{OpenstackNetworksService: p}
}

//
//
func (op *OpenstackNetworksService) NetworkService(id string) *OpenstackNetworkService {
	return NewOpenstackNetworkService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkService(path), nil
	}
	return op.NetworkService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackNetworksService) String() string {
	return fmt.Sprintf("OpenstackNetworksService:%s", op.path)
}

//
//
type OpenstackVolumeProviderService struct {
	BaseService
}

func NewOpenstackVolumeProviderService(connection *Connection, path string) *OpenstackVolumeProviderService {
	var result OpenstackVolumeProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackVolumeProviderServiceGetRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
}

func (p *OpenstackVolumeProviderServiceGetRequest) Header(key, value string) *OpenstackVolumeProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceGetRequest) Query(key, value string) *OpenstackVolumeProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceGetRequest) Follow(follow string) *OpenstackVolumeProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeProviderServiceGetRequest) Send() (*OpenstackVolumeProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProviderServiceGetResponse{provider: result}, nil
}

func (p *OpenstackVolumeProviderServiceGetRequest) MustSend() *OpenstackVolumeProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeProviderServiceGetResponse struct {
	provider *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProviderServiceGetResponse) Provider() (*OpenStackVolumeProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackVolumeProviderServiceGetResponse) MustProvider() *OpenStackVolumeProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
//
func (p *OpenstackVolumeProviderService) Get() *OpenstackVolumeProviderServiceGetRequest {
	return &OpenstackVolumeProviderServiceGetRequest{OpenstackVolumeProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackVolumeProviderServiceImportCertificatesRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	certificates                   *CertificateSlice
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Header(key, value string) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Query(key, value string) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Send() (*OpenstackVolumeProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackVolumeProviderServiceImportCertificatesResponse), nil
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) MustSend() *OpenstackVolumeProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackVolumeProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *OpenstackVolumeProviderService) ImportCertificates() *OpenstackVolumeProviderServiceImportCertificatesRequest {
	return &OpenstackVolumeProviderServiceImportCertificatesRequest{OpenstackVolumeProviderService: p}
}

//
//
type OpenstackVolumeProviderServiceRemoveRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	async                          *bool
	force                          *bool
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Header(key, value string) *OpenstackVolumeProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Query(key, value string) *OpenstackVolumeProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Async(async bool) *OpenstackVolumeProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Force(force bool) *OpenstackVolumeProviderServiceRemoveRequest {
	p.force = &force
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Send() (*OpenstackVolumeProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.force != nil {
		values["force"] = []string{fmt.Sprintf("%v", *p.force)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackVolumeProviderServiceRemoveResponse), nil
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) MustSend() *OpenstackVolumeProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeProviderServiceRemoveResponse struct {
}

//
//
func (p *OpenstackVolumeProviderService) Remove() *OpenstackVolumeProviderServiceRemoveRequest {
	return &OpenstackVolumeProviderServiceRemoveRequest{OpenstackVolumeProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackVolumeProviderServiceTestConnectivityRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	async                          *bool
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Header(key, value string) *OpenstackVolumeProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Query(key, value string) *OpenstackVolumeProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Async(async bool) *OpenstackVolumeProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Send() (*OpenstackVolumeProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackVolumeProviderServiceTestConnectivityResponse), nil
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) MustSend() *OpenstackVolumeProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackVolumeProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *OpenstackVolumeProviderService) TestConnectivity() *OpenstackVolumeProviderServiceTestConnectivityRequest {
	return &OpenstackVolumeProviderServiceTestConnectivityRequest{OpenstackVolumeProviderService: p}
}

//
// Update the specified OpenStack volume provider in the system.
//
type OpenstackVolumeProviderServiceUpdateRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	async                          *bool
	provider                       *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Header(key, value string) *OpenstackVolumeProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Query(key, value string) *OpenstackVolumeProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Async(async bool) *OpenstackVolumeProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Provider(provider *OpenStackVolumeProvider) *OpenstackVolumeProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Send() (*OpenstackVolumeProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackVolumeProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProviderServiceUpdateResponse{provider: result}, nil
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) MustSend() *OpenstackVolumeProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified OpenStack volume provider in the system.
//
type OpenstackVolumeProviderServiceUpdateResponse struct {
	provider *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProviderServiceUpdateResponse) Provider() (*OpenStackVolumeProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackVolumeProviderServiceUpdateResponse) MustProvider() *OpenStackVolumeProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Update the specified OpenStack volume provider in the system.
//
func (p *OpenstackVolumeProviderService) Update() *OpenstackVolumeProviderServiceUpdateRequest {
	return &OpenstackVolumeProviderServiceUpdateRequest{OpenstackVolumeProviderService: p}
}

//
//
func (op *OpenstackVolumeProviderService) AuthenticationKeysService() *OpenstackVolumeAuthenticationKeysService {
	return NewOpenstackVolumeAuthenticationKeysService(op.connection, fmt.Sprintf("%s/authenticationkeys", op.path))
}

//
// A service to view certificates for this external provider.
//
func (op *OpenstackVolumeProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
//
func (op *OpenstackVolumeProviderService) VolumeTypesService() *OpenstackVolumeTypesService {
	return NewOpenstackVolumeTypesService(op.connection, fmt.Sprintf("%s/volumetypes", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "authenticationkeys" {
		return op.AuthenticationKeysService(), nil
	}
	if strings.HasPrefix(path, "authenticationkeys/") {
		return op.AuthenticationKeysService().Service(path[19:])
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "volumetypes" {
		return op.VolumeTypesService(), nil
	}
	if strings.HasPrefix(path, "volumetypes/") {
		return op.VolumeTypesService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackVolumeProviderService) String() string {
	return fmt.Sprintf("OpenstackVolumeProviderService:%s", op.path)
}

//
//
type OpenstackImagesService struct {
	BaseService
}

func NewOpenstackImagesService(connection *Connection, path string) *OpenstackImagesService {
	var result OpenstackImagesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Lists the images of a Glance image storage domain.
// The order of the returned list of images isn't guaranteed.
//
type OpenstackImagesServiceListRequest struct {
	OpenstackImagesService *OpenstackImagesService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
	max                    *int64
}

func (p *OpenstackImagesServiceListRequest) Header(key, value string) *OpenstackImagesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImagesServiceListRequest) Query(key, value string) *OpenstackImagesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImagesServiceListRequest) Follow(follow string) *OpenstackImagesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImagesServiceListRequest) Max(max int64) *OpenstackImagesServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackImagesServiceListRequest) Send() (*OpenstackImagesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImagesService.connection.URL(), p.OpenstackImagesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImagesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImagesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImagesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImagesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImagesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackImagesServiceListResponse{images: result}, nil
}

func (p *OpenstackImagesServiceListRequest) MustSend() *OpenstackImagesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the images of a Glance image storage domain.
// The order of the returned list of images isn't guaranteed.
//
type OpenstackImagesServiceListResponse struct {
	images *OpenStackImageSlice
}

func (p *OpenstackImagesServiceListResponse) Images() (*OpenStackImageSlice, bool) {
	if p.images != nil {
		return p.images, true
	}
	return nil, false
}

func (p *OpenstackImagesServiceListResponse) MustImages() *OpenStackImageSlice {
	if p.images == nil {
		panic("images in response does not exist")
	}
	return p.images
}

//
// Lists the images of a Glance image storage domain.
// The order of the returned list of images isn't guaranteed.
//
func (p *OpenstackImagesService) List() *OpenstackImagesServiceListRequest {
	return &OpenstackImagesServiceListRequest{OpenstackImagesService: p}
}

//
// Returns a reference to the service that manages a specific image.
//
func (op *OpenstackImagesService) ImageService(id string) *OpenstackImageService {
	return NewOpenstackImageService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImagesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ImageService(path), nil
	}
	return op.ImageService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackImagesService) String() string {
	return fmt.Sprintf("OpenstackImagesService:%s", op.path)
}

//
//
type OpenstackVolumeAuthenticationKeysService struct {
	BaseService
}

func NewOpenstackVolumeAuthenticationKeysService(connection *Connection, path string) *OpenstackVolumeAuthenticationKeysService {
	var result OpenstackVolumeAuthenticationKeysService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new authentication key to the OpenStack volume provider.
//
type OpenstackVolumeAuthenticationKeysServiceAddRequest struct {
	OpenstackVolumeAuthenticationKeysService *OpenstackVolumeAuthenticationKeysService
	header                                   map[string]string
	query                                    map[string]string
	key                                      *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Key(key *OpenstackVolumeAuthenticationKey) *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	p.key = key
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Send() (*OpenstackVolumeAuthenticationKeysServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeysService.connection.URL(), p.OpenstackVolumeAuthenticationKeysService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenstackVolumeAuthenticationKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeysServiceAddResponse{key: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) MustSend() *OpenstackVolumeAuthenticationKeysServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new authentication key to the OpenStack volume provider.
//
type OpenstackVolumeAuthenticationKeysServiceAddResponse struct {
	key *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddResponse) Key() (*OpenstackVolumeAuthenticationKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddResponse) MustKey() *OpenstackVolumeAuthenticationKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
// Add a new authentication key to the OpenStack volume provider.
//
func (p *OpenstackVolumeAuthenticationKeysService) Add() *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	return &OpenstackVolumeAuthenticationKeysServiceAddRequest{OpenstackVolumeAuthenticationKeysService: p}
}

//
// Returns the list of authentication keys.
// The order of the returned list of authentication keys isn't guaranteed.
//
type OpenstackVolumeAuthenticationKeysServiceListRequest struct {
	OpenstackVolumeAuthenticationKeysService *OpenstackVolumeAuthenticationKeysService
	header                                   map[string]string
	query                                    map[string]string
	follow                                   *string
	max                                      *int64
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Follow(follow string) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Max(max int64) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Send() (*OpenstackVolumeAuthenticationKeysServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeysService.connection.URL(), p.OpenstackVolumeAuthenticationKeysService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeysServiceListResponse{keys: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) MustSend() *OpenstackVolumeAuthenticationKeysServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of authentication keys.
// The order of the returned list of authentication keys isn't guaranteed.
//
type OpenstackVolumeAuthenticationKeysServiceListResponse struct {
	keys *OpenstackVolumeAuthenticationKeySlice
}

func (p *OpenstackVolumeAuthenticationKeysServiceListResponse) Keys() (*OpenstackVolumeAuthenticationKeySlice, bool) {
	if p.keys != nil {
		return p.keys, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeysServiceListResponse) MustKeys() *OpenstackVolumeAuthenticationKeySlice {
	if p.keys == nil {
		panic("keys in response does not exist")
	}
	return p.keys
}

//
// Returns the list of authentication keys.
// The order of the returned list of authentication keys isn't guaranteed.
//
func (p *OpenstackVolumeAuthenticationKeysService) List() *OpenstackVolumeAuthenticationKeysServiceListRequest {
	return &OpenstackVolumeAuthenticationKeysServiceListRequest{OpenstackVolumeAuthenticationKeysService: p}
}

//
//
func (op *OpenstackVolumeAuthenticationKeysService) KeyService(id string) *OpenstackVolumeAuthenticationKeyService {
	return NewOpenstackVolumeAuthenticationKeyService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeAuthenticationKeysService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KeyService(path), nil
	}
	return op.KeyService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackVolumeAuthenticationKeysService) String() string {
	return fmt.Sprintf("OpenstackVolumeAuthenticationKeysService:%s", op.path)
}

//
// This service manages OpenStack network providers.
//
type OpenstackNetworkProvidersService struct {
	BaseService
}

func NewOpenstackNetworkProvidersService(connection *Connection, path string) *OpenstackNetworkProvidersService {
	var result OpenstackNetworkProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// The operation adds a new network provider to the system.
// If the `type` property is not present, a default value of `NEUTRON` will be used.
//
type OpenstackNetworkProvidersServiceAddRequest struct {
	OpenstackNetworkProvidersService *OpenstackNetworkProvidersService
	header                           map[string]string
	query                            map[string]string
	provider                         *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Header(key, value string) *OpenstackNetworkProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Query(key, value string) *OpenstackNetworkProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Provider(provider *OpenStackNetworkProvider) *OpenstackNetworkProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Send() (*OpenstackNetworkProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProvidersService.connection.URL(), p.OpenstackNetworkProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackNetworkProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProvidersServiceAddResponse{provider: result}, nil
}

func (p *OpenstackNetworkProvidersServiceAddRequest) MustSend() *OpenstackNetworkProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// The operation adds a new network provider to the system.
// If the `type` property is not present, a default value of `NEUTRON` will be used.
//
type OpenstackNetworkProvidersServiceAddResponse struct {
	provider *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProvidersServiceAddResponse) Provider() (*OpenStackNetworkProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackNetworkProvidersServiceAddResponse) MustProvider() *OpenStackNetworkProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// The operation adds a new network provider to the system.
// If the `type` property is not present, a default value of `NEUTRON` will be used.
//
func (p *OpenstackNetworkProvidersService) Add() *OpenstackNetworkProvidersServiceAddRequest {
	return &OpenstackNetworkProvidersServiceAddRequest{OpenstackNetworkProvidersService: p}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackNetworkProvidersServiceListRequest struct {
	OpenstackNetworkProvidersService *OpenstackNetworkProvidersService
	header                           map[string]string
	query                            map[string]string
	follow                           *string
	max                              *int64
	search                           *string
}

func (p *OpenstackNetworkProvidersServiceListRequest) Header(key, value string) *OpenstackNetworkProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Query(key, value string) *OpenstackNetworkProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Follow(follow string) *OpenstackNetworkProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Max(max int64) *OpenstackNetworkProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Search(search string) *OpenstackNetworkProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Send() (*OpenstackNetworkProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProvidersService.connection.URL(), p.OpenstackNetworkProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProvidersServiceListResponse{providers: result}, nil
}

func (p *OpenstackNetworkProvidersServiceListRequest) MustSend() *OpenstackNetworkProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackNetworkProvidersServiceListResponse struct {
	providers *OpenStackNetworkProviderSlice
}

func (p *OpenstackNetworkProvidersServiceListResponse) Providers() (*OpenStackNetworkProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *OpenstackNetworkProvidersServiceListResponse) MustProviders() *OpenStackNetworkProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
func (p *OpenstackNetworkProvidersService) List() *OpenstackNetworkProvidersServiceListRequest {
	return &OpenstackNetworkProvidersServiceListRequest{OpenstackNetworkProvidersService: p}
}

//
// Reference to OpenStack network provider service.
//
func (op *OpenstackNetworkProvidersService) ProviderService(id string) *OpenstackNetworkProviderService {
	return NewOpenstackNetworkProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworkProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackNetworkProvidersService) String() string {
	return fmt.Sprintf("OpenstackNetworkProvidersService:%s", op.path)
}

//
//
type OpenstackNetworkService struct {
	BaseService
}

func NewOpenstackNetworkService(connection *Connection, path string) *OpenstackNetworkService {
	var result OpenstackNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackNetworkServiceGetRequest struct {
	OpenstackNetworkService *OpenstackNetworkService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *OpenstackNetworkServiceGetRequest) Header(key, value string) *OpenstackNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkServiceGetRequest) Query(key, value string) *OpenstackNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkServiceGetRequest) Follow(follow string) *OpenstackNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworkServiceGetRequest) Send() (*OpenstackNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkService.connection.URL(), p.OpenstackNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkServiceGetResponse{network: result}, nil
}

func (p *OpenstackNetworkServiceGetRequest) MustSend() *OpenstackNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackNetworkServiceGetResponse struct {
	network *OpenStackNetwork
}

func (p *OpenstackNetworkServiceGetResponse) Network() (*OpenStackNetwork, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *OpenstackNetworkServiceGetResponse) MustNetwork() *OpenStackNetwork {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
//
func (p *OpenstackNetworkService) Get() *OpenstackNetworkServiceGetRequest {
	return &OpenstackNetworkServiceGetRequest{OpenstackNetworkService: p}
}

//
// This operation imports an external network into {product-name}.
// The network will be added to the specified data center.
//
type OpenstackNetworkServiceImportRequest struct {
	OpenstackNetworkService *OpenstackNetworkService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
	dataCenter              *DataCenter
}

func (p *OpenstackNetworkServiceImportRequest) Header(key, value string) *OpenstackNetworkServiceImportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkServiceImportRequest) Query(key, value string) *OpenstackNetworkServiceImportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkServiceImportRequest) Async(async bool) *OpenstackNetworkServiceImportRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkServiceImportRequest) DataCenter(dataCenter *DataCenter) *OpenstackNetworkServiceImportRequest {
	p.dataCenter = dataCenter
	return p
}

func (p *OpenstackNetworkServiceImportRequest) Send() (*OpenstackNetworkServiceImportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/import", p.OpenstackNetworkService.connection.URL(), p.OpenstackNetworkService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.DataCenter(p.dataCenter)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackNetworkServiceImportResponse), nil
}

func (p *OpenstackNetworkServiceImportRequest) MustSend() *OpenstackNetworkServiceImportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation imports an external network into {product-name}.
// The network will be added to the specified data center.
//
type OpenstackNetworkServiceImportResponse struct {
}

//
// This operation imports an external network into {product-name}.
// The network will be added to the specified data center.
//
func (p *OpenstackNetworkService) Import() *OpenstackNetworkServiceImportRequest {
	return &OpenstackNetworkServiceImportRequest{OpenstackNetworkService: p}
}

//
//
func (op *OpenstackNetworkService) SubnetsService() *OpenstackSubnetsService {
	return NewOpenstackSubnetsService(op.connection, fmt.Sprintf("%s/subnets", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "subnets" {
		return op.SubnetsService(), nil
	}
	if strings.HasPrefix(path, "subnets/") {
		return op.SubnetsService().Service(path[8:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackNetworkService) String() string {
	return fmt.Sprintf("OpenstackNetworkService:%s", op.path)
}

//
//
type OpenstackVolumeProvidersService struct {
	BaseService
}

func NewOpenstackVolumeProvidersService(connection *Connection, path string) *OpenstackVolumeProvidersService {
	var result OpenstackVolumeProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new volume provider.
// For example:
// [source]
// ----
// POST /ovirt-engine/api/openstackvolumeproviders
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_volume_provider>
//   <name>mycinder</name>
//   <url>https://mycinder.example.com:8776</url>
//   <data_center>
//     <name>mydc</name>
//   </data_center>
//   <requires_authentication>true</requires_authentication>
//   <username>admin</username>
//   <password>mypassword</password>
//   <tenant_name>mytenant</tenant_name>
// </openstack_volume_provider>
// ----
//
type OpenstackVolumeProvidersServiceAddRequest struct {
	OpenstackVolumeProvidersService *OpenstackVolumeProvidersService
	header                          map[string]string
	query                           map[string]string
	provider                        *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Header(key, value string) *OpenstackVolumeProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Query(key, value string) *OpenstackVolumeProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Provider(provider *OpenStackVolumeProvider) *OpenstackVolumeProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Send() (*OpenstackVolumeProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProvidersService.connection.URL(), p.OpenstackVolumeProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackVolumeProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProvidersServiceAddResponse{provider: result}, nil
}

func (p *OpenstackVolumeProvidersServiceAddRequest) MustSend() *OpenstackVolumeProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new volume provider.
// For example:
// [source]
// ----
// POST /ovirt-engine/api/openstackvolumeproviders
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_volume_provider>
//   <name>mycinder</name>
//   <url>https://mycinder.example.com:8776</url>
//   <data_center>
//     <name>mydc</name>
//   </data_center>
//   <requires_authentication>true</requires_authentication>
//   <username>admin</username>
//   <password>mypassword</password>
//   <tenant_name>mytenant</tenant_name>
// </openstack_volume_provider>
// ----
//
type OpenstackVolumeProvidersServiceAddResponse struct {
	provider *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProvidersServiceAddResponse) Provider() (*OpenStackVolumeProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackVolumeProvidersServiceAddResponse) MustProvider() *OpenStackVolumeProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Adds a new volume provider.
// For example:
// [source]
// ----
// POST /ovirt-engine/api/openstackvolumeproviders
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_volume_provider>
//   <name>mycinder</name>
//   <url>https://mycinder.example.com:8776</url>
//   <data_center>
//     <name>mydc</name>
//   </data_center>
//   <requires_authentication>true</requires_authentication>
//   <username>admin</username>
//   <password>mypassword</password>
//   <tenant_name>mytenant</tenant_name>
// </openstack_volume_provider>
// ----
//
func (p *OpenstackVolumeProvidersService) Add() *OpenstackVolumeProvidersServiceAddRequest {
	return &OpenstackVolumeProvidersServiceAddRequest{OpenstackVolumeProvidersService: p}
}

//
// Retrieves the list of volume providers.
// The order of the returned list of volume providers isn't guaranteed.
//
type OpenstackVolumeProvidersServiceListRequest struct {
	OpenstackVolumeProvidersService *OpenstackVolumeProvidersService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
	max                             *int64
	search                          *string
}

func (p *OpenstackVolumeProvidersServiceListRequest) Header(key, value string) *OpenstackVolumeProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Query(key, value string) *OpenstackVolumeProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Follow(follow string) *OpenstackVolumeProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Max(max int64) *OpenstackVolumeProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Search(search string) *OpenstackVolumeProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Send() (*OpenstackVolumeProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProvidersService.connection.URL(), p.OpenstackVolumeProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProvidersServiceListResponse{providers: result}, nil
}

func (p *OpenstackVolumeProvidersServiceListRequest) MustSend() *OpenstackVolumeProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the list of volume providers.
// The order of the returned list of volume providers isn't guaranteed.
//
type OpenstackVolumeProvidersServiceListResponse struct {
	providers *OpenStackVolumeProviderSlice
}

func (p *OpenstackVolumeProvidersServiceListResponse) Providers() (*OpenStackVolumeProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *OpenstackVolumeProvidersServiceListResponse) MustProviders() *OpenStackVolumeProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Retrieves the list of volume providers.
// The order of the returned list of volume providers isn't guaranteed.
//
func (p *OpenstackVolumeProvidersService) List() *OpenstackVolumeProvidersServiceListRequest {
	return &OpenstackVolumeProvidersServiceListRequest{OpenstackVolumeProvidersService: p}
}

//
//
func (op *OpenstackVolumeProvidersService) ProviderService(id string) *OpenstackVolumeProviderService {
	return NewOpenstackVolumeProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackVolumeProvidersService) String() string {
	return fmt.Sprintf("OpenstackVolumeProvidersService:%s", op.path)
}

//
// This service manages the OpenStack network provider.
//
type OpenstackNetworkProviderService struct {
	BaseService
}

func NewOpenstackNetworkProviderService(connection *Connection, path string) *OpenstackNetworkProviderService {
	var result OpenstackNetworkProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the representation of the object managed by this service.
// For example, to get the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceGetRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
}

func (p *OpenstackNetworkProviderServiceGetRequest) Header(key, value string) *OpenstackNetworkProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceGetRequest) Query(key, value string) *OpenstackNetworkProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceGetRequest) Follow(follow string) *OpenstackNetworkProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworkProviderServiceGetRequest) Send() (*OpenstackNetworkProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProviderServiceGetResponse{provider: result}, nil
}

func (p *OpenstackNetworkProviderServiceGetRequest) MustSend() *OpenstackNetworkProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the representation of the object managed by this service.
// For example, to get the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceGetResponse struct {
	provider *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProviderServiceGetResponse) Provider() (*OpenStackNetworkProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackNetworkProviderServiceGetResponse) MustProvider() *OpenStackNetworkProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Returns the representation of the object managed by this service.
// For example, to get the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
func (p *OpenstackNetworkProviderService) Get() *OpenstackNetworkProviderServiceGetRequest {
	return &OpenstackNetworkProviderServiceGetRequest{OpenstackNetworkProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackNetworkProviderServiceImportCertificatesRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	certificates                    *CertificateSlice
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Header(key, value string) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Query(key, value string) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Send() (*OpenstackNetworkProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackNetworkProviderServiceImportCertificatesResponse), nil
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) MustSend() *OpenstackNetworkProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackNetworkProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *OpenstackNetworkProviderService) ImportCertificates() *OpenstackNetworkProviderServiceImportCertificatesRequest {
	return &OpenstackNetworkProviderServiceImportCertificatesRequest{OpenstackNetworkProviderService: p}
}

//
// Removes the provider.
// For example, to remove the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceRemoveRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	async                           *bool
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Header(key, value string) *OpenstackNetworkProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Query(key, value string) *OpenstackNetworkProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Async(async bool) *OpenstackNetworkProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Send() (*OpenstackNetworkProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackNetworkProviderServiceRemoveResponse), nil
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) MustSend() *OpenstackNetworkProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the provider.
// For example, to remove the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceRemoveResponse struct {
}

//
// Removes the provider.
// For example, to remove the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
func (p *OpenstackNetworkProviderService) Remove() *OpenstackNetworkProviderServiceRemoveRequest {
	return &OpenstackNetworkProviderServiceRemoveRequest{OpenstackNetworkProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackNetworkProviderServiceTestConnectivityRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	async                           *bool
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Header(key, value string) *OpenstackNetworkProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Query(key, value string) *OpenstackNetworkProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Async(async bool) *OpenstackNetworkProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Send() (*OpenstackNetworkProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackNetworkProviderServiceTestConnectivityResponse), nil
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) MustSend() *OpenstackNetworkProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackNetworkProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *OpenstackNetworkProviderService) TestConnectivity() *OpenstackNetworkProviderServiceTestConnectivityRequest {
	return &OpenstackNetworkProviderServiceTestConnectivityRequest{OpenstackNetworkProviderService: p}
}

//
// Updates the provider.
// For example, to update `provider_name`, `requires_authentication`, `url`, `tenant_name` and `type` properties,
// for the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/openstacknetworkproviders/1234
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_network_provider>
//   <name>ovn-network-provider</name>
//   <requires_authentication>false</requires_authentication>
//   <url>http://some_server_url.domain.com:9696</url>
//   <tenant_name>oVirt</tenant_name>
//   <type>external</type>
// </openstack_network_provider>
// ----
//
type OpenstackNetworkProviderServiceUpdateRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	async                           *bool
	provider                        *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Header(key, value string) *OpenstackNetworkProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Query(key, value string) *OpenstackNetworkProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Async(async bool) *OpenstackNetworkProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Provider(provider *OpenStackNetworkProvider) *OpenstackNetworkProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Send() (*OpenstackNetworkProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackNetworkProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProviderServiceUpdateResponse{provider: result}, nil
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) MustSend() *OpenstackNetworkProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the provider.
// For example, to update `provider_name`, `requires_authentication`, `url`, `tenant_name` and `type` properties,
// for the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/openstacknetworkproviders/1234
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_network_provider>
//   <name>ovn-network-provider</name>
//   <requires_authentication>false</requires_authentication>
//   <url>http://some_server_url.domain.com:9696</url>
//   <tenant_name>oVirt</tenant_name>
//   <type>external</type>
// </openstack_network_provider>
// ----
//
type OpenstackNetworkProviderServiceUpdateResponse struct {
	provider *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProviderServiceUpdateResponse) Provider() (*OpenStackNetworkProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackNetworkProviderServiceUpdateResponse) MustProvider() *OpenStackNetworkProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Updates the provider.
// For example, to update `provider_name`, `requires_authentication`, `url`, `tenant_name` and `type` properties,
// for the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/openstacknetworkproviders/1234
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_network_provider>
//   <name>ovn-network-provider</name>
//   <requires_authentication>false</requires_authentication>
//   <url>http://some_server_url.domain.com:9696</url>
//   <tenant_name>oVirt</tenant_name>
//   <type>external</type>
// </openstack_network_provider>
// ----
//
func (p *OpenstackNetworkProviderService) Update() *OpenstackNetworkProviderServiceUpdateRequest {
	return &OpenstackNetworkProviderServiceUpdateRequest{OpenstackNetworkProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *OpenstackNetworkProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
// Reference to OpenStack networks service.
//
func (op *OpenstackNetworkProviderService) NetworksService() *OpenstackNetworksService {
	return NewOpenstackNetworksService(op.connection, fmt.Sprintf("%s/networks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworkProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "networks" {
		return op.NetworksService(), nil
	}
	if strings.HasPrefix(path, "networks/") {
		return op.NetworksService().Service(path[9:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackNetworkProviderService) String() string {
	return fmt.Sprintf("OpenstackNetworkProviderService:%s", op.path)
}

//
//
type AssignedCpuProfileService struct {
	BaseService
}

func NewAssignedCpuProfileService(connection *Connection, path string) *AssignedCpuProfileService {
	var result AssignedCpuProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type AssignedCpuProfileServiceGetRequest struct {
	AssignedCpuProfileService *AssignedCpuProfileService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
}

func (p *AssignedCpuProfileServiceGetRequest) Header(key, value string) *AssignedCpuProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfileServiceGetRequest) Query(key, value string) *AssignedCpuProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfileServiceGetRequest) Follow(follow string) *AssignedCpuProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedCpuProfileServiceGetRequest) Send() (*AssignedCpuProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfileService.connection.URL(), p.AssignedCpuProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedCpuProfileServiceGetResponse{profile: result}, nil
}

func (p *AssignedCpuProfileServiceGetRequest) MustSend() *AssignedCpuProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedCpuProfileServiceGetResponse struct {
	profile *CpuProfile
}

func (p *AssignedCpuProfileServiceGetResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedCpuProfileServiceGetResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *AssignedCpuProfileService) Get() *AssignedCpuProfileServiceGetRequest {
	return &AssignedCpuProfileServiceGetRequest{AssignedCpuProfileService: p}
}

//
//
type AssignedCpuProfileServiceRemoveRequest struct {
	AssignedCpuProfileService *AssignedCpuProfileService
	header                    map[string]string
	query                     map[string]string
	async                     *bool
}

func (p *AssignedCpuProfileServiceRemoveRequest) Header(key, value string) *AssignedCpuProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfileServiceRemoveRequest) Query(key, value string) *AssignedCpuProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfileServiceRemoveRequest) Async(async bool) *AssignedCpuProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AssignedCpuProfileServiceRemoveRequest) Send() (*AssignedCpuProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfileService.connection.URL(), p.AssignedCpuProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedCpuProfileServiceRemoveResponse), nil
}

func (p *AssignedCpuProfileServiceRemoveRequest) MustSend() *AssignedCpuProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedCpuProfileServiceRemoveResponse struct {
}

//
//
func (p *AssignedCpuProfileService) Remove() *AssignedCpuProfileServiceRemoveRequest {
	return &AssignedCpuProfileServiceRemoveRequest{AssignedCpuProfileService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedCpuProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedCpuProfileService) String() string {
	return fmt.Sprintf("AssignedCpuProfileService:%s", op.path)
}

//
//
type StorageServerConnectionsService struct {
	BaseService
}

func NewStorageServerConnectionsService(connection *Connection, path string) *StorageServerConnectionsService {
	var result StorageServerConnectionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new storage connection.
// For example, to create a new storage connection for the NFS server `mynfs.example.com` and NFS share
// `/export/mydata` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageconnections
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection>
//   <type>nfs</type>
//   <address>mynfs.example.com</address>
//   <path>/export/mydata</path>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_connection>
// ----
//
type StorageServerConnectionsServiceAddRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddRequest) Header(key, value string) *StorageServerConnectionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddRequest) Query(key, value string) *StorageServerConnectionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddRequest) Send() (*StorageServerConnectionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageConnectionWriteOne(writer, p.connection, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageServerConnectionsServiceAddResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddRequest) MustSend() *StorageServerConnectionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new storage connection.
// For example, to create a new storage connection for the NFS server `mynfs.example.com` and NFS share
// `/export/mydata` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageconnections
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection>
//   <type>nfs</type>
//   <address>mynfs.example.com</address>
//   <path>/export/mydata</path>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_connection>
// ----
//
type StorageServerConnectionsServiceAddResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Creates a new storage connection.
// For example, to create a new storage connection for the NFS server `mynfs.example.com` and NFS share
// `/export/mydata` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageconnections
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection>
//   <type>nfs</type>
//   <address>mynfs.example.com</address>
//   <path>/export/mydata</path>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_connection>
// ----
//
func (p *StorageServerConnectionsService) Add() *StorageServerConnectionsServiceAddRequest {
	return &StorageServerConnectionsServiceAddRequest{StorageServerConnectionsService: p}
}

//
// Add a Glusterfs storage connection to the system.
//
type StorageServerConnectionsServiceAddGlusterfsRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Header(key, value string) *StorageServerConnectionsServiceAddGlusterfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Query(key, value string) *StorageServerConnectionsServiceAddGlusterfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddGlusterfsRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Send() (*StorageServerConnectionsServiceAddGlusterfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/glusterfs", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddGlusterfsResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) MustSend() *StorageServerConnectionsServiceAddGlusterfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a Glusterfs storage connection to the system.
//
type StorageServerConnectionsServiceAddGlusterfsResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddGlusterfsResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddGlusterfsResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a Glusterfs storage connection to the system.
//
func (p *StorageServerConnectionsService) AddGlusterfs() *StorageServerConnectionsServiceAddGlusterfsRequest {
	return &StorageServerConnectionsServiceAddGlusterfsRequest{StorageServerConnectionsService: p}
}

//
// Add a iSCSI storage connection to the system.
//
type StorageServerConnectionsServiceAddIscsiRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Header(key, value string) *StorageServerConnectionsServiceAddIscsiRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Query(key, value string) *StorageServerConnectionsServiceAddIscsiRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddIscsiRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Send() (*StorageServerConnectionsServiceAddIscsiResponse, error) {
	rawURL := fmt.Sprintf("%s%s/iscsi", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddIscsiResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) MustSend() *StorageServerConnectionsServiceAddIscsiResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a iSCSI storage connection to the system.
//
type StorageServerConnectionsServiceAddIscsiResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddIscsiResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddIscsiResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a iSCSI storage connection to the system.
//
func (p *StorageServerConnectionsService) AddIscsi() *StorageServerConnectionsServiceAddIscsiRequest {
	return &StorageServerConnectionsServiceAddIscsiRequest{StorageServerConnectionsService: p}
}

//
// Returns the list of storage connections.
// The order of the returned list of connections isn't guaranteed.
//
type StorageServerConnectionsServiceListRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
	max                             *int64
}

func (p *StorageServerConnectionsServiceListRequest) Header(key, value string) *StorageServerConnectionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Query(key, value string) *StorageServerConnectionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Follow(follow string) *StorageServerConnectionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Max(max int64) *StorageServerConnectionsServiceListRequest {
	p.max = &max
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Send() (*StorageServerConnectionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &StorageServerConnectionsServiceListResponse{connections: result}, nil
}

func (p *StorageServerConnectionsServiceListRequest) MustSend() *StorageServerConnectionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of storage connections.
// The order of the returned list of connections isn't guaranteed.
//
type StorageServerConnectionsServiceListResponse struct {
	connections *StorageConnectionSlice
}

func (p *StorageServerConnectionsServiceListResponse) Connections() (*StorageConnectionSlice, bool) {
	if p.connections != nil {
		return p.connections, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceListResponse) MustConnections() *StorageConnectionSlice {
	if p.connections == nil {
		panic("connections in response does not exist")
	}
	return p.connections
}

//
// Returns the list of storage connections.
// The order of the returned list of connections isn't guaranteed.
//
func (p *StorageServerConnectionsService) List() *StorageServerConnectionsServiceListRequest {
	return &StorageServerConnectionsServiceListRequest{StorageServerConnectionsService: p}
}

//
// Add a local storage connection to the system.
//
type StorageServerConnectionsServiceAddLocalRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Header(key, value string) *StorageServerConnectionsServiceAddLocalRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Query(key, value string) *StorageServerConnectionsServiceAddLocalRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddLocalRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Send() (*StorageServerConnectionsServiceAddLocalResponse, error) {
	rawURL := fmt.Sprintf("%s%s/local", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddLocalResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddLocalRequest) MustSend() *StorageServerConnectionsServiceAddLocalResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a local storage connection to the system.
//
type StorageServerConnectionsServiceAddLocalResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddLocalResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddLocalResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a local storage connection to the system.
//
func (p *StorageServerConnectionsService) AddLocal() *StorageServerConnectionsServiceAddLocalRequest {
	return &StorageServerConnectionsServiceAddLocalRequest{StorageServerConnectionsService: p}
}

//
// Add a nfs storage connection to the system.
//
type StorageServerConnectionsServiceAddNfsRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Header(key, value string) *StorageServerConnectionsServiceAddNfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Query(key, value string) *StorageServerConnectionsServiceAddNfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddNfsRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Send() (*StorageServerConnectionsServiceAddNfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/nfs", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddNfsResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddNfsRequest) MustSend() *StorageServerConnectionsServiceAddNfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a nfs storage connection to the system.
//
type StorageServerConnectionsServiceAddNfsResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddNfsResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddNfsResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a nfs storage connection to the system.
//
func (p *StorageServerConnectionsService) AddNfs() *StorageServerConnectionsServiceAddNfsRequest {
	return &StorageServerConnectionsServiceAddNfsRequest{StorageServerConnectionsService: p}
}

//
// Add a vfs storage connection to the system.
//
type StorageServerConnectionsServiceAddVfsRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Header(key, value string) *StorageServerConnectionsServiceAddVfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Query(key, value string) *StorageServerConnectionsServiceAddVfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddVfsRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Send() (*StorageServerConnectionsServiceAddVfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vfs", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddVfsResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddVfsRequest) MustSend() *StorageServerConnectionsServiceAddVfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a vfs storage connection to the system.
//
type StorageServerConnectionsServiceAddVfsResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddVfsResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddVfsResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a vfs storage connection to the system.
//
func (p *StorageServerConnectionsService) AddVfs() *StorageServerConnectionsServiceAddVfsRequest {
	return &StorageServerConnectionsServiceAddVfsRequest{StorageServerConnectionsService: p}
}

//
//
func (op *StorageServerConnectionsService) StorageConnectionService(id string) *StorageServerConnectionService {
	return NewStorageServerConnectionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageServerConnectionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.StorageConnectionService(path), nil
	}
	return op.StorageConnectionService(path[:index]).Service(path[index+1:])
}

func (op *StorageServerConnectionsService) String() string {
	return fmt.Sprintf("StorageServerConnectionsService:%s", op.path)
}

//
//
type CopyableService struct {
	BaseService
}

func NewCopyableService(connection *Connection, path string) *CopyableService {
	var result CopyableService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type CopyableServiceCopyRequest struct {
	CopyableService *CopyableService
	header          map[string]string
	query           map[string]string
	async           *bool
}

func (p *CopyableServiceCopyRequest) Header(key, value string) *CopyableServiceCopyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CopyableServiceCopyRequest) Query(key, value string) *CopyableServiceCopyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CopyableServiceCopyRequest) Async(async bool) *CopyableServiceCopyRequest {
	p.async = &async
	return p
}

func (p *CopyableServiceCopyRequest) Send() (*CopyableServiceCopyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copy", p.CopyableService.connection.URL(), p.CopyableService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CopyableService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CopyableService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CopyableService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CopyableService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CopyableService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(CopyableServiceCopyResponse), nil
}

func (p *CopyableServiceCopyRequest) MustSend() *CopyableServiceCopyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type CopyableServiceCopyResponse struct {
}

//
//
func (p *CopyableService) Copy() *CopyableServiceCopyRequest {
	return &CopyableServiceCopyRequest{CopyableService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *CopyableService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *CopyableService) String() string {
	return fmt.Sprintf("CopyableService:%s", op.path)
}

//
// Provides information about virtual machine user sessions.
//
type VmSessionsService struct {
	BaseService
}

func NewVmSessionsService(connection *Connection, path string) *VmSessionsService {
	var result VmSessionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Lists all user sessions for this virtual machine.
// For example, to retrieve the session information for virtual machine `123` send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/sessions
// ----
// The response body will contain something like this:
// [source,xml]
// ----
// <sessions>
//   <session href="/ovirt-engine/api/vms/123/sessions/456" id="456">
//     <console_user>true</console_user>
//     <ip>
//       <address>192.168.122.1</address>
//     </ip>
//     <user href="/ovirt-engine/api/users/789" id="789"/>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </session>
//   ...
// </sessions>
// ----
// The order of the returned list of sessions isn't guaranteed.
//
type VmSessionsServiceListRequest struct {
	VmSessionsService *VmSessionsService
	header            map[string]string
	query             map[string]string
	follow            *string
	max               *int64
}

func (p *VmSessionsServiceListRequest) Header(key, value string) *VmSessionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmSessionsServiceListRequest) Query(key, value string) *VmSessionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmSessionsServiceListRequest) Follow(follow string) *VmSessionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmSessionsServiceListRequest) Max(max int64) *VmSessionsServiceListRequest {
	p.max = &max
	return p
}

func (p *VmSessionsServiceListRequest) Send() (*VmSessionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmSessionsService.connection.URL(), p.VmSessionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmSessionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmSessionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmSessionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmSessionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmSessionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSessionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmSessionsServiceListResponse{sessions: result}, nil
}

func (p *VmSessionsServiceListRequest) MustSend() *VmSessionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all user sessions for this virtual machine.
// For example, to retrieve the session information for virtual machine `123` send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/sessions
// ----
// The response body will contain something like this:
// [source,xml]
// ----
// <sessions>
//   <session href="/ovirt-engine/api/vms/123/sessions/456" id="456">
//     <console_user>true</console_user>
//     <ip>
//       <address>192.168.122.1</address>
//     </ip>
//     <user href="/ovirt-engine/api/users/789" id="789"/>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </session>
//   ...
// </sessions>
// ----
// The order of the returned list of sessions isn't guaranteed.
//
type VmSessionsServiceListResponse struct {
	sessions *SessionSlice
}

func (p *VmSessionsServiceListResponse) Sessions() (*SessionSlice, bool) {
	if p.sessions != nil {
		return p.sessions, true
	}
	return nil, false
}

func (p *VmSessionsServiceListResponse) MustSessions() *SessionSlice {
	if p.sessions == nil {
		panic("sessions in response does not exist")
	}
	return p.sessions
}

//
// Lists all user sessions for this virtual machine.
// For example, to retrieve the session information for virtual machine `123` send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/sessions
// ----
// The response body will contain something like this:
// [source,xml]
// ----
// <sessions>
//   <session href="/ovirt-engine/api/vms/123/sessions/456" id="456">
//     <console_user>true</console_user>
//     <ip>
//       <address>192.168.122.1</address>
//     </ip>
//     <user href="/ovirt-engine/api/users/789" id="789"/>
//     <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   </session>
//   ...
// </sessions>
// ----
// The order of the returned list of sessions isn't guaranteed.
//
func (p *VmSessionsService) List() *VmSessionsServiceListRequest {
	return &VmSessionsServiceListRequest{VmSessionsService: p}
}

//
// Reference to the service that manages a specific session.
//
func (op *VmSessionsService) SessionService(id string) *VmSessionService {
	return NewVmSessionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmSessionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.SessionService(path), nil
	}
	return op.SessionService(path[:index]).Service(path[index+1:])
}

func (op *VmSessionsService) String() string {
	return fmt.Sprintf("VmSessionsService:%s", op.path)
}

//
//
type VmDiskService struct {
	BaseService
}

func NewVmDiskService(connection *Connection, path string) *VmDiskService {
	var result VmDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type VmDiskServiceActivateRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceActivateRequest) Header(key, value string) *VmDiskServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceActivateRequest) Query(key, value string) *VmDiskServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceActivateRequest) Async(async bool) *VmDiskServiceActivateRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceActivateRequest) Send() (*VmDiskServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceActivateResponse), nil
}

func (p *VmDiskServiceActivateRequest) MustSend() *VmDiskServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceActivateResponse struct {
}

//
//
func (p *VmDiskService) Activate() *VmDiskServiceActivateRequest {
	return &VmDiskServiceActivateRequest{VmDiskService: p}
}

//
//
type VmDiskServiceDeactivateRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceDeactivateRequest) Header(key, value string) *VmDiskServiceDeactivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceDeactivateRequest) Query(key, value string) *VmDiskServiceDeactivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceDeactivateRequest) Async(async bool) *VmDiskServiceDeactivateRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceDeactivateRequest) Send() (*VmDiskServiceDeactivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/deactivate", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceDeactivateResponse), nil
}

func (p *VmDiskServiceDeactivateRequest) MustSend() *VmDiskServiceDeactivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceDeactivateResponse struct {
}

//
//
func (p *VmDiskService) Deactivate() *VmDiskServiceDeactivateRequest {
	return &VmDiskServiceDeactivateRequest{VmDiskService: p}
}

//
//
type VmDiskServiceExportRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	filter        *bool
}

func (p *VmDiskServiceExportRequest) Header(key, value string) *VmDiskServiceExportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceExportRequest) Query(key, value string) *VmDiskServiceExportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceExportRequest) Async(async bool) *VmDiskServiceExportRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceExportRequest) Filter(filter bool) *VmDiskServiceExportRequest {
	p.filter = &filter
	return p
}

func (p *VmDiskServiceExportRequest) Send() (*VmDiskServiceExportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/export", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceExportResponse), nil
}

func (p *VmDiskServiceExportRequest) MustSend() *VmDiskServiceExportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceExportResponse struct {
}

//
//
func (p *VmDiskService) Export() *VmDiskServiceExportRequest {
	return &VmDiskServiceExportRequest{VmDiskService: p}
}

//
//
type VmDiskServiceGetRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	follow        *string
}

func (p *VmDiskServiceGetRequest) Header(key, value string) *VmDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceGetRequest) Query(key, value string) *VmDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceGetRequest) Follow(follow string) *VmDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmDiskServiceGetRequest) Send() (*VmDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmDiskServiceGetResponse{disk: result}, nil
}

func (p *VmDiskServiceGetRequest) MustSend() *VmDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceGetResponse struct {
	disk *Disk
}

func (p *VmDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *VmDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
//
func (p *VmDiskService) Get() *VmDiskServiceGetRequest {
	return &VmDiskServiceGetRequest{VmDiskService: p}
}

//
//
type VmDiskServiceMoveRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	filter        *bool
}

func (p *VmDiskServiceMoveRequest) Header(key, value string) *VmDiskServiceMoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceMoveRequest) Query(key, value string) *VmDiskServiceMoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceMoveRequest) Async(async bool) *VmDiskServiceMoveRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceMoveRequest) Filter(filter bool) *VmDiskServiceMoveRequest {
	p.filter = &filter
	return p
}

func (p *VmDiskServiceMoveRequest) Send() (*VmDiskServiceMoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/move", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceMoveResponse), nil
}

func (p *VmDiskServiceMoveRequest) MustSend() *VmDiskServiceMoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceMoveResponse struct {
}

//
//
func (p *VmDiskService) Move() *VmDiskServiceMoveRequest {
	return &VmDiskServiceMoveRequest{VmDiskService: p}
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
type VmDiskServiceReduceRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceReduceRequest) Header(key, value string) *VmDiskServiceReduceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceReduceRequest) Query(key, value string) *VmDiskServiceReduceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceReduceRequest) Async(async bool) *VmDiskServiceReduceRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceReduceRequest) Send() (*VmDiskServiceReduceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/reduce", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceReduceResponse), nil
}

func (p *VmDiskServiceReduceRequest) MustSend() *VmDiskServiceReduceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
type VmDiskServiceReduceResponse struct {
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
func (p *VmDiskService) Reduce() *VmDiskServiceReduceRequest {
	return &VmDiskServiceReduceRequest{VmDiskService: p}
}

//
// Detach the disk from the virtual machine.
// NOTE: In version 3 of the API this used to also remove the disk completely from the system, but starting with
// version 4 it doesn't. If you need to remove it completely use the <<services/disk/methods/remove,remove
// method of the top level disk service>>.
//
type VmDiskServiceRemoveRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceRemoveRequest) Header(key, value string) *VmDiskServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceRemoveRequest) Query(key, value string) *VmDiskServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceRemoveRequest) Async(async bool) *VmDiskServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceRemoveRequest) Send() (*VmDiskServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VmDiskServiceRemoveResponse), nil
}

func (p *VmDiskServiceRemoveRequest) MustSend() *VmDiskServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Detach the disk from the virtual machine.
// NOTE: In version 3 of the API this used to also remove the disk completely from the system, but starting with
// version 4 it doesn't. If you need to remove it completely use the <<services/disk/methods/remove,remove
// method of the top level disk service>>.
//
type VmDiskServiceRemoveResponse struct {
}

//
// Detach the disk from the virtual machine.
// NOTE: In version 3 of the API this used to also remove the disk completely from the system, but starting with
// version 4 it doesn't. If you need to remove it completely use the <<services/disk/methods/remove,remove
// method of the top level disk service>>.
//
func (p *VmDiskService) Remove() *VmDiskServiceRemoveRequest {
	return &VmDiskServiceRemoveRequest{VmDiskService: p}
}

//
//
type VmDiskServiceUpdateRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	disk          *Disk
}

func (p *VmDiskServiceUpdateRequest) Header(key, value string) *VmDiskServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceUpdateRequest) Query(key, value string) *VmDiskServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceUpdateRequest) Async(async bool) *VmDiskServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceUpdateRequest) Disk(disk *Disk) *VmDiskServiceUpdateRequest {
	p.disk = disk
	return p
}

func (p *VmDiskServiceUpdateRequest) Send() (*VmDiskServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmDiskServiceUpdateResponse{disk: result}, nil
}

func (p *VmDiskServiceUpdateRequest) MustSend() *VmDiskServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceUpdateResponse struct {
	disk *Disk
}

func (p *VmDiskServiceUpdateResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *VmDiskServiceUpdateResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
//
func (p *VmDiskService) Update() *VmDiskServiceUpdateRequest {
	return &VmDiskServiceUpdateRequest{VmDiskService: p}
}

//
//
func (op *VmDiskService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *VmDiskService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmDiskService) String() string {
	return fmt.Sprintf("VmDiskService:%s", op.path)
}

//
// A service to access a particular device of a host.
//
type HostDeviceService struct {
	BaseService
}

func NewHostDeviceService(connection *Connection, path string) *HostDeviceService {
	var result HostDeviceService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
type HostDeviceServiceGetRequest struct {
	HostDeviceService *HostDeviceService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *HostDeviceServiceGetRequest) Header(key, value string) *HostDeviceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostDeviceServiceGetRequest) Query(key, value string) *HostDeviceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostDeviceServiceGetRequest) Follow(follow string) *HostDeviceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostDeviceServiceGetRequest) Send() (*HostDeviceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostDeviceService.connection.URL(), p.HostDeviceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostDeviceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostDeviceServiceGetResponse{device: result}, nil
}

func (p *HostDeviceServiceGetRequest) MustSend() *HostDeviceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
type HostDeviceServiceGetResponse struct {
	device *HostDevice
}

func (p *HostDeviceServiceGetResponse) Device() (*HostDevice, bool) {
	if p.device != nil {
		return p.device, true
	}
	return nil, false
}

func (p *HostDeviceServiceGetResponse) MustDevice() *HostDevice {
	if p.device == nil {
		panic("device in response does not exist")
	}
	return p.device
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
func (p *HostDeviceService) Get() *HostDeviceServiceGetRequest {
	return &HostDeviceServiceGetRequest{HostDeviceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostDeviceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostDeviceService) String() string {
	return fmt.Sprintf("HostDeviceService:%s", op.path)
}

//
//
type AssignedVnicProfileService struct {
	BaseService
}

func NewAssignedVnicProfileService(connection *Connection, path string) *AssignedVnicProfileService {
	var result AssignedVnicProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type AssignedVnicProfileServiceGetRequest struct {
	AssignedVnicProfileService *AssignedVnicProfileService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
}

func (p *AssignedVnicProfileServiceGetRequest) Header(key, value string) *AssignedVnicProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedVnicProfileServiceGetRequest) Query(key, value string) *AssignedVnicProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedVnicProfileServiceGetRequest) Follow(follow string) *AssignedVnicProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedVnicProfileServiceGetRequest) Send() (*AssignedVnicProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedVnicProfileService.connection.URL(), p.AssignedVnicProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedVnicProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedVnicProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedVnicProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedVnicProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedVnicProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedVnicProfileServiceGetResponse{profile: result}, nil
}

func (p *AssignedVnicProfileServiceGetRequest) MustSend() *AssignedVnicProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedVnicProfileServiceGetResponse struct {
	profile *VnicProfile
}

func (p *AssignedVnicProfileServiceGetResponse) Profile() (*VnicProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedVnicProfileServiceGetResponse) MustProfile() *VnicProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *AssignedVnicProfileService) Get() *AssignedVnicProfileServiceGetRequest {
	return &AssignedVnicProfileServiceGetRequest{AssignedVnicProfileService: p}
}

//
//
type AssignedVnicProfileServiceRemoveRequest struct {
	AssignedVnicProfileService *AssignedVnicProfileService
	header                     map[string]string
	query                      map[string]string
	async                      *bool
}

func (p *AssignedVnicProfileServiceRemoveRequest) Header(key, value string) *AssignedVnicProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedVnicProfileServiceRemoveRequest) Query(key, value string) *AssignedVnicProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedVnicProfileServiceRemoveRequest) Async(async bool) *AssignedVnicProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AssignedVnicProfileServiceRemoveRequest) Send() (*AssignedVnicProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedVnicProfileService.connection.URL(), p.AssignedVnicProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedVnicProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedVnicProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedVnicProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedVnicProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedVnicProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedVnicProfileServiceRemoveResponse), nil
}

func (p *AssignedVnicProfileServiceRemoveRequest) MustSend() *AssignedVnicProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AssignedVnicProfileServiceRemoveResponse struct {
}

//
//
func (p *AssignedVnicProfileService) Remove() *AssignedVnicProfileServiceRemoveRequest {
	return &AssignedVnicProfileServiceRemoveRequest{AssignedVnicProfileService: p}
}

//
//
func (op *AssignedVnicProfileService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedVnicProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedVnicProfileService) String() string {
	return fmt.Sprintf("AssignedVnicProfileService:%s", op.path)
}

//
//
type CpuProfileService struct {
	BaseService
}

func NewCpuProfileService(connection *Connection, path string) *CpuProfileService {
	var result CpuProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type CpuProfileServiceGetRequest struct {
	CpuProfileService *CpuProfileService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *CpuProfileServiceGetRequest) Header(key, value string) *CpuProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfileServiceGetRequest) Query(key, value string) *CpuProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfileServiceGetRequest) Follow(follow string) *CpuProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *CpuProfileServiceGetRequest) Send() (*CpuProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfileService.connection.URL(), p.CpuProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &CpuProfileServiceGetResponse{profile: result}, nil
}

func (p *CpuProfileServiceGetRequest) MustSend() *CpuProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type CpuProfileServiceGetResponse struct {
	profile *CpuProfile
}

func (p *CpuProfileServiceGetResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *CpuProfileServiceGetResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *CpuProfileService) Get() *CpuProfileServiceGetRequest {
	return &CpuProfileServiceGetRequest{CpuProfileService: p}
}

//
//
type CpuProfileServiceRemoveRequest struct {
	CpuProfileService *CpuProfileService
	header            map[string]string
	query             map[string]string
	async             *bool
}

func (p *CpuProfileServiceRemoveRequest) Header(key, value string) *CpuProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfileServiceRemoveRequest) Query(key, value string) *CpuProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfileServiceRemoveRequest) Async(async bool) *CpuProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *CpuProfileServiceRemoveRequest) Send() (*CpuProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfileService.connection.URL(), p.CpuProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(CpuProfileServiceRemoveResponse), nil
}

func (p *CpuProfileServiceRemoveRequest) MustSend() *CpuProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type CpuProfileServiceRemoveResponse struct {
}

//
//
func (p *CpuProfileService) Remove() *CpuProfileServiceRemoveRequest {
	return &CpuProfileServiceRemoveRequest{CpuProfileService: p}
}

//
// Update the specified cpu profile in the system.
//
type CpuProfileServiceUpdateRequest struct {
	CpuProfileService *CpuProfileService
	header            map[string]string
	query             map[string]string
	async             *bool
	profile           *CpuProfile
}

func (p *CpuProfileServiceUpdateRequest) Header(key, value string) *CpuProfileServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CpuProfileServiceUpdateRequest) Query(key, value string) *CpuProfileServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CpuProfileServiceUpdateRequest) Async(async bool) *CpuProfileServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *CpuProfileServiceUpdateRequest) Profile(profile *CpuProfile) *CpuProfileServiceUpdateRequest {
	p.profile = profile
	return p
}

func (p *CpuProfileServiceUpdateRequest) Send() (*CpuProfileServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.CpuProfileService.connection.URL(), p.CpuProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLCpuProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CpuProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CpuProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CpuProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CpuProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CpuProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &CpuProfileServiceUpdateResponse{profile: result}, nil
}

func (p *CpuProfileServiceUpdateRequest) MustSend() *CpuProfileServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified cpu profile in the system.
//
type CpuProfileServiceUpdateResponse struct {
	profile *CpuProfile
}

func (p *CpuProfileServiceUpdateResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *CpuProfileServiceUpdateResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Update the specified cpu profile in the system.
//
func (p *CpuProfileService) Update() *CpuProfileServiceUpdateRequest {
	return &CpuProfileServiceUpdateRequest{CpuProfileService: p}
}

//
//
func (op *CpuProfileService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *CpuProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *CpuProfileService) String() string {
	return fmt.Sprintf("CpuProfileService:%s", op.path)
}

//
// A service to manage a specific data center network.
//
type DataCenterNetworkService struct {
	BaseService
}

func NewDataCenterNetworkService(connection *Connection, path string) *DataCenterNetworkService {
	var result DataCenterNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the data center network details.
//
type DataCenterNetworkServiceGetRequest struct {
	DataCenterNetworkService *DataCenterNetworkService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *DataCenterNetworkServiceGetRequest) Header(key, value string) *DataCenterNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworkServiceGetRequest) Query(key, value string) *DataCenterNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworkServiceGetRequest) Follow(follow string) *DataCenterNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DataCenterNetworkServiceGetRequest) Send() (*DataCenterNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworkService.connection.URL(), p.DataCenterNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworkServiceGetResponse{network: result}, nil
}

func (p *DataCenterNetworkServiceGetRequest) MustSend() *DataCenterNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the data center network details.
//
type DataCenterNetworkServiceGetResponse struct {
	network *Network
}

func (p *DataCenterNetworkServiceGetResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *DataCenterNetworkServiceGetResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Retrieves the data center network details.
//
func (p *DataCenterNetworkService) Get() *DataCenterNetworkServiceGetRequest {
	return &DataCenterNetworkServiceGetRequest{DataCenterNetworkService: p}
}

//
// Removes the network.
//
type DataCenterNetworkServiceRemoveRequest struct {
	DataCenterNetworkService *DataCenterNetworkService
	header                   map[string]string
	query                    map[string]string
}

func (p *DataCenterNetworkServiceRemoveRequest) Header(key, value string) *DataCenterNetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworkServiceRemoveRequest) Query(key, value string) *DataCenterNetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworkServiceRemoveRequest) Send() (*DataCenterNetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworkService.connection.URL(), p.DataCenterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DataCenterNetworkServiceRemoveResponse), nil
}

func (p *DataCenterNetworkServiceRemoveRequest) MustSend() *DataCenterNetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the network.
//
type DataCenterNetworkServiceRemoveResponse struct {
}

//
// Removes the network.
//
func (p *DataCenterNetworkService) Remove() *DataCenterNetworkServiceRemoveRequest {
	return &DataCenterNetworkServiceRemoveRequest{DataCenterNetworkService: p}
}

//
// Updates the network in the data center.
//
type DataCenterNetworkServiceUpdateRequest struct {
	DataCenterNetworkService *DataCenterNetworkService
	header                   map[string]string
	query                    map[string]string
	network                  *Network
}

func (p *DataCenterNetworkServiceUpdateRequest) Header(key, value string) *DataCenterNetworkServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworkServiceUpdateRequest) Query(key, value string) *DataCenterNetworkServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworkServiceUpdateRequest) Network(network *Network) *DataCenterNetworkServiceUpdateRequest {
	p.network = network
	return p
}

func (p *DataCenterNetworkServiceUpdateRequest) Send() (*DataCenterNetworkServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworkService.connection.URL(), p.DataCenterNetworkService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworkServiceUpdateResponse{network: result}, nil
}

func (p *DataCenterNetworkServiceUpdateRequest) MustSend() *DataCenterNetworkServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the network in the data center.
//
type DataCenterNetworkServiceUpdateResponse struct {
	network *Network
}

func (p *DataCenterNetworkServiceUpdateResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *DataCenterNetworkServiceUpdateResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Updates the network in the data center.
//
func (p *DataCenterNetworkService) Update() *DataCenterNetworkServiceUpdateRequest {
	return &DataCenterNetworkServiceUpdateRequest{DataCenterNetworkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DataCenterNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DataCenterNetworkService) String() string {
	return fmt.Sprintf("DataCenterNetworkService:%s", op.path)
}

//
// Represents a readonly network filters sub-collection.
// The network filter enables to filter packets send to/from the VM's nic according to defined rules.
// For more information please refer to <<services/network_filter,NetworkFilter>> service documentation
// Network filters are supported in different versions, starting from version 3.0.
// A network filter is defined for each vnic profile.
// A vnic profile is defined for a specific network.
// A network can be assigned to several different clusters. In the future, each network will be defined in
// cluster level.
// Currently, each network is being defined at data center level. Potential network filters for each network
// are determined by the network's data center compatibility version V.
// V must be >= the network filter version in order to configure this network filter for a specific network.
// Please note, that if a network is assigned to cluster with a version supporting a network filter, the filter
// may not be available due to the data center version being smaller then the network filter's version.
// Example of listing all of the supported network filters for a specific cluster:
// [source]
// ----
// GET http://localhost:8080/ovirt-engine/api/clusters/{cluster:id}/networkfilters
// ----
// Output:
// [source,xml]
// ----
// <network_filters>
//   <network_filter id="00000019-0019-0019-0019-00000000026c">
//     <name>example-network-filter-a</name>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
//   <network_filter id="00000019-0019-0019-0019-00000000026b">
//     <name>example-network-filter-b</name>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
//   <network_filter id="00000019-0019-0019-0019-00000000026a">
//     <name>example-network-filter-a</name>
//     <version>
//       <major>3</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
// </network_filters>
// ----
//
type NetworkFiltersService struct {
	BaseService
}

func NewNetworkFiltersService(connection *Connection, path string) *NetworkFiltersService {
	var result NetworkFiltersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
type NetworkFiltersServiceListRequest struct {
	NetworkFiltersService *NetworkFiltersService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *NetworkFiltersServiceListRequest) Header(key, value string) *NetworkFiltersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkFiltersServiceListRequest) Query(key, value string) *NetworkFiltersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkFiltersServiceListRequest) Follow(follow string) *NetworkFiltersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *NetworkFiltersServiceListRequest) Send() (*NetworkFiltersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkFiltersService.connection.URL(), p.NetworkFiltersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkFiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkFiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkFiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkFiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkFiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkFilterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &NetworkFiltersServiceListResponse{filters: result}, nil
}

func (p *NetworkFiltersServiceListRequest) MustSend() *NetworkFiltersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
type NetworkFiltersServiceListResponse struct {
	filters *NetworkFilterSlice
}

func (p *NetworkFiltersServiceListResponse) Filters() (*NetworkFilterSlice, bool) {
	if p.filters != nil {
		return p.filters, true
	}
	return nil, false
}

func (p *NetworkFiltersServiceListResponse) MustFilters() *NetworkFilterSlice {
	if p.filters == nil {
		panic("filters in response does not exist")
	}
	return p.filters
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
func (p *NetworkFiltersService) List() *NetworkFiltersServiceListRequest {
	return &NetworkFiltersServiceListRequest{NetworkFiltersService: p}
}

//
//
func (op *NetworkFiltersService) NetworkFilterService(id string) *NetworkFilterService {
	return NewNetworkFilterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkFiltersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkFilterService(path), nil
	}
	return op.NetworkFilterService(path[:index]).Service(path[index+1:])
}

func (op *NetworkFiltersService) String() string {
	return fmt.Sprintf("NetworkFiltersService:%s", op.path)
}

//
//
type SchedulingPolicyUnitService struct {
	BaseService
}

func NewSchedulingPolicyUnitService(connection *Connection, path string) *SchedulingPolicyUnitService {
	var result SchedulingPolicyUnitService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SchedulingPolicyUnitServiceGetRequest struct {
	SchedulingPolicyUnitService *SchedulingPolicyUnitService
	header                      map[string]string
	query                       map[string]string
	filter                      *bool
	follow                      *string
}

func (p *SchedulingPolicyUnitServiceGetRequest) Header(key, value string) *SchedulingPolicyUnitServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Query(key, value string) *SchedulingPolicyUnitServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Filter(filter bool) *SchedulingPolicyUnitServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Follow(follow string) *SchedulingPolicyUnitServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Send() (*SchedulingPolicyUnitServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyUnitService.connection.URL(), p.SchedulingPolicyUnitService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyUnitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyUnitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyUnitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyUnitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyUnitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyUnitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SchedulingPolicyUnitServiceGetResponse{unit: result}, nil
}

func (p *SchedulingPolicyUnitServiceGetRequest) MustSend() *SchedulingPolicyUnitServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyUnitServiceGetResponse struct {
	unit *SchedulingPolicyUnit
}

func (p *SchedulingPolicyUnitServiceGetResponse) Unit() (*SchedulingPolicyUnit, bool) {
	if p.unit != nil {
		return p.unit, true
	}
	return nil, false
}

func (p *SchedulingPolicyUnitServiceGetResponse) MustUnit() *SchedulingPolicyUnit {
	if p.unit == nil {
		panic("unit in response does not exist")
	}
	return p.unit
}

//
//
func (p *SchedulingPolicyUnitService) Get() *SchedulingPolicyUnitServiceGetRequest {
	return &SchedulingPolicyUnitServiceGetRequest{SchedulingPolicyUnitService: p}
}

//
//
type SchedulingPolicyUnitServiceRemoveRequest struct {
	SchedulingPolicyUnitService *SchedulingPolicyUnitService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Header(key, value string) *SchedulingPolicyUnitServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Query(key, value string) *SchedulingPolicyUnitServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Async(async bool) *SchedulingPolicyUnitServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Send() (*SchedulingPolicyUnitServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyUnitService.connection.URL(), p.SchedulingPolicyUnitService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyUnitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyUnitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyUnitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyUnitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyUnitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(SchedulingPolicyUnitServiceRemoveResponse), nil
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) MustSend() *SchedulingPolicyUnitServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyUnitServiceRemoveResponse struct {
}

//
//
func (p *SchedulingPolicyUnitService) Remove() *SchedulingPolicyUnitServiceRemoveRequest {
	return &SchedulingPolicyUnitServiceRemoveRequest{SchedulingPolicyUnitService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SchedulingPolicyUnitService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SchedulingPolicyUnitService) String() string {
	return fmt.Sprintf("SchedulingPolicyUnitService:%s", op.path)
}

//
// Describes how an external network provider is provisioned by the system on the host.
//
type ExternalNetworkProviderConfigurationService struct {
	BaseService
}

func NewExternalNetworkProviderConfigurationService(connection *Connection, path string) *ExternalNetworkProviderConfigurationService {
	var result ExternalNetworkProviderConfigurationService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the information about an external network provider on the host.
//
type ExternalNetworkProviderConfigurationServiceGetRequest struct {
	ExternalNetworkProviderConfigurationService *ExternalNetworkProviderConfigurationService
	header                                      map[string]string
	query                                       map[string]string
	follow                                      *string
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Header(key, value string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Query(key, value string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Follow(follow string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Send() (*ExternalNetworkProviderConfigurationServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalNetworkProviderConfigurationService.connection.URL(), p.ExternalNetworkProviderConfigurationService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalNetworkProviderConfigurationService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalNetworkProviderConfigurationService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalNetworkProviderConfigurationService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalNetworkProviderConfigurationService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalNetworkProviderConfigurationService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalNetworkProviderConfigurationReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalNetworkProviderConfigurationServiceGetResponse{configuration: result}, nil
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) MustSend() *ExternalNetworkProviderConfigurationServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the information about an external network provider on the host.
//
type ExternalNetworkProviderConfigurationServiceGetResponse struct {
	configuration *ExternalNetworkProviderConfiguration
}

func (p *ExternalNetworkProviderConfigurationServiceGetResponse) Configuration() (*ExternalNetworkProviderConfiguration, bool) {
	if p.configuration != nil {
		return p.configuration, true
	}
	return nil, false
}

func (p *ExternalNetworkProviderConfigurationServiceGetResponse) MustConfiguration() *ExternalNetworkProviderConfiguration {
	if p.configuration == nil {
		panic("configuration in response does not exist")
	}
	return p.configuration
}

//
// Returns the information about an external network provider on the host.
//
func (p *ExternalNetworkProviderConfigurationService) Get() *ExternalNetworkProviderConfigurationServiceGetRequest {
	return &ExternalNetworkProviderConfigurationServiceGetRequest{ExternalNetworkProviderConfigurationService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalNetworkProviderConfigurationService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalNetworkProviderConfigurationService) String() string {
	return fmt.Sprintf("ExternalNetworkProviderConfigurationService:%s", op.path)
}

//
// This service manages the collection of all vNIC profiles.
//
type VnicProfilesService struct {
	BaseService
}

func NewVnicProfilesService(connection *Connection, path string) *VnicProfilesService {
	var result VnicProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a vNIC profile.
// For example to add vNIC profile `123` to network `456` send a request to:
// [source]
// ----
// POST /ovirt-engine/api/networks/456/vnicprofiles
// ----
// With the following body:
// [source,xml]
// ----
// <vnic_profile id="123">
//   <name>new_vNIC_name</name>
//   <pass_through>
//     <mode>disabled</mode>
//   </pass_through>
//   <port_mirroring>false</port_mirroring>
// </vnic_profile>
// ----
// Please note that there is a default network filter to each VNIC profile.
// For more details of how the default network filter is calculated please refer to
// the documentation in <<services/network_filters,NetworkFilters>>.
// NOTE: The automatically created vNIC profile for the external network will be without network filter.
// The output of creating a new VNIC profile depends in the  body  arguments that were given.
// In case no network filter was given, the default network filter will be configured. For example:
// [source,xml]
// ----
// <vnic_profile href="/ovirt-engine/api/vnicprofiles/123" id="123">
//   <name>new_vNIC_name</name>
//   <link href="/ovirt-engine/api/vnicprofiles/123/permissions" rel="permissions"/>
//   <pass_through>
//     <mode>disabled</mode>
//   </pass_through>
//   <port_mirroring>false</port_mirroring>
//   <network href="/ovirt-engine/api/networks/456" id="456"/>
//   <network_filter href="/ovirt-engine/api/networkfilters/789" id="789"/>
// </vnic_profile>
// ----
// In case an empty network filter was given, no network filter will be configured for the specific VNIC profile
// regardless of the VNIC profile's default network filter. For example:
// [source,xml]
// ----
// <vnic_profile>
//   <name>no_network_filter</name>
//   <network_filter/>
// </vnic_profile>
// ----
// In case that a specific valid network filter id was given, the VNIC profile will be configured with the given
// network filter regardless of the VNIC profiles's default network filter. For example:
// [source,xml]
// ----
// <vnic_profile>
//   <name>user_choice_network_filter</name>
//   <network_filter id= "0000001b-001b-001b-001b-0000000001d5"/>
// </vnic_profile>
// ----
//
type VnicProfilesServiceAddRequest struct {
	VnicProfilesService *VnicProfilesService
	header              map[string]string
	query               map[string]string
	profile             *VnicProfile
}

func (p *VnicProfilesServiceAddRequest) Header(key, value string) *VnicProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VnicProfilesServiceAddRequest) Query(key, value string) *VnicProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VnicProfilesServiceAddRequest) Profile(profile *VnicProfile) *VnicProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *VnicProfilesServiceAddRequest) Send() (*VnicProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VnicProfilesService.connection.URL(), p.VnicProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVnicProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VnicProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VnicProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VnicProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VnicProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VnicProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VnicProfilesServiceAddResponse{profile: result}, nil
}

func (p *VnicProfilesServiceAddRequest) MustSend() *VnicProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a vNIC profile.
// For example to add vNIC profile `123` to network `456` send a request to:
// [source]
// ----
// POST /ovirt-engine/api/networks/456/vnicprofiles
// ----
// With the following body:
// [source,xml]
// ----
// <vnic_profile id="123">
//   <name>new_vNIC_name</name>
//   <pass_through>
//     <mode>disabled</mode>
//   </pass_through>
//   <port_mirroring>false</port_mirroring>
// </vnic_profile>
// ----
// Please note that there is a default network filter to each VNIC profile.
// For more details of how the default network filter is calculated please refer to
// the documentation in <<services/network_filters,NetworkFilters>>.
// NOTE: The automatically created vNIC profile for the external network will be without network filter.
// The output of creating a new VNIC profile depends in the  body  arguments that were given.
// In case no network filter was given, the default network filter will be configured. For example:
// [source,xml]
// ----
// <vnic_profile href="/ovirt-engine/api/vnicprofiles/123" id="123">
//   <name>new_vNIC_name</name>
//   <link href="/ovirt-engine/api/vnicprofiles/123/permissions" rel="permissions"/>
//   <pass_through>
//     <mode>disabled</mode>
//   </pass_through>
//   <port_mirroring>false</port_mirroring>
//   <network href="/ovirt-engine/api/networks/456" id="456"/>
//   <network_filter href="/ovirt-engine/api/networkfilters/789" id="789"/>
// </vnic_profile>
// ----
// In case an empty network filter was given, no network filter will be configured for the specific VNIC profile
// regardless of the VNIC profile's default network filter. For example:
// [source,xml]
// ----
// <vnic_profile>
//   <name>no_network_filter</name>
//   <network_filter/>
// </vnic_profile>
// ----
// In case that a specific valid network filter id was given, the VNIC profile will be configured with the given
// network filter regardless of the VNIC profiles's default network filter. For example:
// [source,xml]
// ----
// <vnic_profile>
//   <name>user_choice_network_filter</name>
//   <network_filter id= "0000001b-001b-001b-001b-0000000001d5"/>
// </vnic_profile>
// ----
//
type VnicProfilesServiceAddResponse struct {
	profile *VnicProfile
}

func (p *VnicProfilesServiceAddResponse) Profile() (*VnicProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *VnicProfilesServiceAddResponse) MustProfile() *VnicProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a vNIC profile.
// For example to add vNIC profile `123` to network `456` send a request to:
// [source]
// ----
// POST /ovirt-engine/api/networks/456/vnicprofiles
// ----
// With the following body:
// [source,xml]
// ----
// <vnic_profile id="123">
//   <name>new_vNIC_name</name>
//   <pass_through>
//     <mode>disabled</mode>
//   </pass_through>
//   <port_mirroring>false</port_mirroring>
// </vnic_profile>
// ----
// Please note that there is a default network filter to each VNIC profile.
// For more details of how the default network filter is calculated please refer to
// the documentation in <<services/network_filters,NetworkFilters>>.
// NOTE: The automatically created vNIC profile for the external network will be without network filter.
// The output of creating a new VNIC profile depends in the  body  arguments that were given.
// In case no network filter was given, the default network filter will be configured. For example:
// [source,xml]
// ----
// <vnic_profile href="/ovirt-engine/api/vnicprofiles/123" id="123">
//   <name>new_vNIC_name</name>
//   <link href="/ovirt-engine/api/vnicprofiles/123/permissions" rel="permissions"/>
//   <pass_through>
//     <mode>disabled</mode>
//   </pass_through>
//   <port_mirroring>false</port_mirroring>
//   <network href="/ovirt-engine/api/networks/456" id="456"/>
//   <network_filter href="/ovirt-engine/api/networkfilters/789" id="789"/>
// </vnic_profile>
// ----
// In case an empty network filter was given, no network filter will be configured for the specific VNIC profile
// regardless of the VNIC profile's default network filter. For example:
// [source,xml]
// ----
// <vnic_profile>
//   <name>no_network_filter</name>
//   <network_filter/>
// </vnic_profile>
// ----
// In case that a specific valid network filter id was given, the VNIC profile will be configured with the given
// network filter regardless of the VNIC profiles's default network filter. For example:
// [source,xml]
// ----
// <vnic_profile>
//   <name>user_choice_network_filter</name>
//   <network_filter id= "0000001b-001b-001b-001b-0000000001d5"/>
// </vnic_profile>
// ----
//
func (p *VnicProfilesService) Add() *VnicProfilesServiceAddRequest {
	return &VnicProfilesServiceAddRequest{VnicProfilesService: p}
}

//
// List all vNIC profiles.
// The order of the returned list of vNIC profiles isn't guaranteed.
//
type VnicProfilesServiceListRequest struct {
	VnicProfilesService *VnicProfilesService
	header              map[string]string
	query               map[string]string
	follow              *string
	max                 *int64
}

func (p *VnicProfilesServiceListRequest) Header(key, value string) *VnicProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VnicProfilesServiceListRequest) Query(key, value string) *VnicProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VnicProfilesServiceListRequest) Follow(follow string) *VnicProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VnicProfilesServiceListRequest) Max(max int64) *VnicProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *VnicProfilesServiceListRequest) Send() (*VnicProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VnicProfilesService.connection.URL(), p.VnicProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VnicProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VnicProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VnicProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VnicProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VnicProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VnicProfilesServiceListResponse{profiles: result}, nil
}

func (p *VnicProfilesServiceListRequest) MustSend() *VnicProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all vNIC profiles.
// The order of the returned list of vNIC profiles isn't guaranteed.
//
type VnicProfilesServiceListResponse struct {
	profiles *VnicProfileSlice
}

func (p *VnicProfilesServiceListResponse) Profiles() (*VnicProfileSlice, bool) {
	if p.profiles != nil {
		return p.profiles, true
	}
	return nil, false
}

func (p *VnicProfilesServiceListResponse) MustProfiles() *VnicProfileSlice {
	if p.profiles == nil {
		panic("profiles in response does not exist")
	}
	return p.profiles
}

//
// List all vNIC profiles.
// The order of the returned list of vNIC profiles isn't guaranteed.
//
func (p *VnicProfilesService) List() *VnicProfilesServiceListRequest {
	return &VnicProfilesServiceListRequest{VnicProfilesService: p}
}

//
//
func (op *VnicProfilesService) ProfileService(id string) *VnicProfileService {
	return NewVnicProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VnicProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProfileService(path), nil
	}
	return op.ProfileService(path[:index]).Service(path[index+1:])
}

func (op *VnicProfilesService) String() string {
	return fmt.Sprintf("VnicProfilesService:%s", op.path)
}

//
// This service is used to list and manipulate affinity labels that are
// assigned to supported entities when accessed using entities/affinitylabels.
//
type AssignedAffinityLabelsService struct {
	BaseService
}

func NewAssignedAffinityLabelsService(connection *Connection, path string) *AssignedAffinityLabelsService {
	var result AssignedAffinityLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Attaches a label to an entity.
//
type AssignedAffinityLabelsServiceAddRequest struct {
	AssignedAffinityLabelsService *AssignedAffinityLabelsService
	header                        map[string]string
	query                         map[string]string
	label                         *AffinityLabel
}

func (p *AssignedAffinityLabelsServiceAddRequest) Header(key, value string) *AssignedAffinityLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceAddRequest) Query(key, value string) *AssignedAffinityLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceAddRequest) Label(label *AffinityLabel) *AssignedAffinityLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *AssignedAffinityLabelsServiceAddRequest) Send() (*AssignedAffinityLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelsService.connection.URL(), p.AssignedAffinityLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedAffinityLabelsServiceAddResponse{label: result}, nil
}

func (p *AssignedAffinityLabelsServiceAddRequest) MustSend() *AssignedAffinityLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Attaches a label to an entity.
//
type AssignedAffinityLabelsServiceAddResponse struct {
	label *AffinityLabel
}

func (p *AssignedAffinityLabelsServiceAddResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AssignedAffinityLabelsServiceAddResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Attaches a label to an entity.
//
func (p *AssignedAffinityLabelsService) Add() *AssignedAffinityLabelsServiceAddRequest {
	return &AssignedAffinityLabelsServiceAddRequest{AssignedAffinityLabelsService: p}
}

//
// Lists all labels that are attached to an entity.
// The order of the returned entities isn't guaranteed.
//
type AssignedAffinityLabelsServiceListRequest struct {
	AssignedAffinityLabelsService *AssignedAffinityLabelsService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
}

func (p *AssignedAffinityLabelsServiceListRequest) Header(key, value string) *AssignedAffinityLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceListRequest) Query(key, value string) *AssignedAffinityLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelsServiceListRequest) Follow(follow string) *AssignedAffinityLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedAffinityLabelsServiceListRequest) Send() (*AssignedAffinityLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelsService.connection.URL(), p.AssignedAffinityLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedAffinityLabelsServiceListResponse{label: result}, nil
}

func (p *AssignedAffinityLabelsServiceListRequest) MustSend() *AssignedAffinityLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all labels that are attached to an entity.
// The order of the returned entities isn't guaranteed.
//
type AssignedAffinityLabelsServiceListResponse struct {
	label *AffinityLabelSlice
}

func (p *AssignedAffinityLabelsServiceListResponse) Label() (*AffinityLabelSlice, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AssignedAffinityLabelsServiceListResponse) MustLabel() *AffinityLabelSlice {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Lists all labels that are attached to an entity.
// The order of the returned entities isn't guaranteed.
//
func (p *AssignedAffinityLabelsService) List() *AssignedAffinityLabelsServiceListRequest {
	return &AssignedAffinityLabelsServiceListRequest{AssignedAffinityLabelsService: p}
}

//
// Link to the specific entity-label assignment to allow
// removal.
//
func (op *AssignedAffinityLabelsService) LabelService(id string) *AssignedAffinityLabelService {
	return NewAssignedAffinityLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedAffinityLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *AssignedAffinityLabelsService) String() string {
	return fmt.Sprintf("AssignedAffinityLabelsService:%s", op.path)
}

//
//
type IscsiBondService struct {
	BaseService
}

func NewIscsiBondService(connection *Connection, path string) *IscsiBondService {
	var result IscsiBondService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type IscsiBondServiceGetRequest struct {
	IscsiBondService *IscsiBondService
	header           map[string]string
	query            map[string]string
	follow           *string
}

func (p *IscsiBondServiceGetRequest) Header(key, value string) *IscsiBondServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *IscsiBondServiceGetRequest) Query(key, value string) *IscsiBondServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *IscsiBondServiceGetRequest) Follow(follow string) *IscsiBondServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *IscsiBondServiceGetRequest) Send() (*IscsiBondServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.IscsiBondService.connection.URL(), p.IscsiBondService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.IscsiBondService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.IscsiBondService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.IscsiBondService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.IscsiBondService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.IscsiBondService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLIscsiBondReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &IscsiBondServiceGetResponse{bond: result}, nil
}

func (p *IscsiBondServiceGetRequest) MustSend() *IscsiBondServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type IscsiBondServiceGetResponse struct {
	bond *IscsiBond
}

func (p *IscsiBondServiceGetResponse) Bond() (*IscsiBond, bool) {
	if p.bond != nil {
		return p.bond, true
	}
	return nil, false
}

func (p *IscsiBondServiceGetResponse) MustBond() *IscsiBond {
	if p.bond == nil {
		panic("bond in response does not exist")
	}
	return p.bond
}

//
//
func (p *IscsiBondService) Get() *IscsiBondServiceGetRequest {
	return &IscsiBondServiceGetRequest{IscsiBondService: p}
}

//
// Removes of an existing iSCSI bond.
// For example, to remove the iSCSI bond `456` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123/iscsibonds/456
// ----
//
type IscsiBondServiceRemoveRequest struct {
	IscsiBondService *IscsiBondService
	header           map[string]string
	query            map[string]string
	async            *bool
}

func (p *IscsiBondServiceRemoveRequest) Header(key, value string) *IscsiBondServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *IscsiBondServiceRemoveRequest) Query(key, value string) *IscsiBondServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *IscsiBondServiceRemoveRequest) Async(async bool) *IscsiBondServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *IscsiBondServiceRemoveRequest) Send() (*IscsiBondServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.IscsiBondService.connection.URL(), p.IscsiBondService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.IscsiBondService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.IscsiBondService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.IscsiBondService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.IscsiBondService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.IscsiBondService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(IscsiBondServiceRemoveResponse), nil
}

func (p *IscsiBondServiceRemoveRequest) MustSend() *IscsiBondServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes of an existing iSCSI bond.
// For example, to remove the iSCSI bond `456` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123/iscsibonds/456
// ----
//
type IscsiBondServiceRemoveResponse struct {
}

//
// Removes of an existing iSCSI bond.
// For example, to remove the iSCSI bond `456` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123/iscsibonds/456
// ----
//
func (p *IscsiBondService) Remove() *IscsiBondServiceRemoveRequest {
	return &IscsiBondServiceRemoveRequest{IscsiBondService: p}
}

//
// Updates an iSCSI bond.
// Updating of an iSCSI bond can be done on the `name` and the `description` attributes only. For example, to
// update the iSCSI bond `456` of data center `123`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/iscsibonds/1234
// ----
// The request body should look like this:
// [source,xml]
// ----
// <iscsi_bond>
//    <name>mybond</name>
//    <description>My iSCSI bond</description>
// </iscsi_bond>
// ----
//
type IscsiBondServiceUpdateRequest struct {
	IscsiBondService *IscsiBondService
	header           map[string]string
	query            map[string]string
	async            *bool
	bond             *IscsiBond
}

func (p *IscsiBondServiceUpdateRequest) Header(key, value string) *IscsiBondServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *IscsiBondServiceUpdateRequest) Query(key, value string) *IscsiBondServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *IscsiBondServiceUpdateRequest) Async(async bool) *IscsiBondServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *IscsiBondServiceUpdateRequest) Bond(bond *IscsiBond) *IscsiBondServiceUpdateRequest {
	p.bond = bond
	return p
}

func (p *IscsiBondServiceUpdateRequest) Send() (*IscsiBondServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.IscsiBondService.connection.URL(), p.IscsiBondService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLIscsiBondWriteOne(writer, p.bond, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.IscsiBondService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.IscsiBondService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.IscsiBondService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.IscsiBondService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.IscsiBondService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLIscsiBondReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &IscsiBondServiceUpdateResponse{bond: result}, nil
}

func (p *IscsiBondServiceUpdateRequest) MustSend() *IscsiBondServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates an iSCSI bond.
// Updating of an iSCSI bond can be done on the `name` and the `description` attributes only. For example, to
// update the iSCSI bond `456` of data center `123`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/iscsibonds/1234
// ----
// The request body should look like this:
// [source,xml]
// ----
// <iscsi_bond>
//    <name>mybond</name>
//    <description>My iSCSI bond</description>
// </iscsi_bond>
// ----
//
type IscsiBondServiceUpdateResponse struct {
	bond *IscsiBond
}

func (p *IscsiBondServiceUpdateResponse) Bond() (*IscsiBond, bool) {
	if p.bond != nil {
		return p.bond, true
	}
	return nil, false
}

func (p *IscsiBondServiceUpdateResponse) MustBond() *IscsiBond {
	if p.bond == nil {
		panic("bond in response does not exist")
	}
	return p.bond
}

//
// Updates an iSCSI bond.
// Updating of an iSCSI bond can be done on the `name` and the `description` attributes only. For example, to
// update the iSCSI bond `456` of data center `123`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/iscsibonds/1234
// ----
// The request body should look like this:
// [source,xml]
// ----
// <iscsi_bond>
//    <name>mybond</name>
//    <description>My iSCSI bond</description>
// </iscsi_bond>
// ----
//
func (p *IscsiBondService) Update() *IscsiBondServiceUpdateRequest {
	return &IscsiBondServiceUpdateRequest{IscsiBondService: p}
}

//
//
func (op *IscsiBondService) NetworksService() *NetworksService {
	return NewNetworksService(op.connection, fmt.Sprintf("%s/networks", op.path))
}

//
//
func (op *IscsiBondService) StorageServerConnectionsService() *StorageServerConnectionsService {
	return NewStorageServerConnectionsService(op.connection, fmt.Sprintf("%s/storageserverconnections", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *IscsiBondService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "networks" {
		return op.NetworksService(), nil
	}
	if strings.HasPrefix(path, "networks/") {
		return op.NetworksService().Service(path[9:])
	}
	if path == "storageserverconnections" {
		return op.StorageServerConnectionsService(), nil
	}
	if strings.HasPrefix(path, "storageserverconnections/") {
		return op.StorageServerConnectionsService().Service(path[25:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *IscsiBondService) String() string {
	return fmt.Sprintf("IscsiBondService:%s", op.path)
}

//
//
type StorageDomainContentDiskService struct {
	BaseService
}

func NewStorageDomainContentDiskService(connection *Connection, path string) *StorageDomainContentDiskService {
	var result StorageDomainContentDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageDomainContentDiskServiceGetRequest struct {
	StorageDomainContentDiskService *StorageDomainContentDiskService
	header                          map[string]string
	query                           map[string]string
	filter                          *bool
	follow                          *string
}

func (p *StorageDomainContentDiskServiceGetRequest) Header(key, value string) *StorageDomainContentDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainContentDiskServiceGetRequest) Query(key, value string) *StorageDomainContentDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainContentDiskServiceGetRequest) Filter(filter bool) *StorageDomainContentDiskServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *StorageDomainContentDiskServiceGetRequest) Follow(follow string) *StorageDomainContentDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainContentDiskServiceGetRequest) Send() (*StorageDomainContentDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainContentDiskService.connection.URL(), p.StorageDomainContentDiskService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainContentDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainContentDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainContentDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainContentDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainContentDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainContentDiskServiceGetResponse{disk: result}, nil
}

func (p *StorageDomainContentDiskServiceGetRequest) MustSend() *StorageDomainContentDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageDomainContentDiskServiceGetResponse struct {
	disk *Disk
}

func (p *StorageDomainContentDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *StorageDomainContentDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
//
func (p *StorageDomainContentDiskService) Get() *StorageDomainContentDiskServiceGetRequest {
	return &StorageDomainContentDiskServiceGetRequest{StorageDomainContentDiskService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainContentDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageDomainContentDiskService) String() string {
	return fmt.Sprintf("StorageDomainContentDiskService:%s", op.path)
}

//
//
type NetworkAttachmentService struct {
	BaseService
}

func NewNetworkAttachmentService(connection *Connection, path string) *NetworkAttachmentService {
	var result NetworkAttachmentService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type NetworkAttachmentServiceGetRequest struct {
	NetworkAttachmentService *NetworkAttachmentService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *NetworkAttachmentServiceGetRequest) Header(key, value string) *NetworkAttachmentServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkAttachmentServiceGetRequest) Query(key, value string) *NetworkAttachmentServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkAttachmentServiceGetRequest) Follow(follow string) *NetworkAttachmentServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *NetworkAttachmentServiceGetRequest) Send() (*NetworkAttachmentServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkAttachmentService.connection.URL(), p.NetworkAttachmentService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkAttachmentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkAttachmentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkAttachmentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkAttachmentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkAttachmentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkAttachmentServiceGetResponse{attachment: result}, nil
}

func (p *NetworkAttachmentServiceGetRequest) MustSend() *NetworkAttachmentServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type NetworkAttachmentServiceGetResponse struct {
	attachment *NetworkAttachment
}

func (p *NetworkAttachmentServiceGetResponse) Attachment() (*NetworkAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *NetworkAttachmentServiceGetResponse) MustAttachment() *NetworkAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
//
func (p *NetworkAttachmentService) Get() *NetworkAttachmentServiceGetRequest {
	return &NetworkAttachmentServiceGetRequest{NetworkAttachmentService: p}
}

//
//
type NetworkAttachmentServiceRemoveRequest struct {
	NetworkAttachmentService *NetworkAttachmentService
	header                   map[string]string
	query                    map[string]string
	async                    *bool
}

func (p *NetworkAttachmentServiceRemoveRequest) Header(key, value string) *NetworkAttachmentServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkAttachmentServiceRemoveRequest) Query(key, value string) *NetworkAttachmentServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkAttachmentServiceRemoveRequest) Async(async bool) *NetworkAttachmentServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *NetworkAttachmentServiceRemoveRequest) Send() (*NetworkAttachmentServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkAttachmentService.connection.URL(), p.NetworkAttachmentService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkAttachmentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkAttachmentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkAttachmentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkAttachmentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkAttachmentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(NetworkAttachmentServiceRemoveResponse), nil
}

func (p *NetworkAttachmentServiceRemoveRequest) MustSend() *NetworkAttachmentServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type NetworkAttachmentServiceRemoveResponse struct {
}

//
//
func (p *NetworkAttachmentService) Remove() *NetworkAttachmentServiceRemoveRequest {
	return &NetworkAttachmentServiceRemoveRequest{NetworkAttachmentService: p}
}

//
// Update the specified network attachment on the host.
//
type NetworkAttachmentServiceUpdateRequest struct {
	NetworkAttachmentService *NetworkAttachmentService
	header                   map[string]string
	query                    map[string]string
	async                    *bool
	attachment               *NetworkAttachment
}

func (p *NetworkAttachmentServiceUpdateRequest) Header(key, value string) *NetworkAttachmentServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkAttachmentServiceUpdateRequest) Query(key, value string) *NetworkAttachmentServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkAttachmentServiceUpdateRequest) Async(async bool) *NetworkAttachmentServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *NetworkAttachmentServiceUpdateRequest) Attachment(attachment *NetworkAttachment) *NetworkAttachmentServiceUpdateRequest {
	p.attachment = attachment
	return p
}

func (p *NetworkAttachmentServiceUpdateRequest) Send() (*NetworkAttachmentServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkAttachmentService.connection.URL(), p.NetworkAttachmentService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkAttachmentWriteOne(writer, p.attachment, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkAttachmentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkAttachmentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkAttachmentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkAttachmentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkAttachmentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkAttachmentServiceUpdateResponse{attachment: result}, nil
}

func (p *NetworkAttachmentServiceUpdateRequest) MustSend() *NetworkAttachmentServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified network attachment on the host.
//
type NetworkAttachmentServiceUpdateResponse struct {
	attachment *NetworkAttachment
}

func (p *NetworkAttachmentServiceUpdateResponse) Attachment() (*NetworkAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *NetworkAttachmentServiceUpdateResponse) MustAttachment() *NetworkAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
// Update the specified network attachment on the host.
//
func (p *NetworkAttachmentService) Update() *NetworkAttachmentServiceUpdateRequest {
	return &NetworkAttachmentServiceUpdateRequest{NetworkAttachmentService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkAttachmentService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *NetworkAttachmentService) String() string {
	return fmt.Sprintf("NetworkAttachmentService:%s", op.path)
}

//
//
type VmReportedDeviceService struct {
	BaseService
}

func NewVmReportedDeviceService(connection *Connection, path string) *VmReportedDeviceService {
	var result VmReportedDeviceService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type VmReportedDeviceServiceGetRequest struct {
	VmReportedDeviceService *VmReportedDeviceService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *VmReportedDeviceServiceGetRequest) Header(key, value string) *VmReportedDeviceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmReportedDeviceServiceGetRequest) Query(key, value string) *VmReportedDeviceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmReportedDeviceServiceGetRequest) Follow(follow string) *VmReportedDeviceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmReportedDeviceServiceGetRequest) Send() (*VmReportedDeviceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmReportedDeviceService.connection.URL(), p.VmReportedDeviceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmReportedDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmReportedDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmReportedDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmReportedDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmReportedDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLReportedDeviceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmReportedDeviceServiceGetResponse{reportedDevice: result}, nil
}

func (p *VmReportedDeviceServiceGetRequest) MustSend() *VmReportedDeviceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmReportedDeviceServiceGetResponse struct {
	reportedDevice *ReportedDevice
}

func (p *VmReportedDeviceServiceGetResponse) ReportedDevice() (*ReportedDevice, bool) {
	if p.reportedDevice != nil {
		return p.reportedDevice, true
	}
	return nil, false
}

func (p *VmReportedDeviceServiceGetResponse) MustReportedDevice() *ReportedDevice {
	if p.reportedDevice == nil {
		panic("reportedDevice in response does not exist")
	}
	return p.reportedDevice
}

//
//
func (p *VmReportedDeviceService) Get() *VmReportedDeviceServiceGetRequest {
	return &VmReportedDeviceServiceGetRequest{VmReportedDeviceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmReportedDeviceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmReportedDeviceService) String() string {
	return fmt.Sprintf("VmReportedDeviceService:%s", op.path)
}

//
//
type SnapshotDiskService struct {
	BaseService
}

func NewSnapshotDiskService(connection *Connection, path string) *SnapshotDiskService {
	var result SnapshotDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SnapshotDiskServiceGetRequest struct {
	SnapshotDiskService *SnapshotDiskService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *SnapshotDiskServiceGetRequest) Header(key, value string) *SnapshotDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotDiskServiceGetRequest) Query(key, value string) *SnapshotDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotDiskServiceGetRequest) Follow(follow string) *SnapshotDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SnapshotDiskServiceGetRequest) Send() (*SnapshotDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotDiskService.connection.URL(), p.SnapshotDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SnapshotDiskServiceGetResponse{disk: result}, nil
}

func (p *SnapshotDiskServiceGetRequest) MustSend() *SnapshotDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SnapshotDiskServiceGetResponse struct {
	disk *Disk
}

func (p *SnapshotDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *SnapshotDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
//
func (p *SnapshotDiskService) Get() *SnapshotDiskServiceGetRequest {
	return &SnapshotDiskServiceGetRequest{SnapshotDiskService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SnapshotDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SnapshotDiskService) String() string {
	return fmt.Sprintf("SnapshotDiskService:%s", op.path)
}

//
// Lists the virtual machines of an export storage domain.
// For example, to retrieve the virtual machines that are available in the storage domain with identifier `123` send the
// following request:
// [source]
// ----
// GET /ovirt-engine/api/storagedomains/123/vms
// ----
// This will return the following response body:
// [source,xml]
// ----
// <vms>
//   <vm id="456" href="/api/storagedomains/123/vms/456">
//     <name>vm1</name>
//     ...
//     <storage_domain id="123" href="/api/storagedomains/123"/>
//     <actions>
//       <link rel="import" href="/api/storagedomains/123/vms/456/import"/>
//     </actions>
//   </vm>
// </vms>
// ----
// Virtual machines and templates in these collections have a similar representation to their counterparts in the
// top-level <<types/vm, Vm>> and <<types/template, Template>> collections, except they also contain a
// <<types/storage_domain, StorageDomain>> reference and an <<services/storage_domain_vm/methods/import, import>>
// action.
//
type StorageDomainVmsService struct {
	BaseService
}

func NewStorageDomainVmsService(connection *Connection, path string) *StorageDomainVmsService {
	var result StorageDomainVmsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of virtual machines of the export storage domain.
// The order of the returned list of virtual machines isn't guaranteed.
//
type StorageDomainVmsServiceListRequest struct {
	StorageDomainVmsService *StorageDomainVmsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
	max                     *int64
	unregistered            *bool
}

func (p *StorageDomainVmsServiceListRequest) Header(key, value string) *StorageDomainVmsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainVmsServiceListRequest) Query(key, value string) *StorageDomainVmsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainVmsServiceListRequest) Follow(follow string) *StorageDomainVmsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainVmsServiceListRequest) Max(max int64) *StorageDomainVmsServiceListRequest {
	p.max = &max
	return p
}

func (p *StorageDomainVmsServiceListRequest) Unregistered(unregistered bool) *StorageDomainVmsServiceListRequest {
	p.unregistered = &unregistered
	return p
}

func (p *StorageDomainVmsServiceListRequest) Send() (*StorageDomainVmsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainVmsService.connection.URL(), p.StorageDomainVmsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.unregistered != nil {
		values["unregistered"] = []string{fmt.Sprintf("%v", *p.unregistered)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &StorageDomainVmsServiceListResponse{vm: result}, nil
}

func (p *StorageDomainVmsServiceListRequest) MustSend() *StorageDomainVmsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of virtual machines of the export storage domain.
// The order of the returned list of virtual machines isn't guaranteed.
//
type StorageDomainVmsServiceListResponse struct {
	vm *VmSlice
}

func (p *StorageDomainVmsServiceListResponse) Vm() (*VmSlice, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *StorageDomainVmsServiceListResponse) MustVm() *VmSlice {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// Returns the list of virtual machines of the export storage domain.
// The order of the returned list of virtual machines isn't guaranteed.
//
func (p *StorageDomainVmsService) List() *StorageDomainVmsServiceListRequest {
	return &StorageDomainVmsServiceListRequest{StorageDomainVmsService: p}
}

//
//
func (op *StorageDomainVmsService) VmService(id string) *StorageDomainVmService {
	return NewStorageDomainVmService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainVmsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VmService(path), nil
	}
	return op.VmService(path[:index]).Service(path[index+1:])
}

func (op *StorageDomainVmsService) String() string {
	return fmt.Sprintf("StorageDomainVmsService:%s", op.path)
}

//
//
type InstanceTypesService struct {
	BaseService
}

func NewInstanceTypesService(connection *Connection, path string) *InstanceTypesService {
	var result InstanceTypesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new instance type.
// This requires only a name attribute and can include all hardware configurations of the
// virtual machine.
// [source]
// ----
// POST /ovirt-engine/api/instancetypes
// ----
// With a request body like this:
// [source,xml]
// ----
// <instance_type>
//   <name>myinstancetype</name>
// </template>
// ----
// Creating an instance type with all hardware configurations with a request body like this:
// [source,xml]
// ----
// <instance_type>
//   <name>myinstancetype</name>
//   <console>
//     <enabled>true</enabled>
//   </console>
//   <cpu>
//     <topology>
//       <cores>2</cores>
//       <sockets>2</sockets>
//       <threads>1</threads>
//     </topology>
//   </cpu>
//   <custom_cpu_model>AMD Opteron_G2</custom_cpu_model>
//   <custom_emulated_machine>q35</custom_emulated_machine>
//   <display>
//     <monitors>1</monitors>
//     <single_qxl_pci>true</single_qxl_pci>
//     <smartcard_enabled>true</smartcard_enabled>
//     <type>spice</type>
//   </display>
//   <high_availability>
//     <enabled>true</enabled>
//     <priority>1</priority>
//   </high_availability>
//   <io>
//     <threads>2</threads>
//   </io>
//   <memory>4294967296</memory>
//   <memory_policy>
//     <ballooning>true</ballooning>
//     <guaranteed>268435456</guaranteed>
//   </memory_policy>
//   <migration>
//     <auto_converge>inherit</auto_converge>
//     <compressed>inherit</compressed>
//     <policy id="00000000-0000-0000-0000-000000000000"/>
//   </migration>
//   <migration_downtime>2</migration_downtime>
//   <os>
//     <boot>
//       <devices>
//         <device>hd</device>
//       </devices>
//     </boot>
//   </os>
//   <rng_device>
//     <rate>
//       <bytes>200</bytes>
//       <period>2</period>
//     </rate>
//     <source>urandom</source>
//   </rng_device>
//   <soundcard_enabled>true</soundcard_enabled>
//   <usb>
//     <enabled>true</enabled>
//     <type>native</type>
//   </usb>
//   <virtio_scsi>
//     <enabled>true</enabled>
//   </virtio_scsi>
// </instance_type>
// ----
//
type InstanceTypesServiceAddRequest struct {
	InstanceTypesService *InstanceTypesService
	header               map[string]string
	query                map[string]string
	instanceType         *InstanceType
}

func (p *InstanceTypesServiceAddRequest) Header(key, value string) *InstanceTypesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypesServiceAddRequest) Query(key, value string) *InstanceTypesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypesServiceAddRequest) InstanceType(instanceType *InstanceType) *InstanceTypesServiceAddRequest {
	p.instanceType = instanceType
	return p
}

func (p *InstanceTypesServiceAddRequest) Send() (*InstanceTypesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypesService.connection.URL(), p.InstanceTypesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLInstanceTypeWriteOne(writer, p.instanceType, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLInstanceTypeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &InstanceTypesServiceAddResponse{instanceType: result}, nil
}

func (p *InstanceTypesServiceAddRequest) MustSend() *InstanceTypesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new instance type.
// This requires only a name attribute and can include all hardware configurations of the
// virtual machine.
// [source]
// ----
// POST /ovirt-engine/api/instancetypes
// ----
// With a request body like this:
// [source,xml]
// ----
// <instance_type>
//   <name>myinstancetype</name>
// </template>
// ----
// Creating an instance type with all hardware configurations with a request body like this:
// [source,xml]
// ----
// <instance_type>
//   <name>myinstancetype</name>
//   <console>
//     <enabled>true</enabled>
//   </console>
//   <cpu>
//     <topology>
//       <cores>2</cores>
//       <sockets>2</sockets>
//       <threads>1</threads>
//     </topology>
//   </cpu>
//   <custom_cpu_model>AMD Opteron_G2</custom_cpu_model>
//   <custom_emulated_machine>q35</custom_emulated_machine>
//   <display>
//     <monitors>1</monitors>
//     <single_qxl_pci>true</single_qxl_pci>
//     <smartcard_enabled>true</smartcard_enabled>
//     <type>spice</type>
//   </display>
//   <high_availability>
//     <enabled>true</enabled>
//     <priority>1</priority>
//   </high_availability>
//   <io>
//     <threads>2</threads>
//   </io>
//   <memory>4294967296</memory>
//   <memory_policy>
//     <ballooning>true</ballooning>
//     <guaranteed>268435456</guaranteed>
//   </memory_policy>
//   <migration>
//     <auto_converge>inherit</auto_converge>
//     <compressed>inherit</compressed>
//     <policy id="00000000-0000-0000-0000-000000000000"/>
//   </migration>
//   <migration_downtime>2</migration_downtime>
//   <os>
//     <boot>
//       <devices>
//         <device>hd</device>
//       </devices>
//     </boot>
//   </os>
//   <rng_device>
//     <rate>
//       <bytes>200</bytes>
//       <period>2</period>
//     </rate>
//     <source>urandom</source>
//   </rng_device>
//   <soundcard_enabled>true</soundcard_enabled>
//   <usb>
//     <enabled>true</enabled>
//     <type>native</type>
//   </usb>
//   <virtio_scsi>
//     <enabled>true</enabled>
//   </virtio_scsi>
// </instance_type>
// ----
//
type InstanceTypesServiceAddResponse struct {
	instanceType *InstanceType
}

func (p *InstanceTypesServiceAddResponse) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

func (p *InstanceTypesServiceAddResponse) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("instanceType in response does not exist")
	}
	return p.instanceType
}

//
// Creates a new instance type.
// This requires only a name attribute and can include all hardware configurations of the
// virtual machine.
// [source]
// ----
// POST /ovirt-engine/api/instancetypes
// ----
// With a request body like this:
// [source,xml]
// ----
// <instance_type>
//   <name>myinstancetype</name>
// </template>
// ----
// Creating an instance type with all hardware configurations with a request body like this:
// [source,xml]
// ----
// <instance_type>
//   <name>myinstancetype</name>
//   <console>
//     <enabled>true</enabled>
//   </console>
//   <cpu>
//     <topology>
//       <cores>2</cores>
//       <sockets>2</sockets>
//       <threads>1</threads>
//     </topology>
//   </cpu>
//   <custom_cpu_model>AMD Opteron_G2</custom_cpu_model>
//   <custom_emulated_machine>q35</custom_emulated_machine>
//   <display>
//     <monitors>1</monitors>
//     <single_qxl_pci>true</single_qxl_pci>
//     <smartcard_enabled>true</smartcard_enabled>
//     <type>spice</type>
//   </display>
//   <high_availability>
//     <enabled>true</enabled>
//     <priority>1</priority>
//   </high_availability>
//   <io>
//     <threads>2</threads>
//   </io>
//   <memory>4294967296</memory>
//   <memory_policy>
//     <ballooning>true</ballooning>
//     <guaranteed>268435456</guaranteed>
//   </memory_policy>
//   <migration>
//     <auto_converge>inherit</auto_converge>
//     <compressed>inherit</compressed>
//     <policy id="00000000-0000-0000-0000-000000000000"/>
//   </migration>
//   <migration_downtime>2</migration_downtime>
//   <os>
//     <boot>
//       <devices>
//         <device>hd</device>
//       </devices>
//     </boot>
//   </os>
//   <rng_device>
//     <rate>
//       <bytes>200</bytes>
//       <period>2</period>
//     </rate>
//     <source>urandom</source>
//   </rng_device>
//   <soundcard_enabled>true</soundcard_enabled>
//   <usb>
//     <enabled>true</enabled>
//     <type>native</type>
//   </usb>
//   <virtio_scsi>
//     <enabled>true</enabled>
//   </virtio_scsi>
// </instance_type>
// ----
//
func (p *InstanceTypesService) Add() *InstanceTypesServiceAddRequest {
	return &InstanceTypesServiceAddRequest{InstanceTypesService: p}
}

//
// Lists all existing instance types in the system.
// The order of the returned list of instance types isn't guaranteed.
//
type InstanceTypesServiceListRequest struct {
	InstanceTypesService *InstanceTypesService
	header               map[string]string
	query                map[string]string
	caseSensitive        *bool
	follow               *string
	max                  *int64
	search               *string
}

func (p *InstanceTypesServiceListRequest) Header(key, value string) *InstanceTypesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypesServiceListRequest) Query(key, value string) *InstanceTypesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypesServiceListRequest) CaseSensitive(caseSensitive bool) *InstanceTypesServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *InstanceTypesServiceListRequest) Follow(follow string) *InstanceTypesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *InstanceTypesServiceListRequest) Max(max int64) *InstanceTypesServiceListRequest {
	p.max = &max
	return p
}

func (p *InstanceTypesServiceListRequest) Search(search string) *InstanceTypesServiceListRequest {
	p.search = &search
	return p
}

func (p *InstanceTypesServiceListRequest) Send() (*InstanceTypesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypesService.connection.URL(), p.InstanceTypesService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLInstanceTypeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &InstanceTypesServiceListResponse{instanceType: result}, nil
}

func (p *InstanceTypesServiceListRequest) MustSend() *InstanceTypesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all existing instance types in the system.
// The order of the returned list of instance types isn't guaranteed.
//
type InstanceTypesServiceListResponse struct {
	instanceType *InstanceTypeSlice
}

func (p *InstanceTypesServiceListResponse) InstanceType() (*InstanceTypeSlice, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

func (p *InstanceTypesServiceListResponse) MustInstanceType() *InstanceTypeSlice {
	if p.instanceType == nil {
		panic("instanceType in response does not exist")
	}
	return p.instanceType
}

//
// Lists all existing instance types in the system.
// The order of the returned list of instance types isn't guaranteed.
//
func (p *InstanceTypesService) List() *InstanceTypesServiceListRequest {
	return &InstanceTypesServiceListRequest{InstanceTypesService: p}
}

//
//
func (op *InstanceTypesService) InstanceTypeService(id string) *InstanceTypeService {
	return NewInstanceTypeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *InstanceTypesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.InstanceTypeService(path), nil
	}
	return op.InstanceTypeService(path[:index]).Service(path[index+1:])
}

func (op *InstanceTypesService) String() string {
	return fmt.Sprintf("InstanceTypesService:%s", op.path)
}

//
// A service to list all domain users in the system.
//
type DomainUsersService struct {
	BaseService
}

func NewDomainUsersService(connection *Connection, path string) *DomainUsersService {
	var result DomainUsersService
	result.connection = connection
	result.path = path
	return &result
}

//
// List all the users in the domain.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users
// ....
// Will return the list of users in the domain:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/domains/5678/users/1234" id="1234">
//     <name>admin</name>
//     <namespace>*</namespace>
//     <principal>admin</principal>
//     <user_name>admin@internal-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/5678" id="5678">
//       <name>internal-authz</name>
//     </domain>
//     <groups/>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type DomainUsersServiceListRequest struct {
	DomainUsersService *DomainUsersService
	header             map[string]string
	query              map[string]string
	caseSensitive      *bool
	follow             *string
	max                *int64
	search             *string
}

func (p *DomainUsersServiceListRequest) Header(key, value string) *DomainUsersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainUsersServiceListRequest) Query(key, value string) *DomainUsersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainUsersServiceListRequest) CaseSensitive(caseSensitive bool) *DomainUsersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DomainUsersServiceListRequest) Follow(follow string) *DomainUsersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainUsersServiceListRequest) Max(max int64) *DomainUsersServiceListRequest {
	p.max = &max
	return p
}

func (p *DomainUsersServiceListRequest) Search(search string) *DomainUsersServiceListRequest {
	p.search = &search
	return p
}

func (p *DomainUsersServiceListRequest) Send() (*DomainUsersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainUsersService.connection.URL(), p.DomainUsersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainUsersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainUsersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainUsersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainUsersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainUsersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainUsersServiceListResponse{users: result}, nil
}

func (p *DomainUsersServiceListRequest) MustSend() *DomainUsersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the users in the domain.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users
// ....
// Will return the list of users in the domain:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/domains/5678/users/1234" id="1234">
//     <name>admin</name>
//     <namespace>*</namespace>
//     <principal>admin</principal>
//     <user_name>admin@internal-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/5678" id="5678">
//       <name>internal-authz</name>
//     </domain>
//     <groups/>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type DomainUsersServiceListResponse struct {
	users *UserSlice
}

func (p *DomainUsersServiceListResponse) Users() (*UserSlice, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

func (p *DomainUsersServiceListResponse) MustUsers() *UserSlice {
	if p.users == nil {
		panic("users in response does not exist")
	}
	return p.users
}

//
// List all the users in the domain.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users
// ....
// Will return the list of users in the domain:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/domains/5678/users/1234" id="1234">
//     <name>admin</name>
//     <namespace>*</namespace>
//     <principal>admin</principal>
//     <user_name>admin@internal-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/5678" id="5678">
//       <name>internal-authz</name>
//     </domain>
//     <groups/>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
func (p *DomainUsersService) List() *DomainUsersServiceListRequest {
	return &DomainUsersServiceListRequest{DomainUsersService: p}
}

//
// Reference to a service to view details of a domain user.
//
func (op *DomainUsersService) UserService(id string) *DomainUserService {
	return NewDomainUserService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainUsersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.UserService(path), nil
	}
	return op.UserService(path[:index]).Service(path[index+1:])
}

func (op *DomainUsersService) String() string {
	return fmt.Sprintf("DomainUsersService:%s", op.path)
}

//
//
type DomainGroupService struct {
	BaseService
}

func NewDomainGroupService(connection *Connection, path string) *DomainGroupService {
	var result DomainGroupService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type DomainGroupServiceGetRequest struct {
	DomainGroupService *DomainGroupService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *DomainGroupServiceGetRequest) Header(key, value string) *DomainGroupServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainGroupServiceGetRequest) Query(key, value string) *DomainGroupServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainGroupServiceGetRequest) Follow(follow string) *DomainGroupServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DomainGroupServiceGetRequest) Send() (*DomainGroupServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainGroupService.connection.URL(), p.DomainGroupService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainGroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainGroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainGroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainGroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainGroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DomainGroupServiceGetResponse{get: result}, nil
}

func (p *DomainGroupServiceGetRequest) MustSend() *DomainGroupServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DomainGroupServiceGetResponse struct {
	get *Group
}

func (p *DomainGroupServiceGetResponse) Get() (*Group, bool) {
	if p.get != nil {
		return p.get, true
	}
	return nil, false
}

func (p *DomainGroupServiceGetResponse) MustGet() *Group {
	if p.get == nil {
		panic("get in response does not exist")
	}
	return p.get
}

//
//
func (p *DomainGroupService) Get() *DomainGroupServiceGetRequest {
	return &DomainGroupServiceGetRequest{DomainGroupService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainGroupService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainGroupService) String() string {
	return fmt.Sprintf("DomainGroupService:%s", op.path)
}

//
//
type DomainGroupsService struct {
	BaseService
}

func NewDomainGroupsService(connection *Connection, path string) *DomainGroupsService {
	var result DomainGroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of groups.
// The order of the returned list of groups isn't guaranteed.
//
type DomainGroupsServiceListRequest struct {
	DomainGroupsService *DomainGroupsService
	header              map[string]string
	query               map[string]string
	caseSensitive       *bool
	follow              *string
	max                 *int64
	search              *string
}

func (p *DomainGroupsServiceListRequest) Header(key, value string) *DomainGroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainGroupsServiceListRequest) Query(key, value string) *DomainGroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainGroupsServiceListRequest) CaseSensitive(caseSensitive bool) *DomainGroupsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DomainGroupsServiceListRequest) Follow(follow string) *DomainGroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainGroupsServiceListRequest) Max(max int64) *DomainGroupsServiceListRequest {
	p.max = &max
	return p
}

func (p *DomainGroupsServiceListRequest) Search(search string) *DomainGroupsServiceListRequest {
	p.search = &search
	return p
}

func (p *DomainGroupsServiceListRequest) Send() (*DomainGroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainGroupsService.connection.URL(), p.DomainGroupsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainGroupsServiceListResponse{groups: result}, nil
}

func (p *DomainGroupsServiceListRequest) MustSend() *DomainGroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of groups.
// The order of the returned list of groups isn't guaranteed.
//
type DomainGroupsServiceListResponse struct {
	groups *GroupSlice
}

func (p *DomainGroupsServiceListResponse) Groups() (*GroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *DomainGroupsServiceListResponse) MustGroups() *GroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// Returns the list of groups.
// The order of the returned list of groups isn't guaranteed.
//
func (p *DomainGroupsService) List() *DomainGroupsServiceListRequest {
	return &DomainGroupsServiceListRequest{DomainGroupsService: p}
}

//
//
func (op *DomainGroupsService) GroupService(id string) *DomainGroupService {
	return NewDomainGroupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainGroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.GroupService(path), nil
	}
	return op.GroupService(path[:index]).Service(path[index+1:])
}

func (op *DomainGroupsService) String() string {
	return fmt.Sprintf("DomainGroupsService:%s", op.path)
}

//
// Manages a group of users. Use this service to either get groups details or remove groups. In order
// to add new groups please use <<services/groups, service>> that manages the collection of groups.
//
type GroupService struct {
	BaseService
}

func NewGroupService(connection *Connection, path string) *GroupService {
	var result GroupService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the system group information.
// Usage:
// ....
// GET /ovirt-engine/api/groups/123
// ....
// Will return the group information:
// [source,xml]
// ----
// <group href="/ovirt-engine/api/groups/123" id="123">
//   <name>mygroup</name>
//   <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//   <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//   <namespace>DC=example,DC=com</namespace>
//   <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//     <name>myextension-authz</name>
//   </domain>
// </group>
// ----
//
type GroupServiceGetRequest struct {
	GroupService *GroupService
	header       map[string]string
	query        map[string]string
	follow       *string
}

func (p *GroupServiceGetRequest) Header(key, value string) *GroupServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupServiceGetRequest) Query(key, value string) *GroupServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupServiceGetRequest) Follow(follow string) *GroupServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GroupServiceGetRequest) Send() (*GroupServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupService.connection.URL(), p.GroupService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GroupServiceGetResponse{get: result}, nil
}

func (p *GroupServiceGetRequest) MustSend() *GroupServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the system group information.
// Usage:
// ....
// GET /ovirt-engine/api/groups/123
// ....
// Will return the group information:
// [source,xml]
// ----
// <group href="/ovirt-engine/api/groups/123" id="123">
//   <name>mygroup</name>
//   <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//   <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//   <namespace>DC=example,DC=com</namespace>
//   <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//     <name>myextension-authz</name>
//   </domain>
// </group>
// ----
//
type GroupServiceGetResponse struct {
	get *Group
}

func (p *GroupServiceGetResponse) Get() (*Group, bool) {
	if p.get != nil {
		return p.get, true
	}
	return nil, false
}

func (p *GroupServiceGetResponse) MustGet() *Group {
	if p.get == nil {
		panic("get in response does not exist")
	}
	return p.get
}

//
// Gets the system group information.
// Usage:
// ....
// GET /ovirt-engine/api/groups/123
// ....
// Will return the group information:
// [source,xml]
// ----
// <group href="/ovirt-engine/api/groups/123" id="123">
//   <name>mygroup</name>
//   <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//   <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//   <namespace>DC=example,DC=com</namespace>
//   <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//     <name>myextension-authz</name>
//   </domain>
// </group>
// ----
//
func (p *GroupService) Get() *GroupServiceGetRequest {
	return &GroupServiceGetRequest{GroupService: p}
}

//
// Removes the system group.
// Usage:
// ....
// DELETE /ovirt-engine/api/groups/123
// ....
//
type GroupServiceRemoveRequest struct {
	GroupService *GroupService
	header       map[string]string
	query        map[string]string
	async        *bool
}

func (p *GroupServiceRemoveRequest) Header(key, value string) *GroupServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupServiceRemoveRequest) Query(key, value string) *GroupServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupServiceRemoveRequest) Async(async bool) *GroupServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GroupServiceRemoveRequest) Send() (*GroupServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupService.connection.URL(), p.GroupService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GroupServiceRemoveResponse), nil
}

func (p *GroupServiceRemoveRequest) MustSend() *GroupServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the system group.
// Usage:
// ....
// DELETE /ovirt-engine/api/groups/123
// ....
//
type GroupServiceRemoveResponse struct {
}

//
// Removes the system group.
// Usage:
// ....
// DELETE /ovirt-engine/api/groups/123
// ....
//
func (p *GroupService) Remove() *GroupServiceRemoveRequest {
	return &GroupServiceRemoveRequest{GroupService: p}
}

//
// Reference to the service that manages the collection of permissions assigned to this system group.
//
func (op *GroupService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Reference to the service that manages the collection of roles assigned to this system group.
//
func (op *GroupService) RolesService() *AssignedRolesService {
	return NewAssignedRolesService(op.connection, fmt.Sprintf("%s/roles", op.path))
}

//
// Reference to the service that manages the collection of tags assigned to this system group.
//
func (op *GroupService) TagsService() *AssignedTagsService {
	return NewAssignedTagsService(op.connection, fmt.Sprintf("%s/tags", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GroupService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "roles" {
		return op.RolesService(), nil
	}
	if strings.HasPrefix(path, "roles/") {
		return op.RolesService().Service(path[6:])
	}
	if path == "tags" {
		return op.TagsService(), nil
	}
	if strings.HasPrefix(path, "tags/") {
		return op.TagsService().Service(path[5:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GroupService) String() string {
	return fmt.Sprintf("GroupService:%s", op.path)
}

//
//
type SshPublicKeysService struct {
	BaseService
}

func NewSshPublicKeysService(connection *Connection, path string) *SshPublicKeysService {
	var result SshPublicKeysService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SshPublicKeysServiceAddRequest struct {
	SshPublicKeysService *SshPublicKeysService
	header               map[string]string
	query                map[string]string
	key                  *SshPublicKey
}

func (p *SshPublicKeysServiceAddRequest) Header(key, value string) *SshPublicKeysServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeysServiceAddRequest) Query(key, value string) *SshPublicKeysServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeysServiceAddRequest) Key(key *SshPublicKey) *SshPublicKeysServiceAddRequest {
	p.key = key
	return p
}

func (p *SshPublicKeysServiceAddRequest) Send() (*SshPublicKeysServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeysService.connection.URL(), p.SshPublicKeysService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSshPublicKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SshPublicKeysServiceAddResponse{key: result}, nil
}

func (p *SshPublicKeysServiceAddRequest) MustSend() *SshPublicKeysServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeysServiceAddResponse struct {
	key *SshPublicKey
}

func (p *SshPublicKeysServiceAddResponse) Key() (*SshPublicKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *SshPublicKeysServiceAddResponse) MustKey() *SshPublicKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *SshPublicKeysService) Add() *SshPublicKeysServiceAddRequest {
	return &SshPublicKeysServiceAddRequest{SshPublicKeysService: p}
}

//
// Returns a list of SSH public keys of the user.
// For example, to retrieve the list of SSH keys of user with identifier `123`,
// send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/users/123/sshpublickeys
// ----
// The result will be the following XML document:
// [source,xml]
// ----
// <ssh_public_keys>
//   <ssh_public_key href="/ovirt-engine/api/users/123/sshpublickeys/456" id="456">
//     <content>ssh-rsa ...</content>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//   </ssh_public_key>
// </ssh_public_keys>
// ----
// Or the following JSON object
// [source,json]
// ----
// {
//   "ssh_public_key": [
//     {
//       "content": "ssh-rsa ...",
//       "user": {
//         "href": "/ovirt-engine/api/users/123",
//         "id": "123"
//       },
//       "href": "/ovirt-engine/api/users/123/sshpublickeys/456",
//       "id": "456"
//     }
//   ]
// }
// ----
// The order of the returned list of keys is not guaranteed.
//
type SshPublicKeysServiceListRequest struct {
	SshPublicKeysService *SshPublicKeysService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *SshPublicKeysServiceListRequest) Header(key, value string) *SshPublicKeysServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeysServiceListRequest) Query(key, value string) *SshPublicKeysServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeysServiceListRequest) Follow(follow string) *SshPublicKeysServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SshPublicKeysServiceListRequest) Max(max int64) *SshPublicKeysServiceListRequest {
	p.max = &max
	return p
}

func (p *SshPublicKeysServiceListRequest) Send() (*SshPublicKeysServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeysService.connection.URL(), p.SshPublicKeysService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SshPublicKeysServiceListResponse{keys: result}, nil
}

func (p *SshPublicKeysServiceListRequest) MustSend() *SshPublicKeysServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns a list of SSH public keys of the user.
// For example, to retrieve the list of SSH keys of user with identifier `123`,
// send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/users/123/sshpublickeys
// ----
// The result will be the following XML document:
// [source,xml]
// ----
// <ssh_public_keys>
//   <ssh_public_key href="/ovirt-engine/api/users/123/sshpublickeys/456" id="456">
//     <content>ssh-rsa ...</content>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//   </ssh_public_key>
// </ssh_public_keys>
// ----
// Or the following JSON object
// [source,json]
// ----
// {
//   "ssh_public_key": [
//     {
//       "content": "ssh-rsa ...",
//       "user": {
//         "href": "/ovirt-engine/api/users/123",
//         "id": "123"
//       },
//       "href": "/ovirt-engine/api/users/123/sshpublickeys/456",
//       "id": "456"
//     }
//   ]
// }
// ----
// The order of the returned list of keys is not guaranteed.
//
type SshPublicKeysServiceListResponse struct {
	keys *SshPublicKeySlice
}

func (p *SshPublicKeysServiceListResponse) Keys() (*SshPublicKeySlice, bool) {
	if p.keys != nil {
		return p.keys, true
	}
	return nil, false
}

func (p *SshPublicKeysServiceListResponse) MustKeys() *SshPublicKeySlice {
	if p.keys == nil {
		panic("keys in response does not exist")
	}
	return p.keys
}

//
// Returns a list of SSH public keys of the user.
// For example, to retrieve the list of SSH keys of user with identifier `123`,
// send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/users/123/sshpublickeys
// ----
// The result will be the following XML document:
// [source,xml]
// ----
// <ssh_public_keys>
//   <ssh_public_key href="/ovirt-engine/api/users/123/sshpublickeys/456" id="456">
//     <content>ssh-rsa ...</content>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//   </ssh_public_key>
// </ssh_public_keys>
// ----
// Or the following JSON object
// [source,json]
// ----
// {
//   "ssh_public_key": [
//     {
//       "content": "ssh-rsa ...",
//       "user": {
//         "href": "/ovirt-engine/api/users/123",
//         "id": "123"
//       },
//       "href": "/ovirt-engine/api/users/123/sshpublickeys/456",
//       "id": "456"
//     }
//   ]
// }
// ----
// The order of the returned list of keys is not guaranteed.
//
func (p *SshPublicKeysService) List() *SshPublicKeysServiceListRequest {
	return &SshPublicKeysServiceListRequest{SshPublicKeysService: p}
}

//
//
func (op *SshPublicKeysService) KeyService(id string) *SshPublicKeyService {
	return NewSshPublicKeyService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SshPublicKeysService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KeyService(path), nil
	}
	return op.KeyService(path[:index]).Service(path[index+1:])
}

func (op *SshPublicKeysService) String() string {
	return fmt.Sprintf("SshPublicKeysService:%s", op.path)
}

//
// A service to manage a user in the system.
// Use this service to either get users details or remove users.
// In order to add new users please use
// <<services/users>>.
//
type UserService struct {
	BaseService
}

func NewUserService(connection *Connection, path string) *UserService {
	var result UserService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the system user information.
// Usage:
// ....
// GET /ovirt-engine/api/users/1234
// ....
// Will return the user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//   <department></department>
//   <domain_entry_id>23456</domain_entry_id>
//   <email>user1@domain.com</email>
//   <last_name>Lastname</last_name>
//   <namespace>*</namespace>
//   <principal>user1</principal>
//   <user_name>user1@domain-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/45678" id="45678">
//     <name>domain-authz</name>
//   </domain>
// </user>
// ----
//
type UserServiceGetRequest struct {
	UserService *UserService
	header      map[string]string
	query       map[string]string
	follow      *string
}

func (p *UserServiceGetRequest) Header(key, value string) *UserServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UserServiceGetRequest) Query(key, value string) *UserServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UserServiceGetRequest) Follow(follow string) *UserServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *UserServiceGetRequest) Send() (*UserServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UserService.connection.URL(), p.UserService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UserService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UserService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UserService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UserService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UserService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &UserServiceGetResponse{user: result}, nil
}

func (p *UserServiceGetRequest) MustSend() *UserServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the system user information.
// Usage:
// ....
// GET /ovirt-engine/api/users/1234
// ....
// Will return the user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//   <department></department>
//   <domain_entry_id>23456</domain_entry_id>
//   <email>user1@domain.com</email>
//   <last_name>Lastname</last_name>
//   <namespace>*</namespace>
//   <principal>user1</principal>
//   <user_name>user1@domain-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/45678" id="45678">
//     <name>domain-authz</name>
//   </domain>
// </user>
// ----
//
type UserServiceGetResponse struct {
	user *User
}

func (p *UserServiceGetResponse) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

func (p *UserServiceGetResponse) MustUser() *User {
	if p.user == nil {
		panic("user in response does not exist")
	}
	return p.user
}

//
// Gets the system user information.
// Usage:
// ....
// GET /ovirt-engine/api/users/1234
// ....
// Will return the user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//   <department></department>
//   <domain_entry_id>23456</domain_entry_id>
//   <email>user1@domain.com</email>
//   <last_name>Lastname</last_name>
//   <namespace>*</namespace>
//   <principal>user1</principal>
//   <user_name>user1@domain-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/45678" id="45678">
//     <name>domain-authz</name>
//   </domain>
// </user>
// ----
//
func (p *UserService) Get() *UserServiceGetRequest {
	return &UserServiceGetRequest{UserService: p}
}

//
// Removes the system user.
// Usage:
// ....
// DELETE /ovirt-engine/api/users/1234
// ....
//
type UserServiceRemoveRequest struct {
	UserService *UserService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *UserServiceRemoveRequest) Header(key, value string) *UserServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UserServiceRemoveRequest) Query(key, value string) *UserServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UserServiceRemoveRequest) Async(async bool) *UserServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *UserServiceRemoveRequest) Send() (*UserServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UserService.connection.URL(), p.UserService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UserService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UserService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UserService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UserService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UserService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(UserServiceRemoveResponse), nil
}

func (p *UserServiceRemoveRequest) MustSend() *UserServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the system user.
// Usage:
// ....
// DELETE /ovirt-engine/api/users/1234
// ....
//
type UserServiceRemoveResponse struct {
}

//
// Removes the system user.
// Usage:
// ....
// DELETE /ovirt-engine/api/users/1234
// ....
//
func (p *UserService) Remove() *UserServiceRemoveRequest {
	return &UserServiceRemoveRequest{UserService: p}
}

//
//
func (op *UserService) GroupsService() *DomainUserGroupsService {
	return NewDomainUserGroupsService(op.connection, fmt.Sprintf("%s/groups", op.path))
}

//
//
func (op *UserService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *UserService) RolesService() *AssignedRolesService {
	return NewAssignedRolesService(op.connection, fmt.Sprintf("%s/roles", op.path))
}

//
//
func (op *UserService) SshPublicKeysService() *SshPublicKeysService {
	return NewSshPublicKeysService(op.connection, fmt.Sprintf("%s/sshpublickeys", op.path))
}

//
//
func (op *UserService) TagsService() *AssignedTagsService {
	return NewAssignedTagsService(op.connection, fmt.Sprintf("%s/tags", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *UserService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "groups" {
		return op.GroupsService(), nil
	}
	if strings.HasPrefix(path, "groups/") {
		return op.GroupsService().Service(path[7:])
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "roles" {
		return op.RolesService(), nil
	}
	if strings.HasPrefix(path, "roles/") {
		return op.RolesService().Service(path[6:])
	}
	if path == "sshpublickeys" {
		return op.SshPublicKeysService(), nil
	}
	if strings.HasPrefix(path, "sshpublickeys/") {
		return op.SshPublicKeysService().Service(path[14:])
	}
	if path == "tags" {
		return op.TagsService(), nil
	}
	if strings.HasPrefix(path, "tags/") {
		return op.TagsService().Service(path[5:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *UserService) String() string {
	return fmt.Sprintf("UserService:%s", op.path)
}

//
// Manages the collection of groups of users.
//
type GroupsService struct {
	BaseService
}

func NewGroupsService(connection *Connection, path string) *GroupsService {
	var result GroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add group from a directory service. Please note that domain name is name of the authorization provider.
// For example, to add the `Developers` group from the `internal-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/groups
// ----
// With a request body like this:
// [source,xml]
// ----
// <group>
//   <name>Developers</name>
//   <domain>
//     <name>internal-authz</name>
//   </domain>
// </group>
// ----
//
type GroupsServiceAddRequest struct {
	GroupsService *GroupsService
	header        map[string]string
	query         map[string]string
	group         *Group
}

func (p *GroupsServiceAddRequest) Header(key, value string) *GroupsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupsServiceAddRequest) Query(key, value string) *GroupsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupsServiceAddRequest) Group(group *Group) *GroupsServiceAddRequest {
	p.group = group
	return p
}

func (p *GroupsServiceAddRequest) Send() (*GroupsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupsService.connection.URL(), p.GroupsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLGroupWriteOne(writer, p.group, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GroupsServiceAddResponse{group: result}, nil
}

func (p *GroupsServiceAddRequest) MustSend() *GroupsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add group from a directory service. Please note that domain name is name of the authorization provider.
// For example, to add the `Developers` group from the `internal-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/groups
// ----
// With a request body like this:
// [source,xml]
// ----
// <group>
//   <name>Developers</name>
//   <domain>
//     <name>internal-authz</name>
//   </domain>
// </group>
// ----
//
type GroupsServiceAddResponse struct {
	group *Group
}

func (p *GroupsServiceAddResponse) Group() (*Group, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

func (p *GroupsServiceAddResponse) MustGroup() *Group {
	if p.group == nil {
		panic("group in response does not exist")
	}
	return p.group
}

//
// Add group from a directory service. Please note that domain name is name of the authorization provider.
// For example, to add the `Developers` group from the `internal-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/groups
// ----
// With a request body like this:
// [source,xml]
// ----
// <group>
//   <name>Developers</name>
//   <domain>
//     <name>internal-authz</name>
//   </domain>
// </group>
// ----
//
func (p *GroupsService) Add() *GroupsServiceAddRequest {
	return &GroupsServiceAddRequest{GroupsService: p}
}

//
// List all the groups in the system.
// Usage:
// ....
// GET /ovirt-engine/api/groups
// ....
// Will return the list of groups:
// [source,xml]
// ----
// <groups>
//   <group href="/ovirt-engine/api/groups/123" id="123">
//     <name>mygroup</name>
//     <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//     <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//     <namespace>DC=example,DC=com</namespace>
//     <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//       <name>myextension-authz</name>
//     </domain>
//   </group>
//   ...
// </groups>
// ----
// The order of the returned list of groups isn't guaranteed.
//
type GroupsServiceListRequest struct {
	GroupsService *GroupsService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *GroupsServiceListRequest) Header(key, value string) *GroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupsServiceListRequest) Query(key, value string) *GroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupsServiceListRequest) CaseSensitive(caseSensitive bool) *GroupsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *GroupsServiceListRequest) Follow(follow string) *GroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GroupsServiceListRequest) Max(max int64) *GroupsServiceListRequest {
	p.max = &max
	return p
}

func (p *GroupsServiceListRequest) Search(search string) *GroupsServiceListRequest {
	p.search = &search
	return p
}

func (p *GroupsServiceListRequest) Send() (*GroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupsService.connection.URL(), p.GroupsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GroupsServiceListResponse{groups: result}, nil
}

func (p *GroupsServiceListRequest) MustSend() *GroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the groups in the system.
// Usage:
// ....
// GET /ovirt-engine/api/groups
// ....
// Will return the list of groups:
// [source,xml]
// ----
// <groups>
//   <group href="/ovirt-engine/api/groups/123" id="123">
//     <name>mygroup</name>
//     <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//     <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//     <namespace>DC=example,DC=com</namespace>
//     <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//       <name>myextension-authz</name>
//     </domain>
//   </group>
//   ...
// </groups>
// ----
// The order of the returned list of groups isn't guaranteed.
//
type GroupsServiceListResponse struct {
	groups *GroupSlice
}

func (p *GroupsServiceListResponse) Groups() (*GroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *GroupsServiceListResponse) MustGroups() *GroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// List all the groups in the system.
// Usage:
// ....
// GET /ovirt-engine/api/groups
// ....
// Will return the list of groups:
// [source,xml]
// ----
// <groups>
//   <group href="/ovirt-engine/api/groups/123" id="123">
//     <name>mygroup</name>
//     <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//     <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//     <namespace>DC=example,DC=com</namespace>
//     <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//       <name>myextension-authz</name>
//     </domain>
//   </group>
//   ...
// </groups>
// ----
// The order of the returned list of groups isn't guaranteed.
//
func (p *GroupsService) List() *GroupsServiceListRequest {
	return &GroupsServiceListRequest{GroupsService: p}
}

//
// Reference to the service that manages a specific group.
//
func (op *GroupsService) GroupService(id string) *GroupService {
	return NewGroupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.GroupService(path), nil
	}
	return op.GroupService(path[:index]).Service(path[index+1:])
}

func (op *GroupsService) String() string {
	return fmt.Sprintf("GroupsService:%s", op.path)
}

//
// A service to list all authentication domains in the system.
//
type DomainsService struct {
	BaseService
}

func NewDomainsService(connection *Connection, path string) *DomainsService {
	var result DomainsService
	result.connection = connection
	result.path = path
	return &result
}

//
// List all the authentication domains in the system.
// Usage:
// ....
// GET /ovirt-engine/api/domains
// ....
// Will return the list of domains:
// [source,xml]
// ----
// <domains>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//     <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//     <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//     <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//     <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
//   </domain>
// </domains>
// ----
// The order of the returned list of domains isn't guaranteed.
//
type DomainsServiceListRequest struct {
	DomainsService *DomainsService
	header         map[string]string
	query          map[string]string
	follow         *string
	max            *int64
}

func (p *DomainsServiceListRequest) Header(key, value string) *DomainsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainsServiceListRequest) Query(key, value string) *DomainsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainsServiceListRequest) Follow(follow string) *DomainsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainsServiceListRequest) Max(max int64) *DomainsServiceListRequest {
	p.max = &max
	return p
}

func (p *DomainsServiceListRequest) Send() (*DomainsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainsService.connection.URL(), p.DomainsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDomainReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainsServiceListResponse{domains: result}, nil
}

func (p *DomainsServiceListRequest) MustSend() *DomainsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the authentication domains in the system.
// Usage:
// ....
// GET /ovirt-engine/api/domains
// ....
// Will return the list of domains:
// [source,xml]
// ----
// <domains>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//     <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//     <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//     <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//     <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
//   </domain>
// </domains>
// ----
// The order of the returned list of domains isn't guaranteed.
//
type DomainsServiceListResponse struct {
	domains *DomainSlice
}

func (p *DomainsServiceListResponse) Domains() (*DomainSlice, bool) {
	if p.domains != nil {
		return p.domains, true
	}
	return nil, false
}

func (p *DomainsServiceListResponse) MustDomains() *DomainSlice {
	if p.domains == nil {
		panic("domains in response does not exist")
	}
	return p.domains
}

//
// List all the authentication domains in the system.
// Usage:
// ....
// GET /ovirt-engine/api/domains
// ....
// Will return the list of domains:
// [source,xml]
// ----
// <domains>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//     <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//     <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//     <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//     <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
//   </domain>
// </domains>
// ----
// The order of the returned list of domains isn't guaranteed.
//
func (p *DomainsService) List() *DomainsServiceListRequest {
	return &DomainsServiceListRequest{DomainsService: p}
}

//
// Reference to a service to view details of a domain.
//
func (op *DomainsService) DomainService(id string) *DomainService {
	return NewDomainService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DomainService(path), nil
	}
	return op.DomainService(path[:index]).Service(path[index+1:])
}

func (op *DomainsService) String() string {
	return fmt.Sprintf("DomainsService:%s", op.path)
}

//
// A service that shows a user's group membership in the AAA extension.
//
type DomainUserGroupsService struct {
	BaseService
}

func NewDomainUserGroupsService(connection *Connection, path string) *DomainUserGroupsService {
	var result DomainUserGroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of groups that the user is a member of.
//
type DomainUserGroupsServiceListRequest struct {
	DomainUserGroupsService *DomainUserGroupsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *DomainUserGroupsServiceListRequest) Header(key, value string) *DomainUserGroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainUserGroupsServiceListRequest) Query(key, value string) *DomainUserGroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainUserGroupsServiceListRequest) Follow(follow string) *DomainUserGroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainUserGroupsServiceListRequest) Send() (*DomainUserGroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainUserGroupsService.connection.URL(), p.DomainUserGroupsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainUserGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainUserGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainUserGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainUserGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainUserGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainUserGroupsServiceListResponse{groups: result}, nil
}

func (p *DomainUserGroupsServiceListRequest) MustSend() *DomainUserGroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of groups that the user is a member of.
//
type DomainUserGroupsServiceListResponse struct {
	groups *GroupSlice
}

func (p *DomainUserGroupsServiceListResponse) Groups() (*GroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *DomainUserGroupsServiceListResponse) MustGroups() *GroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// Returns the list of groups that the user is a member of.
//
func (p *DomainUserGroupsService) List() *DomainUserGroupsServiceListRequest {
	return &DomainUserGroupsServiceListRequest{DomainUserGroupsService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainUserGroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainUserGroupsService) String() string {
	return fmt.Sprintf("DomainUserGroupsService:%s", op.path)
}

//
//
type SshPublicKeyService struct {
	BaseService
}

func NewSshPublicKeyService(connection *Connection, path string) *SshPublicKeyService {
	var result SshPublicKeyService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SshPublicKeyServiceGetRequest struct {
	SshPublicKeyService *SshPublicKeyService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *SshPublicKeyServiceGetRequest) Header(key, value string) *SshPublicKeyServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeyServiceGetRequest) Query(key, value string) *SshPublicKeyServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeyServiceGetRequest) Follow(follow string) *SshPublicKeyServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SshPublicKeyServiceGetRequest) Send() (*SshPublicKeyServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeyService.connection.URL(), p.SshPublicKeyService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SshPublicKeyServiceGetResponse{key: result}, nil
}

func (p *SshPublicKeyServiceGetRequest) MustSend() *SshPublicKeyServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeyServiceGetResponse struct {
	key *SshPublicKey
}

func (p *SshPublicKeyServiceGetResponse) Key() (*SshPublicKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *SshPublicKeyServiceGetResponse) MustKey() *SshPublicKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *SshPublicKeyService) Get() *SshPublicKeyServiceGetRequest {
	return &SshPublicKeyServiceGetRequest{SshPublicKeyService: p}
}

//
//
type SshPublicKeyServiceRemoveRequest struct {
	SshPublicKeyService *SshPublicKeyService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *SshPublicKeyServiceRemoveRequest) Header(key, value string) *SshPublicKeyServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeyServiceRemoveRequest) Query(key, value string) *SshPublicKeyServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeyServiceRemoveRequest) Async(async bool) *SshPublicKeyServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *SshPublicKeyServiceRemoveRequest) Send() (*SshPublicKeyServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeyService.connection.URL(), p.SshPublicKeyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(SshPublicKeyServiceRemoveResponse), nil
}

func (p *SshPublicKeyServiceRemoveRequest) MustSend() *SshPublicKeyServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeyServiceRemoveResponse struct {
}

//
//
func (p *SshPublicKeyService) Remove() *SshPublicKeyServiceRemoveRequest {
	return &SshPublicKeyServiceRemoveRequest{SshPublicKeyService: p}
}

//
//
type SshPublicKeyServiceUpdateRequest struct {
	SshPublicKeyService *SshPublicKeyService
	header              map[string]string
	query               map[string]string
	async               *bool
	key                 *SshPublicKey
}

func (p *SshPublicKeyServiceUpdateRequest) Header(key, value string) *SshPublicKeyServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Query(key, value string) *SshPublicKeyServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Async(async bool) *SshPublicKeyServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Key(key *SshPublicKey) *SshPublicKeyServiceUpdateRequest {
	p.key = key
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Send() (*SshPublicKeyServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeyService.connection.URL(), p.SshPublicKeyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSshPublicKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SshPublicKeyServiceUpdateResponse{key: result}, nil
}

func (p *SshPublicKeyServiceUpdateRequest) MustSend() *SshPublicKeyServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeyServiceUpdateResponse struct {
	key *SshPublicKey
}

func (p *SshPublicKeyServiceUpdateResponse) Key() (*SshPublicKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *SshPublicKeyServiceUpdateResponse) MustKey() *SshPublicKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *SshPublicKeyService) Update() *SshPublicKeyServiceUpdateRequest {
	return &SshPublicKeyServiceUpdateRequest{SshPublicKeyService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SshPublicKeyService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SshPublicKeyService) String() string {
	return fmt.Sprintf("SshPublicKeyService:%s", op.path)
}

//
// A service to manage the users in the system.
//
type UsersService struct {
	BaseService
}

func NewUsersService(connection *Connection, path string) *UsersService {
	var result UsersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add user from a directory service.
// For example, to add the `myuser` user from the `myextension-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/users
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//   <user_name>myuser@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
// In case you are working with Active Directory you have to pass user principal name (UPN) as `username`, followed
// by authorization provider name. Due to https://bugzilla.redhat.com/1147900[bug 1147900] you need to provide
// also `principal` parameter set to UPN of the user.
// For example, to add the user with UPN `myuser@mysubdomain.mydomain.com` from the `myextension-authz`
// authorization provider send a request body like this:
// [source,xml]
// ----
// <user>
//   <principal>myuser@mysubdomain.mydomain.com</principal>
//   <user_name>myuser@mysubdomain.mydomain.com@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
//
type UsersServiceAddRequest struct {
	UsersService *UsersService
	header       map[string]string
	query        map[string]string
	user         *User
}

func (p *UsersServiceAddRequest) Header(key, value string) *UsersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UsersServiceAddRequest) Query(key, value string) *UsersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UsersServiceAddRequest) User(user *User) *UsersServiceAddRequest {
	p.user = user
	return p
}

func (p *UsersServiceAddRequest) Send() (*UsersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UsersService.connection.URL(), p.UsersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLUserWriteOne(writer, p.user, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UsersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UsersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UsersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UsersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UsersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &UsersServiceAddResponse{user: result}, nil
}

func (p *UsersServiceAddRequest) MustSend() *UsersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add user from a directory service.
// For example, to add the `myuser` user from the `myextension-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/users
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//   <user_name>myuser@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
// In case you are working with Active Directory you have to pass user principal name (UPN) as `username`, followed
// by authorization provider name. Due to https://bugzilla.redhat.com/1147900[bug 1147900] you need to provide
// also `principal` parameter set to UPN of the user.
// For example, to add the user with UPN `myuser@mysubdomain.mydomain.com` from the `myextension-authz`
// authorization provider send a request body like this:
// [source,xml]
// ----
// <user>
//   <principal>myuser@mysubdomain.mydomain.com</principal>
//   <user_name>myuser@mysubdomain.mydomain.com@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
//
type UsersServiceAddResponse struct {
	user *User
}

func (p *UsersServiceAddResponse) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

func (p *UsersServiceAddResponse) MustUser() *User {
	if p.user == nil {
		panic("user in response does not exist")
	}
	return p.user
}

//
// Add user from a directory service.
// For example, to add the `myuser` user from the `myextension-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/users
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//   <user_name>myuser@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
// In case you are working with Active Directory you have to pass user principal name (UPN) as `username`, followed
// by authorization provider name. Due to https://bugzilla.redhat.com/1147900[bug 1147900] you need to provide
// also `principal` parameter set to UPN of the user.
// For example, to add the user with UPN `myuser@mysubdomain.mydomain.com` from the `myextension-authz`
// authorization provider send a request body like this:
// [source,xml]
// ----
// <user>
//   <principal>myuser@mysubdomain.mydomain.com</principal>
//   <user_name>myuser@mysubdomain.mydomain.com@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
//
func (p *UsersService) Add() *UsersServiceAddRequest {
	return &UsersServiceAddRequest{UsersService: p}
}

//
// List all the users in the system.
// Usage:
// ....
// GET /ovirt-engine/api/users
// ....
// Will return the list of users:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/users/1234" id="1234">
//     <name>admin</name>
//     <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//     <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//     <domain_entry_id>23456</domain_entry_id>
//     <namespace>*</namespace>
//     <principal>user1</principal>
//     <user_name>user1@domain-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/45678" id="45678">
//       <name>domain-authz</name>
//     </domain>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type UsersServiceListRequest struct {
	UsersService  *UsersService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *UsersServiceListRequest) Header(key, value string) *UsersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UsersServiceListRequest) Query(key, value string) *UsersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UsersServiceListRequest) CaseSensitive(caseSensitive bool) *UsersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *UsersServiceListRequest) Follow(follow string) *UsersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *UsersServiceListRequest) Max(max int64) *UsersServiceListRequest {
	p.max = &max
	return p
}

func (p *UsersServiceListRequest) Search(search string) *UsersServiceListRequest {
	p.search = &search
	return p
}

func (p *UsersServiceListRequest) Send() (*UsersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UsersService.connection.URL(), p.UsersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UsersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UsersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UsersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UsersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UsersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &UsersServiceListResponse{users: result}, nil
}

func (p *UsersServiceListRequest) MustSend() *UsersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the users in the system.
// Usage:
// ....
// GET /ovirt-engine/api/users
// ....
// Will return the list of users:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/users/1234" id="1234">
//     <name>admin</name>
//     <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//     <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//     <domain_entry_id>23456</domain_entry_id>
//     <namespace>*</namespace>
//     <principal>user1</principal>
//     <user_name>user1@domain-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/45678" id="45678">
//       <name>domain-authz</name>
//     </domain>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type UsersServiceListResponse struct {
	users *UserSlice
}

func (p *UsersServiceListResponse) Users() (*UserSlice, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

func (p *UsersServiceListResponse) MustUsers() *UserSlice {
	if p.users == nil {
		panic("users in response does not exist")
	}
	return p.users
}

//
// List all the users in the system.
// Usage:
// ....
// GET /ovirt-engine/api/users
// ....
// Will return the list of users:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/users/1234" id="1234">
//     <name>admin</name>
//     <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//     <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//     <domain_entry_id>23456</domain_entry_id>
//     <namespace>*</namespace>
//     <principal>user1</principal>
//     <user_name>user1@domain-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/45678" id="45678">
//       <name>domain-authz</name>
//     </domain>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
func (p *UsersService) List() *UsersServiceListRequest {
	return &UsersServiceListRequest{UsersService: p}
}

//
//
func (op *UsersService) UserService(id string) *UserService {
	return NewUserService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *UsersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.UserService(path), nil
	}
	return op.UserService(path[:index]).Service(path[index+1:])
}

func (op *UsersService) String() string {
	return fmt.Sprintf("UsersService:%s", op.path)
}

//
// A service to view a domain user in the system.
//
type DomainUserService struct {
	BaseService
}

func NewDomainUserService(connection *Connection, path string) *DomainUserService {
	var result DomainUserService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the domain user information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users/1234
// ....
// Will return the domain user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <namespace>*</namespace>
//   <principal>admin</principal>
//   <user_name>admin@internal-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//   </domain>
//   <groups/>
// </user>
// ----
//
type DomainUserServiceGetRequest struct {
	DomainUserService *DomainUserService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *DomainUserServiceGetRequest) Header(key, value string) *DomainUserServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainUserServiceGetRequest) Query(key, value string) *DomainUserServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainUserServiceGetRequest) Follow(follow string) *DomainUserServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DomainUserServiceGetRequest) Send() (*DomainUserServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainUserService.connection.URL(), p.DomainUserService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainUserService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainUserService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainUserService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainUserService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainUserService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DomainUserServiceGetResponse{user: result}, nil
}

func (p *DomainUserServiceGetRequest) MustSend() *DomainUserServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the domain user information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users/1234
// ....
// Will return the domain user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <namespace>*</namespace>
//   <principal>admin</principal>
//   <user_name>admin@internal-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//   </domain>
//   <groups/>
// </user>
// ----
//
type DomainUserServiceGetResponse struct {
	user *User
}

func (p *DomainUserServiceGetResponse) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

func (p *DomainUserServiceGetResponse) MustUser() *User {
	if p.user == nil {
		panic("user in response does not exist")
	}
	return p.user
}

//
// Gets the domain user information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users/1234
// ....
// Will return the domain user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <namespace>*</namespace>
//   <principal>admin</principal>
//   <user_name>admin@internal-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//   </domain>
//   <groups/>
// </user>
// ----
//
func (p *DomainUserService) Get() *DomainUserServiceGetRequest {
	return &DomainUserServiceGetRequest{DomainUserService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainUserService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainUserService) String() string {
	return fmt.Sprintf("DomainUserService:%s", op.path)
}

//
// A service to view details of an authentication domain in the system.
//
type DomainService struct {
	BaseService
}

func NewDomainService(connection *Connection, path string) *DomainService {
	var result DomainService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the authentication domain information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678
// ....
// Will return the domain information:
// [source,xml]
// ----
// <domain href="/ovirt-engine/api/domains/5678" id="5678">
//   <name>internal-authz</name>
//   <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//   <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//   <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//   <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
// </domain>
// ----
//
type DomainServiceGetRequest struct {
	DomainService *DomainService
	header        map[string]string
	query         map[string]string
	follow        *string
}

func (p *DomainServiceGetRequest) Header(key, value string) *DomainServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainServiceGetRequest) Query(key, value string) *DomainServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainServiceGetRequest) Follow(follow string) *DomainServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DomainServiceGetRequest) Send() (*DomainServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainService.connection.URL(), p.DomainService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DomainServiceGetResponse{domain: result}, nil
}

func (p *DomainServiceGetRequest) MustSend() *DomainServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the authentication domain information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678
// ....
// Will return the domain information:
// [source,xml]
// ----
// <domain href="/ovirt-engine/api/domains/5678" id="5678">
//   <name>internal-authz</name>
//   <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//   <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//   <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//   <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
// </domain>
// ----
//
type DomainServiceGetResponse struct {
	domain *Domain
}

func (p *DomainServiceGetResponse) Domain() (*Domain, bool) {
	if p.domain != nil {
		return p.domain, true
	}
	return nil, false
}

func (p *DomainServiceGetResponse) MustDomain() *Domain {
	if p.domain == nil {
		panic("domain in response does not exist")
	}
	return p.domain
}

//
// Gets the authentication domain information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678
// ....
// Will return the domain information:
// [source,xml]
// ----
// <domain href="/ovirt-engine/api/domains/5678" id="5678">
//   <name>internal-authz</name>
//   <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//   <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//   <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//   <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
// </domain>
// ----
//
func (p *DomainService) Get() *DomainServiceGetRequest {
	return &DomainServiceGetRequest{DomainService: p}
}

//
// Reference to a service to manage domain groups.
//
func (op *DomainService) GroupsService() *DomainGroupsService {
	return NewDomainGroupsService(op.connection, fmt.Sprintf("%s/groups", op.path))
}

//
// Reference to a service to manage domain users.
//
func (op *DomainService) UsersService() *DomainUsersService {
	return NewDomainUsersService(op.connection, fmt.Sprintf("%s/users", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "groups" {
		return op.GroupsService(), nil
	}
	if strings.HasPrefix(path, "groups/") {
		return op.GroupsService().Service(path[7:])
	}
	if path == "users" {
		return op.UsersService(), nil
	}
	if strings.HasPrefix(path, "users/") {
		return op.UsersService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainService) String() string {
	return fmt.Sprintf("DomainService:%s", op.path)
}

//
// The details of a single affinity label.
//
type AffinityLabelService struct {
	BaseService
}

func NewAffinityLabelService(connection *Connection, path string) *AffinityLabelService {
	var result AffinityLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the details of a label.
//
type AffinityLabelServiceGetRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *AffinityLabelServiceGetRequest) Header(key, value string) *AffinityLabelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceGetRequest) Query(key, value string) *AffinityLabelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceGetRequest) Follow(follow string) *AffinityLabelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelServiceGetRequest) Send() (*AffinityLabelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelServiceGetResponse{label: result}, nil
}

func (p *AffinityLabelServiceGetRequest) MustSend() *AffinityLabelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the details of a label.
//
type AffinityLabelServiceGetResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelServiceGetResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelServiceGetResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Retrieves the details of a label.
//
func (p *AffinityLabelService) Get() *AffinityLabelServiceGetRequest {
	return &AffinityLabelServiceGetRequest{AffinityLabelService: p}
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
type AffinityLabelServiceRemoveRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
}

func (p *AffinityLabelServiceRemoveRequest) Header(key, value string) *AffinityLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceRemoveRequest) Query(key, value string) *AffinityLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceRemoveRequest) Send() (*AffinityLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityLabelServiceRemoveResponse), nil
}

func (p *AffinityLabelServiceRemoveRequest) MustSend() *AffinityLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
type AffinityLabelServiceRemoveResponse struct {
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
func (p *AffinityLabelService) Remove() *AffinityLabelServiceRemoveRequest {
	return &AffinityLabelServiceRemoveRequest{AffinityLabelService: p}
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
type AffinityLabelServiceUpdateRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
	label                *AffinityLabel
}

func (p *AffinityLabelServiceUpdateRequest) Header(key, value string) *AffinityLabelServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Query(key, value string) *AffinityLabelServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Label(label *AffinityLabel) *AffinityLabelServiceUpdateRequest {
	p.label = label
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Send() (*AffinityLabelServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelServiceUpdateResponse{label: result}, nil
}

func (p *AffinityLabelServiceUpdateRequest) MustSend() *AffinityLabelServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
type AffinityLabelServiceUpdateResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelServiceUpdateResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelServiceUpdateResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
func (p *AffinityLabelService) Update() *AffinityLabelServiceUpdateRequest {
	return &AffinityLabelServiceUpdateRequest{AffinityLabelService: p}
}

//
// List all hosts with this label.
//
func (op *AffinityLabelService) HostsService() *AffinityLabelHostsService {
	return NewAffinityLabelHostsService(op.connection, fmt.Sprintf("%s/hosts", op.path))
}

//
// List all virtual machines with this label.
//
func (op *AffinityLabelService) VmsService() *AffinityLabelVmsService {
	return NewAffinityLabelVmsService(op.connection, fmt.Sprintf("%s/vms", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "hosts" {
		return op.HostsService(), nil
	}
	if strings.HasPrefix(path, "hosts/") {
		return op.HostsService().Service(path[6:])
	}
	if path == "vms" {
		return op.VmsService(), nil
	}
	if strings.HasPrefix(path, "vms/") {
		return op.VmsService().Service(path[4:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityLabelService) String() string {
	return fmt.Sprintf("AffinityLabelService:%s", op.path)
}

//
// A service to manage a network interface of a host.
//
type HostNicService struct {
	BaseService
}

func NewHostNicService(connection *Connection, path string) *HostNicService {
	var result HostNicService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type HostNicServiceGetRequest struct {
	HostNicService *HostNicService
	header         map[string]string
	query          map[string]string
	follow         *string
}

func (p *HostNicServiceGetRequest) Header(key, value string) *HostNicServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNicServiceGetRequest) Query(key, value string) *HostNicServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNicServiceGetRequest) Follow(follow string) *HostNicServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostNicServiceGetRequest) Send() (*HostNicServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostNicService.connection.URL(), p.HostNicService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostNicServiceGetResponse{nic: result}, nil
}

func (p *HostNicServiceGetRequest) MustSend() *HostNicServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type HostNicServiceGetResponse struct {
	nic *HostNic
}

func (p *HostNicServiceGetResponse) Nic() (*HostNic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *HostNicServiceGetResponse) MustNic() *HostNic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
//
func (p *HostNicService) Get() *HostNicServiceGetRequest {
	return &HostNicServiceGetRequest{HostNicService: p}
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
type HostNicServiceUpdateVirtualFunctionsConfigurationRequest struct {
	HostNicService                *HostNicService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
	virtualFunctionsConfiguration *HostNicVirtualFunctionsConfiguration
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Header(key, value string) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Query(key, value string) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Async(async bool) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	p.async = &async
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) VirtualFunctionsConfiguration(virtualFunctionsConfiguration *HostNicVirtualFunctionsConfiguration) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	p.virtualFunctionsConfiguration = virtualFunctionsConfiguration
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Send() (*HostNicServiceUpdateVirtualFunctionsConfigurationResponse, error) {
	rawURL := fmt.Sprintf("%s%s/updatevirtualfunctionsconfiguration", p.HostNicService.connection.URL(), p.HostNicService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.VirtualFunctionsConfiguration(p.virtualFunctionsConfiguration)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostNicServiceUpdateVirtualFunctionsConfigurationResponse), nil
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) MustSend() *HostNicServiceUpdateVirtualFunctionsConfigurationResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
type HostNicServiceUpdateVirtualFunctionsConfigurationResponse struct {
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
func (p *HostNicService) UpdateVirtualFunctionsConfiguration() *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	return &HostNicServiceUpdateVirtualFunctionsConfigurationRequest{HostNicService: p}
}

//
// A reference to information elements received by LLDP on the NIC.
//
func (op *HostNicService) LinkLayerDiscoveryProtocolElementsService() *LinkLayerDiscoveryProtocolService {
	return NewLinkLayerDiscoveryProtocolService(op.connection, fmt.Sprintf("%s/linklayerdiscoveryprotocolelements", op.path))
}

//
// Reference to the service that manages the network attachments assigned to this network interface.
//
func (op *HostNicService) NetworkAttachmentsService() *NetworkAttachmentsService {
	return NewNetworkAttachmentsService(op.connection, fmt.Sprintf("%s/networkattachments", op.path))
}

//
// Reference to the service that manages the network labels assigned to this network interface.
//
func (op *HostNicService) NetworkLabelsService() *NetworkLabelsService {
	return NewNetworkLabelsService(op.connection, fmt.Sprintf("%s/networklabels", op.path))
}

//
//
func (op *HostNicService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Retrieves sub-collection resource of network labels that are allowed on an the virtual functions
// in case that the current resource represents an SR-IOV physical function NIC.
//
func (op *HostNicService) VirtualFunctionAllowedLabelsService() *NetworkLabelsService {
	return NewNetworkLabelsService(op.connection, fmt.Sprintf("%s/virtualfunctionallowedlabels", op.path))
}

//
// Retrieves sub-collection resource of networks that are allowed on an the virtual functions
// in case that the current resource represents an SR-IOV physical function NIC.
//
func (op *HostNicService) VirtualFunctionAllowedNetworksService() *VirtualFunctionAllowedNetworksService {
	return NewVirtualFunctionAllowedNetworksService(op.connection, fmt.Sprintf("%s/virtualfunctionallowednetworks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostNicService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "linklayerdiscoveryprotocolelements" {
		return op.LinkLayerDiscoveryProtocolElementsService(), nil
	}
	if strings.HasPrefix(path, "linklayerdiscoveryprotocolelements/") {
		return op.LinkLayerDiscoveryProtocolElementsService().Service(path[35:])
	}
	if path == "networkattachments" {
		return op.NetworkAttachmentsService(), nil
	}
	if strings.HasPrefix(path, "networkattachments/") {
		return op.NetworkAttachmentsService().Service(path[19:])
	}
	if path == "networklabels" {
		return op.NetworkLabelsService(), nil
	}
	if strings.HasPrefix(path, "networklabels/") {
		return op.NetworkLabelsService().Service(path[14:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	if path == "virtualfunctionallowedlabels" {
		return op.VirtualFunctionAllowedLabelsService(), nil
	}
	if strings.HasPrefix(path, "virtualfunctionallowedlabels/") {
		return op.VirtualFunctionAllowedLabelsService().Service(path[29:])
	}
	if path == "virtualfunctionallowednetworks" {
		return op.VirtualFunctionAllowedNetworksService(), nil
	}
	if strings.HasPrefix(path, "virtualfunctionallowednetworks/") {
		return op.VirtualFunctionAllowedNetworksService().Service(path[31:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostNicService) String() string {
	return fmt.Sprintf("HostNicService:%s", op.path)
}

//
// Manages a single disk available in a storage domain attached to a data center.
// IMPORTANT: Since version 4.2 of the engine this service is intended only to list disks available in the storage
// domain, and to register unregistered disks. All the other operations, like copying a disk, moving a disk, etc, have
// been deprecated and will be removed in the future. To perform those operations use the <<services/disks, service
// that manages all the disks of the system>>, or the <<services/disk, service that manages an specific disk>>.
//
type AttachedStorageDomainDiskService struct {
	BaseService
}

func NewAttachedStorageDomainDiskService(connection *Connection, path string) *AttachedStorageDomainDiskService {
	var result AttachedStorageDomainDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceCopyRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	disk                             *Disk
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Header(key, value string) *AttachedStorageDomainDiskServiceCopyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Query(key, value string) *AttachedStorageDomainDiskServiceCopyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Disk(disk *Disk) *AttachedStorageDomainDiskServiceCopyRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceCopyRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Send() (*AttachedStorageDomainDiskServiceCopyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copy", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceCopyResponse), nil
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) MustSend() *AttachedStorageDomainDiskServiceCopyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceCopyResponse struct {
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Copy() *AttachedStorageDomainDiskServiceCopyRequest {
	return &AttachedStorageDomainDiskServiceCopyRequest{AttachedStorageDomainDiskService: p}
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceExportRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Header(key, value string) *AttachedStorageDomainDiskServiceExportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Query(key, value string) *AttachedStorageDomainDiskServiceExportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceExportRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Send() (*AttachedStorageDomainDiskServiceExportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/export", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceExportResponse), nil
}

func (p *AttachedStorageDomainDiskServiceExportRequest) MustSend() *AttachedStorageDomainDiskServiceExportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceExportResponse struct {
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Export() *AttachedStorageDomainDiskServiceExportRequest {
	return &AttachedStorageDomainDiskServiceExportRequest{AttachedStorageDomainDiskService: p}
}

//
// Retrieves the description of the disk.
//
type AttachedStorageDomainDiskServiceGetRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	follow                           *string
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Header(key, value string) *AttachedStorageDomainDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Query(key, value string) *AttachedStorageDomainDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Follow(follow string) *AttachedStorageDomainDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Send() (*AttachedStorageDomainDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDiskServiceGetResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDiskServiceGetRequest) MustSend() *AttachedStorageDomainDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the description of the disk.
//
type AttachedStorageDomainDiskServiceGetResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Retrieves the description of the disk.
//
func (p *AttachedStorageDomainDiskService) Get() *AttachedStorageDomainDiskServiceGetRequest {
	return &AttachedStorageDomainDiskServiceGetRequest{AttachedStorageDomainDiskService: p}
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceMoveRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	async                            *bool
	filter                           *bool
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Header(key, value string) *AttachedStorageDomainDiskServiceMoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Query(key, value string) *AttachedStorageDomainDiskServiceMoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Async(async bool) *AttachedStorageDomainDiskServiceMoveRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Filter(filter bool) *AttachedStorageDomainDiskServiceMoveRequest {
	p.filter = &filter
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceMoveRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Send() (*AttachedStorageDomainDiskServiceMoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/move", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceMoveResponse), nil
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) MustSend() *AttachedStorageDomainDiskServiceMoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceMoveResponse struct {
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Move() *AttachedStorageDomainDiskServiceMoveRequest {
	return &AttachedStorageDomainDiskServiceMoveRequest{AttachedStorageDomainDiskService: p}
}

//
// Registers an unregistered disk.
//
type AttachedStorageDomainDiskServiceRegisterRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Header(key, value string) *AttachedStorageDomainDiskServiceRegisterRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Query(key, value string) *AttachedStorageDomainDiskServiceRegisterRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Send() (*AttachedStorageDomainDiskServiceRegisterResponse, error) {
	rawURL := fmt.Sprintf("%s%s/register", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceRegisterResponse), nil
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) MustSend() *AttachedStorageDomainDiskServiceRegisterResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Registers an unregistered disk.
//
type AttachedStorageDomainDiskServiceRegisterResponse struct {
}

//
// Registers an unregistered disk.
//
func (p *AttachedStorageDomainDiskService) Register() *AttachedStorageDomainDiskServiceRegisterRequest {
	return &AttachedStorageDomainDiskServiceRegisterRequest{AttachedStorageDomainDiskService: p}
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceRemoveRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Header(key, value string) *AttachedStorageDomainDiskServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Query(key, value string) *AttachedStorageDomainDiskServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Send() (*AttachedStorageDomainDiskServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AttachedStorageDomainDiskServiceRemoveResponse), nil
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) MustSend() *AttachedStorageDomainDiskServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceRemoveResponse struct {
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Remove() *AttachedStorageDomainDiskServiceRemoveRequest {
	return &AttachedStorageDomainDiskServiceRemoveRequest{AttachedStorageDomainDiskService: p}
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceSparsifyRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Header(key, value string) *AttachedStorageDomainDiskServiceSparsifyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Query(key, value string) *AttachedStorageDomainDiskServiceSparsifyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Send() (*AttachedStorageDomainDiskServiceSparsifyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/sparsify", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceSparsifyResponse), nil
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) MustSend() *AttachedStorageDomainDiskServiceSparsifyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceSparsifyResponse struct {
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Sparsify() *AttachedStorageDomainDiskServiceSparsifyRequest {
	return &AttachedStorageDomainDiskServiceSparsifyRequest{AttachedStorageDomainDiskService: p}
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceUpdateRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	disk                             *Disk
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Header(key, value string) *AttachedStorageDomainDiskServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Query(key, value string) *AttachedStorageDomainDiskServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Disk(disk *Disk) *AttachedStorageDomainDiskServiceUpdateRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Send() (*AttachedStorageDomainDiskServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDiskServiceUpdateResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) MustSend() *AttachedStorageDomainDiskServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceUpdateResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDiskServiceUpdateResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDiskServiceUpdateResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Update() *AttachedStorageDomainDiskServiceUpdateRequest {
	return &AttachedStorageDomainDiskServiceUpdateRequest{AttachedStorageDomainDiskService: p}
}

//
// Reference to the service that manages the permissions assigned to the disk.
//
func (op *AttachedStorageDomainDiskService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *AttachedStorageDomainDiskService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AttachedStorageDomainDiskService) String() string {
	return fmt.Sprintf("AttachedStorageDomainDiskService:%s", op.path)
}

//
//
type InstanceTypeNicService struct {
	BaseService
}

func NewInstanceTypeNicService(connection *Connection, path string) *InstanceTypeNicService {
	var result InstanceTypeNicService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets network interface configuration of the instance type.
//
type InstanceTypeNicServiceGetRequest struct {
	InstanceTypeNicService *InstanceTypeNicService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *InstanceTypeNicServiceGetRequest) Header(key, value string) *InstanceTypeNicServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeNicServiceGetRequest) Query(key, value string) *InstanceTypeNicServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeNicServiceGetRequest) Follow(follow string) *InstanceTypeNicServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *InstanceTypeNicServiceGetRequest) Send() (*InstanceTypeNicServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeNicService.connection.URL(), p.InstanceTypeNicService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &InstanceTypeNicServiceGetResponse{nic: result}, nil
}

func (p *InstanceTypeNicServiceGetRequest) MustSend() *InstanceTypeNicServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets network interface configuration of the instance type.
//
type InstanceTypeNicServiceGetResponse struct {
	nic *Nic
}

func (p *InstanceTypeNicServiceGetResponse) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *InstanceTypeNicServiceGetResponse) MustNic() *Nic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
// Gets network interface configuration of the instance type.
//
func (p *InstanceTypeNicService) Get() *InstanceTypeNicServiceGetRequest {
	return &InstanceTypeNicServiceGetRequest{InstanceTypeNicService: p}
}

//
// Remove the network interface from the instance type.
//
type InstanceTypeNicServiceRemoveRequest struct {
	InstanceTypeNicService *InstanceTypeNicService
	header                 map[string]string
	query                  map[string]string
	async                  *bool
}

func (p *InstanceTypeNicServiceRemoveRequest) Header(key, value string) *InstanceTypeNicServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeNicServiceRemoveRequest) Query(key, value string) *InstanceTypeNicServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeNicServiceRemoveRequest) Async(async bool) *InstanceTypeNicServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *InstanceTypeNicServiceRemoveRequest) Send() (*InstanceTypeNicServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeNicService.connection.URL(), p.InstanceTypeNicService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(InstanceTypeNicServiceRemoveResponse), nil
}

func (p *InstanceTypeNicServiceRemoveRequest) MustSend() *InstanceTypeNicServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove the network interface from the instance type.
//
type InstanceTypeNicServiceRemoveResponse struct {
}

//
// Remove the network interface from the instance type.
//
func (p *InstanceTypeNicService) Remove() *InstanceTypeNicServiceRemoveRequest {
	return &InstanceTypeNicServiceRemoveRequest{InstanceTypeNicService: p}
}

//
// Updates the network interface configuration of the instance type.
//
type InstanceTypeNicServiceUpdateRequest struct {
	InstanceTypeNicService *InstanceTypeNicService
	header                 map[string]string
	query                  map[string]string
	async                  *bool
	nic                    *Nic
}

func (p *InstanceTypeNicServiceUpdateRequest) Header(key, value string) *InstanceTypeNicServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeNicServiceUpdateRequest) Query(key, value string) *InstanceTypeNicServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeNicServiceUpdateRequest) Async(async bool) *InstanceTypeNicServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *InstanceTypeNicServiceUpdateRequest) Nic(nic *Nic) *InstanceTypeNicServiceUpdateRequest {
	p.nic = nic
	return p
}

func (p *InstanceTypeNicServiceUpdateRequest) Send() (*InstanceTypeNicServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeNicService.connection.URL(), p.InstanceTypeNicService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNicWriteOne(writer, p.nic, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &InstanceTypeNicServiceUpdateResponse{nic: result}, nil
}

func (p *InstanceTypeNicServiceUpdateRequest) MustSend() *InstanceTypeNicServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the network interface configuration of the instance type.
//
type InstanceTypeNicServiceUpdateResponse struct {
	nic *Nic
}

func (p *InstanceTypeNicServiceUpdateResponse) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *InstanceTypeNicServiceUpdateResponse) MustNic() *Nic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
// Updates the network interface configuration of the instance type.
//
func (p *InstanceTypeNicService) Update() *InstanceTypeNicServiceUpdateRequest {
	return &InstanceTypeNicServiceUpdateRequest{InstanceTypeNicService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *InstanceTypeNicService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *InstanceTypeNicService) String() string {
	return fmt.Sprintf("InstanceTypeNicService:%s", op.path)
}

//
// Manages the affinity labels available in the system.
//
type AffinityLabelsService struct {
	BaseService
}

func NewAffinityLabelsService(connection *Connection, path string) *AffinityLabelsService {
	var result AffinityLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new label. The label is automatically attached
// to all entities mentioned in the vms or hosts lists.
//
type AffinityLabelsServiceAddRequest struct {
	AffinityLabelsService *AffinityLabelsService
	header                map[string]string
	query                 map[string]string
	label                 *AffinityLabel
}

func (p *AffinityLabelsServiceAddRequest) Header(key, value string) *AffinityLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelsServiceAddRequest) Query(key, value string) *AffinityLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelsServiceAddRequest) Label(label *AffinityLabel) *AffinityLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *AffinityLabelsServiceAddRequest) Send() (*AffinityLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelsService.connection.URL(), p.AffinityLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelsServiceAddResponse{label: result}, nil
}

func (p *AffinityLabelsServiceAddRequest) MustSend() *AffinityLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new label. The label is automatically attached
// to all entities mentioned in the vms or hosts lists.
//
type AffinityLabelsServiceAddResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelsServiceAddResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelsServiceAddResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Creates a new label. The label is automatically attached
// to all entities mentioned in the vms or hosts lists.
//
func (p *AffinityLabelsService) Add() *AffinityLabelsServiceAddRequest {
	return &AffinityLabelsServiceAddRequest{AffinityLabelsService: p}
}

//
// Lists all labels present in the system.
// The order of the returned labels isn't guaranteed.
//
type AffinityLabelsServiceListRequest struct {
	AffinityLabelsService *AffinityLabelsService
	header                map[string]string
	query                 map[string]string
	follow                *string
	max                   *int64
}

func (p *AffinityLabelsServiceListRequest) Header(key, value string) *AffinityLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelsServiceListRequest) Query(key, value string) *AffinityLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelsServiceListRequest) Follow(follow string) *AffinityLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelsServiceListRequest) Max(max int64) *AffinityLabelsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityLabelsServiceListRequest) Send() (*AffinityLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelsService.connection.URL(), p.AffinityLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityLabelsServiceListResponse{labels: result}, nil
}

func (p *AffinityLabelsServiceListRequest) MustSend() *AffinityLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all labels present in the system.
// The order of the returned labels isn't guaranteed.
//
type AffinityLabelsServiceListResponse struct {
	labels *AffinityLabelSlice
}

func (p *AffinityLabelsServiceListResponse) Labels() (*AffinityLabelSlice, bool) {
	if p.labels != nil {
		return p.labels, true
	}
	return nil, false
}

func (p *AffinityLabelsServiceListResponse) MustLabels() *AffinityLabelSlice {
	if p.labels == nil {
		panic("labels in response does not exist")
	}
	return p.labels
}

//
// Lists all labels present in the system.
// The order of the returned labels isn't guaranteed.
//
func (p *AffinityLabelsService) List() *AffinityLabelsServiceListRequest {
	return &AffinityLabelsServiceListRequest{AffinityLabelsService: p}
}

//
// Link to a single label details.
//
func (op *AffinityLabelsService) LabelService(id string) *AffinityLabelService {
	return NewAffinityLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *AffinityLabelsService) String() string {
	return fmt.Sprintf("AffinityLabelsService:%s", op.path)
}

//
//
type TemplateWatchdogService struct {
	BaseService
}

func NewTemplateWatchdogService(connection *Connection, path string) *TemplateWatchdogService {
	var result TemplateWatchdogService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type TemplateWatchdogServiceGetRequest struct {
	TemplateWatchdogService *TemplateWatchdogService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *TemplateWatchdogServiceGetRequest) Header(key, value string) *TemplateWatchdogServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateWatchdogServiceGetRequest) Query(key, value string) *TemplateWatchdogServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateWatchdogServiceGetRequest) Follow(follow string) *TemplateWatchdogServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *TemplateWatchdogServiceGetRequest) Send() (*TemplateWatchdogServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateWatchdogService.connection.URL(), p.TemplateWatchdogService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateWatchdogService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateWatchdogService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateWatchdogService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateWatchdogService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateWatchdogService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TemplateWatchdogServiceGetResponse{watchdog: result}, nil
}

func (p *TemplateWatchdogServiceGetRequest) MustSend() *TemplateWatchdogServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type TemplateWatchdogServiceGetResponse struct {
	watchdog *Watchdog
}

func (p *TemplateWatchdogServiceGetResponse) Watchdog() (*Watchdog, bool) {
	if p.watchdog != nil {
		return p.watchdog, true
	}
	return nil, false
}

func (p *TemplateWatchdogServiceGetResponse) MustWatchdog() *Watchdog {
	if p.watchdog == nil {
		panic("watchdog in response does not exist")
	}
	return p.watchdog
}

//
//
func (p *TemplateWatchdogService) Get() *TemplateWatchdogServiceGetRequest {
	return &TemplateWatchdogServiceGetRequest{TemplateWatchdogService: p}
}

//
//
type TemplateWatchdogServiceRemoveRequest struct {
	TemplateWatchdogService *TemplateWatchdogService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
}

func (p *TemplateWatchdogServiceRemoveRequest) Header(key, value string) *TemplateWatchdogServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateWatchdogServiceRemoveRequest) Query(key, value string) *TemplateWatchdogServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateWatchdogServiceRemoveRequest) Async(async bool) *TemplateWatchdogServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *TemplateWatchdogServiceRemoveRequest) Send() (*TemplateWatchdogServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateWatchdogService.connection.URL(), p.TemplateWatchdogService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateWatchdogService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateWatchdogService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateWatchdogService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateWatchdogService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateWatchdogService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(TemplateWatchdogServiceRemoveResponse), nil
}

func (p *TemplateWatchdogServiceRemoveRequest) MustSend() *TemplateWatchdogServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type TemplateWatchdogServiceRemoveResponse struct {
}

//
//
func (p *TemplateWatchdogService) Remove() *TemplateWatchdogServiceRemoveRequest {
	return &TemplateWatchdogServiceRemoveRequest{TemplateWatchdogService: p}
}

//
// Update the watchdog for the template identified by the given id.
//
type TemplateWatchdogServiceUpdateRequest struct {
	TemplateWatchdogService *TemplateWatchdogService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
	watchdog                *Watchdog
}

func (p *TemplateWatchdogServiceUpdateRequest) Header(key, value string) *TemplateWatchdogServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateWatchdogServiceUpdateRequest) Query(key, value string) *TemplateWatchdogServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateWatchdogServiceUpdateRequest) Async(async bool) *TemplateWatchdogServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *TemplateWatchdogServiceUpdateRequest) Watchdog(watchdog *Watchdog) *TemplateWatchdogServiceUpdateRequest {
	p.watchdog = watchdog
	return p
}

func (p *TemplateWatchdogServiceUpdateRequest) Send() (*TemplateWatchdogServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateWatchdogService.connection.URL(), p.TemplateWatchdogService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLWatchdogWriteOne(writer, p.watchdog, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateWatchdogService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateWatchdogService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateWatchdogService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateWatchdogService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateWatchdogService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TemplateWatchdogServiceUpdateResponse{watchdog: result}, nil
}

func (p *TemplateWatchdogServiceUpdateRequest) MustSend() *TemplateWatchdogServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the watchdog for the template identified by the given id.
//
type TemplateWatchdogServiceUpdateResponse struct {
	watchdog *Watchdog
}

func (p *TemplateWatchdogServiceUpdateResponse) Watchdog() (*Watchdog, bool) {
	if p.watchdog != nil {
		return p.watchdog, true
	}
	return nil, false
}

func (p *TemplateWatchdogServiceUpdateResponse) MustWatchdog() *Watchdog {
	if p.watchdog == nil {
		panic("watchdog in response does not exist")
	}
	return p.watchdog
}

//
// Update the watchdog for the template identified by the given id.
//
func (p *TemplateWatchdogService) Update() *TemplateWatchdogServiceUpdateRequest {
	return &TemplateWatchdogServiceUpdateRequest{TemplateWatchdogService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *TemplateWatchdogService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *TemplateWatchdogService) String() string {
	return fmt.Sprintf("TemplateWatchdogService:%s", op.path)
}

//
//
type VmsService struct {
	BaseService
}

func NewVmsService(connection *Connection, path string) *VmsService {
	var result VmsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new virtual machine.
// The virtual machine can be created in different ways:
// - From a template. In this case the identifier or name of the template must be provided. For example, using a
//   plain shell script and XML:
// [source,bash]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <vm>
//   <name>myvm</name>
//   <template>
//     <name>Blank</name>
//   </template>
//   <cluster>
//     <name>mycluster</name>
//   </cluster>
// </vm>
// ' \
// "${url}/vms"
// ----
// - From a snapshot. In this case the identifier of the snapshot has to be provided. For example, using a plain
//   shel script and XML:
// [source,bash]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <vm>
//   <name>myvm</name>
//   <snapshots>
//     <snapshot id="266742a5-6a65-483c-816d-d2ce49746680"/>
//   </snapshots>
//   <cluster>
//     <name>mycluster</name>
//   </cluster>
// </vm>
// ' \
// "${url}/vms"
// ----
// When creating a virtual machine from a template or from a snapshot it is usually useful to explicitly indicate
// in what storage domain to create the disks for the virtual machine. If the virtual machine is created from
// a template then this is achieved passing a set of `disk_attachment` elements that indicate the mapping:
// [source,xml]
// ----
// <vm>
//   ...
//   <disk_attachments>
//     <disk_attachment>
//       <disk id="8d4bd566-6c86-4592-a4a7-912dbf93c298">
//         <storage_domains>
//           <storage_domain id="9cb6cb0a-cf1d-41c2-92ca-5a6d665649c9"/>
//         </storage_domains>
//       </disk>
//     <disk_attachment>
//   </disk_attachments>
// </vm>
// ----
// When the virtual machine is created from a snapshot this set of disks is slightly different, it uses the
// `image_id` attribute instead of `id`.
// [source,xml]
// ----
// <vm>
//   ...
//   <disk_attachments>
//     <disk_attachment>
//       <disk>
//         <image_id>8d4bd566-6c86-4592-a4a7-912dbf93c298</image_id>
//         <storage_domains>
//           <storage_domain id="9cb6cb0a-cf1d-41c2-92ca-5a6d665649c9"/>
//         </storage_domains>
//       </disk>
//     <disk_attachment>
//   </disk_attachments>
// </vm>
// ----
// It is possible to specify additional virtual machine parameters in the XML description, e.g. a virtual machine
// of `desktop` type, with 2 GiB of RAM and additional description can be added sending a request body like the
// following:
// [source,xml]
// ----
// <vm>
//   <name>myvm</name>
//   <description>My Desktop Virtual Machine</description>
//   <type>desktop</type>
//   <memory>2147483648</memory>
//   ...
// </vm>
// ----
// A bootable CDROM device can be set like this:
// [source,xml]
// ----
// <vm>
//   ...
//   <os>
//     <boot dev="cdrom"/>
//   </os>
// </vm>
// ----
// In order to boot from CDROM, you first need to insert a disk, as described in the
// <<services/vm_cdrom, CDROM service>>. Then booting from that CDROM can be specified using the `os.boot.devices`
// attribute:
// [source,xml]
// ----
// <vm>
//   ...
//   <os>
//     <boot>
//       <devices>
//         <device>cdrom</device>
//       </devices>
//     </boot>
//   </os>
// </vm>
// ----
// In all cases the name or identifier of the cluster where the virtual machine will be created is mandatory.
//
type VmsServiceAddRequest struct {
	VmsService       *VmsService
	header           map[string]string
	query            map[string]string
	clone            *bool
	clonePermissions *bool
	filter           *bool
	vm               *Vm
}

func (p *VmsServiceAddRequest) Header(key, value string) *VmsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmsServiceAddRequest) Query(key, value string) *VmsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmsServiceAddRequest) Clone(clone bool) *VmsServiceAddRequest {
	p.clone = &clone
	return p
}

func (p *VmsServiceAddRequest) ClonePermissions(clonePermissions bool) *VmsServiceAddRequest {
	p.clonePermissions = &clonePermissions
	return p
}

func (p *VmsServiceAddRequest) Filter(filter bool) *VmsServiceAddRequest {
	p.filter = &filter
	return p
}

func (p *VmsServiceAddRequest) Vm(vm *Vm) *VmsServiceAddRequest {
	p.vm = vm
	return p
}

func (p *VmsServiceAddRequest) Send() (*VmsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmsService.connection.URL(), p.VmsService.path)
	values := make(url.Values)
	if p.clone != nil {
		values["clone"] = []string{fmt.Sprintf("%v", *p.clone)}
	}

	if p.clonePermissions != nil {
		values["clone_permissions"] = []string{fmt.Sprintf("%v", *p.clonePermissions)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVmWriteOne(writer, p.vm, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmsServiceAddResponse{vm: result}, nil
}

func (p *VmsServiceAddRequest) MustSend() *VmsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new virtual machine.
// The virtual machine can be created in different ways:
// - From a template. In this case the identifier or name of the template must be provided. For example, using a
//   plain shell script and XML:
// [source,bash]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <vm>
//   <name>myvm</name>
//   <template>
//     <name>Blank</name>
//   </template>
//   <cluster>
//     <name>mycluster</name>
//   </cluster>
// </vm>
// ' \
// "${url}/vms"
// ----
// - From a snapshot. In this case the identifier of the snapshot has to be provided. For example, using a plain
//   shel script and XML:
// [source,bash]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <vm>
//   <name>myvm</name>
//   <snapshots>
//     <snapshot id="266742a5-6a65-483c-816d-d2ce49746680"/>
//   </snapshots>
//   <cluster>
//     <name>mycluster</name>
//   </cluster>
// </vm>
// ' \
// "${url}/vms"
// ----
// When creating a virtual machine from a template or from a snapshot it is usually useful to explicitly indicate
// in what storage domain to create the disks for the virtual machine. If the virtual machine is created from
// a template then this is achieved passing a set of `disk_attachment` elements that indicate the mapping:
// [source,xml]
// ----
// <vm>
//   ...
//   <disk_attachments>
//     <disk_attachment>
//       <disk id="8d4bd566-6c86-4592-a4a7-912dbf93c298">
//         <storage_domains>
//           <storage_domain id="9cb6cb0a-cf1d-41c2-92ca-5a6d665649c9"/>
//         </storage_domains>
//       </disk>
//     <disk_attachment>
//   </disk_attachments>
// </vm>
// ----
// When the virtual machine is created from a snapshot this set of disks is slightly different, it uses the
// `image_id` attribute instead of `id`.
// [source,xml]
// ----
// <vm>
//   ...
//   <disk_attachments>
//     <disk_attachment>
//       <disk>
//         <image_id>8d4bd566-6c86-4592-a4a7-912dbf93c298</image_id>
//         <storage_domains>
//           <storage_domain id="9cb6cb0a-cf1d-41c2-92ca-5a6d665649c9"/>
//         </storage_domains>
//       </disk>
//     <disk_attachment>
//   </disk_attachments>
// </vm>
// ----
// It is possible to specify additional virtual machine parameters in the XML description, e.g. a virtual machine
// of `desktop` type, with 2 GiB of RAM and additional description can be added sending a request body like the
// following:
// [source,xml]
// ----
// <vm>
//   <name>myvm</name>
//   <description>My Desktop Virtual Machine</description>
//   <type>desktop</type>
//   <memory>2147483648</memory>
//   ...
// </vm>
// ----
// A bootable CDROM device can be set like this:
// [source,xml]
// ----
// <vm>
//   ...
//   <os>
//     <boot dev="cdrom"/>
//   </os>
// </vm>
// ----
// In order to boot from CDROM, you first need to insert a disk, as described in the
// <<services/vm_cdrom, CDROM service>>. Then booting from that CDROM can be specified using the `os.boot.devices`
// attribute:
// [source,xml]
// ----
// <vm>
//   ...
//   <os>
//     <boot>
//       <devices>
//         <device>cdrom</device>
//       </devices>
//     </boot>
//   </os>
// </vm>
// ----
// In all cases the name or identifier of the cluster where the virtual machine will be created is mandatory.
//
type VmsServiceAddResponse struct {
	vm *Vm
}

func (p *VmsServiceAddResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *VmsServiceAddResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// Creates a new virtual machine.
// The virtual machine can be created in different ways:
// - From a template. In this case the identifier or name of the template must be provided. For example, using a
//   plain shell script and XML:
// [source,bash]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <vm>
//   <name>myvm</name>
//   <template>
//     <name>Blank</name>
//   </template>
//   <cluster>
//     <name>mycluster</name>
//   </cluster>
// </vm>
// ' \
// "${url}/vms"
// ----
// - From a snapshot. In this case the identifier of the snapshot has to be provided. For example, using a plain
//   shel script and XML:
// [source,bash]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <vm>
//   <name>myvm</name>
//   <snapshots>
//     <snapshot id="266742a5-6a65-483c-816d-d2ce49746680"/>
//   </snapshots>
//   <cluster>
//     <name>mycluster</name>
//   </cluster>
// </vm>
// ' \
// "${url}/vms"
// ----
// When creating a virtual machine from a template or from a snapshot it is usually useful to explicitly indicate
// in what storage domain to create the disks for the virtual machine. If the virtual machine is created from
// a template then this is achieved passing a set of `disk_attachment` elements that indicate the mapping:
// [source,xml]
// ----
// <vm>
//   ...
//   <disk_attachments>
//     <disk_attachment>
//       <disk id="8d4bd566-6c86-4592-a4a7-912dbf93c298">
//         <storage_domains>
//           <storage_domain id="9cb6cb0a-cf1d-41c2-92ca-5a6d665649c9"/>
//         </storage_domains>
//       </disk>
//     <disk_attachment>
//   </disk_attachments>
// </vm>
// ----
// When the virtual machine is created from a snapshot this set of disks is slightly different, it uses the
// `image_id` attribute instead of `id`.
// [source,xml]
// ----
// <vm>
//   ...
//   <disk_attachments>
//     <disk_attachment>
//       <disk>
//         <image_id>8d4bd566-6c86-4592-a4a7-912dbf93c298</image_id>
//         <storage_domains>
//           <storage_domain id="9cb6cb0a-cf1d-41c2-92ca-5a6d665649c9"/>
//         </storage_domains>
//       </disk>
//     <disk_attachment>
//   </disk_attachments>
// </vm>
// ----
// It is possible to specify additional virtual machine parameters in the XML description, e.g. a virtual machine
// of `desktop` type, with 2 GiB of RAM and additional description can be added sending a request body like the
// following:
// [source,xml]
// ----
// <vm>
//   <name>myvm</name>
//   <description>My Desktop Virtual Machine</description>
//   <type>desktop</type>
//   <memory>2147483648</memory>
//   ...
// </vm>
// ----
// A bootable CDROM device can be set like this:
// [source,xml]
// ----
// <vm>
//   ...
//   <os>
//     <boot dev="cdrom"/>
//   </os>
// </vm>
// ----
// In order to boot from CDROM, you first need to insert a disk, as described in the
// <<services/vm_cdrom, CDROM service>>. Then booting from that CDROM can be specified using the `os.boot.devices`
// attribute:
// [source,xml]
// ----
// <vm>
//   ...
//   <os>
//     <boot>
//       <devices>
//         <device>cdrom</device>
//       </devices>
//     </boot>
//   </os>
// </vm>
// ----
// In all cases the name or identifier of the cluster where the virtual machine will be created is mandatory.
//
func (p *VmsService) Add() *VmsServiceAddRequest {
	return &VmsServiceAddRequest{VmsService: p}
}

//
// add a virtual machine to the system from a configuration - requires the configuration type and the configuration data
//
type VmsServiceAddFromConfigurationRequest struct {
	VmsService       *VmsService
	header           map[string]string
	query            map[string]string
	clone            *bool
	clonePermissions *bool
	filter           *bool
	vm               *Vm
}

func (p *VmsServiceAddFromConfigurationRequest) Header(key, value string) *VmsServiceAddFromConfigurationRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmsServiceAddFromConfigurationRequest) Query(key, value string) *VmsServiceAddFromConfigurationRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmsServiceAddFromConfigurationRequest) Clone(clone bool) *VmsServiceAddFromConfigurationRequest {
	p.clone = &clone
	return p
}

func (p *VmsServiceAddFromConfigurationRequest) ClonePermissions(clonePermissions bool) *VmsServiceAddFromConfigurationRequest {
	p.clonePermissions = &clonePermissions
	return p
}

func (p *VmsServiceAddFromConfigurationRequest) Filter(filter bool) *VmsServiceAddFromConfigurationRequest {
	p.filter = &filter
	return p
}

func (p *VmsServiceAddFromConfigurationRequest) Vm(vm *Vm) *VmsServiceAddFromConfigurationRequest {
	p.vm = vm
	return p
}

func (p *VmsServiceAddFromConfigurationRequest) Send() (*VmsServiceAddFromConfigurationResponse, error) {
	rawURL := fmt.Sprintf("%s%s/fromconfiguration", p.VmsService.connection.URL(), p.VmsService.path)
	actionBuilder := NewActionBuilder()
	if p.clone != nil {
		actionBuilder.Clone(*p.clone)
	}
	if p.clonePermissions != nil {
		actionBuilder.ClonePermissions(*p.clonePermissions)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.Vm(p.vm)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustVm()
	return &VmsServiceAddFromConfigurationResponse{vm: result}, nil
}

func (p *VmsServiceAddFromConfigurationRequest) MustSend() *VmsServiceAddFromConfigurationResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// add a virtual machine to the system from a configuration - requires the configuration type and the configuration data
//
type VmsServiceAddFromConfigurationResponse struct {
	vm *Vm
}

func (p *VmsServiceAddFromConfigurationResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *VmsServiceAddFromConfigurationResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// add a virtual machine to the system from a configuration - requires the configuration type and the configuration data
//
func (p *VmsService) AddFromConfiguration() *VmsServiceAddFromConfigurationRequest {
	return &VmsServiceAddFromConfigurationRequest{VmsService: p}
}

//
// add a virtual machine to the system from scratch
//
type VmsServiceAddFromScratchRequest struct {
	VmsService       *VmsService
	header           map[string]string
	query            map[string]string
	clone            *bool
	clonePermissions *bool
	filter           *bool
	vm               *Vm
}

func (p *VmsServiceAddFromScratchRequest) Header(key, value string) *VmsServiceAddFromScratchRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmsServiceAddFromScratchRequest) Query(key, value string) *VmsServiceAddFromScratchRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmsServiceAddFromScratchRequest) Clone(clone bool) *VmsServiceAddFromScratchRequest {
	p.clone = &clone
	return p
}

func (p *VmsServiceAddFromScratchRequest) ClonePermissions(clonePermissions bool) *VmsServiceAddFromScratchRequest {
	p.clonePermissions = &clonePermissions
	return p
}

func (p *VmsServiceAddFromScratchRequest) Filter(filter bool) *VmsServiceAddFromScratchRequest {
	p.filter = &filter
	return p
}

func (p *VmsServiceAddFromScratchRequest) Vm(vm *Vm) *VmsServiceAddFromScratchRequest {
	p.vm = vm
	return p
}

func (p *VmsServiceAddFromScratchRequest) Send() (*VmsServiceAddFromScratchResponse, error) {
	rawURL := fmt.Sprintf("%s%s/fromscratch", p.VmsService.connection.URL(), p.VmsService.path)
	actionBuilder := NewActionBuilder()
	if p.clone != nil {
		actionBuilder.Clone(*p.clone)
	}
	if p.clonePermissions != nil {
		actionBuilder.ClonePermissions(*p.clonePermissions)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.Vm(p.vm)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustVm()
	return &VmsServiceAddFromScratchResponse{vm: result}, nil
}

func (p *VmsServiceAddFromScratchRequest) MustSend() *VmsServiceAddFromScratchResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// add a virtual machine to the system from scratch
//
type VmsServiceAddFromScratchResponse struct {
	vm *Vm
}

func (p *VmsServiceAddFromScratchResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *VmsServiceAddFromScratchResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// add a virtual machine to the system from scratch
//
func (p *VmsService) AddFromScratch() *VmsServiceAddFromScratchRequest {
	return &VmsServiceAddFromScratchRequest{VmsService: p}
}

//
// add a virtual machine to the system by cloning from a snapshot
//
type VmsServiceAddFromSnapshotRequest struct {
	VmsService       *VmsService
	header           map[string]string
	query            map[string]string
	clone            *bool
	clonePermissions *bool
	filter           *bool
	vm               *Vm
}

func (p *VmsServiceAddFromSnapshotRequest) Header(key, value string) *VmsServiceAddFromSnapshotRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmsServiceAddFromSnapshotRequest) Query(key, value string) *VmsServiceAddFromSnapshotRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmsServiceAddFromSnapshotRequest) Clone(clone bool) *VmsServiceAddFromSnapshotRequest {
	p.clone = &clone
	return p
}

func (p *VmsServiceAddFromSnapshotRequest) ClonePermissions(clonePermissions bool) *VmsServiceAddFromSnapshotRequest {
	p.clonePermissions = &clonePermissions
	return p
}

func (p *VmsServiceAddFromSnapshotRequest) Filter(filter bool) *VmsServiceAddFromSnapshotRequest {
	p.filter = &filter
	return p
}

func (p *VmsServiceAddFromSnapshotRequest) Vm(vm *Vm) *VmsServiceAddFromSnapshotRequest {
	p.vm = vm
	return p
}

func (p *VmsServiceAddFromSnapshotRequest) Send() (*VmsServiceAddFromSnapshotResponse, error) {
	rawURL := fmt.Sprintf("%s%s/fromsnapshot", p.VmsService.connection.URL(), p.VmsService.path)
	actionBuilder := NewActionBuilder()
	if p.clone != nil {
		actionBuilder.Clone(*p.clone)
	}
	if p.clonePermissions != nil {
		actionBuilder.ClonePermissions(*p.clonePermissions)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.Vm(p.vm)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustVm()
	return &VmsServiceAddFromSnapshotResponse{vm: result}, nil
}

func (p *VmsServiceAddFromSnapshotRequest) MustSend() *VmsServiceAddFromSnapshotResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// add a virtual machine to the system by cloning from a snapshot
//
type VmsServiceAddFromSnapshotResponse struct {
	vm *Vm
}

func (p *VmsServiceAddFromSnapshotResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *VmsServiceAddFromSnapshotResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// add a virtual machine to the system by cloning from a snapshot
//
func (p *VmsService) AddFromSnapshot() *VmsServiceAddFromSnapshotRequest {
	return &VmsServiceAddFromSnapshotRequest{VmsService: p}
}

//
// Returns the list of virtual machines of the system.
// The order of the returned list of virtual machines is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
type VmsServiceListRequest struct {
	VmsService    *VmsService
	header        map[string]string
	query         map[string]string
	allContent    *bool
	caseSensitive *bool
	filter        *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *VmsServiceListRequest) Header(key, value string) *VmsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmsServiceListRequest) Query(key, value string) *VmsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmsServiceListRequest) AllContent(allContent bool) *VmsServiceListRequest {
	p.allContent = &allContent
	return p
}

func (p *VmsServiceListRequest) CaseSensitive(caseSensitive bool) *VmsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *VmsServiceListRequest) Filter(filter bool) *VmsServiceListRequest {
	p.filter = &filter
	return p
}

func (p *VmsServiceListRequest) Follow(follow string) *VmsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmsServiceListRequest) Max(max int64) *VmsServiceListRequest {
	p.max = &max
	return p
}

func (p *VmsServiceListRequest) Search(search string) *VmsServiceListRequest {
	p.search = &search
	return p
}

func (p *VmsServiceListRequest) Send() (*VmsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmsService.connection.URL(), p.VmsService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmsServiceListResponse{vms: result}, nil
}

func (p *VmsServiceListRequest) MustSend() *VmsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of virtual machines of the system.
// The order of the returned list of virtual machines is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
type VmsServiceListResponse struct {
	vms *VmSlice
}

func (p *VmsServiceListResponse) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *VmsServiceListResponse) MustVms() *VmSlice {
	if p.vms == nil {
		panic("vms in response does not exist")
	}
	return p.vms
}

//
// Returns the list of virtual machines of the system.
// The order of the returned list of virtual machines is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
func (p *VmsService) List() *VmsServiceListRequest {
	return &VmsServiceListRequest{VmsService: p}
}

//
//
func (op *VmsService) VmService(id string) *VmService {
	return NewVmService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VmService(path), nil
	}
	return op.VmService(path[:index]).Service(path[index+1:])
}

func (op *VmsService) String() string {
	return fmt.Sprintf("VmsService:%s", op.path)
}

//
// Provides a way for clients to list available files.
// This service is specifically targeted to ISO storage domains, which contain ISO images and virtual floppy disks
// (VFDs) that an administrator uploads.
// The addition of a CD-ROM device to a virtual machine requires an ISO image from the files of an ISO storage domain.
//
type FilesService struct {
	BaseService
}

func NewFilesService(connection *Connection, path string) *FilesService {
	var result FilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
type FilesServiceListRequest struct {
	FilesService  *FilesService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	refresh       *bool
	search        *string
}

func (p *FilesServiceListRequest) Header(key, value string) *FilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FilesServiceListRequest) Query(key, value string) *FilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FilesServiceListRequest) CaseSensitive(caseSensitive bool) *FilesServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *FilesServiceListRequest) Follow(follow string) *FilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *FilesServiceListRequest) Max(max int64) *FilesServiceListRequest {
	p.max = &max
	return p
}

func (p *FilesServiceListRequest) Refresh(refresh bool) *FilesServiceListRequest {
	p.refresh = &refresh
	return p
}

func (p *FilesServiceListRequest) Search(search string) *FilesServiceListRequest {
	p.search = &search
	return p
}

func (p *FilesServiceListRequest) Send() (*FilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FilesService.connection.URL(), p.FilesService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.refresh != nil {
		values["refresh"] = []string{fmt.Sprintf("%v", *p.refresh)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &FilesServiceListResponse{file: result}, nil
}

func (p *FilesServiceListRequest) MustSend() *FilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
type FilesServiceListResponse struct {
	file *FileSlice
}

func (p *FilesServiceListResponse) File() (*FileSlice, bool) {
	if p.file != nil {
		return p.file, true
	}
	return nil, false
}

func (p *FilesServiceListResponse) MustFile() *FileSlice {
	if p.file == nil {
		panic("file in response does not exist")
	}
	return p.file
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
func (p *FilesService) List() *FilesServiceListRequest {
	return &FilesServiceListRequest{FilesService: p}
}

//
//
func (op *FilesService) FileService(id string) *FileService {
	return NewFileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FileService(path), nil
	}
	return op.FileService(path[:index]).Service(path[index+1:])
}

func (op *FilesService) String() string {
	return fmt.Sprintf("FilesService:%s", op.path)
}

//
// A service managing a network
//
type NetworkService struct {
	BaseService
}

func NewNetworkService(connection *Connection, path string) *NetworkService {
	var result NetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets a logical network.
// For example:
// [source]
// ----
// GET /ovirt-engine/api/networks/123
// ----
// Will respond:
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/123" id="123">
//   <name>ovirtmgmt</name>
//   <description>Default Management Network</description>
//   <link href="/ovirt-engine/api/networks/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/123/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/123/networklabels" rel="networklabels"/>
//   <mtu>0</mtu>
//   <stp>false</stp>
//   <usages>
//     <usage>vm</usage>
//   </usages>
//   <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
// </network>
// ----
//
type NetworkServiceGetRequest struct {
	NetworkService *NetworkService
	header         map[string]string
	query          map[string]string
	follow         *string
}

func (p *NetworkServiceGetRequest) Header(key, value string) *NetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkServiceGetRequest) Query(key, value string) *NetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkServiceGetRequest) Follow(follow string) *NetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *NetworkServiceGetRequest) Send() (*NetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkService.connection.URL(), p.NetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkServiceGetResponse{network: result}, nil
}

func (p *NetworkServiceGetRequest) MustSend() *NetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets a logical network.
// For example:
// [source]
// ----
// GET /ovirt-engine/api/networks/123
// ----
// Will respond:
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/123" id="123">
//   <name>ovirtmgmt</name>
//   <description>Default Management Network</description>
//   <link href="/ovirt-engine/api/networks/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/123/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/123/networklabels" rel="networklabels"/>
//   <mtu>0</mtu>
//   <stp>false</stp>
//   <usages>
//     <usage>vm</usage>
//   </usages>
//   <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
// </network>
// ----
//
type NetworkServiceGetResponse struct {
	network *Network
}

func (p *NetworkServiceGetResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *NetworkServiceGetResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Gets a logical network.
// For example:
// [source]
// ----
// GET /ovirt-engine/api/networks/123
// ----
// Will respond:
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/123" id="123">
//   <name>ovirtmgmt</name>
//   <description>Default Management Network</description>
//   <link href="/ovirt-engine/api/networks/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/123/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/123/networklabels" rel="networklabels"/>
//   <mtu>0</mtu>
//   <stp>false</stp>
//   <usages>
//     <usage>vm</usage>
//   </usages>
//   <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
// </network>
// ----
//
func (p *NetworkService) Get() *NetworkServiceGetRequest {
	return &NetworkServiceGetRequest{NetworkService: p}
}

//
// Removes a logical network, or the association of a logical network to a data center.
// For example, to remove the logical network `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/networks/123
// ----
// Each network is bound exactly to one data center. So if we disassociate network with data center it has the same
// result as if we would just remove that network. However it might be more specific to say we're removing network
// `456` of data center `123`.
// For example, to remove the association of network `456` to data center `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123/networks/456
// ----
// NOTE: To remove an external logical network, the network has to be removed directly from its provider by
// https://developer.openstack.org/api-ref/network[OpenStack Networking API].
// The entity representing the external network inside {product-name} is removed automatically,
// if <<types/open_stack_network_provider/attributes/auto_sync,`auto_sync`>> is enabled for the provider,
// otherwise the entity has to be removed using this method.
//
type NetworkServiceRemoveRequest struct {
	NetworkService *NetworkService
	header         map[string]string
	query          map[string]string
	async          *bool
}

func (p *NetworkServiceRemoveRequest) Header(key, value string) *NetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkServiceRemoveRequest) Query(key, value string) *NetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkServiceRemoveRequest) Async(async bool) *NetworkServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *NetworkServiceRemoveRequest) Send() (*NetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkService.connection.URL(), p.NetworkService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(NetworkServiceRemoveResponse), nil
}

func (p *NetworkServiceRemoveRequest) MustSend() *NetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a logical network, or the association of a logical network to a data center.
// For example, to remove the logical network `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/networks/123
// ----
// Each network is bound exactly to one data center. So if we disassociate network with data center it has the same
// result as if we would just remove that network. However it might be more specific to say we're removing network
// `456` of data center `123`.
// For example, to remove the association of network `456` to data center `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123/networks/456
// ----
// NOTE: To remove an external logical network, the network has to be removed directly from its provider by
// https://developer.openstack.org/api-ref/network[OpenStack Networking API].
// The entity representing the external network inside {product-name} is removed automatically,
// if <<types/open_stack_network_provider/attributes/auto_sync,`auto_sync`>> is enabled for the provider,
// otherwise the entity has to be removed using this method.
//
type NetworkServiceRemoveResponse struct {
}

//
// Removes a logical network, or the association of a logical network to a data center.
// For example, to remove the logical network `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/networks/123
// ----
// Each network is bound exactly to one data center. So if we disassociate network with data center it has the same
// result as if we would just remove that network. However it might be more specific to say we're removing network
// `456` of data center `123`.
// For example, to remove the association of network `456` to data center `123` send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123/networks/456
// ----
// NOTE: To remove an external logical network, the network has to be removed directly from its provider by
// https://developer.openstack.org/api-ref/network[OpenStack Networking API].
// The entity representing the external network inside {product-name} is removed automatically,
// if <<types/open_stack_network_provider/attributes/auto_sync,`auto_sync`>> is enabled for the provider,
// otherwise the entity has to be removed using this method.
//
func (p *NetworkService) Remove() *NetworkServiceRemoveRequest {
	return &NetworkServiceRemoveRequest{NetworkService: p}
}

//
// Updates a logical network.
// The `name`, `description`, `ip`, `vlan`, `stp` and `display` attributes can be updated.
// For example, to update the description of the logical network `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/networks/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <network>
//   <description>My updated description</description>
// </network>
// ----
// The maximum transmission unit of a network is set using a PUT request to
// specify the integer value of the `mtu` attribute.
// For example, to set the maximum transmission unit send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/networks/456
// ----
// With a request body like this:
// [source,xml]
// ----
// <network>
//   <mtu>1500</mtu>
// </network>
// ----
// NOTE: Updating external networks is not propagated to the provider.
//
type NetworkServiceUpdateRequest struct {
	NetworkService *NetworkService
	header         map[string]string
	query          map[string]string
	async          *bool
	network        *Network
}

func (p *NetworkServiceUpdateRequest) Header(key, value string) *NetworkServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkServiceUpdateRequest) Query(key, value string) *NetworkServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkServiceUpdateRequest) Async(async bool) *NetworkServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *NetworkServiceUpdateRequest) Network(network *Network) *NetworkServiceUpdateRequest {
	p.network = network
	return p
}

func (p *NetworkServiceUpdateRequest) Send() (*NetworkServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkService.connection.URL(), p.NetworkService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkServiceUpdateResponse{network: result}, nil
}

func (p *NetworkServiceUpdateRequest) MustSend() *NetworkServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates a logical network.
// The `name`, `description`, `ip`, `vlan`, `stp` and `display` attributes can be updated.
// For example, to update the description of the logical network `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/networks/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <network>
//   <description>My updated description</description>
// </network>
// ----
// The maximum transmission unit of a network is set using a PUT request to
// specify the integer value of the `mtu` attribute.
// For example, to set the maximum transmission unit send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/networks/456
// ----
// With a request body like this:
// [source,xml]
// ----
// <network>
//   <mtu>1500</mtu>
// </network>
// ----
// NOTE: Updating external networks is not propagated to the provider.
//
type NetworkServiceUpdateResponse struct {
	network *Network
}

func (p *NetworkServiceUpdateResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *NetworkServiceUpdateResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Updates a logical network.
// The `name`, `description`, `ip`, `vlan`, `stp` and `display` attributes can be updated.
// For example, to update the description of the logical network `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/networks/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <network>
//   <description>My updated description</description>
// </network>
// ----
// The maximum transmission unit of a network is set using a PUT request to
// specify the integer value of the `mtu` attribute.
// For example, to set the maximum transmission unit send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/networks/456
// ----
// With a request body like this:
// [source,xml]
// ----
// <network>
//   <mtu>1500</mtu>
// </network>
// ----
// NOTE: Updating external networks is not propagated to the provider.
//
func (p *NetworkService) Update() *NetworkServiceUpdateRequest {
	return &NetworkServiceUpdateRequest{NetworkService: p}
}

//
// Reference to the service that manages the network labels assigned to this network.
//
func (op *NetworkService) NetworkLabelsService() *NetworkLabelsService {
	return NewNetworkLabelsService(op.connection, fmt.Sprintf("%s/networklabels", op.path))
}

//
// Reference to the service that manages the permissions assigned to this network.
//
func (op *NetworkService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Reference to the service that manages the vNIC profiles assigned to this network.
//
func (op *NetworkService) VnicProfilesService() *AssignedVnicProfilesService {
	return NewAssignedVnicProfilesService(op.connection, fmt.Sprintf("%s/vnicprofiles", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "networklabels" {
		return op.NetworkLabelsService(), nil
	}
	if strings.HasPrefix(path, "networklabels/") {
		return op.NetworkLabelsService().Service(path[14:])
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "vnicprofiles" {
		return op.VnicProfilesService(), nil
	}
	if strings.HasPrefix(path, "vnicprofiles/") {
		return op.VnicProfilesService().Service(path[13:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *NetworkService) String() string {
	return fmt.Sprintf("NetworkService:%s", op.path)
}

//
//
type IscsiBondsService struct {
	BaseService
}

func NewIscsiBondsService(connection *Connection, path string) *IscsiBondsService {
	var result IscsiBondsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Create a new iSCSI bond on a data center.
// For example, to create a new iSCSI bond on data center `123` using storage connections `456` and `789`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/iscsibonds
// ----
// The request body should look like this:
// [source,xml]
// ----
// <iscsi_bond>
//   <name>mybond</name>
//   <storage_connections>
//     <storage_connection id="456"/>
//     <storage_connection id="789"/>
//   </storage_connections>
//   <networks>
//     <network id="abc"/>
//   </networks>
// </iscsi_bond>
// ----
//
type IscsiBondsServiceAddRequest struct {
	IscsiBondsService *IscsiBondsService
	header            map[string]string
	query             map[string]string
	bond              *IscsiBond
}

func (p *IscsiBondsServiceAddRequest) Header(key, value string) *IscsiBondsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *IscsiBondsServiceAddRequest) Query(key, value string) *IscsiBondsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *IscsiBondsServiceAddRequest) Bond(bond *IscsiBond) *IscsiBondsServiceAddRequest {
	p.bond = bond
	return p
}

func (p *IscsiBondsServiceAddRequest) Send() (*IscsiBondsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.IscsiBondsService.connection.URL(), p.IscsiBondsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLIscsiBondWriteOne(writer, p.bond, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.IscsiBondsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.IscsiBondsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.IscsiBondsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.IscsiBondsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.IscsiBondsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLIscsiBondReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &IscsiBondsServiceAddResponse{bond: result}, nil
}

func (p *IscsiBondsServiceAddRequest) MustSend() *IscsiBondsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Create a new iSCSI bond on a data center.
// For example, to create a new iSCSI bond on data center `123` using storage connections `456` and `789`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/iscsibonds
// ----
// The request body should look like this:
// [source,xml]
// ----
// <iscsi_bond>
//   <name>mybond</name>
//   <storage_connections>
//     <storage_connection id="456"/>
//     <storage_connection id="789"/>
//   </storage_connections>
//   <networks>
//     <network id="abc"/>
//   </networks>
// </iscsi_bond>
// ----
//
type IscsiBondsServiceAddResponse struct {
	bond *IscsiBond
}

func (p *IscsiBondsServiceAddResponse) Bond() (*IscsiBond, bool) {
	if p.bond != nil {
		return p.bond, true
	}
	return nil, false
}

func (p *IscsiBondsServiceAddResponse) MustBond() *IscsiBond {
	if p.bond == nil {
		panic("bond in response does not exist")
	}
	return p.bond
}

//
// Create a new iSCSI bond on a data center.
// For example, to create a new iSCSI bond on data center `123` using storage connections `456` and `789`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/iscsibonds
// ----
// The request body should look like this:
// [source,xml]
// ----
// <iscsi_bond>
//   <name>mybond</name>
//   <storage_connections>
//     <storage_connection id="456"/>
//     <storage_connection id="789"/>
//   </storage_connections>
//   <networks>
//     <network id="abc"/>
//   </networks>
// </iscsi_bond>
// ----
//
func (p *IscsiBondsService) Add() *IscsiBondsServiceAddRequest {
	return &IscsiBondsServiceAddRequest{IscsiBondsService: p}
}

//
// Returns the list of iSCSI bonds configured in the data center.
// The order of the returned list of iSCSI bonds isn't guaranteed.
//
type IscsiBondsServiceListRequest struct {
	IscsiBondsService *IscsiBondsService
	header            map[string]string
	query             map[string]string
	follow            *string
	max               *int64
}

func (p *IscsiBondsServiceListRequest) Header(key, value string) *IscsiBondsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *IscsiBondsServiceListRequest) Query(key, value string) *IscsiBondsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *IscsiBondsServiceListRequest) Follow(follow string) *IscsiBondsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *IscsiBondsServiceListRequest) Max(max int64) *IscsiBondsServiceListRequest {
	p.max = &max
	return p
}

func (p *IscsiBondsServiceListRequest) Send() (*IscsiBondsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.IscsiBondsService.connection.URL(), p.IscsiBondsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.IscsiBondsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.IscsiBondsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.IscsiBondsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.IscsiBondsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.IscsiBondsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLIscsiBondReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &IscsiBondsServiceListResponse{bonds: result}, nil
}

func (p *IscsiBondsServiceListRequest) MustSend() *IscsiBondsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of iSCSI bonds configured in the data center.
// The order of the returned list of iSCSI bonds isn't guaranteed.
//
type IscsiBondsServiceListResponse struct {
	bonds *IscsiBondSlice
}

func (p *IscsiBondsServiceListResponse) Bonds() (*IscsiBondSlice, bool) {
	if p.bonds != nil {
		return p.bonds, true
	}
	return nil, false
}

func (p *IscsiBondsServiceListResponse) MustBonds() *IscsiBondSlice {
	if p.bonds == nil {
		panic("bonds in response does not exist")
	}
	return p.bonds
}

//
// Returns the list of iSCSI bonds configured in the data center.
// The order of the returned list of iSCSI bonds isn't guaranteed.
//
func (p *IscsiBondsService) List() *IscsiBondsServiceListRequest {
	return &IscsiBondsServiceListRequest{IscsiBondsService: p}
}

//
//
func (op *IscsiBondsService) IscsiBondService(id string) *IscsiBondService {
	return NewIscsiBondService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *IscsiBondsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.IscsiBondService(path), nil
	}
	return op.IscsiBondService(path[:index]).Service(path[index+1:])
}

func (op *IscsiBondsService) String() string {
	return fmt.Sprintf("IscsiBondsService:%s", op.path)
}

//
// A service to access host devices.
//
type HostDevicesService struct {
	BaseService
}

func NewHostDevicesService(connection *Connection, path string) *HostDevicesService {
	var result HostDevicesService
	result.connection = connection
	result.path = path
	return &result
}

//
// List the devices of a host.
// The order of the returned list of devices isn't guaranteed.
//
type HostDevicesServiceListRequest struct {
	HostDevicesService *HostDevicesService
	header             map[string]string
	query              map[string]string
	follow             *string
	max                *int64
}

func (p *HostDevicesServiceListRequest) Header(key, value string) *HostDevicesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostDevicesServiceListRequest) Query(key, value string) *HostDevicesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostDevicesServiceListRequest) Follow(follow string) *HostDevicesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostDevicesServiceListRequest) Max(max int64) *HostDevicesServiceListRequest {
	p.max = &max
	return p
}

func (p *HostDevicesServiceListRequest) Send() (*HostDevicesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostDevicesService.connection.URL(), p.HostDevicesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostDevicesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostDevicesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostDevicesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostDevicesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostDevicesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostDeviceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostDevicesServiceListResponse{devices: result}, nil
}

func (p *HostDevicesServiceListRequest) MustSend() *HostDevicesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the devices of a host.
// The order of the returned list of devices isn't guaranteed.
//
type HostDevicesServiceListResponse struct {
	devices *HostDeviceSlice
}

func (p *HostDevicesServiceListResponse) Devices() (*HostDeviceSlice, bool) {
	if p.devices != nil {
		return p.devices, true
	}
	return nil, false
}

func (p *HostDevicesServiceListResponse) MustDevices() *HostDeviceSlice {
	if p.devices == nil {
		panic("devices in response does not exist")
	}
	return p.devices
}

//
// List the devices of a host.
// The order of the returned list of devices isn't guaranteed.
//
func (p *HostDevicesService) List() *HostDevicesServiceListRequest {
	return &HostDevicesServiceListRequest{HostDevicesService: p}
}

//
// Reference to the service that can be used to access a specific host device.
//
func (op *HostDevicesService) DeviceService(id string) *HostDeviceService {
	return NewHostDeviceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostDevicesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DeviceService(path), nil
	}
	return op.DeviceService(path[:index]).Service(path[index+1:])
}

func (op *HostDevicesService) String() string {
	return fmt.Sprintf("HostDevicesService:%s", op.path)
}

//
//
type VmReportedDevicesService struct {
	BaseService
}

func NewVmReportedDevicesService(connection *Connection, path string) *VmReportedDevicesService {
	var result VmReportedDevicesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of reported devices of the virtual machine.
// The order of the returned list of devices isn't guaranteed.
//
type VmReportedDevicesServiceListRequest struct {
	VmReportedDevicesService *VmReportedDevicesService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
	max                      *int64
}

func (p *VmReportedDevicesServiceListRequest) Header(key, value string) *VmReportedDevicesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmReportedDevicesServiceListRequest) Query(key, value string) *VmReportedDevicesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmReportedDevicesServiceListRequest) Follow(follow string) *VmReportedDevicesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmReportedDevicesServiceListRequest) Max(max int64) *VmReportedDevicesServiceListRequest {
	p.max = &max
	return p
}

func (p *VmReportedDevicesServiceListRequest) Send() (*VmReportedDevicesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmReportedDevicesService.connection.URL(), p.VmReportedDevicesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmReportedDevicesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmReportedDevicesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmReportedDevicesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmReportedDevicesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmReportedDevicesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLReportedDeviceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmReportedDevicesServiceListResponse{reportedDevice: result}, nil
}

func (p *VmReportedDevicesServiceListRequest) MustSend() *VmReportedDevicesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of reported devices of the virtual machine.
// The order of the returned list of devices isn't guaranteed.
//
type VmReportedDevicesServiceListResponse struct {
	reportedDevice *ReportedDeviceSlice
}

func (p *VmReportedDevicesServiceListResponse) ReportedDevice() (*ReportedDeviceSlice, bool) {
	if p.reportedDevice != nil {
		return p.reportedDevice, true
	}
	return nil, false
}

func (p *VmReportedDevicesServiceListResponse) MustReportedDevice() *ReportedDeviceSlice {
	if p.reportedDevice == nil {
		panic("reportedDevice in response does not exist")
	}
	return p.reportedDevice
}

//
// Returns the list of reported devices of the virtual machine.
// The order of the returned list of devices isn't guaranteed.
//
func (p *VmReportedDevicesService) List() *VmReportedDevicesServiceListRequest {
	return &VmReportedDevicesServiceListRequest{VmReportedDevicesService: p}
}

//
//
func (op *VmReportedDevicesService) ReportedDeviceService(id string) *VmReportedDeviceService {
	return NewVmReportedDeviceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmReportedDevicesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ReportedDeviceService(path), nil
	}
	return op.ReportedDeviceService(path[:index]).Service(path[index+1:])
}

func (op *VmReportedDevicesService) String() string {
	return fmt.Sprintf("VmReportedDevicesService:%s", op.path)
}

//
// A service to manage individual host device attached to a virtual machine.
//
type VmHostDeviceService struct {
	BaseService
}

func NewVmHostDeviceService(connection *Connection, path string) *VmHostDeviceService {
	var result VmHostDeviceService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieve information about particular host device attached to given virtual machine.
// Example:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/hostdevices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//   <name>pci_0000_04_00_0</name>
//   <capability>pci</capability>
//   <iommu_group>30</iommu_group>
//   <placeholder>true</placeholder>
//   <product id="0x13ba">
//     <name>GM107GL [Quadro K2200]</name>
//   </product>
//   <vendor id="0x10de">
//     <name>NVIDIA Corporation</name>
//   </vendor>
//   <host href="/ovirt-engine/api/hosts/543" id="543"/>
//   <parent_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//     <name>pci_0000_00_03_0</name>
//   </parent_device>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </host_device>
// ----
//
type VmHostDeviceServiceGetRequest struct {
	VmHostDeviceService *VmHostDeviceService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *VmHostDeviceServiceGetRequest) Header(key, value string) *VmHostDeviceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmHostDeviceServiceGetRequest) Query(key, value string) *VmHostDeviceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmHostDeviceServiceGetRequest) Follow(follow string) *VmHostDeviceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmHostDeviceServiceGetRequest) Send() (*VmHostDeviceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmHostDeviceService.connection.URL(), p.VmHostDeviceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmHostDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmHostDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmHostDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmHostDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmHostDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostDeviceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmHostDeviceServiceGetResponse{device: result}, nil
}

func (p *VmHostDeviceServiceGetRequest) MustSend() *VmHostDeviceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve information about particular host device attached to given virtual machine.
// Example:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/hostdevices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//   <name>pci_0000_04_00_0</name>
//   <capability>pci</capability>
//   <iommu_group>30</iommu_group>
//   <placeholder>true</placeholder>
//   <product id="0x13ba">
//     <name>GM107GL [Quadro K2200]</name>
//   </product>
//   <vendor id="0x10de">
//     <name>NVIDIA Corporation</name>
//   </vendor>
//   <host href="/ovirt-engine/api/hosts/543" id="543"/>
//   <parent_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//     <name>pci_0000_00_03_0</name>
//   </parent_device>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </host_device>
// ----
//
type VmHostDeviceServiceGetResponse struct {
	device *HostDevice
}

func (p *VmHostDeviceServiceGetResponse) Device() (*HostDevice, bool) {
	if p.device != nil {
		return p.device, true
	}
	return nil, false
}

func (p *VmHostDeviceServiceGetResponse) MustDevice() *HostDevice {
	if p.device == nil {
		panic("device in response does not exist")
	}
	return p.device
}

//
// Retrieve information about particular host device attached to given virtual machine.
// Example:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/hostdevices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//   <name>pci_0000_04_00_0</name>
//   <capability>pci</capability>
//   <iommu_group>30</iommu_group>
//   <placeholder>true</placeholder>
//   <product id="0x13ba">
//     <name>GM107GL [Quadro K2200]</name>
//   </product>
//   <vendor id="0x10de">
//     <name>NVIDIA Corporation</name>
//   </vendor>
//   <host href="/ovirt-engine/api/hosts/543" id="543"/>
//   <parent_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//     <name>pci_0000_00_03_0</name>
//   </parent_device>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </host_device>
// ----
//
func (p *VmHostDeviceService) Get() *VmHostDeviceServiceGetRequest {
	return &VmHostDeviceServiceGetRequest{VmHostDeviceService: p}
}

//
// Remove the attachment of this host device from given virtual machine.
// NOTE: In case this device serves as an IOMMU placeholder, it cannot be removed (remove will result only
// in setting its `placeholder` flag to `true`). Note that all IOMMU placeholder devices will be removed
// automatically as soon as there will be no more non-placeholder devices (all devices from given IOMMU
// group are detached).
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/hostdevices/456
// ----
//
type VmHostDeviceServiceRemoveRequest struct {
	VmHostDeviceService *VmHostDeviceService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *VmHostDeviceServiceRemoveRequest) Header(key, value string) *VmHostDeviceServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmHostDeviceServiceRemoveRequest) Query(key, value string) *VmHostDeviceServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmHostDeviceServiceRemoveRequest) Async(async bool) *VmHostDeviceServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VmHostDeviceServiceRemoveRequest) Send() (*VmHostDeviceServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmHostDeviceService.connection.URL(), p.VmHostDeviceService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmHostDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmHostDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmHostDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmHostDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmHostDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VmHostDeviceServiceRemoveResponse), nil
}

func (p *VmHostDeviceServiceRemoveRequest) MustSend() *VmHostDeviceServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove the attachment of this host device from given virtual machine.
// NOTE: In case this device serves as an IOMMU placeholder, it cannot be removed (remove will result only
// in setting its `placeholder` flag to `true`). Note that all IOMMU placeholder devices will be removed
// automatically as soon as there will be no more non-placeholder devices (all devices from given IOMMU
// group are detached).
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/hostdevices/456
// ----
//
type VmHostDeviceServiceRemoveResponse struct {
}

//
// Remove the attachment of this host device from given virtual machine.
// NOTE: In case this device serves as an IOMMU placeholder, it cannot be removed (remove will result only
// in setting its `placeholder` flag to `true`). Note that all IOMMU placeholder devices will be removed
// automatically as soon as there will be no more non-placeholder devices (all devices from given IOMMU
// group are detached).
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/hostdevices/456
// ----
//
func (p *VmHostDeviceService) Remove() *VmHostDeviceServiceRemoveRequest {
	return &VmHostDeviceServiceRemoveRequest{VmHostDeviceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmHostDeviceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmHostDeviceService) String() string {
	return fmt.Sprintf("VmHostDeviceService:%s", op.path)
}

//
// Manages the filters used by an scheduling policy.
//
type FiltersService struct {
	BaseService
}

func NewFiltersService(connection *Connection, path string) *FiltersService {
	var result FiltersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a filter to a specified user defined scheduling policy.
//
type FiltersServiceAddRequest struct {
	FiltersService *FiltersService
	header         map[string]string
	query          map[string]string
	filter         *Filter
}

func (p *FiltersServiceAddRequest) Header(key, value string) *FiltersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FiltersServiceAddRequest) Query(key, value string) *FiltersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FiltersServiceAddRequest) Filter(filter *Filter) *FiltersServiceAddRequest {
	p.filter = filter
	return p
}

func (p *FiltersServiceAddRequest) Send() (*FiltersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FiltersService.connection.URL(), p.FiltersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLFilterWriteOne(writer, p.filter, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFilterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FiltersServiceAddResponse{filter: result}, nil
}

func (p *FiltersServiceAddRequest) MustSend() *FiltersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a filter to a specified user defined scheduling policy.
//
type FiltersServiceAddResponse struct {
	filter *Filter
}

func (p *FiltersServiceAddResponse) Filter() (*Filter, bool) {
	if p.filter != nil {
		return p.filter, true
	}
	return nil, false
}

func (p *FiltersServiceAddResponse) MustFilter() *Filter {
	if p.filter == nil {
		panic("filter in response does not exist")
	}
	return p.filter
}

//
// Add a filter to a specified user defined scheduling policy.
//
func (p *FiltersService) Add() *FiltersServiceAddRequest {
	return &FiltersServiceAddRequest{FiltersService: p}
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
type FiltersServiceListRequest struct {
	FiltersService *FiltersService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
	max            *int64
}

func (p *FiltersServiceListRequest) Header(key, value string) *FiltersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FiltersServiceListRequest) Query(key, value string) *FiltersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FiltersServiceListRequest) Filter(filter bool) *FiltersServiceListRequest {
	p.filter = &filter
	return p
}

func (p *FiltersServiceListRequest) Follow(follow string) *FiltersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *FiltersServiceListRequest) Max(max int64) *FiltersServiceListRequest {
	p.max = &max
	return p
}

func (p *FiltersServiceListRequest) Send() (*FiltersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FiltersService.connection.URL(), p.FiltersService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFilterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &FiltersServiceListResponse{filters: result}, nil
}

func (p *FiltersServiceListRequest) MustSend() *FiltersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
type FiltersServiceListResponse struct {
	filters *FilterSlice
}

func (p *FiltersServiceListResponse) Filters() (*FilterSlice, bool) {
	if p.filters != nil {
		return p.filters, true
	}
	return nil, false
}

func (p *FiltersServiceListResponse) MustFilters() *FilterSlice {
	if p.filters == nil {
		panic("filters in response does not exist")
	}
	return p.filters
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
func (p *FiltersService) List() *FiltersServiceListRequest {
	return &FiltersServiceListRequest{FiltersService: p}
}

//
//
func (op *FiltersService) FilterService(id string) *FilterService {
	return NewFilterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FiltersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FilterService(path), nil
	}
	return op.FilterService(path[:index]).Service(path[index+1:])
}

func (op *FiltersService) String() string {
	return fmt.Sprintf("FiltersService:%s", op.path)
}

//
// Manages the set of storage limits configured for a quota.
//
type QuotaStorageLimitsService struct {
	BaseService
}

func NewQuotaStorageLimitsService(connection *Connection, path string) *QuotaStorageLimitsService {
	var result QuotaStorageLimitsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a storage limit to a specified quota.
// To create a 100GiB storage limit for all storage domains in a data center, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas/456/quotastoragelimits
// ----
// With a request body like this:
// [source,xml]
// ----
// <quota_storage_limit>
//   <limit>100</limit>
// </quota_storage_limit>
// ----
// To create a 50GiB storage limit for a storage domain with the ID `000`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas/456/quotastoragelimits
// ----
// With a request body like this:
// [source,xml]
// ----
// <quota_storage_limit>
//   <limit>50</limit>
//   <storage_domain id="000"/>
// </quota_storage_limit>
// ----
//
type QuotaStorageLimitsServiceAddRequest struct {
	QuotaStorageLimitsService *QuotaStorageLimitsService
	header                    map[string]string
	query                     map[string]string
	limit                     *QuotaStorageLimit
}

func (p *QuotaStorageLimitsServiceAddRequest) Header(key, value string) *QuotaStorageLimitsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaStorageLimitsServiceAddRequest) Query(key, value string) *QuotaStorageLimitsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaStorageLimitsServiceAddRequest) Limit(limit *QuotaStorageLimit) *QuotaStorageLimitsServiceAddRequest {
	p.limit = limit
	return p
}

func (p *QuotaStorageLimitsServiceAddRequest) Send() (*QuotaStorageLimitsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaStorageLimitsService.connection.URL(), p.QuotaStorageLimitsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLQuotaStorageLimitWriteOne(writer, p.limit, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaStorageLimitsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaStorageLimitsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaStorageLimitsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaStorageLimitsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaStorageLimitsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaStorageLimitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &QuotaStorageLimitsServiceAddResponse{limit: result}, nil
}

func (p *QuotaStorageLimitsServiceAddRequest) MustSend() *QuotaStorageLimitsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a storage limit to a specified quota.
// To create a 100GiB storage limit for all storage domains in a data center, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas/456/quotastoragelimits
// ----
// With a request body like this:
// [source,xml]
// ----
// <quota_storage_limit>
//   <limit>100</limit>
// </quota_storage_limit>
// ----
// To create a 50GiB storage limit for a storage domain with the ID `000`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas/456/quotastoragelimits
// ----
// With a request body like this:
// [source,xml]
// ----
// <quota_storage_limit>
//   <limit>50</limit>
//   <storage_domain id="000"/>
// </quota_storage_limit>
// ----
//
type QuotaStorageLimitsServiceAddResponse struct {
	limit *QuotaStorageLimit
}

func (p *QuotaStorageLimitsServiceAddResponse) Limit() (*QuotaStorageLimit, bool) {
	if p.limit != nil {
		return p.limit, true
	}
	return nil, false
}

func (p *QuotaStorageLimitsServiceAddResponse) MustLimit() *QuotaStorageLimit {
	if p.limit == nil {
		panic("limit in response does not exist")
	}
	return p.limit
}

//
// Adds a storage limit to a specified quota.
// To create a 100GiB storage limit for all storage domains in a data center, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas/456/quotastoragelimits
// ----
// With a request body like this:
// [source,xml]
// ----
// <quota_storage_limit>
//   <limit>100</limit>
// </quota_storage_limit>
// ----
// To create a 50GiB storage limit for a storage domain with the ID `000`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas/456/quotastoragelimits
// ----
// With a request body like this:
// [source,xml]
// ----
// <quota_storage_limit>
//   <limit>50</limit>
//   <storage_domain id="000"/>
// </quota_storage_limit>
// ----
//
func (p *QuotaStorageLimitsService) Add() *QuotaStorageLimitsServiceAddRequest {
	return &QuotaStorageLimitsServiceAddRequest{QuotaStorageLimitsService: p}
}

//
// Returns the list of storage limits configured for the quota.
// The order of the returned list of storage limits is not guaranteed.
//
type QuotaStorageLimitsServiceListRequest struct {
	QuotaStorageLimitsService *QuotaStorageLimitsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *QuotaStorageLimitsServiceListRequest) Header(key, value string) *QuotaStorageLimitsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaStorageLimitsServiceListRequest) Query(key, value string) *QuotaStorageLimitsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaStorageLimitsServiceListRequest) Follow(follow string) *QuotaStorageLimitsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *QuotaStorageLimitsServiceListRequest) Max(max int64) *QuotaStorageLimitsServiceListRequest {
	p.max = &max
	return p
}

func (p *QuotaStorageLimitsServiceListRequest) Send() (*QuotaStorageLimitsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaStorageLimitsService.connection.URL(), p.QuotaStorageLimitsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaStorageLimitsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaStorageLimitsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaStorageLimitsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaStorageLimitsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaStorageLimitsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaStorageLimitReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &QuotaStorageLimitsServiceListResponse{limits: result}, nil
}

func (p *QuotaStorageLimitsServiceListRequest) MustSend() *QuotaStorageLimitsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of storage limits configured for the quota.
// The order of the returned list of storage limits is not guaranteed.
//
type QuotaStorageLimitsServiceListResponse struct {
	limits *QuotaStorageLimitSlice
}

func (p *QuotaStorageLimitsServiceListResponse) Limits() (*QuotaStorageLimitSlice, bool) {
	if p.limits != nil {
		return p.limits, true
	}
	return nil, false
}

func (p *QuotaStorageLimitsServiceListResponse) MustLimits() *QuotaStorageLimitSlice {
	if p.limits == nil {
		panic("limits in response does not exist")
	}
	return p.limits
}

//
// Returns the list of storage limits configured for the quota.
// The order of the returned list of storage limits is not guaranteed.
//
func (p *QuotaStorageLimitsService) List() *QuotaStorageLimitsServiceListRequest {
	return &QuotaStorageLimitsServiceListRequest{QuotaStorageLimitsService: p}
}

//
//
func (op *QuotaStorageLimitsService) LimitService(id string) *QuotaStorageLimitService {
	return NewQuotaStorageLimitService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *QuotaStorageLimitsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LimitService(path), nil
	}
	return op.LimitService(path[:index]).Service(path[index+1:])
}

func (op *QuotaStorageLimitsService) String() string {
	return fmt.Sprintf("QuotaStorageLimitsService:%s", op.path)
}

//
//
type SchedulingPolicyService struct {
	BaseService
}

func NewSchedulingPolicyService(connection *Connection, path string) *SchedulingPolicyService {
	var result SchedulingPolicyService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SchedulingPolicyServiceGetRequest struct {
	SchedulingPolicyService *SchedulingPolicyService
	header                  map[string]string
	query                   map[string]string
	filter                  *bool
	follow                  *string
}

func (p *SchedulingPolicyServiceGetRequest) Header(key, value string) *SchedulingPolicyServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyServiceGetRequest) Query(key, value string) *SchedulingPolicyServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyServiceGetRequest) Filter(filter bool) *SchedulingPolicyServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *SchedulingPolicyServiceGetRequest) Follow(follow string) *SchedulingPolicyServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SchedulingPolicyServiceGetRequest) Send() (*SchedulingPolicyServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyService.connection.URL(), p.SchedulingPolicyService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SchedulingPolicyServiceGetResponse{policy: result}, nil
}

func (p *SchedulingPolicyServiceGetRequest) MustSend() *SchedulingPolicyServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyServiceGetResponse struct {
	policy *SchedulingPolicy
}

func (p *SchedulingPolicyServiceGetResponse) Policy() (*SchedulingPolicy, bool) {
	if p.policy != nil {
		return p.policy, true
	}
	return nil, false
}

func (p *SchedulingPolicyServiceGetResponse) MustPolicy() *SchedulingPolicy {
	if p.policy == nil {
		panic("policy in response does not exist")
	}
	return p.policy
}

//
//
func (p *SchedulingPolicyService) Get() *SchedulingPolicyServiceGetRequest {
	return &SchedulingPolicyServiceGetRequest{SchedulingPolicyService: p}
}

//
//
type SchedulingPolicyServiceRemoveRequest struct {
	SchedulingPolicyService *SchedulingPolicyService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
}

func (p *SchedulingPolicyServiceRemoveRequest) Header(key, value string) *SchedulingPolicyServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyServiceRemoveRequest) Query(key, value string) *SchedulingPolicyServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyServiceRemoveRequest) Async(async bool) *SchedulingPolicyServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *SchedulingPolicyServiceRemoveRequest) Send() (*SchedulingPolicyServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyService.connection.URL(), p.SchedulingPolicyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(SchedulingPolicyServiceRemoveResponse), nil
}

func (p *SchedulingPolicyServiceRemoveRequest) MustSend() *SchedulingPolicyServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyServiceRemoveResponse struct {
}

//
//
func (p *SchedulingPolicyService) Remove() *SchedulingPolicyServiceRemoveRequest {
	return &SchedulingPolicyServiceRemoveRequest{SchedulingPolicyService: p}
}

//
// Update the specified user defined scheduling policy in the system.
//
type SchedulingPolicyServiceUpdateRequest struct {
	SchedulingPolicyService *SchedulingPolicyService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
	policy                  *SchedulingPolicy
}

func (p *SchedulingPolicyServiceUpdateRequest) Header(key, value string) *SchedulingPolicyServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyServiceUpdateRequest) Query(key, value string) *SchedulingPolicyServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyServiceUpdateRequest) Async(async bool) *SchedulingPolicyServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *SchedulingPolicyServiceUpdateRequest) Policy(policy *SchedulingPolicy) *SchedulingPolicyServiceUpdateRequest {
	p.policy = policy
	return p
}

func (p *SchedulingPolicyServiceUpdateRequest) Send() (*SchedulingPolicyServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyService.connection.URL(), p.SchedulingPolicyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSchedulingPolicyWriteOne(writer, p.policy, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SchedulingPolicyServiceUpdateResponse{policy: result}, nil
}

func (p *SchedulingPolicyServiceUpdateRequest) MustSend() *SchedulingPolicyServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified user defined scheduling policy in the system.
//
type SchedulingPolicyServiceUpdateResponse struct {
	policy *SchedulingPolicy
}

func (p *SchedulingPolicyServiceUpdateResponse) Policy() (*SchedulingPolicy, bool) {
	if p.policy != nil {
		return p.policy, true
	}
	return nil, false
}

func (p *SchedulingPolicyServiceUpdateResponse) MustPolicy() *SchedulingPolicy {
	if p.policy == nil {
		panic("policy in response does not exist")
	}
	return p.policy
}

//
// Update the specified user defined scheduling policy in the system.
//
func (p *SchedulingPolicyService) Update() *SchedulingPolicyServiceUpdateRequest {
	return &SchedulingPolicyServiceUpdateRequest{SchedulingPolicyService: p}
}

//
//
func (op *SchedulingPolicyService) BalancesService() *BalancesService {
	return NewBalancesService(op.connection, fmt.Sprintf("%s/balances", op.path))
}

//
//
func (op *SchedulingPolicyService) FiltersService() *FiltersService {
	return NewFiltersService(op.connection, fmt.Sprintf("%s/filters", op.path))
}

//
//
func (op *SchedulingPolicyService) WeightsService() *WeightsService {
	return NewWeightsService(op.connection, fmt.Sprintf("%s/weights", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SchedulingPolicyService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "balances" {
		return op.BalancesService(), nil
	}
	if strings.HasPrefix(path, "balances/") {
		return op.BalancesService().Service(path[9:])
	}
	if path == "filters" {
		return op.FiltersService(), nil
	}
	if strings.HasPrefix(path, "filters/") {
		return op.FiltersService().Service(path[8:])
	}
	if path == "weights" {
		return op.WeightsService(), nil
	}
	if strings.HasPrefix(path, "weights/") {
		return op.WeightsService().Service(path[8:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SchedulingPolicyService) String() string {
	return fmt.Sprintf("SchedulingPolicyService:%s", op.path)
}

//
//
type VmBackupDiskService struct {
	BaseService
}

func NewVmBackupDiskService(connection *Connection, path string) *VmBackupDiskService {
	var result VmBackupDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the description of the disk.
//
type VmBackupDiskServiceGetRequest struct {
	VmBackupDiskService *VmBackupDiskService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *VmBackupDiskServiceGetRequest) Header(key, value string) *VmBackupDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmBackupDiskServiceGetRequest) Query(key, value string) *VmBackupDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmBackupDiskServiceGetRequest) Follow(follow string) *VmBackupDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmBackupDiskServiceGetRequest) Send() (*VmBackupDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmBackupDiskService.connection.URL(), p.VmBackupDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmBackupDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmBackupDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmBackupDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmBackupDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmBackupDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmBackupDiskServiceGetResponse{disk: result}, nil
}

func (p *VmBackupDiskServiceGetRequest) MustSend() *VmBackupDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the description of the disk.
//
type VmBackupDiskServiceGetResponse struct {
	disk *Disk
}

func (p *VmBackupDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *VmBackupDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Retrieves the description of the disk.
//
func (p *VmBackupDiskService) Get() *VmBackupDiskServiceGetRequest {
	return &VmBackupDiskServiceGetRequest{VmBackupDiskService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmBackupDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmBackupDiskService) String() string {
	return fmt.Sprintf("VmBackupDiskService:%s", op.path)
}

//
// A service to manage a host.
//
type HostService struct {
	BaseService
}

func NewHostService(connection *Connection, path string) *HostService {
	var result HostService
	result.connection = connection
	result.path = path
	return &result
}

//
// Activates the host for use, for example to run virtual machines.
//
type HostServiceActivateRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceActivateRequest) Header(key, value string) *HostServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceActivateRequest) Query(key, value string) *HostServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceActivateRequest) Async(async bool) *HostServiceActivateRequest {
	p.async = &async
	return p
}

func (p *HostServiceActivateRequest) Send() (*HostServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceActivateResponse), nil
}

func (p *HostServiceActivateRequest) MustSend() *HostServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Activates the host for use, for example to run virtual machines.
//
type HostServiceActivateResponse struct {
}

//
// Activates the host for use, for example to run virtual machines.
//
func (p *HostService) Activate() *HostServiceActivateRequest {
	return &HostServiceActivateRequest{HostService: p}
}

//
// Approve a pre-installed Hypervisor host for usage in the virtualization environment.
// This action also accepts an optional cluster element to define the target cluster for this host.
//
type HostServiceApproveRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	activate    *bool
	async       *bool
	cluster     *Cluster
	host        *Host
}

func (p *HostServiceApproveRequest) Header(key, value string) *HostServiceApproveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceApproveRequest) Query(key, value string) *HostServiceApproveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceApproveRequest) Activate(activate bool) *HostServiceApproveRequest {
	p.activate = &activate
	return p
}

func (p *HostServiceApproveRequest) Async(async bool) *HostServiceApproveRequest {
	p.async = &async
	return p
}

func (p *HostServiceApproveRequest) Cluster(cluster *Cluster) *HostServiceApproveRequest {
	p.cluster = cluster
	return p
}

func (p *HostServiceApproveRequest) Host(host *Host) *HostServiceApproveRequest {
	p.host = host
	return p
}

func (p *HostServiceApproveRequest) Send() (*HostServiceApproveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/approve", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.activate != nil {
		actionBuilder.Activate(*p.activate)
	}
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Cluster(p.cluster)
	actionBuilder.Host(p.host)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceApproveResponse), nil
}

func (p *HostServiceApproveRequest) MustSend() *HostServiceApproveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Approve a pre-installed Hypervisor host for usage in the virtualization environment.
// This action also accepts an optional cluster element to define the target cluster for this host.
//
type HostServiceApproveResponse struct {
}

//
// Approve a pre-installed Hypervisor host for usage in the virtualization environment.
// This action also accepts an optional cluster element to define the target cluster for this host.
//
func (p *HostService) Approve() *HostServiceApproveRequest {
	return &HostServiceApproveRequest{HostService: p}
}

//
// Marks the network configuration as good and persists it inside the host.
// An API user commits the network configuration to persist a host network interface attachment or detachment, or
// persist the creation and deletion of a bonded interface.
// IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
// not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
// automatically reverts to the previous networking configuration.
// For example, to commit the network configuration of host with id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/commitnetconfig
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// IMPORTANT: Since {engine-name} 4.3, it is possible to also specify `commit_on_success` in
// the <<services/host/methods/setup_networks, setupnetworks>> request, in which case the new
// configuration is automatically saved in the {hypervisor-name} upon completing the setup and
// re-establishing connectivity between the {hypervisor-name} and {engine-name}, and without
// waiting for a separate <<services/host/methods/commit_net_config, commitnetconfig>> request.
//
type HostServiceCommitNetConfigRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceCommitNetConfigRequest) Header(key, value string) *HostServiceCommitNetConfigRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceCommitNetConfigRequest) Query(key, value string) *HostServiceCommitNetConfigRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceCommitNetConfigRequest) Async(async bool) *HostServiceCommitNetConfigRequest {
	p.async = &async
	return p
}

func (p *HostServiceCommitNetConfigRequest) Send() (*HostServiceCommitNetConfigResponse, error) {
	rawURL := fmt.Sprintf("%s%s/commitnetconfig", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceCommitNetConfigResponse), nil
}

func (p *HostServiceCommitNetConfigRequest) MustSend() *HostServiceCommitNetConfigResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Marks the network configuration as good and persists it inside the host.
// An API user commits the network configuration to persist a host network interface attachment or detachment, or
// persist the creation and deletion of a bonded interface.
// IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
// not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
// automatically reverts to the previous networking configuration.
// For example, to commit the network configuration of host with id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/commitnetconfig
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// IMPORTANT: Since {engine-name} 4.3, it is possible to also specify `commit_on_success` in
// the <<services/host/methods/setup_networks, setupnetworks>> request, in which case the new
// configuration is automatically saved in the {hypervisor-name} upon completing the setup and
// re-establishing connectivity between the {hypervisor-name} and {engine-name}, and without
// waiting for a separate <<services/host/methods/commit_net_config, commitnetconfig>> request.
//
type HostServiceCommitNetConfigResponse struct {
}

//
// Marks the network configuration as good and persists it inside the host.
// An API user commits the network configuration to persist a host network interface attachment or detachment, or
// persist the creation and deletion of a bonded interface.
// IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
// not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
// automatically reverts to the previous networking configuration.
// For example, to commit the network configuration of host with id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/commitnetconfig
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// IMPORTANT: Since {engine-name} 4.3, it is possible to also specify `commit_on_success` in
// the <<services/host/methods/setup_networks, setupnetworks>> request, in which case the new
// configuration is automatically saved in the {hypervisor-name} upon completing the setup and
// re-establishing connectivity between the {hypervisor-name} and {engine-name}, and without
// waiting for a separate <<services/host/methods/commit_net_config, commitnetconfig>> request.
//
func (p *HostService) CommitNetConfig() *HostServiceCommitNetConfigRequest {
	return &HostServiceCommitNetConfigRequest{HostService: p}
}

//
// Deactivates the host to perform maintenance tasks.
//
type HostServiceDeactivateRequest struct {
	HostService        *HostService
	header             map[string]string
	query              map[string]string
	async              *bool
	reason             *string
	stopGlusterService *bool
}

func (p *HostServiceDeactivateRequest) Header(key, value string) *HostServiceDeactivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceDeactivateRequest) Query(key, value string) *HostServiceDeactivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceDeactivateRequest) Async(async bool) *HostServiceDeactivateRequest {
	p.async = &async
	return p
}

func (p *HostServiceDeactivateRequest) Reason(reason string) *HostServiceDeactivateRequest {
	p.reason = &reason
	return p
}

func (p *HostServiceDeactivateRequest) StopGlusterService(stopGlusterService bool) *HostServiceDeactivateRequest {
	p.stopGlusterService = &stopGlusterService
	return p
}

func (p *HostServiceDeactivateRequest) Send() (*HostServiceDeactivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/deactivate", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.reason != nil {
		actionBuilder.Reason(*p.reason)
	}
	if p.stopGlusterService != nil {
		actionBuilder.StopGlusterService(*p.stopGlusterService)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceDeactivateResponse), nil
}

func (p *HostServiceDeactivateRequest) MustSend() *HostServiceDeactivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Deactivates the host to perform maintenance tasks.
//
type HostServiceDeactivateResponse struct {
}

//
// Deactivates the host to perform maintenance tasks.
//
func (p *HostService) Deactivate() *HostServiceDeactivateRequest {
	return &HostServiceDeactivateRequest{HostService: p}
}

//
// Enrolls the certificate of the host. Useful in case you get a warning that it is about to expire or has already
// expired.
//
type HostServiceEnrollCertificateRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceEnrollCertificateRequest) Header(key, value string) *HostServiceEnrollCertificateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceEnrollCertificateRequest) Query(key, value string) *HostServiceEnrollCertificateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceEnrollCertificateRequest) Async(async bool) *HostServiceEnrollCertificateRequest {
	p.async = &async
	return p
}

func (p *HostServiceEnrollCertificateRequest) Send() (*HostServiceEnrollCertificateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/enrollcertificate", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceEnrollCertificateResponse), nil
}

func (p *HostServiceEnrollCertificateRequest) MustSend() *HostServiceEnrollCertificateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Enrolls the certificate of the host. Useful in case you get a warning that it is about to expire or has already
// expired.
//
type HostServiceEnrollCertificateResponse struct {
}

//
// Enrolls the certificate of the host. Useful in case you get a warning that it is about to expire or has already
// expired.
//
func (p *HostService) EnrollCertificate() *HostServiceEnrollCertificateRequest {
	return &HostServiceEnrollCertificateRequest{HostService: p}
}

//
// Controls the host's power management device.
// For example, to start the host. This can be done via:
// [source]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <action>
//   <fence_type>start</fence_type>
// </action>
// ' \
// "${url}/hosts/123/fence"
// ----
//
type HostServiceFenceRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
	fenceType   *string
}

func (p *HostServiceFenceRequest) Header(key, value string) *HostServiceFenceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceFenceRequest) Query(key, value string) *HostServiceFenceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceFenceRequest) Async(async bool) *HostServiceFenceRequest {
	p.async = &async
	return p
}

func (p *HostServiceFenceRequest) FenceType(fenceType string) *HostServiceFenceRequest {
	p.fenceType = &fenceType
	return p
}

func (p *HostServiceFenceRequest) Send() (*HostServiceFenceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/fence", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.fenceType != nil {
		actionBuilder.FenceType(*p.fenceType)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPowerManagement()
	return &HostServiceFenceResponse{powerManagement: result}, nil
}

func (p *HostServiceFenceRequest) MustSend() *HostServiceFenceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Controls the host's power management device.
// For example, to start the host. This can be done via:
// [source]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <action>
//   <fence_type>start</fence_type>
// </action>
// ' \
// "${url}/hosts/123/fence"
// ----
//
type HostServiceFenceResponse struct {
	powerManagement *PowerManagement
}

func (p *HostServiceFenceResponse) PowerManagement() (*PowerManagement, bool) {
	if p.powerManagement != nil {
		return p.powerManagement, true
	}
	return nil, false
}

func (p *HostServiceFenceResponse) MustPowerManagement() *PowerManagement {
	if p.powerManagement == nil {
		panic("powerManagement in response does not exist")
	}
	return p.powerManagement
}

//
// Controls the host's power management device.
// For example, to start the host. This can be done via:
// [source]
// ----
// #!/bin/sh -ex
// url="https://engine.example.com/ovirt-engine/api"
// user="admin@internal"
// password="..."
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --user "${user}:${password}" \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --data '
// <action>
//   <fence_type>start</fence_type>
// </action>
// ' \
// "${url}/hosts/123/fence"
// ----
//
func (p *HostService) Fence() *HostServiceFenceRequest {
	return &HostServiceFenceRequest{HostService: p}
}

//
// To manually set a host as the storage pool manager (SPM).
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/forceselectspm
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type HostServiceForceSelectSpmRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *HostServiceForceSelectSpmRequest) Header(key, value string) *HostServiceForceSelectSpmRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceForceSelectSpmRequest) Query(key, value string) *HostServiceForceSelectSpmRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceForceSelectSpmRequest) Async(async bool) *HostServiceForceSelectSpmRequest {
	p.async = &async
	return p
}

func (p *HostServiceForceSelectSpmRequest) Send() (*HostServiceForceSelectSpmResponse, error) {
	rawURL := fmt.Sprintf("%s%s/forceselectspm", p.HostService.connection.URL(), p.HostService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostServiceForceSelectSpmResponse), nil
}

func (p *HostServiceForceSelectSpmRequest) MustSend() *HostServiceForceSelectSpmResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// To manually set a host as the storage pool manager (SPM).
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/forceselectspm
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
type HostServiceForceSelectSpmResponse struct {
}

//
// To manually set a host as the storage pool manager (SPM).
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/forceselectspm
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *HostService) ForceSelectSpm() *HostServiceForceSelectSpmRequest {
	return &HostServiceForceSelectSpmRequest{HostService: p}
}

//
// Gets the host details.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123
// ----
//
type HostServiceGetRequest struct {
	HostService *HostService
	header      map[string]string
	query       map[string]string
	allContent  *bool
	filter      *bool
	follow      *string
}

func (p *HostServiceGetRequest) Header(key, value string) *HostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceGetRequest) Query(key, value string) *HostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceGetRequest) AllContent(allContent bool) *HostServiceGetRequest {
	p.allContent = &allContent
	return p
}

func (p *HostServiceGetRequest) Filter(filter bool) *HostServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *HostServiceGetRequest) Follow(follow string) *HostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostServiceGetRequest) Send() (*HostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostService.connection.URL(), p.HostService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostServiceGetResponse{host: result}, nil
}

func (p *HostServiceGetRequest) MustSend() *HostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the host details.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123
// ----
//
type HostServiceGetResponse struct {
	host *Host
}

func (p *HostServiceGetResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *HostServiceGetResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Gets the host details.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123
// ----
//
func (p *HostService) Get() *HostServiceGetRequest {
	return &HostServiceGetRequest{HostService: p}
}

//
// Installs the latest version of VDSM and related software on the host.
// The action also performs every configuration steps on the host which is done during adding host to the engine:
// kdump configuration, hosted-engine deploy, kernel options changes, etc.
// The host type defines additional parameters for the action.
// Example of installing a host, using `curl` and JSON, plain:
// [source,bash]
// ----
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --request PUT \
// --header "Content-Type: application/json" \
// --header "Accept: application/json" \
// --header "Version: 4" \
// --user "admin@internal:..." \
// --data '
// {
//   "root_password": "myrootpassword"
// }
// ' \
// "https://engine.example.com/ovirt-engine/api/hosts/123"
// ----
// Example of installing a host, using `curl` and JSON, with hosted engine components:
// [source,bash]
// ----
// curl \
// curl \
// --verbose \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --request PUT \
// --header "Content-Type: application/json" \
// --header "Accept: application/json" \
// --header "Version: 4" \
// --user "admin@internal:..." \
// --data '
// {
//   "root_password": "myrootpassword"
// }
// ' \
// "https://engine.example.com/ovirt-engine/api/hosts/123?deploy_hosted_engine=true"
// ----
// IMPORTANT: Since version 4.1.2 of the engine when a host is reinstalled we override the host firewall
// definitions by default.
//
type HostServiceInstallRequest struct {
	HostService          *HostService
	header               map[string]string
	query                map[string]string
	activate             *bool
	async                *bool
	deployHostedEngine   *bool
	host                 *Host
	image                *string
	rootPassword         *string
	ssh                  *Ssh
	undeployHostedEngine *bool
}

func (p *HostServiceInstallRequest) Header(key, value string) *HostServiceInstallRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostServiceInstallRequest) Query(key, value string) *HostServiceInstallRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostServiceInstallRequest) Activate(activate bool) *HostServiceInstallRequest {
	p.activate = &activate
	return p
}

func (p *HostServiceInstallRequest) Async(async bool) *HostServiceInstallRequest {
	p.async = &async
	return p
}

func (p *HostServiceInstallRequest) DeployHostedEngine(deployHostedEngine bool) *HostServiceInstallRequest {
	p.deployHostedEngine = &deployHo