/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.lang.String;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ClusterBuilder;
import org.ovirt.engine.sdk4.builders.HostBuilder;
import org.ovirt.engine.sdk4.builders.HostNicBuilder;
import org.ovirt.engine.sdk4.builders.IscsiDetailsBuilder;
import org.ovirt.engine.sdk4.builders.NetworkAttachmentBuilder;
import org.ovirt.engine.sdk4.builders.NetworkLabelBuilder;
import org.ovirt.engine.sdk4.builders.SshBuilder;
import org.ovirt.engine.sdk4.types.Cluster;
import org.ovirt.engine.sdk4.types.Host;
import org.ovirt.engine.sdk4.types.HostNic;
import org.ovirt.engine.sdk4.types.IscsiDetails;
import org.ovirt.engine.sdk4.types.NetworkAttachment;
import org.ovirt.engine.sdk4.types.NetworkLabel;
import org.ovirt.engine.sdk4.types.PowerManagement;
import org.ovirt.engine.sdk4.types.Ssh;
import org.ovirt.engine.sdk4.types.StorageDomain;

/**
 * A service to manage a host.
 */
public interface HostService extends MeasurableService {
    /**
     * Activate the host for use, such as running virtual machines.
     */
    public interface ActivateRequest extends Request<ActivateRequest, ActivateResponse> {
        /**
         * Indicates if the activation should be performed asynchronously.
         */
        ActivateRequest async(Boolean async);
    }
    
    /**
     * Activate the host for use, such as running virtual machines.
     */
    public interface ActivateResponse extends Response {
    }
    
    /**
     * Activate the host for use, such as running virtual machines.
     */
    ActivateRequest activate();
    
    /**
     * Approve a pre-installed Hypervisor host for usage in the virtualization environment.
     * 
     * This action also accepts an optional cluster element to define the target cluster for this host.
     */
    public interface ApproveRequest extends Request<ApproveRequest, ApproveResponse> {
        /**
         * Indicates if the approval should be performed asynchronously.
         */
        ApproveRequest async(Boolean async);
        ApproveRequest cluster(Cluster cluster);
        /**
         * Cluster where the host will be added after approving it.
         */
        ApproveRequest cluster(ClusterBuilder cluster);
        ApproveRequest host(Host host);
        /**
         * Host to approve.
         */
        ApproveRequest host(HostBuilder host);
    }
    
    /**
     * Approve a pre-installed Hypervisor host for usage in the virtualization environment.
     * 
     * This action also accepts an optional cluster element to define the target cluster for this host.
     */
    public interface ApproveResponse extends Response {
    }
    
    /**
     * Approve a pre-installed Hypervisor host for usage in the virtualization environment.
     * 
     * This action also accepts an optional cluster element to define the target cluster for this host.
     */
    ApproveRequest approve();
    
    /**
     * Marks the network configuration as good and persists it inside the host.
     * 
     * An API user commits the network configuration to persist a host network interface attachment or detachment, or
     * persist the creation and deletion of a bonded interface.
     * 
     * IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
     * not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
     * automatically reverts to the previous networking configuration.
     * 
     * For example, to commit the network configuration of host with id `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts/123/commitnetconfig
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action/>
     * ----
     */
    public interface CommitNetConfigRequest extends Request<CommitNetConfigRequest, CommitNetConfigResponse> {
        /**
         * Indicates if the action should be performed asynchronously.
         */
        CommitNetConfigRequest async(Boolean async);
    }
    
    /**
     * Marks the network configuration as good and persists it inside the host.
     * 
     * An API user commits the network configuration to persist a host network interface attachment or detachment, or
     * persist the creation and deletion of a bonded interface.
     * 
     * IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
     * not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
     * automatically reverts to the previous networking configuration.
     * 
     * For example, to commit the network configuration of host with id `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts/123/commitnetconfig
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action/>
     * ----
     */
    public interface CommitNetConfigResponse extends Response {
    }
    
    /**
     * Marks the network configuration as good and persists it inside the host.
     * 
     * An API user commits the network configuration to persist a host network interface attachment or detachment, or
     * persist the creation and deletion of a bonded interface.
     * 
     * IMPORTANT: Networking configuration is only committed after the engine has established that host connectivity is
     * not lost as a result of the configuration changes. If host connectivity is lost, the host requires a reboot and
     * automatically reverts to the previous networking configuration.
     * 
     * For example, to commit the network configuration of host with id `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts/123/commitnetconfig
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action/>
     * ----
     */
    CommitNetConfigRequest commitNetConfig();
    
    /**
     * Deactivate the host to perform maintenance tasks.
     */
    public interface DeactivateRequest extends Request<DeactivateRequest, DeactivateResponse> {
        /**
         * Indicates if the deactivation should be performed asynchronously.
         */
        DeactivateRequest async(Boolean async);
        DeactivateRequest reason(String reason);
        /**
         * Indicates if the gluster service should be stopped as part of deactivating the host. It can be used while
         * performing maintenance operations on the gluster host. Default value for this variable is `false`.
         */
        DeactivateRequest stopGlusterService(Boolean stopGlusterService);
    }
    
    /**
     * Deactivate the host to perform maintenance tasks.
     */
    public interface DeactivateResponse extends Response {
    }
    
    /**
     * Deactivate the host to perform maintenance tasks.
     */
    DeactivateRequest deactivate();
    
    /**
     * Enroll certificate of the host. Useful in case you get a warning that it is about to, or already expired.
     */
    public interface EnrollCertificateRequest extends Request<EnrollCertificateRequest, EnrollCertificateResponse> {
        /**
         * Indicates if the enrollment should be performed asynchronously.
         */
        EnrollCertificateRequest async(Boolean async);
    }
    
    /**
     * Enroll certificate of the host. Useful in case you get a warning that it is about to, or already expired.
     */
    public interface EnrollCertificateResponse extends Response {
    }
    
    /**
     * Enroll certificate of the host. Useful in case you get a warning that it is about to, or already expired.
     */
    EnrollCertificateRequest enrollCertificate();
    
    /**
     * Controls host's power management device.
     * 
     * For example, let's assume you want to start the host. This can be done via:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <fence_type>start</fence_type>
     * </action>
     * ' \
     * "${url}/hosts/123/fence"
     * ----
     */
    public interface FenceRequest extends Request<FenceRequest, FenceResponse> {
        /**
         * Indicates if the fencing should be performed asynchronously.
         */
        FenceRequest async(Boolean async);
        FenceRequest fenceType(String fenceType);
    }
    
    /**
     * Controls host's power management device.
     * 
     * For example, let's assume you want to start the host. This can be done via:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <fence_type>start</fence_type>
     * </action>
     * ' \
     * "${url}/hosts/123/fence"
     * ----
     */
    public interface FenceResponse extends Response {
        PowerManagement powerManagement();
    }
    
    /**
     * Controls host's power management device.
     * 
     * For example, let's assume you want to start the host. This can be done via:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <fence_type>start</fence_type>
     * </action>
     * ' \
     * "${url}/hosts/123/fence"
     * ----
     */
    FenceRequest fence();
    
    /**
     * Manually set a host as the storage pool manager (SPM).
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts/123/forceselectspm
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action/>
     * ----
     */
    public interface ForceSelectSpmRequest extends Request<ForceSelectSpmRequest, ForceSelectSpmResponse> {
        /**
         * Indicates if the action should be performed asynchronously.
         */
        ForceSelectSpmRequest async(Boolean async);
    }
    
    /**
     * Manually set a host as the storage pool manager (SPM).
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts/123/forceselectspm
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action/>
     * ----
     */
    public interface ForceSelectSpmResponse extends Response {
    }
    
    /**
     * Manually set a host as the storage pool manager (SPM).
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts/123/forceselectspm
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action/>
     * ----
     */
    ForceSelectSpmRequest forceSelectSpm();
    
    /**
     * Get the host details.
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/hosts/123
     * ----
     */
    public interface GetRequest extends Request<GetRequest, GetResponse> {
        /**
         * Indicates if all of the attributes of the host should be included in the response.
         * 
         * By default the following attributes are excluded:
         * 
         * - `hosted_engine`
         * 
         * For example, to retrieve the complete representation of host '123':
         * 
         * ....
         * GET /ovirt-engine/api/hosts/123?all_content=true
         * ....
         * 
         * NOTE: These attributes are not included by default because retrieving them impacts performance. They are
         * seldom used and require additional queries to the database. Use this parameter with caution and only when
         * specifically required.
         */
        GetRequest allContent(Boolean allContent);
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        GetRequest filter(Boolean filter);
    }
    
    /**
     * Get the host details.
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/hosts/123
     * ----
     */
    public interface GetResponse extends Response {
        /**
         * Queried host.
         */
        Host host();
    }
    
    /**
     * Get the host details.
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/hosts/123
     * ----
     */
    GetRequest get();
    
    /**
     * Install VDSM and related software on the host. The host type defines additional parameters for the action.
     * 
     * Example of installing a host, using `curl` and JSON, plain:
     * 
     * [source,bash]
     * ----
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --request PUT \
     * --header "Content-Type: application/json" \
     * --header "Accept: application/json" \
     * --header "Version: 4" \
     * --user "admin@internal:..." \
     * --data '
     * {
         *   "root_password": "myrootpassword"
     * }
     * ' \
     * "https://engine.example.com/ovirt-engine/api/hosts/123"
     * ----
     * 
     * Example of installing a host, using `curl` and JSON, with hosted engine components:
     * 
     * [source,bash]
     * ----
     * curl \
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --request PUT \
     * --header "Content-Type: application/json" \
     * --header "Accept: application/json" \
     * --header "Version: 4" \
     * --user "admin@internal:..." \
     * --data '
     * {
         *   "root_password": "myrootpassword"
     * }
     * ' \
     * "https://engine.example.com/ovirt-engine/api/hosts/123?deploy_hosted_engine=true"
     * ----
     * 
     * IMPORTANT: Since version 4.1.2 of the engine when a host is reinstalled we override the host firewall
     * definitions by default.
     */
    public interface InstallRequest extends Request<InstallRequest, InstallResponse> {
        /**
         * Indicates if the installation should be performed asynchronously.
         */
        InstallRequest async(Boolean async);
        /**
         * When set to `true` it means this host should deploy also hosted
         * engine components. Missing value is treated as `true` i.e deploy.
         * Omitting this parameter means `false` and will perform no operation
         * in hosted engine area.
         */
        InstallRequest deployHostedEngine(Boolean deployHostedEngine);
        InstallRequest host(Host host);
        /**
         * This `override_iptables` property is used to indicate if the firewall configuration should be
         * replaced by the default one.
         */
        InstallRequest host(HostBuilder host);
        /**
         * When installing an oVirt node a image ISO file is needed.
         */
        InstallRequest image(String image);
        /**
         * The password of of the `root` user, used to connect to the host via SSH.
         */
        InstallRequest rootPassword(String rootPassword);
        InstallRequest ssh(Ssh ssh);
        /**
         * The SSH details used to connect to the host.
         */
        InstallRequest ssh(SshBuilder ssh);
        /**
         * When set to `true` it means this host should un-deploy hosted engine
         * components and this host will not function as part of the High
         * Availability cluster. Missing value is treated as `true` i.e un-deploy
         * Omitting this parameter means `false` and will perform no operation
         * in hosted engine area.
         */
        InstallRequest undeployHostedEngine(Boolean undeployHostedEngine);
    }
    
    /**
     * Install VDSM and related software on the host. The host type defines additional parameters for the action.
     * 
     * Example of installing a host, using `curl` and JSON, plain:
     * 
     * [source,bash]
     * ----
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --request PUT \
     * --header "Content-Type: application/json" \
     * --header "Accept: application/json" \
     * --header "Version: 4" \
     * --user "admin@internal:..." \
     * --data '
     * {
         *   "root_password": "myrootpassword"
     * }
     * ' \
     * "https://engine.example.com/ovirt-engine/api/hosts/123"
     * ----
     * 
     * Example of installing a host, using `curl` and JSON, with hosted engine components:
     * 
     * [source,bash]
     * ----
     * curl \
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --request PUT \
     * --header "Content-Type: application/json" \
     * --header "Accept: application/json" \
     * --header "Version: 4" \
     * --user "admin@internal:..." \
     * --data '
     * {
         *   "root_password": "myrootpassword"
     * }
     * ' \
     * "https://engine.example.com/ovirt-engine/api/hosts/123?deploy_hosted_engine=true"
     * ----
     * 
     * IMPORTANT: Since version 4.1.2 of the engine when a host is reinstalled we override the host firewall
     * definitions by default.
     */
    public interface InstallResponse extends Response {
    }
    
    /**
     * Install VDSM and related software on the host. The host type defines additional parameters for the action.
     * 
     * Example of installing a host, using `curl` and JSON, plain:
     * 
     * [source,bash]
     * ----
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --request PUT \
     * --header "Content-Type: application/json" \
     * --header "Accept: application/json" \
     * --header "Version: 4" \
     * --user "admin@internal:..." \
     * --data '
     * {
         *   "root_password": "myrootpassword"
     * }
     * ' \
     * "https://engine.example.com/ovirt-engine/api/hosts/123"
     * ----
     * 
     * Example of installing a host, using `curl` and JSON, with hosted engine components:
     * 
     * [source,bash]
     * ----
     * curl \
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --request PUT \
     * --header "Content-Type: application/json" \
     * --header "Accept: application/json" \
     * --header "Version: 4" \
     * --user "admin@internal:..." \
     * --data '
     * {
         *   "root_password": "myrootpassword"
     * }
     * ' \
     * "https://engine.example.com/ovirt-engine/api/hosts/123?deploy_hosted_engine=true"
     * ----
     * 
     * IMPORTANT: Since version 4.1.2 of the engine when a host is reinstalled we override the host firewall
     * definitions by default.
     */
    InstallRequest install();
    
    /**
     * Discover iSCSI targets on the host, using the initiator details.
     */
    public interface IscsiDiscoverRequest extends Request<IscsiDiscoverRequest, IscsiDiscoverResponse> {
        /**
         * Indicates if the discovery should be performed asynchronously.
         */
        IscsiDiscoverRequest async(Boolean async);
        IscsiDiscoverRequest iscsi(IscsiDetails iscsi);
        /**
         * The target iSCSI device.
         */
        IscsiDiscoverRequest iscsi(IscsiDetailsBuilder iscsi);
    }
    
    /**
     * Discover iSCSI targets on the host, using the initiator details.
     */
    public interface IscsiDiscoverResponse extends Response {
        /**
         * The iSCSI targets.
         */
        List<String> iscsiTargets();
    }
    
    /**
     * Discover iSCSI targets on the host, using the initiator details.
     */
    IscsiDiscoverRequest iscsiDiscover();
    
    /**
     * Login to iSCSI targets on the host, using the target details.
     */
    public interface IscsiLoginRequest extends Request<IscsiLoginRequest, IscsiLoginResponse> {
        /**
         * Indicates if the login should be performed asynchronously.
         */
        IscsiLoginRequest async(Boolean async);
        IscsiLoginRequest iscsi(IscsiDetails iscsi);
        /**
         * The target iSCSI device.
         */
        IscsiLoginRequest iscsi(IscsiDetailsBuilder iscsi);
    }
    
    /**
     * Login to iSCSI targets on the host, using the target details.
     */
    public interface IscsiLoginResponse extends Response {
    }
    
    /**
     * Login to iSCSI targets on the host, using the target details.
     */
    IscsiLoginRequest iscsiLogin();
    
    /**
     * Refresh the host devices and capabilities.
     */
    public interface RefreshRequest extends Request<RefreshRequest, RefreshResponse> {
        /**
         * Indicates if the refresh should be performed asynchronously.
         */
        RefreshRequest async(Boolean async);
    }
    
    /**
     * Refresh the host devices and capabilities.
     */
    public interface RefreshResponse extends Response {
    }
    
    /**
     * Refresh the host devices and capabilities.
     */
    RefreshRequest refresh();
    
    /**
     * Remove the host from the system.
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request DELETE \
     * --header "Version: 4" \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc"
     * ----
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    /**
     * Remove the host from the system.
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request DELETE \
     * --header "Version: 4" \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc"
     * ----
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Remove the host from the system.
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request DELETE \
     * --header "Version: 4" \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc"
     * ----
     */
    RemoveRequest remove();
    
    /**
     * This method is used to change the configuration of the network interfaces of a host.
     * 
     * For example, lets assume that you have a host with three network interfaces `eth0`, `eth1` and `eth2` and that
     * you want to configure a new bond using `eth0` and `eth1`, and put a VLAN on top of it. Using a simple shell
     * script and the `curl` command line HTTP client that can be done as follows:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <modified_bonds>
     *     <host_nic>
     *       <name>bond0</name>
     *       <bonding>
     *         <options>
     *           <option>
     *             <name>mode</name>
     *             <value>4</value>
     *           </option>
     *           <option>
     *             <name>miimon</name>
     *             <value>100</value>
     *           </option>
     *         </options>
     *         <slaves>
     *           <host_nic>
     *             <name>eth1</name>
     *           </host_nic>
     *           <host_nic>
     *             <name>eth2</name>
     *           </host_nic>
     *         </slaves>
     *       </bonding>
     *     </host_nic>
     *   </modified_bonds>
     *   <modified_network_attachments>
     *     <network_attachment>
     *       <network>
     *         <name>myvlan</name>
     *       </network>
     *       <host_nic>
     *         <name>bond0</name>
     *       </host_nic>
     *       <ip_address_assignments>
     *         <assignment_method>static</assignment_method>
     *         <ip_address_assignment>
     *           <ip>
     *             <address>192.168.122.10</address>
     *             <netmask>255.255.255.0</netmask>
     *           </ip>
     *         </ip_address_assignment>
     *       </ip_address_assignments>
     *       <dns_resolver_configuration>
     *         <name_servers>
     *           <name_server>1.1.1.1</name_server>
     *           <name_server>2.2.2.2</name_server>
     *         </name_servers>
     *       </dns_resolver_configuration>
     *     </network_attachment>
     *   </modified_network_attachments>
     *  </action>
     * ' \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc/setupnetworks"
     * ----
     * 
     * Note that this is valid for version 4 of the API. In previous versions some elements were represented as XML
     * attributes instead of XML elements. In particular the `options` and `ip` elements were represented as follows:
     * 
     * [source,xml]
     * ----
     * <options name="mode" value="4"/>
     * <options name="miimon" value="100"/>
     * <ip address="192.168.122.10" netmask="255.255.255.0"/>
     * ----
     * 
     * Using the Python SDK the same can be done with the following code:
     * 
     * [source,python]
     * ----
     * # Find the service that manages the collection of hosts:
     * hosts_service = connection.system_service().hosts_service()
     * 
     * # Find the host:
     * host = hosts_service.list(search='name=myhost')[0]
     * 
     * # Find the service that manages the host:
     * host_service = hosts_service.host_service(host.id)
     * 
     * # Configure the network adding a bond with two slaves and attaching it to a
     * # network with an static IP address:
     * host_service.setup_networks(
     *     modified_bonds=[
     *         types.HostNic(
     *             name='bond0',
     *             bonding=types.Bonding(
     *                 options=[
     *                     types.Option(
     *                         name='mode',
     *                         value='4',
     *                     ),
     *                     types.Option(
     *                         name='miimon',
     *                         value='100',
     *                     ),
     *                 ],
     *                 slaves=[
     *                     types.HostNic(
     *                         name='eth1',
     *                     ),
     *                     types.HostNic(
     *                         name='eth2',
     *                     ),
     *                 ],
     *             ),
     *         ),
     *     ],
     *     modified_network_attachments=[
     *         types.NetworkAttachment(
     *             network=types.Network(
     *                 name='myvlan',
     *             ),
     *             host_nic=types.HostNic(
     *                 name='bond0',
     *             ),
     *             ip_address_assignments=[
     *                 types.IpAddressAssignment(
     *                     assignment_method=types.BootProtocol.STATIC,
     *                     ip=types.Ip(
     *                         address='192.168.122.10',
     *                         netmask='255.255.255.0',
     *                     ),
     *                 ),
     *             ],
     *             dns_resolver_configuration=types.DnsResolverConfiguration(
     *                 name_servers=[
     *                     '1.1.1.1',
     *                     '2.2.2.2',
     *                 ],
     *             ),
     *         ),
     *     ],
     * )
     * 
     * # After modifying the network configuration it is very important to make it
     * # persistent:
     * host_service.commit_net_config()
     * ----
     * 
     * IMPORTANT: To make sure that the network configuration has been saved in the host, and that it will be applied
     * when the host is rebooted, remember to call <<services/host/methods/commit_net_config, commitnetconfig>>.
     */
    public interface SetupNetworksRequest extends Request<SetupNetworksRequest, SetupNetworksResponse> {
        /**
         * Indicates if the action should be performed asynchronously.
         */
        SetupNetworksRequest async(Boolean async);
        SetupNetworksRequest checkConnectivity(Boolean checkConnectivity);
        SetupNetworksRequest connectivityTimeout(Integer connectivityTimeout);
        SetupNetworksRequest connectivityTimeout(Long connectivityTimeout);
        SetupNetworksRequest connectivityTimeout(BigInteger connectivityTimeout);
        SetupNetworksRequest modifiedBonds(List<HostNic> modifiedBonds);
        SetupNetworksRequest modifiedBonds(HostNic... modifiedBonds);
        SetupNetworksRequest modifiedBonds(HostNicBuilder... modifiedBonds);
        SetupNetworksRequest modifiedLabels(List<NetworkLabel> modifiedLabels);
        SetupNetworksRequest modifiedLabels(NetworkLabel... modifiedLabels);
        SetupNetworksRequest modifiedLabels(NetworkLabelBuilder... modifiedLabels);
        SetupNetworksRequest modifiedNetworkAttachments(List<NetworkAttachment> modifiedNetworkAttachments);
        SetupNetworksRequest modifiedNetworkAttachments(NetworkAttachment... modifiedNetworkAttachments);
        SetupNetworksRequest modifiedNetworkAttachments(NetworkAttachmentBuilder... modifiedNetworkAttachments);
        SetupNetworksRequest removedBonds(List<HostNic> removedBonds);
        SetupNetworksRequest removedBonds(HostNic... removedBonds);
        SetupNetworksRequest removedBonds(HostNicBuilder... removedBonds);
        SetupNetworksRequest removedLabels(List<NetworkLabel> removedLabels);
        SetupNetworksRequest removedLabels(NetworkLabel... removedLabels);
        SetupNetworksRequest removedLabels(NetworkLabelBuilder... removedLabels);
        SetupNetworksRequest removedNetworkAttachments(List<NetworkAttachment> removedNetworkAttachments);
        SetupNetworksRequest removedNetworkAttachments(NetworkAttachment... removedNetworkAttachments);
        SetupNetworksRequest removedNetworkAttachments(NetworkAttachmentBuilder... removedNetworkAttachments);
        SetupNetworksRequest synchronizedNetworkAttachments(List<NetworkAttachment> synchronizedNetworkAttachments);
        SetupNetworksRequest synchronizedNetworkAttachments(NetworkAttachment... synchronizedNetworkAttachments);
        SetupNetworksRequest synchronizedNetworkAttachments(NetworkAttachmentBuilder... synchronizedNetworkAttachments);
    }
    
    /**
     * This method is used to change the configuration of the network interfaces of a host.
     * 
     * For example, lets assume that you have a host with three network interfaces `eth0`, `eth1` and `eth2` and that
     * you want to configure a new bond using `eth0` and `eth1`, and put a VLAN on top of it. Using a simple shell
     * script and the `curl` command line HTTP client that can be done as follows:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <modified_bonds>
     *     <host_nic>
     *       <name>bond0</name>
     *       <bonding>
     *         <options>
     *           <option>
     *             <name>mode</name>
     *             <value>4</value>
     *           </option>
     *           <option>
     *             <name>miimon</name>
     *             <value>100</value>
     *           </option>
     *         </options>
     *         <slaves>
     *           <host_nic>
     *             <name>eth1</name>
     *           </host_nic>
     *           <host_nic>
     *             <name>eth2</name>
     *           </host_nic>
     *         </slaves>
     *       </bonding>
     *     </host_nic>
     *   </modified_bonds>
     *   <modified_network_attachments>
     *     <network_attachment>
     *       <network>
     *         <name>myvlan</name>
     *       </network>
     *       <host_nic>
     *         <name>bond0</name>
     *       </host_nic>
     *       <ip_address_assignments>
     *         <assignment_method>static</assignment_method>
     *         <ip_address_assignment>
     *           <ip>
     *             <address>192.168.122.10</address>
     *             <netmask>255.255.255.0</netmask>
     *           </ip>
     *         </ip_address_assignment>
     *       </ip_address_assignments>
     *       <dns_resolver_configuration>
     *         <name_servers>
     *           <name_server>1.1.1.1</name_server>
     *           <name_server>2.2.2.2</name_server>
     *         </name_servers>
     *       </dns_resolver_configuration>
     *     </network_attachment>
     *   </modified_network_attachments>
     *  </action>
     * ' \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc/setupnetworks"
     * ----
     * 
     * Note that this is valid for version 4 of the API. In previous versions some elements were represented as XML
     * attributes instead of XML elements. In particular the `options` and `ip` elements were represented as follows:
     * 
     * [source,xml]
     * ----
     * <options name="mode" value="4"/>
     * <options name="miimon" value="100"/>
     * <ip address="192.168.122.10" netmask="255.255.255.0"/>
     * ----
     * 
     * Using the Python SDK the same can be done with the following code:
     * 
     * [source,python]
     * ----
     * # Find the service that manages the collection of hosts:
     * hosts_service = connection.system_service().hosts_service()
     * 
     * # Find the host:
     * host = hosts_service.list(search='name=myhost')[0]
     * 
     * # Find the service that manages the host:
     * host_service = hosts_service.host_service(host.id)
     * 
     * # Configure the network adding a bond with two slaves and attaching it to a
     * # network with an static IP address:
     * host_service.setup_networks(
     *     modified_bonds=[
     *         types.HostNic(
     *             name='bond0',
     *             bonding=types.Bonding(
     *                 options=[
     *                     types.Option(
     *                         name='mode',
     *                         value='4',
     *                     ),
     *                     types.Option(
     *                         name='miimon',
     *                         value='100',
     *                     ),
     *                 ],
     *                 slaves=[
     *                     types.HostNic(
     *                         name='eth1',
     *                     ),
     *                     types.HostNic(
     *                         name='eth2',
     *                     ),
     *                 ],
     *             ),
     *         ),
     *     ],
     *     modified_network_attachments=[
     *         types.NetworkAttachment(
     *             network=types.Network(
     *                 name='myvlan',
     *             ),
     *             host_nic=types.HostNic(
     *                 name='bond0',
     *             ),
     *             ip_address_assignments=[
     *                 types.IpAddressAssignment(
     *                     assignment_method=types.BootProtocol.STATIC,
     *                     ip=types.Ip(
     *                         address='192.168.122.10',
     *                         netmask='255.255.255.0',
     *                     ),
     *                 ),
     *             ],
     *             dns_resolver_configuration=types.DnsResolverConfiguration(
     *                 name_servers=[
     *                     '1.1.1.1',
     *                     '2.2.2.2',
     *                 ],
     *             ),
     *         ),
     *     ],
     * )
     * 
     * # After modifying the network configuration it is very important to make it
     * # persistent:
     * host_service.commit_net_config()
     * ----
     * 
     * IMPORTANT: To make sure that the network configuration has been saved in the host, and that it will be applied
     * when the host is rebooted, remember to call <<services/host/methods/commit_net_config, commitnetconfig>>.
     */
    public interface SetupNetworksResponse extends Response {
    }
    
    /**
     * This method is used to change the configuration of the network interfaces of a host.
     * 
     * For example, lets assume that you have a host with three network interfaces `eth0`, `eth1` and `eth2` and that
     * you want to configure a new bond using `eth0` and `eth1`, and put a VLAN on top of it. Using a simple shell
     * script and the `curl` command line HTTP client that can be done as follows:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <modified_bonds>
     *     <host_nic>
     *       <name>bond0</name>
     *       <bonding>
     *         <options>
     *           <option>
     *             <name>mode</name>
     *             <value>4</value>
     *           </option>
     *           <option>
     *             <name>miimon</name>
     *             <value>100</value>
     *           </option>
     *         </options>
     *         <slaves>
     *           <host_nic>
     *             <name>eth1</name>
     *           </host_nic>
     *           <host_nic>
     *             <name>eth2</name>
     *           </host_nic>
     *         </slaves>
     *       </bonding>
     *     </host_nic>
     *   </modified_bonds>
     *   <modified_network_attachments>
     *     <network_attachment>
     *       <network>
     *         <name>myvlan</name>
     *       </network>
     *       <host_nic>
     *         <name>bond0</name>
     *       </host_nic>
     *       <ip_address_assignments>
     *         <assignment_method>static</assignment_method>
     *         <ip_address_assignment>
     *           <ip>
     *             <address>192.168.122.10</address>
     *             <netmask>255.255.255.0</netmask>
     *           </ip>
     *         </ip_address_assignment>
     *       </ip_address_assignments>
     *       <dns_resolver_configuration>
     *         <name_servers>
     *           <name_server>1.1.1.1</name_server>
     *           <name_server>2.2.2.2</name_server>
     *         </name_servers>
     *       </dns_resolver_configuration>
     *     </network_attachment>
     *   </modified_network_attachments>
     *  </action>
     * ' \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc/setupnetworks"
     * ----
     * 
     * Note that this is valid for version 4 of the API. In previous versions some elements were represented as XML
     * attributes instead of XML elements. In particular the `options` and `ip` elements were represented as follows:
     * 
     * [source,xml]
     * ----
     * <options name="mode" value="4"/>
     * <options name="miimon" value="100"/>
     * <ip address="192.168.122.10" netmask="255.255.255.0"/>
     * ----
     * 
     * Using the Python SDK the same can be done with the following code:
     * 
     * [source,python]
     * ----
     * # Find the service that manages the collection of hosts:
     * hosts_service = connection.system_service().hosts_service()
     * 
     * # Find the host:
     * host = hosts_service.list(search='name=myhost')[0]
     * 
     * # Find the service that manages the host:
     * host_service = hosts_service.host_service(host.id)
     * 
     * # Configure the network adding a bond with two slaves and attaching it to a
     * # network with an static IP address:
     * host_service.setup_networks(
     *     modified_bonds=[
     *         types.HostNic(
     *             name='bond0',
     *             bonding=types.Bonding(
     *                 options=[
     *                     types.Option(
     *                         name='mode',
     *                         value='4',
     *                     ),
     *                     types.Option(
     *                         name='miimon',
     *                         value='100',
     *                     ),
     *                 ],
     *                 slaves=[
     *                     types.HostNic(
     *                         name='eth1',
     *                     ),
     *                     types.HostNic(
     *                         name='eth2',
     *                     ),
     *                 ],
     *             ),
     *         ),
     *     ],
     *     modified_network_attachments=[
     *         types.NetworkAttachment(
     *             network=types.Network(
     *                 name='myvlan',
     *             ),
     *             host_nic=types.HostNic(
     *                 name='bond0',
     *             ),
     *             ip_address_assignments=[
     *                 types.IpAddressAssignment(
     *                     assignment_method=types.BootProtocol.STATIC,
     *                     ip=types.Ip(
     *                         address='192.168.122.10',
     *                         netmask='255.255.255.0',
     *                     ),
     *                 ),
     *             ],
     *             dns_resolver_configuration=types.DnsResolverConfiguration(
     *                 name_servers=[
     *                     '1.1.1.1',
     *                     '2.2.2.2',
     *                 ],
     *             ),
     *         ),
     *     ],
     * )
     * 
     * # After modifying the network configuration it is very important to make it
     * # persistent:
     * host_service.commit_net_config()
     * ----
     * 
     * IMPORTANT: To make sure that the network configuration has been saved in the host, and that it will be applied
     * when the host is rebooted, remember to call <<services/host/methods/commit_net_config, commitnetconfig>>.
     */
    SetupNetworksRequest setupNetworks();
    
    /**
     * Discover the block Storage Domains which are candidates to be imported to the setup. For FCP no arguments are needed.
     */
    public interface UnregisteredStorageDomainsDiscoverRequest extends Request<UnregisteredStorageDomainsDiscoverRequest, UnregisteredStorageDomainsDiscoverResponse> {
        /**
         * Indicates if the discovery should be performed asynchronously.
         */
        UnregisteredStorageDomainsDiscoverRequest async(Boolean async);
        UnregisteredStorageDomainsDiscoverRequest iscsi(IscsiDetails iscsi);
        UnregisteredStorageDomainsDiscoverRequest iscsi(IscsiDetailsBuilder iscsi);
    }
    
    /**
     * Discover the block Storage Domains which are candidates to be imported to the setup. For FCP no arguments are needed.
     */
    public interface UnregisteredStorageDomainsDiscoverResponse extends Response {
        List<StorageDomain> storageDomains();
    }
    
    /**
     * Discover the block Storage Domains which are candidates to be imported to the setup. For FCP no arguments are needed.
     */
    UnregisteredStorageDomainsDiscoverRequest unregisteredStorageDomainsDiscover();
    
    /**
     * Update the host properties.
     * 
     * For example, to update a the kernel command line of a host send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/hosts/123
     * ----
     * 
     * With request body like this:
     * 
     * [source, xml]
     * ----
     * <host>
     *   <os>
     *     <custom_kernel_cmdline>vfio_iommu_type1.allow_unsafe_interrupts=1</custom_kernel_cmdline>
     *   </os>
     * </host>
     * ----
     */
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        /**
         * Indicates if the update should be performed asynchronously.
         */
        UpdateRequest async(Boolean async);
        UpdateRequest host(Host host);
        UpdateRequest host(HostBuilder host);
    }
    
    /**
     * Update the host properties.
     * 
     * For example, to update a the kernel command line of a host send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/hosts/123
     * ----
     * 
     * With request body like this:
     * 
     * [source, xml]
     * ----
     * <host>
     *   <os>
     *     <custom_kernel_cmdline>vfio_iommu_type1.allow_unsafe_interrupts=1</custom_kernel_cmdline>
     *   </os>
     * </host>
     * ----
     */
    public interface UpdateResponse extends Response {
        Host host();
    }
    
    /**
     * Update the host properties.
     * 
     * For example, to update a the kernel command line of a host send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/hosts/123
     * ----
     * 
     * With request body like this:
     * 
     * [source, xml]
     * ----
     * <host>
     *   <os>
     *     <custom_kernel_cmdline>vfio_iommu_type1.allow_unsafe_interrupts=1</custom_kernel_cmdline>
     *   </os>
     * </host>
     * ----
     */
    UpdateRequest update();
    
    /**
     * Upgrade VDSM and selected software on the host.
     */
    public interface UpgradeRequest extends Request<UpgradeRequest, UpgradeResponse> {
        /**
         * Indicates if the upgrade should be performed asynchronously.
         */
        UpgradeRequest async(Boolean async);
    }
    
    /**
     * Upgrade VDSM and selected software on the host.
     */
    public interface UpgradeResponse extends Response {
    }
    
    /**
     * Upgrade VDSM and selected software on the host.
     */
    UpgradeRequest upgrade();
    
    /**
     * Check if there are upgrades available for the host. If there are upgrades
     * available an icon will be displayed next to host status icon in the webadmin.
     * Audit log messages are also added to indicate the availability of upgrades.
     * The upgrade can be started from the webadmin or by using the
     * <<services/host/methods/upgrade, upgrade>> host action.
     */
    public interface UpgradeCheckRequest extends Request<UpgradeCheckRequest, UpgradeCheckResponse> {
    }
    
    /**
     * Check if there are upgrades available for the host. If there are upgrades
     * available an icon will be displayed next to host status icon in the webadmin.
     * Audit log messages are also added to indicate the availability of upgrades.
     * The upgrade can be started from the webadmin or by using the
     * <<services/host/methods/upgrade, upgrade>> host action.
     */
    public interface UpgradeCheckResponse extends Response {
    }
    
    /**
     * Check if there are upgrades available for the host. If there are upgrades
     * available an icon will be displayed next to host status icon in the webadmin.
     * Audit log messages are also added to indicate the availability of upgrades.
     * The upgrade can be started from the webadmin or by using the
     * <<services/host/methods/upgrade, upgrade>> host action.
     */
    UpgradeCheckRequest upgradeCheck();
    
    /**
     * List of scheduling labels assigned to this host.
     */
    AssignedAffinityLabelsService affinityLabelsService();
    /**
     * Reference to the host devices service.
     * Use this service to view the devices of the host object.
     */
    HostDevicesService devicesService();
    /**
     * Reference to the fence agents service.
     * Use this service to manage fence and power management agents on the host object.
     */
    FenceAgentsService fenceAgentsService();
    /**
     * Reference to the host hooks service.
     * Use this service to view the hooks available in the host object.
     */
    HostHooksService hooksService();
    /**
     * Reference to the service that can show the applicable errata available on the host.
     * This information is taken from Katello.
     */
    KatelloErrataService katelloErrataService();
    /**
     * Reference to the network attachments service. You can use this service to attach
     * Logical networks to host interfaces.
     */
    NetworkAttachmentsService networkAttachmentsService();
    /**
     * Reference to the service that manages the network interface devices on the host.
     */
    HostNicsService nicsService();
    /**
     * Reference to the service that manage NUMA nodes for the host.
     */
    HostNumaNodesService numaNodesService();
    /**
     * Reference to the host permission service.
     * Use this service to manage permissions on the host object.
     */
    AssignedPermissionsService permissionsService();
    /**
     * Reference to the service that manage hosts storage.
     */
    HostStorageService storageService();
    /**
     * Reference to storage connection extensions.
     */
    StorageServerConnectionExtensionsService storageConnectionExtensionsService();
    /**
     * Reference to the host tags service.
     * Use this service to manage tags on the host object.
     */
    AssignedTagsService tagsService();
    /**
     * Reference to unmanaged networks.
     */
    UnmanagedNetworksService unmanagedNetworksService();
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

