package org.ovirt.engine.core.bll.storage;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.ovirt.engine.core.bll.LockIdNameAttribute;
import org.ovirt.engine.core.bll.LockMessagesMatchUtil;
import org.ovirt.engine.core.bll.NonTransactiveCommandAttribute;
import org.ovirt.engine.core.bll.context.CompensationContext;
import org.ovirt.engine.core.common.action.StorageServerConnectionParametersBase;
import org.ovirt.engine.core.common.businessentities.LUNs;
import org.ovirt.engine.core.common.businessentities.StorageDomain;
import org.ovirt.engine.core.common.businessentities.StorageDomainSharedStatus;
import org.ovirt.engine.core.common.businessentities.StorageDomainStatus;
import org.ovirt.engine.core.common.businessentities.StorageDomainType;
import org.ovirt.engine.core.common.businessentities.StoragePoolIsoMap;
import org.ovirt.engine.core.common.businessentities.StorageServerConnections;
import org.ovirt.engine.core.common.businessentities.StorageType;
import org.ovirt.engine.core.common.businessentities.VM;
import org.ovirt.engine.core.common.businessentities.VMStatus;
import org.ovirt.engine.core.common.errors.VdcBllMessages;
import org.ovirt.engine.core.common.locks.LockingGroup;
import org.ovirt.engine.core.common.utils.Pair;
import org.ovirt.engine.core.common.validation.NfsMountPointConstraint;
import org.ovirt.engine.core.common.vdscommands.StorageServerConnectionManagementVDSParameters;
import org.ovirt.engine.core.common.vdscommands.GetStorageDomainStatsVDSCommandParameters;
import org.ovirt.engine.core.common.vdscommands.VDSCommandType;
import org.ovirt.engine.core.common.vdscommands.VDSReturnValue;
import org.ovirt.engine.core.compat.Guid;
import org.ovirt.engine.core.dao.StorageDomainDynamicDAO;
import org.ovirt.engine.core.dao.StoragePoolIsoMapDAO;
import org.ovirt.engine.core.utils.transaction.TransactionMethod;
import org.ovirt.engine.core.utils.transaction.TransactionSupport;

@NonTransactiveCommandAttribute(forceCompensation = true)
@LockIdNameAttribute
public class UpdateStorageServerConnectionCommand<T extends StorageServerConnectionParametersBase> extends ConnectStorageToVdsCommand<T> {
    private List<StorageDomain> domains = new ArrayList<>();
    private List<LUNs> luns = new ArrayList<>();

    public UpdateStorageServerConnectionCommand(T parameters) {
        super(parameters);
    }

    @Override
    protected boolean canDoAction() {
        StorageServerConnections newConnectionDetails = getConnection();
        StorageType storageType = newConnectionDetails.getstorage_type();
        if ((!storageType.isFileDomain() && !storageType.equals(StorageType.ISCSI))
                || storageType.equals(StorageType.GLUSTERFS)) {
            return failCanDoAction(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_UNSUPPORTED_ACTION_FOR_STORAGE_TYPE);
        }

        // Check if the NFS path has a valid format
        if (newConnectionDetails.getstorage_type() == StorageType.NFS
                && !new NfsMountPointConstraint().isValid(newConnectionDetails.getconnection(), null)) {
            return failCanDoAction(VdcBllMessages.VALIDATION_STORAGE_CONNECTION_INVALID);
        }

        if (newConnectionDetails.getstorage_type() == StorageType.POSIXFS
                && (StringUtils.isEmpty(newConnectionDetails.getVfsType()))) {
            return failCanDoAction(VdcBllMessages.VALIDATION_STORAGE_CONNECTION_EMPTY_VFSTYPE);
        }

        if (checkIsConnectionFieldEmpty(newConnectionDetails)) {
            return false;
        }

        // Check if connection exists by id, otherwise there's nothing to update
        String connectionId = newConnectionDetails.getid();

        StorageServerConnections oldConnection = getStorageConnDao().get(connectionId);

        if (oldConnection == null) {
            return failCanDoAction(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_NOT_EXIST);
        }

        if (!newConnectionDetails.getstorage_type().equals(oldConnection.getstorage_type())) {
            return failCanDoAction(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_UNSUPPORTED_CHANGE_STORAGE_TYPE);
        }

        if (isConnWithSameDetailsExists(newConnectionDetails)) {
            return failCanDoAction(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_ALREADY_EXISTS);
        }

        if (doDomainsUseConnection(newConnectionDetails) || doLunsUseConnection()) {
            if (storageType.isFileDomain() && domains.size() > 1) {
                String domainNames = createDomainNamesList(domains);
                addCanDoActionMessage(String.format("$domainNames %1$s", domainNames));
                return failCanDoAction(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_BELONGS_TO_SEVERAL_STORAGE_DOMAINS);
            }
            // Check that the storage domain is in proper state to be edited
            if (!isConnectionEditable(newConnectionDetails)) {
                return false;
            }
        }
        return super.canDoAction();
    }

    protected String createDomainNamesList(List<StorageDomain> domains) {
        // Build domain names list to display in the error
        StringBuilder domainNames = new StringBuilder();
        for (StorageDomain domain : domains) {
            domainNames.append(domain.getStorageName());
            domainNames.append(",");
        }
        // Remove the last comma after the last domain
        domainNames.deleteCharAt(domainNames.length() - 1);
        return domainNames.toString();
    }

    protected List<LUNs> getLuns() {
        if (luns.isEmpty()) {
            luns = getLunDao().getAllForStorageServerConnection(getConnection().getid());
        }
        return luns;
    }

    protected boolean isConnectionEditable(StorageServerConnections connection) {
        if (connection.getstorage_type().isFileDomain()) {
            boolean isConnectionEditable = isDomainInEditState(domains.get(0));
            if (!isConnectionEditable) {
                addCanDoActionMessage(String.format("$domainNames %1$s", domains.get(0).getStorageName()));
                addCanDoActionMessage(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_UNSUPPORTED_ACTION_FOR_DOMAINS_STATUS);
            }
            return isConnectionEditable;
        }
        if (!getLuns().isEmpty()) {
            List<String> problematicVMNames = new ArrayList<>();
            List<String> problematicDomainNames = new ArrayList<>();
            for (LUNs lun : getLuns()) {
                Guid diskId = lun.getDiskId();
                if (diskId != null) {
                    Map<Boolean, List<VM>> vmsMap = getVmDAO().getForDisk(diskId);
                    List<VM> pluggedVms = vmsMap.get(Boolean.TRUE);
                    if (pluggedVms != null && !pluggedVms.isEmpty()) {
                        for (VM vm : pluggedVms) {
                            if (!vm.getStatus().equals(VMStatus.Down)) {
                                problematicVMNames.add(vm.getName());
                            }
                        }
                    }
                }
                Guid storageDomainId = lun.getStorageDomainId();
                if (storageDomainId != null) {
                    StorageDomain domain = getStorageDomainDao().get(storageDomainId);
                    if (!domain.getStorageDomainSharedStatus().equals(StorageDomainSharedStatus.Unattached)) {
                        for (StoragePoolIsoMap map : getStoragePoolIsoMap(domain)) {
                            if (!map.getStatus().equals(StorageDomainStatus.Maintenance)) {
                                String domainName = domain.getStorageName();
                                problematicDomainNames.add(domainName);
                            } else {
                                domains.add(domain);
                            }
                        }
                    }
                    else { //unattached domain, edit allowed
                        domains.add(domain);
                    }

                }
            }
            if (!problematicVMNames.isEmpty()) {
                if (problematicDomainNames.isEmpty()) {
                    addCanDoActionMessage(String.format("$vmNames %1$s", prepareEntityNamesForMessage(problematicVMNames)));
                    addCanDoActionMessage(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_UNSUPPORTED_ACTION_FOR_RUNNING_VMS);
                } else {
                    addCanDoActionMessage(String.format("$vmNames %1$s", prepareEntityNamesForMessage(problematicVMNames)));
                    addCanDoActionMessage(String.format("$domainNames %1$s", prepareEntityNamesForMessage(problematicDomainNames)));
                    addCanDoActionMessage(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_UNSUPPORTED_ACTION_FOR_RUNNING_VMS_AND_DOMAINS_STATUS);
                }
                return false;
            }

            if (!problematicDomainNames.isEmpty()) {
                addCanDoActionMessage(String.format("$domainNames %1$s", prepareEntityNamesForMessage(problematicDomainNames)));
                addCanDoActionMessage(VdcBllMessages.ACTION_TYPE_FAILED_STORAGE_CONNECTION_UNSUPPORTED_ACTION_FOR_DOMAINS_STATUS);
                return false;
            }
        }
        return true;
    }

    private String prepareEntityNamesForMessage(List<String> entityNames) {
        return StringUtils.join(entityNames, ",");
    }

    protected boolean isDomainInEditState(StorageDomain storageDomain) {
        boolean isEditable = (storageDomain.getStorageDomainType() == StorageDomainType.Data || storageDomain.getStorageDomainType() == StorageDomainType.Master)
                && (storageDomain.getStatus() == StorageDomainStatus.Maintenance || storageDomain.getStorageDomainSharedStatus() == StorageDomainSharedStatus.Unattached);
        return isEditable;
    }

    @Override
    protected void executeCommand() {
        boolean isDomainUpdateRequired = !Guid.isNullOrEmpty(getVdsId()) && doDomainsUseConnection(getConnection());
        List<StorageDomain> updatedDomains = new ArrayList<>();
        boolean hasConnectStorageSucceeded = false;
        if (isDomainUpdateRequired) {
            hasConnectStorageSucceeded = connectToStorage();
            VDSReturnValue returnValueUpdatedStorageDomain = null;
            if (hasConnectStorageSucceeded) {
                changeStorageDomainStatusInTransaction(StorageDomainStatus.Locked);
                for (StorageDomain domain : domains) {
                    // update info such as free space - because we switched to a different server
                    returnValueUpdatedStorageDomain = getStatsForDomain(domain);
                    if (returnValueUpdatedStorageDomain.getSucceeded()) {
                        StorageDomain updatedStorageDomain =
                                (StorageDomain) returnValueUpdatedStorageDomain.getReturnValue();
                        updatedDomains.add(updatedStorageDomain);
                    }
                }
                if (!updatedDomains.isEmpty()) {
                   updateStorageDomain(updatedDomains);
                }
            }
        }
        getStorageConnDao().update(getParameters().getStorageServerConnection());
        if (isDomainUpdateRequired) {
            for (StorageDomain domain : domains) {
                for (StoragePoolIsoMap map : getStoragePoolIsoMap(domain)) {
                    restoreStateAfterUpdate(map);
                }
            }
            if (hasConnectStorageSucceeded) {
                disconnectFromStorage();
            }
        }
        setSucceeded(true);
    }

    protected void restoreStateAfterUpdate(StoragePoolIsoMap map) {
        updateStatus(map, StorageDomainStatus.Maintenance);
    }

    protected boolean doDomainsUseConnection(StorageServerConnections connection) {
        if (domains == null || domains.isEmpty()) {
            domains = getStorageDomainsByConnId(connection.getid());
        }
        return domains != null && !domains.isEmpty();
    }

    protected boolean doLunsUseConnection() {
        return !getLuns().isEmpty();
    }

    protected Collection<StoragePoolIsoMap> getStoragePoolIsoMap(StorageDomain storageDomain) {
        return getStoragePoolIsoMapDao().getAllForStorage(storageDomain.getId());
    }

    protected void changeStorageDomainStatusInTransaction(final StorageDomainStatus status) {
        executeInNewTransaction(new TransactionMethod<Void>() {
            @Override
            public Void runInTransaction() {
                CompensationContext context = getCompensationContext();
                for (StorageDomain domain : domains) {
                    for (StoragePoolIsoMap map : getStoragePoolIsoMap(domain)) {
                        context.snapshotEntityStatus(map);
                        updateStatus(map, status);
                    }
                }
                getCompensationContext().stateChanged();
                return null;
            }
        });
    }

    protected void updateStorageDomain(final List<StorageDomain> storageDomainsToUpdate) {
        executeInNewTransaction(new TransactionMethod<Void>() {
            @Override
            public Void runInTransaction() {
                for (StorageDomain domainToUpdate : storageDomainsToUpdate) {
                        CompensationContext context = getCompensationContext();
                        context.snapshotEntity(domainToUpdate.getStorageDynamicData());
                        getStorageDomainDynamicDao().update(domainToUpdate.getStorageDynamicData());
                        getCompensationContext().stateChanged();
                }
                return null;
            }
        });
    }

    protected void updateStatus(StoragePoolIsoMap map, StorageDomainStatus status) {
        log.infoFormat("Setting domain %s to status $s", map.getId(), status.name());
        map.setStatus(status);
        getStoragePoolIsoMapDao().updateStatus(map.getId(), map.getStatus());
    }

    protected void executeInNewTransaction(TransactionMethod<?> method) {
        TransactionSupport.executeInNewTransaction(method);
    }

    protected boolean connectToStorage() {
         Pair<Boolean, Integer> result = connectHostToStorage();
         return result.getFirst();
    }

    protected void disconnectFromStorage() {
        StorageServerConnectionManagementVDSParameters connectionParametersForVdsm =
                createParametersForVdsm(getParameters().getVdsId(),
                        Guid.Empty,
                        getConnection().getstorage_type(),
                        getConnection());
        boolean isDisconnectSucceeded =
                runVdsCommand(VDSCommandType.DisconnectStorageServer, connectionParametersForVdsm).getSucceeded();
        if (!isDisconnectSucceeded) {
            log.warn("Failed to disconnect storage connection " + getConnection());
        }
    }

    protected VDSReturnValue getStatsForDomain(StorageDomain storageDomain) {
        return runVdsCommand(VDSCommandType.GetStorageDomainStats,
                new GetStorageDomainStatsVDSCommandParameters(getVds().getId(), storageDomain.getId()));
    }

    protected StorageServerConnectionManagementVDSParameters createParametersForVdsm(Guid vdsmId,
                                                                               Guid storagePoolId,
                                                                               StorageType storageType,
                                                                               StorageServerConnections storageServerConnection) {
        StorageServerConnectionManagementVDSParameters newConnectionParametersForVdsm =
                new StorageServerConnectionManagementVDSParameters(vdsmId, storagePoolId, storageType,
                        new ArrayList<StorageServerConnections>(Arrays.asList(storageServerConnection)));
        return newConnectionParametersForVdsm;
    }

    protected StorageDomainDynamicDAO getStorageDomainDynamicDao() {
        return getDbFacade().getStorageDomainDynamicDao();
    }

    protected StoragePoolIsoMapDAO getStoragePoolIsoMapDao() {
        return getDbFacade().getStoragePoolIsoMapDao();
    }

    @Override
    protected Map<String, Pair<String, String>> getExclusiveLocks() {
        Map<String, Pair<String, String>> locks = new HashMap<String, Pair<String, String>>();
        domains = getStorageDomainsByConnId(getConnection().getid());
        if (!domains.isEmpty()) {
            for (StorageDomain domain : domains) {
                locks.put(domain.getId().toString(),
                        LockMessagesMatchUtil.makeLockingPair(LockingGroup.STORAGE,
                                VdcBllMessages.ACTION_TYPE_FAILED_OBJECT_LOCKED));
            }
        }
        if (getConnection().getstorage_type().isFileDomain()) {
           // lock the path to avoid at the same time if some other user tries to
           // add new storage connection to same path or edit another storage server connection to point to same path
           locks.put(getConnection().getconnection(),
                    LockMessagesMatchUtil.makeLockingPair(LockingGroup.STORAGE_CONNECTION,
                            VdcBllMessages.ACTION_TYPE_FAILED_OBJECT_LOCKED));
        }
        else {
          // for block domains, locking the target details
          locks.put(getConnection().getconnection() + ";" + getConnection().getiqn() + ";" + getConnection().getport() + ";" + getConnection().getuser_name(),
          LockMessagesMatchUtil.makeLockingPair(LockingGroup.STORAGE_CONNECTION,
                    VdcBllMessages.ACTION_TYPE_FAILED_OBJECT_LOCKED));

          //lock lun disks and domains, not VMs , no need to load from db.
          if(getLuns()!=null) {
              for(LUNs lun : getLuns()) {
                Guid diskId = lun.getDiskId();
                Guid storageDomainId = lun.getStorageDomainId();
                if(diskId != null) {
                       locks.put(diskId.toString(),LockMessagesMatchUtil.makeLockingPair(LockingGroup.DISK,
                       VdcBllMessages.ACTION_TYPE_FAILED_OBJECT_LOCKED));
                }
                if(storageDomainId != null) {
                       locks.put(storageDomainId.toString(),LockMessagesMatchUtil.makeLockingPair(LockingGroup.STORAGE,
                       VdcBllMessages.ACTION_TYPE_FAILED_OBJECT_LOCKED));
                }

              }
          }

        }

        // lock connection's id to avoid editing or removing this connection at the same time
        // by another user
        locks.put(getConnection().getid(),
                LockMessagesMatchUtil.makeLockingPair(LockingGroup.STORAGE_CONNECTION,
                        VdcBllMessages.ACTION_TYPE_FAILED_OBJECT_LOCKED));
        return locks;
    }

    @Override
    protected void setActionMessageParameters() {
        addCanDoActionMessage(VdcBllMessages.VAR__ACTION__UPDATE);
        addCanDoActionMessage(VdcBllMessages.VAR__TYPE__STORAGE__CONNECTION);
    }
}
