package org.ovirt.engine.api.restapi.types;

import static org.ovirt.engine.api.restapi.utils.VersionUtils.greaterOrEqual;

import java.util.LinkedHashMap;

import org.ovirt.engine.api.model.CPU;
import org.ovirt.engine.api.model.Cluster;
import org.ovirt.engine.api.model.DataCenter;
import org.ovirt.engine.api.model.ErrorHandling;
import org.ovirt.engine.api.model.MemoryOverCommit;
import org.ovirt.engine.api.model.MemoryPolicy;
import org.ovirt.engine.api.model.MigrateOnError;
import org.ovirt.engine.api.model.SchedulingPolicy;
import org.ovirt.engine.api.model.SchedulingPolicyThresholds;
import org.ovirt.engine.api.model.SchedulingPolicyType;
import org.ovirt.engine.api.model.TransparentHugePages;
import org.ovirt.engine.api.model.Version;
import org.ovirt.engine.api.restapi.utils.GuidUtils;
import org.ovirt.engine.core.common.businessentities.MigrateOnErrorOptions;
import org.ovirt.engine.core.common.businessentities.StoragePool;
import org.ovirt.engine.core.common.businessentities.VDSGroup;

public class ClusterMapper {
    private static final String CPU_OVER_COMMIT_DURATION_MINUTES = "CpuOverCommitDurationMinutes";
    private static final String HIGH_UTILIZATION = "HighUtilization";
    private static final String LOW_UTILIZATION = "LowUtilization";
    static final org.ovirt.engine.core.compat.Version min_thp_version = new org.ovirt.engine.core.compat.Version(3,0);

    @Mapping(from = Cluster.class, to = VDSGroup.class)
    public static VDSGroup map(Cluster model, VDSGroup template) {
        VDSGroup entity = template != null ? template : new VDSGroup();
        if (model.isSetId()) {
            entity.setId(GuidUtils.asGuid(model.getId()));
        }
        if (model.isSetName()) {
            entity.setName(model.getName());
        }
        if (model.isSetDescription()) {
            entity.setdescription(model.getDescription());
        }
        if (model.isSetComment()) {
            entity.setComment(model.getComment());
        }
        if (model.isSetCpu() && model.getCpu().isSetId()) {
            entity.setcpu_name(model.getCpu().getId());
        }
        if (model.isSetDataCenter() && model.getDataCenter().isSetId()) {
            entity.setStoragePoolId(GuidUtils.asGuid(model.getDataCenter().getId()));
        }
        if (model.isSetVersion() && model.getVersion().getMajor()!=null && model.getVersion().getMinor()!=null) {
            entity.setcompatibility_version(new org.ovirt.engine.core.compat.Version(model.getVersion().getMajor(),
                                                                                model.getVersion().getMinor()));
        }
        if (model.isSetMemoryPolicy()) {
            entity = map(model.getMemoryPolicy(), entity);
        } else if (model.isSetVersion() && greaterOrEqual(model.getVersion() , min_thp_version)){
            entity.setTransparentHugepages(true);
        }
        if (model.isSetSchedulingPolicy()) {
            entity = map(model.getSchedulingPolicy(), entity);
        }
        if (model.isSetErrorHandling() && model.getErrorHandling().isSetOnError()) {
            entity.setMigrateOnError(map(model.getErrorHandling().getOnError(), null));
        }
        if(model.isSetVirtService()) {
            entity.setVirtService(model.isVirtService());
        }
        if(model.isSetGlusterService()) {
            entity.setGlusterService(model.isGlusterService());
        }
        if (model.isSetThreadsAsCores()) {
            entity.setCountThreadsAsCores(model.isThreadsAsCores());
        }
        if (model.isSetTunnelMigration()) {
            entity.setTunnelMigration(model.isTunnelMigration());
        }
        if (model.isSetTrustedService()){
            entity.setTrustedService(model.isTrustedService());
        }
        if (model.isSetBallooningEnabled()) {
            entity.setEnableBallooning(model.isBallooningEnabled());
        }
        return entity;
    }

    @Mapping(from = VDSGroup.class, to = Cluster.class)
    public static Cluster map(VDSGroup entity, Cluster template) {
        Cluster model = template != null ? template : new Cluster();
        model.setId(entity.getId().toString());
        model.setName(entity.getName());
        model.setDescription(entity.getdescription());
        model.setComment(entity.getComment());
        if (entity.getcpu_name() != null) {
            CPU cpu = new CPU();
            cpu.setId(entity.getcpu_name());
            model.setCpu(cpu);
        }
        if (entity.getStoragePoolId() != null) {
            DataCenter dataCenter = new DataCenter();
            dataCenter.setId(entity.getStoragePoolId().toString());
            model.setDataCenter(dataCenter);
        }
        if (entity.getcompatibility_version() != null) {
            model.setVersion(new Version());
            model.getVersion().setMajor(entity.getcompatibility_version().getMajor());
            model.getVersion().setMinor(entity.getcompatibility_version().getMinor());
        }
        model.setMemoryPolicy(map(entity, (MemoryPolicy)null));
        model.setSchedulingPolicy(map(entity, (SchedulingPolicy)null));
        model.setErrorHandling(map(entity.getMigrateOnError(), (ErrorHandling)null));
        model.setVirtService(entity.supportsVirtService());
        model.setGlusterService(entity.supportsGlusterService());
        model.setThreadsAsCores(entity.getCountThreadsAsCores());
        model.setTunnelMigration(entity.isTunnelMigration());
        model.setTrustedService(entity.supportsTrustedService());
        model.setBallooningEnabled(entity.isEnableBallooning());

        return model;
    }

    @Mapping(from = MemoryPolicy.class, to = VDSGroup.class)
    public static VDSGroup map(MemoryPolicy model, VDSGroup template) {
        VDSGroup entity = template != null ? template : new VDSGroup();
        if (model.isSetOverCommit() && model.getOverCommit().getPercent()!=null) {
            entity.setmax_vds_memory_over_commit(model.getOverCommit().getPercent());
        }
        if (model.isSetTransparentHugepages() &&
            model.getTransparentHugepages().isSetEnabled()) {
            entity.setTransparentHugepages(model.getTransparentHugepages().isEnabled());
        } else if (template != null && greaterOrEqual(template.getcompatibility_version(), min_thp_version)){
            entity.setTransparentHugepages(true);
        }
        return entity;
    }

    @Mapping(from = VDSGroup.class, to = MemoryPolicy.class)
    public static MemoryPolicy map(VDSGroup entity, MemoryPolicy template) {
        MemoryPolicy model = template != null ? template : new MemoryPolicy();
        model.setOverCommit(new MemoryOverCommit());
        model.getOverCommit().setPercent(entity.getmax_vds_memory_over_commit());
        model.setTransparentHugepages(new TransparentHugePages());
        model.getTransparentHugepages().setEnabled(entity.getTransparentHugepages());
        return model;
    }

    @Mapping(from = SchedulingPolicy.class, to = VDSGroup.class)
    public static VDSGroup map(SchedulingPolicy model, VDSGroup template) {
        VDSGroup entity = template != null ? template : new VDSGroup();
        if (model.isSetPolicy()) {
            SchedulingPolicyType policyType = SchedulingPolicyType.fromValue(model.getPolicy());
            if (policyType != null) {
                entity.setClusterPolicyId(null);
                entity.setClusterPolicyName(policyType.name().toLowerCase());
            }
        }
        if (model.isSetThresholds()) {
            SchedulingPolicyThresholds thresholds = model.getThresholds();
            if (entity.getClusterPolicyProperties() == null) {
                entity.setClusterPolicyProperties(new LinkedHashMap<String, String>());
            }
            if (thresholds.getLow() != null) {
                entity.getClusterPolicyProperties().put(LOW_UTILIZATION, thresholds.getLow().toString());
            }
            if (thresholds.getHigh() != null) {
                entity.getClusterPolicyProperties().put(HIGH_UTILIZATION, thresholds.getHigh().toString());
            }
            if (thresholds.getDuration() != null) {
                int round = Math.round(thresholds.getDuration() / 60.0f);
                entity.getClusterPolicyProperties().put(CPU_OVER_COMMIT_DURATION_MINUTES, Integer.toString(round));
            }
        }
        return entity;
    }

    @Mapping(from = VDSGroup.class, to = SchedulingPolicy.class)
    public static SchedulingPolicy map(VDSGroup entity, SchedulingPolicy template) {
        SchedulingPolicy model = template != null ? template : new SchedulingPolicy();
        if (entity.getClusterPolicyName() != null) {
            model.setPolicy(entity.getClusterPolicyName().toLowerCase());
            if (entity.getClusterPolicyProperties() != null && !entity.getClusterPolicyProperties().isEmpty()) {
                model.setThresholds(new SchedulingPolicyThresholds());
                String lowUtilization = entity.getClusterPolicyProperties().get(LOW_UTILIZATION);
                String highUtilization = entity.getClusterPolicyProperties().get(HIGH_UTILIZATION);
                String cpuOverCommitDurationMinutes =
                        entity.getClusterPolicyProperties().get(CPU_OVER_COMMIT_DURATION_MINUTES);
                if (lowUtilization != null) {
                    model.getThresholds().setLow(Integer.parseInt(lowUtilization));
                }
                if (highUtilization != null) {
                    model.getThresholds().setHigh(Integer.parseInt(highUtilization));
                }
                if (cpuOverCommitDurationMinutes != null) {
                    int duration = Integer.parseInt(cpuOverCommitDurationMinutes) * 60;
                    model.getThresholds().setDuration(duration);
                }
            }
        }
        return model;
    }

    @Mapping(from = StoragePool.class, to = VDSGroup.class)
    public static VDSGroup map(StoragePool pool, VDSGroup template) {
        VDSGroup entity = template != null ? template : new VDSGroup();
        if (pool.getcompatibility_version() != null) {
            entity.setcompatibility_version(pool.getcompatibility_version());
        }
        return entity;
    }

    @Mapping(from = MigrateOnError.class, to = MigrateOnErrorOptions.class)
    public static MigrateOnErrorOptions map(MigrateOnError model, MigrateOnErrorOptions template) {
        if (model==null) {
            return null;
        }
        switch (model) {
        case MIGRATE:
            return MigrateOnErrorOptions.YES;
        case DO_NOT_MIGRATE:
            return MigrateOnErrorOptions.NO;
        case MIGRATE_HIGHLY_AVAILABLE:
            return MigrateOnErrorOptions.HA_ONLY;
        default:
            assert(false) : "unknown migrate-on-error value: " + model.toString();
            return null;
        }
    }

    @Mapping(from = MigrateOnErrorOptions.class, to = MigrateOnError.class)
    public static MigrateOnError map(MigrateOnErrorOptions model, MigrateOnError template) {
        if (model==null) {
            return null;
        }
        switch (model) {
        case YES:
            return MigrateOnError.MIGRATE;
        case NO:
            return MigrateOnError.DO_NOT_MIGRATE;
        case HA_ONLY:
            return MigrateOnError.MIGRATE_HIGHLY_AVAILABLE;
        default:
            assert(false) : "unknown migrate-on-error value: " + model.toString();
            return null;
        }
    }

    @Mapping(from = String.class, to = MigrateOnErrorOptions.class)
    private static MigrateOnErrorOptions map(String migrateOnError, MigrateOnErrorOptions template) {
        try {
            MigrateOnError value = MigrateOnError.fromValue(migrateOnError);
            return map(value, template);
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    @Mapping(from = MigrateOnErrorOptions.class, to = ErrorHandling.class)
    private static ErrorHandling map(MigrateOnErrorOptions migrateOnError, ErrorHandling template) {
        MigrateOnError value = map(migrateOnError, (MigrateOnError)null);
        if (value==null) {
            return null;
        } else {
            template = template==null ? new ErrorHandling() : template;
            template.setOnError(value.value());
            return template;
        }
    }

}
